/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of MapAnalyser.h
*/
#include "CommonIncludes.h"

#include <algorithm>
#include <cmath>
#include <deque>
#include <set>

#include "BaseLocation.h"
#include "BaseManager.h"
#include "BuildingManager.h"
#include "Clustering.h"
#include "Distances.h"
#include "MapAnalyser.h"
#include "MapGraph.h"
#include "MathConstants.h"
#include "Point2D.h"
#include "Tuples.h"

#pragma warning( push )
#pragma warning( disable:4018 )
#pragma warning( disable:4244 )

using namespace BWAPI;
using namespace std;

/*
	MAP ANALYSER
*/

const WalkPosition MapAnalyser::NULL_TILE(-1, -1);

MapAnalyser::MapAnalyser()
{
	mapWidth = Broodwar->mapWidth() * 4;
	mapHeight = Broodwar->mapHeight() * 4;
	numWalkTiles = mapWidth*mapHeight;

	walkableAreas = vector<MapConnectedArea*>();
	obstacles = vector<MapConnectedArea*>();
	chokepoints = vector<MapChokePoint*>();
	mapRegions = vector<MapRegion*>();
	baseLocations = vector<BaseLocation*>();

	graph;

	walkable = new bool[numWalkTiles];

	chokeTiles = new bool[numWalkTiles];
	ArrayUtils::initializeBoolArray(chokeTiles, numWalkTiles, false);

	clearances = new float[numWalkTiles];
	ArrayUtils::initializeFloatArray(clearances, numWalkTiles, -1.0f);

	connectivities = new short[numWalkTiles];
	ArrayUtils::initializeShortArray(connectivities, numWalkTiles, 0);

	nearestObstacleTiles = new WalkPosition[numWalkTiles];
	// no need to intialize, will always assign some value before reading values from this array

	regionNumbers = new short[numWalkTiles];
	ArrayUtils::initializeShortArray(regionNumbers, numWalkTiles, 0);
}

MapAnalyser::~MapAnalyser()
{
	for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
	{
		delete (*it);
	}

	for(auto it = mapRegions.begin(); it != mapRegions.end(); ++it)
	{
		delete (*it);
	}

	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		delete (*it);
	}

	if(walkable != nullptr)
		delete[] walkable;

	if(chokeTiles != nullptr)
		delete[] chokeTiles;

	if(clearances != nullptr)
		delete[] clearances;

	if(connectivities != nullptr)
		delete[] connectivities;

	if(nearestObstacleTiles != nullptr)
		delete[] nearestObstacleTiles;

	if(regionNumbers != nullptr)
		delete[] regionNumbers;
}

const bool MapAnalyser::tileExists(const int x, const int y) const
{
	return (
				x	>=		0				&& 
				y	>=		0				&& 
				x	<		mapWidth		&& 
				y	<		mapHeight
			);
}

const bool MapAnalyser::tileExists(const TilePosition& position) const
{
	return (ArrayUtils::boundsCheck(position.x, Broodwar->mapWidth()) && 
			ArrayUtils::boundsCheck(position.y, Broodwar->mapHeight())	);
}

const bool MapAnalyser::tileExists(const WalkPosition& position) const
{
	return tileExists(position.x, position.y);
}

// Analyses the current map. Algorithm used: http://legionbot.blogspot.nl/2012/09/visualizing-map-information-what-would.html
void MapAnalyser::analyseCurrentMap()
{
	LOG_MESSAGE(StringBuilder() << "MAASCRAFT playing on map: " << Broodwar->mapFileName())
	LOG_MESSAGE(StringBuilder()<< "Map Hash = " << Broodwar->mapHash())

	computeWalkability();
	computeConnectivity();
	computeClearance();
	computeRegions();
	connectChokesToRegions();
	pruneRegions();
	updateRegionNumbers();
	findBaseLocations();
	identifyStartLocations();
	graph.initialize(mapRegions, chokepoints, baseLocations);

	cleanup();
}

/*
	Private methods
*/

void MapAnalyser::computeWalkability()
{
	for(size_t y = 0; y < mapHeight; ++y)
	{
		for(size_t x = 0; x < mapWidth; ++x)
		{
			ArrayUtils::setMatrixValue(walkable, y, x, mapWidth, Broodwar->isWalkable(x, y));
		}
	}

	// on most maps, BWAPI's isWalkable() suffices. So far only the Fortress map gives problems.
	// Fortress has a number of static impassable doors (http://code.google.com/p/bwapi/wiki/UnitTypes#Doors)
	// BWAPI's functions do not seem to accurately give the positions of these doors, so hardcoding the locations
	// they occupy as being unwalkable instead.
	//
	// Note: these doors ARE passable using a StarCraft glitch (http://wiki.teamliquid.net/starcraft/Bugs#Mineral_Walk)
	// but during map analysis this bot doesn't care about that.
	if(Broodwar->mapHash() == "83320e505f35c65324e93510ce2eafbaa71c9aa1" || Broodwar->mapFileName() == "(4)Fortress.scx")
	{
		ArrayUtils::setMatrixValue(walkable, 95, 77, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 94, 78, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 94, 79, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 93, 80, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 93, 81, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 92, 82, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 107, 101, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 106, 102, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 106, 103, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 105, 104, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 105, 105, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 104, 106, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 115, 394, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 114, 393, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 114, 392, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 113, 391, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 113, 390, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 113, 389, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 103, 418, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 102, 417, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 102, 416, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 101, 415, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 101, 414, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 101, 413, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 403, 122, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 402, 121, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 402, 120, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 401, 119, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 401, 118, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 400, 117, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 415, 98, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 414, 97, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 414, 96, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 413, 95, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 413, 94, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 412, 93, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 411, 405, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 410, 406, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 410, 407, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 409, 408, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 409, 409, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 408, 410, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 408, 411, mapWidth, false);

		ArrayUtils::setMatrixValue(walkable, 423, 429, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 422, 430, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 422, 431, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 421, 432, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 421, 433, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 420, 434, mapWidth, false);
	}
	else if(Broodwar->mapHash() == "af618ea3ed8a8926ca7b17619eebcb9126f0d8b1" || Broodwar->mapFileName() == "(2)Benzene.scx")		// similar problem on Benzene with some static neutral buildings
	{															// in the case of Benzene, they cannot be passed using a glitch, but they CAN be destroyed
		// first ''row'' of left bottom obstacles
		ArrayUtils::setMatrixValue(walkable, 261, 0, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 261, 1, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 261, 2, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 261, 3, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 261, 4, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 5, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 6, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 7, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 8, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 9, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 10, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 11, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 12, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 13, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 14, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 15, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 16, mapWidth, false);

		// second ''row'' of left bottom obstacles
		ArrayUtils::setMatrixValue(walkable, 262, 0, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 1, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 2, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 3, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 4, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 262, 5, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 6, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 7, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 8, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 263, 9, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 10, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 11, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 12, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 13, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 264, 14, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 265, 15, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 265, 16, mapWidth, false);

		// first ''row'' of right top obstacles
		ArrayUtils::setMatrixValue(walkable, 180, 495, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 496, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 497, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 498, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 499, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 500, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 501, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 502, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 503, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 504, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 505, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 506, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 183, 507, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 183, 508, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 183, 509, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 183, 510, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 183, 511, mapWidth, false);

		// second ''row'' of right top obstacles
		ArrayUtils::setMatrixValue(walkable, 179, 495, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 179, 496, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 179, 497, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 179, 498, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 499, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 500, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 501, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 180, 502, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 503, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 504, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 505, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 181, 506, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 507, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 508, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 509, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 510, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 182, 511, mapWidth, false);
	}
	else if(Broodwar->mapHash() == "1e983eb6bcfa02ef7d75bd572cb59ad3aab49285" || Broodwar->mapFileName() == "(4)Andromeda.scx")
	{
		// lower right
		ArrayUtils::setMatrixValue(walkable, 449, 330, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 331, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 332, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 333, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 334, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 335, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 336, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 337, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 338, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 449, 339, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 340, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 341, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 342, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 343, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 344, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 345, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 346, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 347, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 348, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 349, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 350, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 351, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 352, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 353, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 354, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 355, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 356, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 357, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 358, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 359, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 360, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 361, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 362, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 363, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 364, mapWidth, false);

		// upper left
		ArrayUtils::setMatrixValue(walkable, 70, 147, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 148, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 149, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 150, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 151, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 152, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 153, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 70, 154, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 155, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 156, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 157, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 158, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 159, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 160, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 161, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 69, 162, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 163, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 164, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 165, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 166, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 167, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 168, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 68, 169, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 170, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 171, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 172, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 173, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 174, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 67, 175, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 176, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 177, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 178, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 179, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 180, mapWidth, false);

		// upper right
		ArrayUtils::setMatrixValue(walkable, 66, 330, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 331, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 332, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 333, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 334, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 335, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 66, 336, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 337, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 338, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 339, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 340, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 341, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 342, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 65, 343, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 344, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 345, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 346, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 347, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 348, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 349, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 64, 350, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 351, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 352, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 353, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 354, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 355, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 63, 356, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 62, 357, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 62, 358, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 62, 359, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 62, 360, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 62, 361, mapWidth, false);

		// lower left
		ArrayUtils::setMatrixValue(walkable, 446, 146, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 147, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 148, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 149, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 150, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 151, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 152, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 153, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 154, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 446, 155, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 156, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 157, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 158, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 159, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 160, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 161, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 162, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 163, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 164, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 447, 165, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 166, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 167, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 168, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 169, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 170, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 171, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 172, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 173, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 174, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 175, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 176, mapWidth, false);
		ArrayUtils::setMatrixValue(walkable, 448, 177, mapWidth, false);
	}
}

// Computes connectivity. 0 = not processed yet, negative number = unwalkable, positive = walkable.
// Same number (unless 0) = connected
void MapAnalyser::computeConnectivity()
{
	short nextPos = 1;
	short nextNeg = -1;

	for(size_t y = 0; y < mapHeight; ++y)
	{
		for(size_t x = 0; x < mapWidth; ++x)
		{
			if(getConnectivity(x, y) == Connectivities::UNASSIGNED)
			{
				if(isWalkable(x, y))
				{
					processConnectivity(x, y, nextPos);
					nextPos++;
				}
				else
				{
					processConnectivity(x, y, nextNeg);
					nextNeg--;
				}
			}
		}
	}

	// Right now, each obstacle has a different connectivity number.
	// Now want to assign -1 to all small obstacles, and -2 to all large obstacles
	const size_t MIN_OBSTACLE_SIZE = Options::MapAnalysis::CHOKE_MIN_OBSTACLE_SIZE;

	for(auto it_obstacles = obstacles.begin(); it_obstacles != obstacles.end(); ++it_obstacles)
	{
		MapConnectedArea* obstacle = *it_obstacles;
		vector<WalkPosition> obstacleTiles = obstacle->getTiles();

		if(obstacleTiles.size() < MIN_OBSTACLE_SIZE)
		{
			for(auto it_tile = obstacleTiles.begin(); it_tile != obstacleTiles.end(); ++it_tile)
			{
				setConnectivity((*it_tile).x, (*it_tile).y, Connectivities::SMALL_OBSTACLE);
			}
		}
		else
		{
			for(auto it_tile = obstacleTiles.begin(); it_tile != obstacleTiles.end(); ++it_tile)
			{
				setConnectivity((*it_tile).x, (*it_tile).y, Connectivities::LARGE_OBSTACLE);
			}
		}
	}
}

void MapAnalyser::processConnectivity(int x, int y, short connectivity)
{
	vector<WalkPosition> queue;
	queue.push_back(WalkPosition(x, y));

	bool walkable = isWalkable(x, y);
	MapConnectedArea* area = new MapConnectedArea(!walkable);

	while(queue.size() > 0)
	{
		WalkPosition tile = queue.back();
		queue.pop_back();

		x = tile.x;
		y = tile.y;

		if(getConnectivity(x, y) != Connectivities::UNASSIGNED)		// already processed
			continue;

		if(walkable == isWalkable(x, y))
		{
			setConnectivity(x, y, connectivity);
			area->addTile(tile);

			// since we will hit this part of code exactly once for each tile which is part of an obstacle,
			// we might as well save some work in computeClearance() and already tell this tile what it's shortest
			// distance to obstacle and the obstacle itself are
			if(!walkable)
			{
				setClearance(x, y, 0.0f);
				setNearestObstacleTile(x, y, tile);
			}

			if(tileExists(x,		y - 1))		queue.push_back(WalkPosition(x,		y - 1));
			if(tileExists(x + 1,	y))			queue.push_back(WalkPosition(x + 1,	y));
			if(tileExists(x,		y + 1))		queue.push_back(WalkPosition(x,		y + 1));
			if(tileExists(x - 1,	y))			queue.push_back(WalkPosition(x - 1,	y));
		}
	}

	if(walkable)
		walkableAreas.push_back(area);
	else
		obstacles.push_back(area);
}

// using this 4-tuple as follows; first = new tile, second = from tile, third = distance from first to second, fourth = wave-direction
typedef Tuple_4<WalkPosition, WalkPosition, float, int> Tile_Origin_Dist_Dir;

void MapAnalyser::computeClearance()
{
	// All tiles which are part of obstacles already know their distance to nearest obstacle
	// and have a pointer to obstacle (namely the obstacle they're a part of).
	//
	// All other tiles still have a 0 pointer and distance = -1.0

	// For each obstacle, we will loop through the tiles which are part of that obstacle and
	// for each of those tiles, send a ''pulse'' out to assign shortest distances to walkable tiles
	// The ''pulse'' is blocked by all tiles which already have a shorter (non-negative) shortest distance to some other obstacle

	// Distances are measured in walk-tile units!

	const size_t MIN_OBSTACLE_SIZE = Options::MapAnalysis::CHOKE_MIN_OBSTACLE_SIZE;

	const int	NORTH_WEST = 1,		NORTH = 2,			NORTH_EAST = 3,
				EAST = 4,			SOUTH_EAST = 5,		SOUTH = 6,
				SOUTH_WEST = 7,		WEST = 8;


	// First emit waves from the map boundaries. Testing indicated that doing this before emitting from obstacles is faster than other way around

	// starting moving from west to east
	int boundaryX = (mapWidth + 1) / 2;		// only need to check left half (rounding up in case of uneven size), because we know right half will be closer to right border
	float distBorder = 0.0f;

	int maxY = mapHeight;

	for(int x = 0; x < boundaryX; ++x)
	{
		distBorder += 1.0f;

		for(int y = x; y < maxY; ++y)
		{
			float dist = getClearance(x, y);
			if(dist >= 0.0f && dist < distBorder)
				continue;

			setClearance(x, y, distBorder);
			setNearestObstacleTile(x, y, WalkPosition(0, y));
		}

		maxY--;
	}

	// reversed version of above, for east to west
	boundaryX = (mapWidth - 1) / 2;
	distBorder = 0.0f;

	int minY = 0;
	maxY = mapHeight;

	for(int x = mapWidth - 1; x > boundaryX; x--)
	{
		distBorder += 1.0f;

		for(int y = minY; y < maxY; ++y)
		{
			float dist = getClearance(x, y);
			if(dist >= 0.0f && dist < distBorder)
				continue;

			setClearance(x, y, distBorder);
			setNearestObstacleTile(x, y, WalkPosition(mapWidth - 1, y));
		}

		maxY--;
		minY++;
	}

	// similar thing to move from north to south
	int boundaryY = (mapHeight + 1) / 2;
	distBorder = 0.0f;

	int maxX = mapWidth;

	for(int y = 0; y < boundaryY; ++y)
	{
		distBorder += 1.0f;

		for(int x = y; x < maxX; ++x)
		{
			float dist = getClearance(x, y);
			if(dist >= 0.0f && dist < distBorder)
				continue;

			setClearance(x, y, distBorder);
			setNearestObstacleTile(x, y, WalkPosition(x, 0));
		}

		maxX--;
	}

	// reversed from above, to move from south to north
	boundaryY = (mapHeight - 1) / 2;
	distBorder = 0.0f;

	int minX = 0;
	maxX = mapWidth;

	for(int y = mapHeight - 1; y > boundaryY; y--)
	{
		distBorder += 1.0f;

		for(int x = minX; x < maxX; ++x)
		{
			float dist = getClearance(x, y);
			if(dist >= 0.0f && dist < distBorder)
				continue;

			setClearance(x, y, distBorder);
			setNearestObstacleTile(x, y, WalkPosition(x, mapHeight - 1));
		}

		maxX--;
		minX++;
	}

	// Now use actual obstacles
	for(auto it_obstacles = obstacles.begin(); it_obstacles != obstacles.end(); ++it_obstacles)
	{
		MapConnectedArea* obstacle = *it_obstacles;
		vector<WalkPosition> obstacleTiles = obstacle->getTiles();

		if(obstacleTiles.size() < MIN_OBSTACLE_SIZE)		// obstacle too small, ignore it
			continue;

		for(auto it_tiles = obstacleTiles.begin(); it_tiles != obstacleTiles.end(); ++it_tiles)
		{
			WalkPosition tile = *it_tiles;
			int x = tile.x;
			int y = tile.y;

			// FIFO queue with everything we need to check
			deque<Tile_Origin_Dist_Dir> tiles_to_check;

			if(tileExists(x - 1,	y - 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x - 1,	y - 1),		tile, MathConstants::SQRT_2F,		SOUTH_WEST));
			if(tileExists(x,		y - 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x,		y - 1),		tile, 1.0f,							SOUTH));
			if(tileExists(x + 1,	y - 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x + 1,	y - 1),		tile, MathConstants::SQRT_2F,		SOUTH_EAST));

			if(tileExists(x + 1,	y))	
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x + 1,	y),			tile, 1.0f,							EAST));
			if(tileExists(x + 1,	y + 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x + 1,	y + 1),		tile, MathConstants::SQRT_2F,		NORTH_EAST));

			if(tileExists(x,		y + 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x,		y + 1),		tile, 1.0f,							NORTH));
			if(tileExists(x - 1,	y + 1))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x - 1,	y + 1),		tile, MathConstants::SQRT_2F,		NORTH_WEST));

			if(tileExists(x - 1,	y))
				tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(x - 1,	y),			tile, 1.0f,							WEST));

			while(!tiles_to_check.empty())
			{
				Tile_Origin_Dist_Dir tuple_4 = tiles_to_check.front();
				tiles_to_check.pop_front();

				WalkPosition t = tuple_4.first;
				float tempSmallestDist = getClearance(t.x, t.y);

				WalkPosition origin = tuple_4.second;
				float newDist = tuple_4.third;

				int dir = tuple_4.fourth;

				// if tempSmallestDist < 0, we have reached a non-obstacle tile which we never processed before, so need to process
				// if tempSmallestDist == 0, we have reached an obstacle tile and should not update that tile but should still continue the wave, in case the
				//	obstacle tile is part of only a small obstacle
				// if tempSmallestDist > newDist, we have reached a non-obstacle tile which was already processed before, but should be updated because
				//	the current obstacle is closer
				if(tempSmallestDist > newDist || tempSmallestDist <= 0.0f)
				{
					if(tempSmallestDist != 0.0f)		// don't update values if it's an obstacle
					{
						setClearance(t.x, t.y, newDist);
						setNearestObstacleTile(t.x, t.y, tile);
					}
					else if(!isSmallObstacle(t))	// wave is only allowed to pass through small obstacles
						continue;

					int tx = t.x;
					int ty = t.y;

					// again add 8 surrounding tiles. This time, only send wave in the same direction as we're going (NW, NE, SW and SE will send 3 waves, N, E, W and S only send single waves)
					if((dir==SOUTH_WEST) && tileExists(tx - 1,	ty - 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx - 1,	ty - 1),	t, newDist + MathConstants::SQRT_2F,	SOUTH_WEST));

					if((dir==SOUTH ||
						dir==SOUTH_WEST ||
						dir==SOUTH_EAST) && tileExists(tx,		ty - 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx,		ty - 1),	t, newDist + 1.0f,						SOUTH));

					if((dir==SOUTH_EAST) && tileExists(tx + 1,	ty - 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx + 1,	ty - 1),	t, newDist + MathConstants::SQRT_2F,	SOUTH_EAST));

					if((dir==EAST ||
						dir==SOUTH_EAST ||
						dir==NORTH_EAST) && tileExists(tx + 1,	ty))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx + 1,	ty),		t, newDist + 1.0f,						EAST));

					if((dir==NORTH_EAST) && tileExists(tx + 1,	ty + 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx + 1,	ty + 1),	t, newDist + MathConstants::SQRT_2F,	NORTH_EAST));

					if((dir==NORTH ||
						dir==NORTH_EAST ||
						dir==NORTH_WEST) && tileExists(tx,		ty + 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx,		ty + 1),	t, newDist + 1.0f,						NORTH));

					if((dir==NORTH_WEST) && tileExists(tx - 1,	ty + 1))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx - 1,	ty + 1),	t, newDist + MathConstants::SQRT_2F,	NORTH_WEST));

					if((dir==WEST ||
						dir==NORTH_WEST ||
						dir==SOUTH_WEST) && tileExists(tx - 1,	ty))
							tiles_to_check.push_back(Tile_Origin_Dist_Dir(WalkPosition(tx - 1,	ty),		t, newDist + 1.0f,						WEST));
				}
			}
		}
	}
}

// Ordering predicate to order tiles from highest to lowest clearance
struct MapAnalyser::TileOrderingClearance
{
	bool operator() (WalkPosition t1, WalkPosition t2) const
	{
		float clearance1 = MapAnalyser::Instance()->getClearance(t1.x, t1.y);
		float clearance2 = MapAnalyser::Instance()->getClearance(t2.x, t2.y);

		// if clearance is 0.0f, set to sufficiently large number to explore beyond these obstacles first.
		if(clearance1 == 0.0)
			clearance1 = 1000000.0f;

		if(clearance2 == 0.0f)
			clearance2 = 1000000.0f;

		return (clearance1 > clearance2);
	}
};

// set of walktiles ordered by predicate above
typedef multiset<WalkPosition, MapAnalyser::TileOrderingClearance> OrderedWalkTileSet;

/*
================================================ PSEUDOCODE ================================================
High-level pseudocode of Algorithm used to partition map in Regions and Chokepoints:

NOTE: I wrote this pseudocode quite a while ago, some bug fixes and all kinds of small details
found during implementation will be missing from the pseudocode

	tileSet = All walkable tiles of the entire map
	start_tile = null

	while TRUE:
		if start_tile == null:
			start_tile = largest clearance tile not assigned to a region
			
		if start_tile == null:
			break
			
		lowest_clearance_tile = start_tile
	
		region = new Region
		chokepoint_found = False
	
		propagationTiles.add(start_tile)

		while not chokepoint_found && not propagationTiles.isEmpty:
			propagate from start_tile with brushfire algorithm, spreading to larger clearance tiles first, ...
			... skipping tiles already marked as chokepoints
		
			if current_tile.clearance < lowest_clearance_tile.clearance:
				lowest_clearance_tile = current_tile
				region.add(current_tile)
			else if	(lowest_clearance < 0.9 * start_clearance) 					&& 
					(lowest_clearance < 0.91 * current_clearance)				&&
					manhattan_dist(current_tile, start_tile) >= 32				&&
					manhattan_dist(current_tile, lowest_clearance_tile) >= 32	&&
					current_tile.clearance >= 120									:
				chokepoint_found = True
				CHOKEPOINT = all tiles in straight line between the 2 nearest obstacle-tiles
			else:
				region.add(current_tile)
				
		if chokepoint_found:
			delete region
			continue
		
		start_tile = null
		mapRegions.add(region)

================================================ TEXTUAL DESCRIPTION ================================================
Textual description of the algorithm used, written by the author of SkyNet bot in some forum, with some adaptations 
by the author of the http://legionbot.blogspot.nl/ blog:

in a loop, continue until there are no unvisited tiles remaining:

Find the tile with the largest clearance(distance to obstacle) that hasn't been visited yet. 
Propagate from it tile to it's neighbours in the same way as step 2, though only travel to the 4-connected tiles, using aBrushfire Algorithm 
to travel through the tiles with a larger clearance first, skipping those already marked as chokepoints. 
While traveling, store the last tile that had the lowest clearance along the current path.

At each step compare the clearance of the start tile, the clearance of the current tile and the clearance of the lowest clearance tile so far. 
If certain conditions are met the lowest clearance tile will be marked as a choke point. 
The check I have so far is if the lowest value is less that 90% of the start tile or 91% of the current tile, then it is a chokepoint. 
The tiles between the obstacles that form the chokepoint are then marked off in a straight line, and all tiles on the other side of the chokepoint are removed from the current brushfire search. 
Then the burshfire algorithm continues until it cannot spread to any more tiles due to unwalkable tiles or chokepoint tiles.
The loop then continues, giving you after each step all the tiles that form this Region, the "center" of this region(the original tile) and the chokepoints connected to this region." 
*/
void MapAnalyser::computeRegions()
{
	OrderedWalkTileSet tileSet;

	// fill tileSet with all walkable tiles
	for(auto it_area = walkableAreas.begin(); it_area < walkableAreas.end(); ++it_area)
	{
		const vector<WalkPosition>& tiles = (*it_area)->tiles;

		for(auto it_tile = tiles.begin(); it_tile < tiles.end(); ++it_tile)
		{
			tileSet.insert(*it_tile);
		}
	}

	vector<vector<int>> visited;
	visited.reserve(mapHeight);

	for(int i = 0; i < mapHeight; ++i)
	{
		vector<int> row;
		visited.push_back(row);
	}

	// Algorithm truly begins here, above is just set-up
	WalkPosition startTile = NULL_TILE;
	short regionNumber = 1;
	
	while(true)
	{
		if(startTile == NULL_TILE)		// need to find largest-clearance tile which does not have a MapRegion assigned yet
		{
			while(startTile == NULL_TILE && !tileSet.empty())
			{
				WalkPosition candidateStartTile = *(tileSet.begin());
				tileSet.erase(tileSet.begin());

				if(!isChokeTile(candidateStartTile.x, candidateStartTile.y) && getRegionNumber(candidateStartTile.x, candidateStartTile.y) == 0)
					startTile = candidateStartTile;
			}
		}

		if(startTile == NULL_TILE)		// if still a nullptr, every tile has a MapRegion assigned to it, so we're finished
			break;

		// reset entire map to not-visited
		for(int i = 0; i < mapHeight; ++i)
		{
			VectorUtils::fillVector(visited[i], mapWidth, 0);
		}

		int x = startTile.x;
		int y = startTile.y;
		vector<WalkPosition> lowestClearanceTiles;
		lowestClearanceTiles.push_back(startTile);
		float lowestClearance = getClearance(x, y);
		float regionClearance = lowestClearance;
		visited[y][x] = 1;

		MapRegion* region = new MapRegion(startTile);
		bool chokepointFound = false;
		region->addTile(startTile);

		OrderedWalkTileSet propagationTiles;

		// start with all walkable neighbours of startTile
		if(tileExists(x,		y - 1) && (tilesConnected(startTile, x, y - 1) || isSmallObstacle(WalkPosition(x,	y - 1))) && !isChokeTile(x, y - 1))
			propagationTiles.insert(WalkPosition(x,		y - 1));

		if(tileExists(x + 1,	y) && (tilesConnected(startTile, x + 1, y) || isSmallObstacle(WalkPosition(x + 1,	y))) && !isChokeTile(x + 1, y))
			propagationTiles.insert(WalkPosition(x + 1,	y));

		if(tileExists(x,		y + 1) && (tilesConnected(startTile, x, y + 1) || isSmallObstacle(WalkPosition(x,	y + 1))) && !isChokeTile(x, y + 1))
			propagationTiles.insert(WalkPosition(x,		y + 1));

		if(tileExists(x - 1,	y) && (tilesConnected(startTile, x - 1, y) || isSmallObstacle(WalkPosition(x - 1,	y))) && !isChokeTile(x - 1, y))
			propagationTiles.insert(WalkPosition(x - 1,	y));

		while(!chokepointFound && !propagationTiles.empty())
		{
			WalkPosition currentTile = *propagationTiles.begin();
			propagationTiles.erase(propagationTiles.begin());

			int tx = currentTile.x;
			int ty = currentTile.y;

			if(visited[ty][tx] == 1)
				continue;

			visited[ty][tx] = 1;

			float currentClearance = getClearance(currentTile.x, currentTile.y);

			if(currentClearance > 0.0f)
			{
				if(currentClearance == lowestClearance)
				{
					lowestClearanceTiles.push_back(currentTile);
					region->addTile(currentTile);
				}
				else if(currentClearance < lowestClearance)
				{
					lowestClearance = currentClearance;
					lowestClearanceTiles.clear();
					lowestClearanceTiles.push_back(currentTile);
					region->addTile(currentTile);
				}
				else
				{
					// Find the closest lowest clearance tile
					WalkPosition lowestClearanceTile = NULL_TILE;
					float smallestDistSq = mapWidth*mapWidth + mapHeight*mapHeight;

					for(auto it = lowestClearanceTiles.begin(); it != lowestClearanceTiles.end(); ++it)
					{
						float distSq = Distances::getSquaredDistanceF((*it).x, (*it).y, currentTile.x, currentTile.y);
						if(distSq < smallestDistSq)
							lowestClearanceTile = (*it);
					}

					if(lowestClearance < 0.90f * regionClearance															&&
						lowestClearance < 0.91f * currentClearance															&&
						Distances::getManhattanDistanceInt(x, y, lowestClearanceTile.x, lowestClearanceTile.y) >= 32		&&
						Distances::getManhattanDistanceInt(tx, ty, lowestClearanceTile.x, lowestClearanceTile.y) >= 32		&&
						currentClearance >= 12.0f																				)
					{
						// First condition ensures that the chokepoint is small enough relative to the region
						// Second condition ensures that there is a big enough region on other side of chokepoint
						// Third condition ensures that chokepoint is far enough away from centroid of region
						// Fourth condition ensures that region on other side of chokepoint has a centroid far enough away from chokepoint
						// Fifth condition ensures... kind of similar thing to second I guess?
						//
						// Sources of magic numbers:
						// 0.9 and 0.91 were described to be experimentally tuned by the author of http://legionbot.blogspot.nl/
						// The 32's and 12 were found in Skynet source code (I suppose these were experimentally determined by Skynet author)
						chokepointFound = true;

						// Mark off all tiles in a straight line between 2 nearest obstacles as chokepoint
						const Pair<WalkPosition, WalkPosition> chokeSides = findChokeSides(lowestClearanceTile);

						MapChokePoint* chokepoint = new MapChokePoint(lowestClearanceTile);
						chokepoint->addTile(chokeSides.second);
						setChokeTile(chokeSides.second.x, chokeSides.second.y, true);

						int x0 = chokeSides.second.x;
						int y0 = chokeSides.second.y;

						int x1 = lowestClearanceTile.x;
						int y1 = lowestClearanceTile.y;

						int dx = abs(x1 - x0);
						int dy = abs(y1 - y0);

						int sx = x0 < x1 ? 1 : -1;
						int sy = y0 < y1 ? 1 : -1;

						int error = dx - dy;

						while(true)
						{
							if(x0 == x1 && y0 == y1)
								break;

							WalkPosition t(x0, y0);

							if(x0 >= 0 && y0 >= 0 && x0 < mapWidth && y0 < mapHeight && (isWalkable(x0, y0) || isSmallObstacle(t)) && getRegionNumber(x0, y0) == 0)
							{
								chokepoint->addTile(t);
								setChokeTile(x0, y0, true);
							}

							int e2 = error*2;
							if(e2 > -dy)
							{
								error -= dy;
								x0 += sx;
							}
							if(e2 < dx)
							{
								error += dx;
								y0 += sy;
							}
						}

						x0 = lowestClearanceTile.x;
						y0 = lowestClearanceTile.y;

						x1 = chokeSides.first.x;
						y1 = chokeSides.first.y;

						dx = abs(x1 - x0);
						dy = abs(y1 - y0);

						sx = x0 < x1 ? 1 : -1;
						sy = y0 < y1 ? 1 : -1;

						error = dx - dy;

						while(true)
						{
							WalkPosition t(x0, y0);

							if(x0 >= 0 && y0 >= 0 && x0 < mapWidth && y0 < mapHeight && (isWalkable(x0, y0) || isSmallObstacle(t)) && getRegionNumber(x0, y0) == 0)
							{
								chokepoint->addTile(t);
								setChokeTile(x0, y0, true);
							}

							if(x0 == x1 && y0 == y1)
								break;

							int e2 = error*2;
							if(e2 > -dy)
							{
								error -= dy;
								x0 += sx;
							}
							if(e2 < dx)
							{
								error += dx;
								y0 += sy;
							}
						}

						chokepoints.push_back(chokepoint);
						break;	// need to start region-finding procedure over again
					}
					else
					{
						region->addTile(currentTile);
					}
				}
			}

			if(tileExists(tx,		ty - 1) && (tilesConnected(startTile, tx, ty - 1) || isSmallObstacle(WalkPosition(tx,	ty - 1))) && !isChokeTile(tx, ty - 1) && visited[ty - 1][tx] == 0)		
				propagationTiles.insert(WalkPosition(tx,		ty - 1));

			if(tileExists(tx + 1,	ty) && (tilesConnected(startTile, tx + 1, ty) || isSmallObstacle(WalkPosition(tx + 1,	ty))) && !isChokeTile(tx + 1, ty) && visited[ty][tx + 1] == 0)			
				propagationTiles.insert(WalkPosition(tx + 1,	ty));

			if(tileExists(tx,		ty + 1) && (tilesConnected(startTile, tx, ty + 1) || isSmallObstacle(WalkPosition(tx,	ty + 1))) && !isChokeTile(tx, ty + 1) && visited[ty + 1][tx] == 0)		
				propagationTiles.insert(WalkPosition(tx,		ty + 1));

			if(tileExists(tx - 1,	ty) && (tilesConnected(startTile, tx - 1, ty) || isSmallObstacle(WalkPosition(tx - 1,	ty))) && !isChokeTile(tx - 1, ty) && visited[ty][tx - 1] == 0)			
				propagationTiles.insert(WalkPosition(tx - 1,	ty));

		}

		if(chokepointFound)
		{
			delete region;
			continue;
		}

		startTile = NULL_TILE;
		mapRegions.push_back(region);
		vector<WalkPosition> regionTiles = region->getTiles();

		for(auto it = regionTiles.begin(); it != regionTiles.end(); ++it)
		{
			setRegionNumber((*it).x, (*it).y, regionNumber);
		}

		regionNumber++;
	}
}

// Thanks to Skynet source code for implementation of this :D
const Pair<WalkPosition, WalkPosition> MapAnalyser::findChokeSides(const WalkPosition chokeCenter) const
{
	WalkPosition side1 = getNearestObstacleTile(chokeCenter.x, chokeCenter.y);

#ifdef MAASCRAFT_DEBUG
	if(side1 == chokeCenter)
	{
		LOG_WARNING("MapAnalyser : Nearest obstacle to choke center equals the center itself!")
		return Pair<WalkPosition, WalkPosition>(side1, side1);
	}
#endif	// MAASCRAFT_DEBUG

	Vector2D side1Direction(side1.x - chokeCenter.x, side1.y - chokeCenter.y);
	side1Direction.normalize();

	int x0 = side1.x;
	int y0 = side1.y;

	int x1 = chokeCenter.x;
	int y1 = chokeCenter.y;

	int dx = abs(x1 - x0);
	int dy = abs(y1 - y0);

	int sx = x0 < x1 ? 1 : -1;
	int sy = y0 < y1 ? 1 : -1;

	x0 = x1;
	y0 = y1;

	int error = dx - dy;

	while(true)
	{
		WalkPosition tile(x0, y0);

		if(x0 < 0 || y0 < 0 || x0 >= mapWidth || y0 >= mapHeight || (!isWalkable(x0, y0) && !isSmallObstacle(tile)))
			return Pair<WalkPosition, WalkPosition>(side1, tile);

		WalkPosition side2 = getNearestObstacleTile(x0, y0);

		Vector2D side2Direction(side2.x - chokeCenter.x, side2.y - chokeCenter.y);
		side2Direction.normalize();

		float dot = side2Direction*(side1Direction);
		float angle = acos(dot);
		if(angle > 2.0f && !isSmallObstacle(tile))
			return Pair<WalkPosition, WalkPosition>(side1, side2);

		int e2 = error*2;
		if(e2 > -dy)
		{
			error -= dy;
			x0 += sx;
		}
		if(e2 < dx)
		{
			error += dx;
			y0 += sy;
		}
	}
}

const bool MapAnalyser::canReach(const Unit unit, const WalkPosition goal) const
{
	if(!tileExists(goal))
		return false;

	if(unit->getType().isFlyer())
		return true;

	WalkPosition unitPos = PositionUtils::toWalkPosition(unit->getPosition());

	return (getConnectivity(goal.x, goal.y) == getConnectivity(unitPos.x, unitPos.y));
}

const bool MapAnalyser::pathExists(const WalkPosition& p1, const WalkPosition& p2) const
{
	return (getConnectivity(p1.x, p1.y) == getConnectivity(p2.x, p2.y));
}

void MapAnalyser::connectChokesToRegions()
{
	for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
	{
		MapChokePoint* chokepoint = (*it);
		BWAPI::WalkPosition center = chokepoint->center;
		int cx = center.x;
		int cy = center.y;

		if(!isChokeTile(cx - 1, cy) && isWalkable(cx - 1, cy))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy) - 1];	// should be connected to this region
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx + 1, cy) && isWalkable(cx + 1, cy))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx - 1, cy - 1) && isWalkable(cx - 1, cy - 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy - 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx, cy - 1) && isWalkable(cx, cy - 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx, cy - 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx + 1, cy - 1) && isWalkable(cx + 1, cy - 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy - 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx - 1, cy + 1) && isWalkable(cx - 1, cy + 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy + 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx, cy + 1) && isWalkable(cx, cy + 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx, cy + 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		if(!isChokeTile(cx + 1, cy + 1) && isWalkable(cx + 1, cy + 1))
		{
			MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy + 1) - 1];
			if(chokepoint->regions.first == nullptr)
			{
				chokepoint->regions.first = region;
				region->addChokepoint(chokepoint);

				region->addTiles(chokepoint->getTiles());
			}
			else if(chokepoint->regions.second == nullptr)
			{
				MapRegion* first = chokepoint->regions.first;
				if(first->centroid != region->centroid)			// not the same region we already tried adding
				{
					chokepoint->regions.second = region;
					region->addChokepoint(chokepoint);
					continue;
				}
			}
		}

		// rare case where center of chokepoint is on one side completely adjacent
		// to obstacles. Try choke tiles other than center
		if(chokepoint->regions.second == nullptr)
		{
			vector<WalkPosition> chokeTiles = chokepoint->tiles;

			for(auto it_tile = chokeTiles.begin(); it_tile != chokeTiles.end(); ++it_tile)
			{
				BWAPI::WalkPosition tile = (*it_tile);
				cx = tile.x;
				cy = tile.y;

				if(!isChokeTile(cx - 1, cy) && isWalkable(cx - 1, cy))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy) - 1];	// should be connected to this region
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx + 1, cy) && isWalkable(cx + 1, cy))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx - 1, cy - 1) && isWalkable(cx - 1, cy - 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy - 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx, cy - 1) && isWalkable(cx, cy - 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx, cy - 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx + 1, cy - 1) && isWalkable(cx + 1, cy - 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy - 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx - 1, cy + 1) && isWalkable(cx - 1, cy + 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx - 1, cy + 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx, cy + 1) && isWalkable(cx, cy + 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx, cy + 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}

				if(!isChokeTile(cx + 1, cy + 1) && isWalkable(cx + 1, cy + 1))
				{
					MapRegion* region = mapRegions[getRegionNumber(cx + 1, cy + 1) - 1];
					if(chokepoint->regions.first == nullptr)
					{
						chokepoint->regions.first = region;
						region->addChokepoint(chokepoint);

						region->addTiles(chokepoint->getTiles());
					}
					else if(chokepoint->regions.second == nullptr)
					{
						MapRegion* first = chokepoint->regions.first;
						if(first->centroid != region->centroid)			// not the same region we already tried adding
						{
							chokepoint->regions.second = region;
							region->addChokepoint(chokepoint);
							break;
						}
					}
				}
			}
#ifdef MAASCRAFT_DEBUG
			if(chokepoint->regions.first == nullptr)
			{
				LOG_WARNING("Failed to connect a choke to a first region!")
				LOG_MESSAGE(StringBuilder() << " Choke: " << chokepoint->center.x << ", " << chokepoint->center.y)
			}

			LOG_WARNING("Failed to connect a choke to a second region!")
			LOG_MESSAGE(StringBuilder() << " Choke: " << chokepoint->center.x << ", " << chokepoint->center.y)
#endif
		}
	}
}

// Algorithm to prune small regions:
// Region size threshold = 1300
// If Region size < threshold:
//	If merging with a neighbouring region would result in a region still below threshold:
//		merge with smallest region for which that holds
//	Else:
//		merge with largest neighbouring region
//
//	DO NOT ALLOW MERGING PAST CHOKEPOINTS OF SIZE < 10
void MapAnalyser::pruneRegions()
{
	vector<MapRegion*> prunableRegions;
	prunableRegions.reserve(mapRegions.size());

	// store all regions which could potentially be pruned at some point in time
	for(auto it = mapRegions.begin(); it != mapRegions.end(); ++it)
	{
		MapRegion* region = (*it);

		if(region->chokepoints.empty())
			continue;

		if(region->tiles.size() > Options::MapAnalysis::PRUNE_REGION_SIZE_THRESHOLD)
			continue;

		prunableRegions.push_back(region);
	}

	// main pruning algorithm
	while(prunableRegions.size() > 0)
	{
		// find smallest region
		auto smallestRegionIt = prunableRegions.begin();
		size_t smallestSize = (*smallestRegionIt)->tiles.size();

		auto it = prunableRegions.begin();
		++it;
		for(; it != prunableRegions.end(); ++it)
		{
			size_t size = (*it)->tiles.size();

			if(size < smallestSize)
			{
				smallestRegionIt = it;
				smallestSize = size;
			}
		}

		if(smallestSize > Options::MapAnalysis::PRUNE_REGION_SIZE_THRESHOLD)	// finished if only regions remaining which are too large
			break;

		// now smallestRegionIt points to smallest region
		// check if we can indeed prune this region
		MapRegion* smallestRegion = (*smallestRegionIt);
		vector<MapChokePoint*> chokepoints = smallestRegion->chokepoints;

		MapRegion* largestPotentialMergeRegion = nullptr;
		MapRegion* smallestPotentialMergeRegion = nullptr;

		int largestPotentialMergeRegionSize = MathConstants::MIN_INT;
		int smallestPotentialMergeRegionSize = MathConstants::MAX_INT;

		MapChokePoint* largetPotentialMergeChoke = nullptr;
		MapChokePoint* smallestPotentialMergeChoke = nullptr;

		for(auto choke_it = chokepoints.begin(); choke_it != chokepoints.end(); ++choke_it)
		{
			MapChokePoint* choke = (*choke_it);

			if(choke->tiles.size() <= Options::MapAnalysis::PRUNE_CHOKE_SIZE_THRESHOLD)		// choke too small
				continue;

			MapRegion* candidateRegion = choke->getOtherRegion(smallestRegion);
			int candidateSize = (int) candidateRegion->tiles.size();

			if(candidateSize < Options::MapAnalysis::PRUNE_REGION_SIZE_THRESHOLD)
			{
				if(candidateSize < smallestPotentialMergeRegionSize)
				{
					smallestPotentialMergeRegionSize = candidateSize;
					smallestPotentialMergeRegion = candidateRegion;
					smallestPotentialMergeChoke = choke;
				}
			}

			if(candidateSize > largestPotentialMergeRegionSize)
			{
				largestPotentialMergeRegionSize = candidateSize;
				largestPotentialMergeRegion = candidateRegion;
				largetPotentialMergeChoke = choke;
			}
		}

		// first try to merge with smallest adjacent region if that would result in another prunable region
		if(smallestPotentialMergeRegion != nullptr)
		{
			if((smallestRegion->tiles.size() + smallestPotentialMergeRegionSize) < Options::MapAnalysis::PRUNE_REGION_SIZE_THRESHOLD)
			{
				prunableRegions.erase(smallestRegionIt);
				mergeRegionInto(smallestRegion, smallestPotentialMergeRegion, smallestPotentialMergeChoke);
				continue;
			}
		}

		// if not succesfully merged above, try merging with largest adjacent region
		if(largestPotentialMergeRegion != nullptr)
		{
			prunableRegions.erase(smallestRegionIt);
			mergeRegionInto(smallestRegion, largestPotentialMergeRegion, largetPotentialMergeChoke);
			continue;
		}

		// could not merge at all, just remove it from prunableRegions list
		prunableRegions.erase(smallestRegionIt);
	}
}

void MapAnalyser::mergeRegionInto(MapRegion* region1, MapRegion* region2, MapChokePoint* choke)
{
	// remove region and choke from the lists kept by MapAnalyser
	mapRegions.erase(find(mapRegions.begin(), mapRegions.end(), region1));
	chokepoints.erase(find(chokepoints.begin(), chokepoints.end(), choke));

	// transfer tiles to new region
	region2->addTiles(region1->tiles);
	const short newRegionNumber = getRegionNumber(region2->tiles[0].x, region2->tiles[0].y);
	for(auto it = region1->tiles.begin(); it != region1->tiles.end(); ++it)
	{
		WalkPosition pos = (*it);
		setRegionNumber(pos.x, pos.y, newRegionNumber);
	}

	// Remove merged choke from region2
	region2->chokepoints.erase(find(region2->chokepoints.begin(), region2->chokepoints.end(), choke));

	// Add all other chokes of region1 to region2 and update them
	vector<MapChokePoint*> region1Chokes = region1->chokepoints;
	for(auto it = region1Chokes.begin(); it != region1Chokes.end(); ++it)
	{
		MapChokePoint* c = (*it);

		if(c != choke)
		{
			if(c->regions.first == region1)
				c->regions.first = region2;
			else if(c->regions.second == region1)
				c->regions.second = region2;
#ifdef MAASCRAFT_DEBUG
			else
				LOG_WARNING("CANNOT UPDATE CHOKE CORRECTLY IN REGION MERGE")
#endif MAASCRAFT_DEBUG

			// add choke if it is still necessary (maybe it is surrounded by the same region on both sides)
			if(c->regions.first != c->regions.second)
				region2->addChokepoint(c);
			else	// didn't add choke, meaning it disappears, so need to reset choke-status
			{
				for(auto it_c = c->tiles.begin(); it_c != c->tiles.end(); ++it_c)
				{
					WalkPosition pos = (*it_c);
					setChokeTile(pos.x, pos.y, false);
					setRegionNumber(pos.x, pos.y, newRegionNumber);
				}
			}
		}
	}

	// Reset chokepoint status of tiles which we merged
	for(auto it = choke->tiles.begin(); it != choke->tiles.end(); ++it)
	{
		WalkPosition pos = (*it);
		setChokeTile(pos.x, pos.y, false);
		setRegionNumber(pos.x, pos.y, newRegionNumber);
	}

	// clean up memory
	delete choke;
	delete region1;
}

// Update region numbers because pruning may have removed some region numbers in between
void MapAnalyser::updateRegionNumbers()
{
	for(short i = 0; i < mapRegions.size(); ++i)
	{
		MapRegion* region = mapRegions[i];
		vector<WalkPosition> tiles = region->tiles;

		for(auto it = tiles.begin(); it != tiles.end(); ++it)
		{
			setRegionNumber(it->x, it->y, i + 1);
		}
	}
}

void MapAnalyser::findBaseLocations()
{
	const int	NORTH_WEST = 1,		NORTH = 2,			NORTH_EAST = 3,
				EAST = 4,			SOUTH_EAST = 5,		SOUTH = 6,
				SOUTH_WEST = 7,		WEST = 8;

	Unitset resources = Broodwar->getStaticMinerals();

	for(auto it = resources.begin(); it != resources.end(); /**/)
	{
		Unit mineral = *it;
		if(mineral->getResources() < 250)
			resources.remove(it);
		else
			++it;
	}

	resources.push_back(Broodwar->getStaticGeysers());

	// parameters from: http://legionbot.blogspot.nl/2012/09/finding-base-locations.html
	vector<Unitset> clusters = Clustering::getClustersForUnitsSameRegion(resources, 450.0, 3);

	const int mapWidth = Broodwar->mapWidth();
	const int numBuildTiles = mapWidth * Broodwar->mapHeight();
	int* resourceProximityMap = new int[numBuildTiles];

	for(auto cluster_it = clusters.begin(); cluster_it != clusters.end(); ++cluster_it)
	{
		Unitset cluster = (*cluster_it);
		ArrayUtils::initializeIntArray(resourceProximityMap, numBuildTiles, 0);

		int highestProximity = 0;
		TilePosition bestTile = TilePositions::Invalid;

		for(auto resource_it = cluster.begin(); resource_it != cluster.end(); ++resource_it)
		{
			Unit resource = (*resource_it);
			const TilePosition resourcePos = resource->getInitialTilePosition();
			const bool isMineral = (resource->getType() != UnitTypes::Resource_Vespene_Geyser);
			const int startX = resourcePos.x;
			const int startY = resourcePos.y;

			// FLOOD FILL START
			vector<Pair<TilePosition, int>> queue;
			queue.reserve(64);

			if(tileExists(TilePosition(startX - 1,	startY - 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX - 1,	startY - 1), NORTH_WEST));
			if(tileExists(TilePosition(startX,		startY - 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX,		startY - 1), NORTH));
			if(tileExists(TilePosition(startX + 1,	startY - 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX + 1,	startY - 1), NORTH_EAST));

			if(tileExists(TilePosition(startX + 1,	startY)))		
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX + 1,	startY), EAST));
			if(tileExists(TilePosition(startX + 1,	startY + 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX + 1,	startY + 1), SOUTH_EAST));

			if(tileExists(TilePosition(startX,		startY + 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX,		startY + 1), SOUTH));
			if(tileExists(TilePosition(startX - 1,	startY + 1)))	
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX - 1,	startY + 1), SOUTH_WEST));

			if(tileExists(TilePosition(startX - 1,	startY)))		
				queue.push_back(Pair<TilePosition, int>(TilePosition(startX - 1,	startY), WEST));

			while(queue.size() > 0)
			{
				Pair<TilePosition, int> pair = queue.back();
				queue.pop_back();

				const int x = pair.first.x;
				const int y = pair.first.y;
				const int dir = pair.second;
				const int propagationProx = 50 - Distances::getDistance(startX, startY, x, y);

				if(propagationProx > 0)
				{
					if(!BuildingManager::Instance()->canBuildAtTileStatic(UnitTypes::Protoss_Nexus, pair.first))
					{
						ArrayUtils::setMatrixValue(resourceProximityMap, y, x, mapWidth, -1);
					}
					else
					{
						int oldProx = ArrayUtils::getMatrixValue(resourceProximityMap, y, x, mapWidth);

						if(oldProx >= 0)
						{
							const int newProx = propagationProx + oldProx;
							ArrayUtils::setMatrixValue(resourceProximityMap, y, x, mapWidth, newProx);

							if(newProx > highestProximity)
							{
								highestProximity = newProx;
								bestTile = pair.first;
							}
						}
					}

					if(tileExists(TilePosition(x - 1,	y - 1)) && (dir == NORTH_WEST) && tilesConnected(TilePosition(x - 1,	y - 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x - 1,	y - 1), NORTH_WEST));
					if(tileExists(TilePosition(x,		y - 1)) && (dir == NORTH_WEST || dir == NORTH || dir == NORTH_EAST) && tilesConnected(TilePosition(x,	y - 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x,		y - 1), NORTH));
					if(tileExists(TilePosition(x + 1,	y - 1)) && (dir == NORTH_EAST) && tilesConnected(TilePosition(x + 1,	y - 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x + 1,	y - 1), NORTH_EAST));

					if(tileExists(TilePosition(x + 1,	y)) && (dir == NORTH_EAST || dir == EAST || dir == SOUTH_EAST) && tilesConnected(TilePosition(x + 1,	y), resourcePos))		
						queue.push_back(Pair<TilePosition, int>(TilePosition(x + 1,	y), EAST));
					if(tileExists(TilePosition(x + 1,	y + 1)) && (dir == SOUTH_EAST) && tilesConnected(TilePosition(x + 1,	y + 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x + 1,	y + 1), SOUTH_EAST));

					if(tileExists(TilePosition(x,		y + 1)) && (dir == SOUTH_WEST || dir == SOUTH || dir == SOUTH_EAST) && tilesConnected(TilePosition(x,	y + 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x,		y + 1), SOUTH));
					if(tileExists(TilePosition(x - 1,	y + 1)) && (dir == SOUTH_WEST) && tilesConnected(TilePosition(x - 1,	y + 1), resourcePos))	
						queue.push_back(Pair<TilePosition, int>(TilePosition(x - 1,	y + 1), SOUTH_WEST));

					if(tileExists(TilePosition(x - 1,	y)) && (dir == NORTH_WEST || dir == WEST || dir == SOUTH_WEST) && tilesConnected(TilePosition(x - 1,	y), resourcePos))		
						queue.push_back(Pair<TilePosition, int>(TilePosition(x - 1,	y), WEST));
				}
			}
			// FLOOD FILL END
		}
		if(bestTile != TilePositions::Invalid)
		{
			WalkPosition pos = PositionUtils::toWalkPosition(bestTile);
			const short regionNumber = getRegionNumber(pos.x, pos.y);
			baseLocations.push_back(new BaseLocation(cluster, bestTile, regionNumber));
		}
	}

	delete[] resourceProximityMap;
}

void MapAnalyser::identifyStartLocations()
{
	TilePosition::set startLocations = Broodwar->getStartLocations();
	const int numBaseLocations = baseLocations.size();
	const int numStartLocations = startLocations.size();

	for(size_t i = 0; i < numStartLocations; ++i)
	{
		int minDistSq = MathConstants::MAX_INT;
		BaseLocation* nearestBaseLocation = nullptr;

		for(size_t j = 0; j < numBaseLocations; ++j)
		{
			const int distSq = Distances::getSquaredDistance(startLocations[i], baseLocations[j]->getOptimalDepotLocation());

			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				nearestBaseLocation = baseLocations[j];
			}
		}

		nearestBaseLocation->setStartLocation(true);

		if(startLocations[i] == Broodwar->self()->getStartLocation())
		{
			BaseManager::Instance()->addBase(Base(nearestBaseLocation, Broodwar->self()));
			BaseManager::Instance()->setMainBaseLocation(nearestBaseLocation);
		}
	}
}

// All kinds of memory clean-up and memory optimization
void MapAnalyser::cleanup()
{
	// no longer need walkable areas
	for(auto it = walkableAreas.begin(); it != walkableAreas.end(); ++it)
	{
		delete (*it);
	}
	walkableAreas.clear();
	walkableAreas.resize(0);

	// no longer need obstacles
	for(auto it = obstacles.begin(); it != obstacles.end(); ++it)
	{
		delete (*it);
	}
	obstacles.clear();
	obstacles.resize(0);

	// no longer need nearest obstacle tiles
	delete[] nearestObstacleTiles;
	nearestObstacleTiles = nullptr;

	for(auto it = mapRegions.begin(); it != mapRegions.end(); ++it)
	{
		(*it)->optimizeMemory();
	}

	for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
	{
		(*it)->optimizeMemory();
	}

	mapRegions.shrink_to_fit();
	chokepoints.shrink_to_fit();
	baseLocations.shrink_to_fit();
}

#pragma warning( pop )