/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	See MaasCraft.h
*/

#include "CommonIncludes.h"

#include <iostream>
#include <vector>

#include "ArmyManager.h"
#include "BaseManager.h"
#include "BuildingManager.h"
#include "MaasCraft.h"
#include "MapAnalyser.h"
#include "OpponentTracker.h"
#include "ProductionManager.h"
#include "ScoutManager.h"
#include "StrategyManager.h"
#include "Timer.hpp"
#include "UnitTracker.h"
#include "UserCommandProcessor.h"
#include "WorkerManager.h"

// TESTING

using namespace BWAPI;
using namespace Filter;

#pragma warning( push )
#pragma warning( disable:4244 )

void MaasCraft::onStart()
{
#ifdef MAASCRAFT_DEBUG
	if (Options::Debug::DEBUG_MANUAL_CONTROL)
	{
		Broodwar->enableFlag(Flag::UserInput);
	}
#endif

	if(!Broodwar->isReplay())
	{
		UnitTracker::Instance()->validateSelfUnits();

		if(Broodwar->self()->getRace() != BWAPI::Races::Enum::Protoss)
		{
			LOG_WARNING("Races other than Protoss are currently not supported for this bot!")
			Broodwar->sendText("Races other than Protoss are currently not supported for this bot!");
		}

		//Broodwar->setCommandOptimizationLevel(1);	// this optimization level was used by Skynet in AIIDE 2012, so seems safe

		if(Broodwar->getRevision() != BWAPI_getRevision())
		{
			LOG_WARNING(StringBuilder() << "This version of MaasCraft was compiled for BWAPI revision " << BWAPI_getRevision())
			LOG_WARNING(StringBuilder() << "Detected BWAPI revision currently running: " << Broodwar->getRevision())
		}

		Broodwar->setLocalSpeed(Options::GAME_SPEED);

		MapAnalyser::Instance()->analyseCurrentMap();

		BuildingManager::Instance()->onStart();
		StrategyManager::Instance()->onStart();

#ifdef MAASCRAFT_DEBUG
		if(Options::Debug::DEBUG_USER_COMMANDS)
			Broodwar << "Type \"/help <text>\" in order to show a list of known commands containing <text>." << std::endl;
#endif	// MAASCRAFT_DEBUG
	}
}

void MaasCraft::onEnd(bool isWinner)
{
	if(Broodwar->isReplay())
		return;					// if it is a replay, isWinner will always be false, so cannot do anything useful here

	// log game results here if needed
}

void MaasCraft::onFrame()
{
	if ( Broodwar->isReplay() || Broodwar->isPaused() || !Broodwar->self() )
		return;

	static Timer frameTimer;
	frameTimer.start();

	UnitTracker::Instance()->validateSelfUnits();

	StrategyManager::Instance()->onFrame();

	OpponentTracker::Instance()->onFrame();
	BaseManager::Instance()->onFrame();
	ProductionManager::Instance()->onFrame();
	BuildingManager::Instance()->onFrame();
	ScoutManager::Instance()->onFrame();
	WorkerManager::Instance()->onFrame();
	ArmyManager::Instance()->onFrame(Options::CompetitionRules::MAX_MS_PER_FRAME - frameTimer.getElapsedTimeInMilliSec());

#ifdef MAASCRAFT_DEBUG
	if(Options::Debug::DEBUG_USER_COMMANDS)
		UserCommandProcessor::onFrame();

	if(Options::Debug::ScreenInfo::DEBUG_DRAW_FPS)
	{
		Broodwar->drawTextScreen(200, 0,  "FPS: %d", Broodwar->getFPS() );
		Broodwar->drawTextScreen(200, 20, "Average FPS: %f", Broodwar->getAverageFPS() );
		Broodwar->drawTextScreen(200, 40, "Time left = %f ms", Options::CompetitionRules::MAX_MS_PER_FRAME - frameTimer.getElapsedTimeInMilliSec());

		int frameCount = Broodwar->getFrameCount();
		int hours = frameCount / (24 * 60 * 60);
		frameCount -= hours*24*60*60;
		int mins = frameCount / (24 * 60);
		frameCount -= mins*24*60;
		int seconds = frameCount / 24;

		Broodwar->drawTextScreen(200, 60, "Clock: %02d:%02d:%02d", hours, mins, seconds);
		Broodwar->drawTextScreen(200, 80, "Frame: %d", Broodwar->getFrameCount());
	}

//	while(frameTimer.getElapsedTimeInMilliSec() < Options::CompetitionRules::MAX_MS_PER_FRAME)
//	{/* loop to enforce certain FPS cap */}

#endif	// MAASCRAFT_DEBUG
}

void MaasCraft::onUnitDestroy(Unit unit)
{
	if(unit->getPlayer()->isNeutral() || Broodwar->isReplay())
		return;

	if(unit->getPlayer() == Broodwar->enemy())
		OpponentTracker::Instance()->onUnitDestroy(unit);
	else if(unit->getType().isBuilding() || unit->getType().isAddon())
	{
		BaseManager::Instance()->onSelfBuildingDestroy(unit);
		BuildingManager::Instance()->onBuildingDestroy(unit);
		ProductionManager::Instance()->onBuildingDestroy(unit);
		MapAnalyser::Instance()->onBuildingDestroy(unit->getPosition(), unit->getType());
	}
	else
	{
		UnitOwner* owner = UnitTracker::Instance()->getUnitOwner(unit);
		if(owner)
			owner->removeOwnership(unit);
	}
}

void MaasCraft::onUnitComplete(Unit unit)
{
	if(unit->getPlayer()->isNeutral() || Broodwar->isReplay())
		return;

	if(unit->getPlayer() == Broodwar->self())
	{
		UnitType type = unit->getType();

		if(type.isBuilding())
		{
			BaseManager::Instance()->onSelfBuildingComplete(unit);
			ProductionManager::Instance()->onBuildingComplete(unit);

			if(type == UnitTypes::Protoss_Pylon)
				BuildingManager::Instance()->onPylonFinishedBuilding();
		}
		else
		{
			if(type.isWorker())
				WorkerManager::Instance()->receiveOwnership(unit);
			else
				ArmyManager::Instance()->onUnitComplete(unit);

			ProductionManager::Instance()->onFinishProduction(type);
		}
	}
}

void MaasCraft::onUnitCreate(Unit unit)
{
	if(unit->getPlayer()->isNeutral() || Broodwar->isReplay())
		return;

	if(unit->getPlayer() == Broodwar->self() && unit->getType().isBuilding())
	{
		MapAnalyser::Instance()->onBuildingShow(unit);

		if(unit->getType() == UnitTypes::Protoss_Pylon)
			BuildingManager::Instance()->onPylonStartBuilding();
	}
}

void MaasCraft::onSendText(std::string text)
{
	if(Broodwar->isReplay())
		return;

#ifdef MAASCRAFT_DEBUG
	if(Options::Debug::DEBUG_USER_COMMANDS)
		UserCommandProcessor::processCommand(text);
#endif	// MAASCRAFT_DEBUG
}

void MaasCraft::onUnitHide(Unit unit)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onUnitMorph(Unit unit)
{
	if(Broodwar->isReplay())
		return;

	if(unit->getPlayer() == Broodwar->self() && unit->getType() == UnitTypes::Protoss_Assimilator)
	{
		BaseManager::Instance()->onSelfBuildingComplete(unit);
	}
}

void MaasCraft::onUnitShow(Unit unit)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onNukeDetect(Position target)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onUnitDiscover(Unit unit)
{
	if(Broodwar->isReplay())
		return;
	// not using this; using onUnitShow() instead
}

void MaasCraft::onUnitEvade(Unit unit)
{
	if(Broodwar->isReplay())
		return;
	// not using this; using onUnitHide() instead
}

// Called when receiving message from other player
void MaasCraft::onReceiveText(Player player, std::string text)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onPlayerLeft(Player player)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onUnitRenegade(Unit unit)
{
	if(Broodwar->isReplay())
		return;
}

void MaasCraft::onSaveGame(std::string gameName)
{
	if(Broodwar->isReplay())
		return;
}


/*
=============== E X A M P L E   C O D E ===============
The following code was in onFrame() of the Example AI:

// Iterate through all the units that we own
	Unitset myUnits = Broodwar->self()->getUnits();
	for ( Unitset::iterator u = myUnits.begin(); u != myUnits.end(); ++u )
	{
		// Ignore the unit if it no longer exists
		// Make sure to include this block when handling any Unit pointer!
	   if ( !u->exists() )
			continue;

		// Ignore the unit if it has one of the following status ailments
		if ( u->isLockedDown() || u->isMaelstrommed() || u->isStasised() )
			continue;

		// Ignore the unit if it is in one of the following states
		if ( u->isLoaded() || !u->isPowered() || u->isStuck() )
			continue;

		// Ignore the unit if it is incomplete or busy constructing
		if ( !u->isCompleted() || u->isConstructing() )
			continue;


		// Finally make the unit do some stuff!


		// If the unit is a worker unit
		if ( u->getType().isWorker() )
		{
			// if our worker is idle
			if ( u->isIdle() )
			{
				// Order workers carrying a resource to return them to the center,
				// otherwise find a mineral patch to harvest.
				if ( u->isCarryingGas() || u->isCarryingMinerals() )
				{
					u->returnCargo();
				}
				else if ( !u->getPowerUp() )  // The worker cannot harvest anything if it
				{                             // is carrying a powerup such as a flag
					// Harvest from the nearest mineral patch or gas refinery
					if ( !u->gather( u->getClosestUnit( IsMineralField || IsRefinery )) )
					{
						// If the call fails, then print the last error message
						Broodwar << Broodwar->getLastError() << std::endl;
					}

				} // closure: has no powerup
			} // closure: if idle

		}
		else if ( u->getType().isResourceDepot() ) // A resource depot is a Command Center, Nexus, or Hatchery
		{

			// Order the depot to construct more workers! But only when it is idle.
			if ( u->isIdle() && !u->train(u->getType().getRace().getWorker()) )
			{
				// If that fails, draw the error at the location so that you can visibly see what went wrong!
				// However, drawing the error once will only appear for a single frame
				// so create an event that keeps it on the screen for some frames
				Position pos = u->getPosition();
				Error lastErr = Broodwar->getLastError();
				Broodwar->registerEvent([pos,lastErr](Game*){ Broodwar->drawTextMap(pos, "%c%s", Text::White, lastErr.c_str()); },   // action
										nullptr,    // condition
										Broodwar->getLatencyFrames());  // frames to run

				// Retrieve the supply provider type in the case that we have run out of supplies
				UnitType supplyProviderType = u->getType().getRace().getSupplyProvider();
				static int lastChecked = 0;

				// If we are supply blocked and haven't tried constructing more recently
				if (  lastErr == Errors::Insufficient_Supply &&
					lastChecked + 400 < Broodwar->getFrameCount() &&
					Broodwar->self()->incompleteUnitCount(supplyProviderType) == 0 )
				{
					lastChecked = Broodwar->getFrameCount();

					// Retrieve a unit that is capable of constructing the supply needed
					Unit supplyBuilder = u->getClosestUnit(  GetType == supplyProviderType.whatBuilds().first &&
														(IsIdle || IsGatheringMinerals) &&
														IsOwned);
					// If a unit was found
					if ( supplyBuilder )
					{
						if ( supplyProviderType.isBuilding() )
						{
							TilePosition targetBuildLocation = Broodwar->getBuildLocation(supplyProviderType, supplyBuilder->getTilePosition());
							if ( targetBuildLocation )
							{
								// Register an event that draws the target build location
								Broodwar->registerEvent([targetBuildLocation,supplyProviderType](Game*)
														{
															Broodwar->drawBoxMap( Position(targetBuildLocation),
																					Position(targetBuildLocation + supplyProviderType.tileSize()),
																					Colors::Blue);
														},
														nullptr,  // condition
														supplyProviderType.buildTime() + 100 );  // frames to run

								// Order the builder to construct the supply structure
								supplyBuilder->build( supplyProviderType, targetBuildLocation );
							}
						}
						else
						{
							// Train the supply provider (Overlord) if the provider is not a structure
							supplyBuilder->train( supplyProviderType );
						}
					} // closure: supplyBuilder is valid
				} // closure: insufficient supply
			} // closure: failed to train idle unit

		}
	} // closure: unit iterator

=============== E X A M P L E   C O D E ===============
*/

#pragma warning( pop )