/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of GameState.h
*/

#include "CommonIncludes.h"

#ifdef ARMY_MANAGER_UCT

#ifdef MAASCRAFT_DEBUG
#include "DebugDrawing.h"
#endif

#include "BaseManager.h"
#include "Distances.h"
#include "GameState.h"
#include "MapGraphEdge.h"
#include "MapGraphNode.h"
#include "MathConstants.h"
#include "OpponentTracker.h"
#include "RNG.h"

#pragma warning( push )
#pragma warning( disable:4018 )
#pragma warning( disable:4244 )

using namespace BWAPI;
using std::vector;

GameState::GameState() : 
	scheduledActions(), battles(), selfSquads(), enemySquads(), selfBases(), enemyBases(),
	playerToMove(nullptr), frame(0), enemyReverseActions(0), selfReverseActions(0), terminate(false)
{}

GameState::~GameState()
{}

void GameState::applyAction(Action action)
{
	playerToMove = nullptr;

	if(action.type() == RESOLVE_ACTIONS)
	{
		resolveActions();
	}
	else
	{
		scheduledActions.insert(action);
		const int squadIndex = action.squadIndex();

#ifdef MAASCRAFT_DEBUG
		if(action.squadIndex() < 0)
		{
			LOG_MESSAGE(StringBuilder() << "frame = " << frame)
			LOG_MESSAGE(StringBuilder() << "num self squads = " << selfSquads.size())
			LOG_MESSAGE(StringBuilder() << "num enemy squads = " << enemySquads.size())
			LOG_MESSAGE(StringBuilder() << "num scheduled actions = " << scheduledActions.size())
		}
#endif

		if(squadIndex >= selfSquads.size())
		{
			enemySquads.at(squadIndex - selfSquads.size()).setIdle(false);

			if(action.type() == RETREAT_TO)
				enemySquads.at(squadIndex - selfSquads.size()).setRetreating(true);
		}
		else
		{
			selfSquads.at(squadIndex).setIdle(false);

			if(action.type() == RETREAT_TO)
				selfSquads.at(squadIndex).setRetreating(true);
		}
	}
}

void GameState::computeBattleDurations()
{
	for(auto it = battles.begin(); it != battles.end(); ++it)
	{
		SimBattle& battle = *it;
		battle.recomputeRemainingDuration(*this);
	}
}

void GameState::computeBattleLocations()
{
	battles.clear();

	// reset combat status of all squads
	for(size_t i = 0; i < selfSquads.size(); ++i)
	{
		SimSquad& squad = selfSquads.at(i);
		squad.setInCombat(false);
	}

	for(size_t i = 0; i < enemySquads.size(); ++i)
	{
		SimSquad& squad = enemySquads.at(i);
		squad.setInCombat(false);
	}

	// reset combat status of all bases
	for(size_t i = 0; i < selfBases.size(); ++i)
	{
		SimSquad& base = selfBases.at(i);
		base.setInCombat(false);
	}

	for(size_t i = 0; i < enemyBases.size(); ++i)
	{
		SimSquad& base = enemyBases.at(i);
		base.setInCombat(false);
	}

	// create new battles
	for(size_t i = 0; i < selfSquads.size(); ++i)
	{
		SimSquad& squad = selfSquads.at(i);

		if(squad.getAirHP() + squad.getGroundHP() <= 0)
			continue;

		if(!squad.isInCombat())		// this squad not added to a battle yet
		{
			MapGraphNode* node = squad.getGraphNode();
			SimBattle battle(node);
			battle.addAllySquad(i, *this);
			squad.setInCombat(true);
			
			if(frame == 0)
			{
				squad.setStartsInCombat(true);
			}

			for(size_t j = 0; j < enemySquads.size(); ++j)	// add opponents
			{
				SimSquad& enemy = enemySquads.at(j);

				if(enemy.getAirHP() + enemy.getGroundHP() <= 0)
					continue;

				MapGraphNode* enemyNode = enemy.getGraphNode();

				if(node == enemyNode)
				{
					battle.addEnemySquad(j, *this);
					enemy.setInCombat(true);

					if(frame == 0)
					{
						enemy.setStartsInCombat(true);
					}
				}
			}

			if(!battle.isFinished())	// found at least one opponent
			{
				for(size_t ii = i + 1; ii < selfSquads.size(); ++ii)	// add allies
				{
					SimSquad& squad_2 = selfSquads.at(ii);

					if(squad_2.getAirHP() + squad_2.getGroundHP() <= 0)
						continue;

					MapGraphNode* node_2 = squad_2.getGraphNode();

					if(node == node_2)
					{
						battle.addAllySquad(ii, *this);
						squad_2.setInCombat(true);

						if(frame == 0)
						{
							squad_2.setStartsInCombat(true);
						}
					}
				}

				battles.push_back(battle);
			}
			else		// found no opponent squads, so try opponent bases instead
			{
				for(size_t j = 0; j < enemyBases.size(); ++j)	// add opponent bases
				{
					SimSquad& base = enemyBases.at(j);

					if(base.getAirHP() + base.getGroundHP() <= 0)
						continue;

					MapGraphNode* enemyNode = base.getGraphNode();

					if(node == enemyNode)
					{
						battle.addEnemyBase(j, *this);
						base.setInCombat(true);

						if(frame == 0)
						{
							base.setStartsInCombat(true);
						}
					}
				}

				if(!battle.isFinished())	// found at least one base
				{
					for(size_t ii = i + 1; ii < selfSquads.size(); ++ii)	// add allies
					{
						SimSquad& squad_2 = selfSquads.at(ii);

						if(squad_2.getAirHP() + squad_2.getGroundHP() <= 0)
							continue;

						MapGraphNode* node_2 = squad_2.getGraphNode();

						if(node == node_2)
						{
							battle.addAllySquad(ii, *this);
							squad_2.setInCombat(true);

							if(frame == 0)
							{
								squad_2.setStartsInCombat(true);
							}
						}
					}

					battles.push_back(battle);
				}
				else
				{
					squad.setInCombat(false);

					if(frame == 0)
					{
						squad.setStartsInCombat(false);
					}
				}
			}
		}
	}
}

GameStateHitPoints GameState::computeHitPoints() const
{
	int enemyHP = 0;
	int selfHP = 0;

	// squads
	for(size_t i = 0; i < selfSquads.size(); ++i)
	{
		const SimSquad& squad = selfSquads.at(i);
		selfHP += squad.getTrueAirHP();
		selfHP += squad.getTrueGroundHP();
	}

	for(size_t i = 0; i < enemySquads.size(); ++i)
	{
		const SimSquad& squad = enemySquads.at(i);
		enemyHP += squad.getTrueAirHP();
		enemyHP += squad.getTrueGroundHP();
	}

	// bases
	for(size_t i = 0; i < selfBases.size(); ++i)
	{
		const SimSquad& base = selfBases.at(i);

		if(base.getTrueGroundHP() > 0)
			selfHP += base.getTrueGroundHP();
	}

	for(size_t i = 0; i < enemyBases.size(); ++i)
	{
		const SimSquad& base = enemyBases.at(i);

		if(base.getTrueGroundHP() > 0)
			enemyHP += base.getTrueGroundHP();
	}

	return GameStateHitPoints(enemyHP, selfHP);
}

const int GameState::getFrame() const
{
	return frame;
}

vector<Action> GameState::getActions() const
{
	for(size_t i = 0; i < selfSquads.size(); ++i)
	{
		const SimSquad& squad = selfSquads.at(i);

		if(!squad.isIdle() || (squad.getAirHP() + squad.getGroundHP() <= 0))
			continue;

		// found idle squad, so generate actions for this squad and return
		playerToMove = Broodwar->self();
		return getActionsAllied(squad, i);
	}

	// no more allied idle squads, so search enemy squads
	for(size_t i = 0; i < enemySquads.size(); ++i)
	{
		const SimSquad& squad = enemySquads.at(i);

		if(!squad.isIdle() || (squad.getAirHP() + squad.getGroundHP() <= 0))
			continue;

		// found idle squad, so generate actions for this squad and return
		playerToMove = Broodwar->enemy();
		return getActionsEnemy(squad, i + selfSquads.size());
	}

	// no idle squads at all, so return just the action indicating that all squads have chosen an action
	// and actions should now be resolved
	playerToMove = Broodwar->neutral();
	vector<Action> v;
	v.push_back(Action(nullptr, nullptr, RESOLVE_ACTIONS, 0, -1));

	return v;
}

vector<Action> GameState::getActionsAllied(const SimSquad& squad, const int index) const
{
	vector<Action> actions;

	MapGraphNode* node = squad.getGraphNode();

	if(squad.isInCombat())
	{
		if(squad.getTopSpeed() > 0.0 && BaseManager::Instance()->getSelfBases().size() > 0 && squad.allowRetreat())
		{
			vector<MapGraphEdge*> edges = node->getEdges();
			Position squadPos = squad.getPosition();

			// remove edges where enemy squads are closer to the destination than we are
			for(auto it = edges.begin(); it != edges.end(); /**/)
			{
				MapGraphEdge* edge = *it;
				Position posEdge = edge->getOther(node)->getPosition();

				const int distSq = Distances::getSquaredDistance(squadPos, posEdge);

				bool remove = false;
				for(auto it_enemy = enemySquads.begin(); it_enemy != enemySquads.end(); ++it_enemy)
				{
					const SimSquad& enemySquad = *it_enemy;
					Position enemyPos = enemySquad.getPosition();
					const int enemyDistSq = Distances::getSquaredDistance(enemyPos, posEdge);

					if(enemyDistSq < distSq)
					{
						remove = true;
						break;
					}
				}

				if(remove)
					it = edges.erase(it);
				else
					++it;
			}

			// only allow retreat to edge leading to our base
			Base base = BaseManager::Instance()->getSelfBases().front();
			Position basePos = base.getDepotCenter();

			int minDistSq = MathConstants::MAX_INT;
			MapGraphEdge* bestEdge = nullptr;

			for(auto it = edges.begin(); it != edges.end(); ++it)
			{
				MapGraphEdge* edge = *it;
				int distSq = Distances::getSquaredDistance(basePos, edge->getOther(node)->getPosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					bestEdge = edge;
				}
			}

			// create retreat action
			if(bestEdge)
			{
				int distance = 0;		// make retreat distance less if we're already a bit away from the enemy squads
				for(auto it = enemySquads.begin(); it != enemySquads.end(); ++it)
				{
					const SimSquad& enemy = *it;
					int potentialDist = 1.25 * bestEdge->getWalkPixelLength() - Distances::getDistance(squad.getPosition(), enemy.getPosition());
					distance = std::max(distance, potentialDist);
				}

				actions.push_back(Action(bestEdge, bestEdge->getOther(node), RETREAT_TO, distance / squad.getTopSpeed(), index));
			}
		}

		// finally, an action to simply keep fighting
		actions.push_back(Action(nullptr, node, FIGHT, -1, index));
	}
	else	// not in combat
	{
		if(squad.getTopSpeed() > 0.0)
		{
			const vector<MapGraphEdge*>& edges = node->getEdges();

			if(squad.flyersOnly())
			{
				for(auto it = edges.begin(); it != edges.end(); ++it)
				{
					MapGraphEdge* edge = *it;
					actions.push_back(Action(edge, edge->getOther(node), MOVE_TO, edge->getFlyPixelLength() / squad.getTopSpeed(), index));
				}
			}
			else
			{
				// action to directly go to our front squad (= squad furthest away from our bases)
				int front = -1;
				int minDist = MathConstants::MAX_INT;

				for(size_t i = 0; i < selfSquads.size(); ++i)
				{
					const SimSquad& candidateFront = selfSquads.at(i);

					int dist = 0;
					for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
					{
						const SimSquad& base = *it;
						dist += Distances::getDistance(candidateFront.getPosition(), base.getPosition());
					}

					if(dist < minDist)
					{
						minDist = dist;
						front = i;
					}
				}

				if(front >= 0 && front != index)
				{
					// an action to simply do nothing
					actions.push_back(Action(nullptr, node, DO_NOTHING, -1, index));

					// an action to move directly to front squad
					const SimSquad& frontSquad = selfSquads.at(front);
					actions.push_back(Action(nullptr, frontSquad.getGraphNode(), MOVE_TO, 
									(Distances::getDistance(frontSquad.getPosition(), squad.getPosition())) / squad.getTopSpeed(), index));

					// create actions to go to enemy squads directly
					for(auto it = enemySquads.begin(); it != enemySquads.end(); ++it)
					{
						const SimSquad& enemy = *it;
						actions.push_back(Action(nullptr, enemy.getGraphNode(), MOVE_TO, 
									(Distances::getDistance(enemy.getPosition(), squad.getPosition())) / squad.getTopSpeed(), index));
					}
				}
				else
				{
					// an action to simply do nothing
					actions.push_back(Action(nullptr, node, DO_NOTHING, -1, index));

					// create actions to go to adjacent nodes
					for(auto it = edges.begin(); it != edges.end(); ++it)
					{
						MapGraphEdge* edge = *it;
						actions.push_back(Action(edge, edge->getOther(node), MOVE_TO, edge->getWalkPixelLength() / squad.getTopSpeed(), index));
					}
				}
			}
		}
	}

	return actions;
}

vector<Action> GameState::getActionsEnemy(const SimSquad& squad, const int index) const
{
	vector<Action> actions;

	MapGraphNode* node = squad.getGraphNode();

	if(squad.isInCombat())
	{
		if(squad.getTopSpeed() > 0.0)
		{
			vector<MapGraphEdge*> edges = node->getEdges();
			Position squadPos = squad.getPosition();

			// remove edges where ours squads are closer to the destination than the enemy is
			for(auto it = edges.begin(); it != edges.end(); /**/)
			{
				MapGraphEdge* edge = *it;
				Position posEdge = edge->getOther(node)->getPosition();

				const int distSq = Distances::getSquaredDistance(squadPos, posEdge);

				bool remove = false;
				for(auto it_self = selfSquads.begin(); it_self != selfSquads.end(); ++it_self)
				{
					const SimSquad& selfSquad = *it_self;
					Position selfPos = selfSquad.getPosition();
					const int selfDistSq = Distances::getSquaredDistance(selfPos, posEdge);

					if(selfDistSq < distSq)
					{
						remove = true;
						break;
					}
				}

				if(remove)
					it = edges.erase(it);
				else
					++it;
			}

			// create retreat actions
			for(auto it = edges.begin(); it != edges.end(); ++it)
			{
				MapGraphEdge* edge = *it;
				actions.push_back(Action(edge, edge->getOther(node), RETREAT_TO, edge->getWalkPixelLength() / squad.getTopSpeed(), index));
			}
		}

		// finally, a action to simply keep fighting
		actions.push_back(Action(nullptr, node, FIGHT, -1, index));
	}
	else	// not in combat
	{
		// a action to simply do nothing
		actions.push_back(Action(nullptr, node, DO_NOTHING, -1, index));

		if(squad.getTopSpeed() > 0.0)
		{
			const vector<MapGraphEdge*>& edges = node->getEdges();

			// create actions to go to adjacent nodes
			for(auto it = edges.begin(); it != edges.end(); ++it)
			{
				MapGraphEdge* edge = *it;
				actions.push_back(Action(edge, edge->getOther(node), MOVE_TO, edge->getFlyPixelLength() / squad.getTopSpeed(), index));
			} 
		}
	}

	return actions;
}

Player GameState::getPlayerToMove() const
{
#ifdef MAASCRAFT_DEBUG
	if(!playerToMove)
	{
		LOG_WARNING("GameState::getPlayerToMove() called but playerToMove was not cached yet!")
	}
#endif

	return playerToMove;
}

Action GameState::getRandomAction() const
{
	vector<Action> actions = getActions();
	return actions.at(RNG::randomInt(0, actions.size() - 1));
}

const double GameState::getReward(GameState& startState, GameStateHitPoints& startStateHP) const
{
	const static double epsilon = 0.0000000000001;
	double baseDistancesScore = 0.0;
	double squadCoherenceScore = 0.0;

	int baseNumDifference = 0;

	const vector<SimSquad>& startStateSelfBases = startState.getSelfBases();
	const vector<SimSquad>& startStateEnemyBases = startState.getEnemyBases();

	for(auto it = startStateSelfBases.begin(); it != startStateSelfBases.end(); ++it)
	{
		const SimSquad& base = *it;
		baseNumDifference -= 1;

		// also add distance from each non-base squad to this base to the score in order to
		// reward moving away from our bases
		double maxAllowedDistance = MathConstants::MAX_DOUBLE;

		if(startStateEnemyBases.size() > 0)
		{
			const SimSquad& enemyBase = startStateEnemyBases.at(0);
			maxAllowedDistance = Distances::getDistance(enemyBase.getPosition(), base.getPosition());
		}

		for(auto it_squad = selfSquads.begin(); it_squad != selfSquads.end(); ++it_squad)
		{
			const SimSquad& squad = *it_squad;

			if(squad.isBase())
				continue;

			baseDistancesScore += epsilon * std::min(maxAllowedDistance, Distances::getDistance(base.getPosition(), squad.getPosition())) / startStateSelfBases.size();
		}
	}

	for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
	{
		const SimSquad& base = *it;

		if(base.getGroundHP() > 0)
		{
			baseNumDifference += 1;
		}
	}

	// determine front squad
	int front = -1;
	int minDist = MathConstants::MAX_INT;

	for(size_t i = 0; i < selfSquads.size(); ++i)
	{
		const SimSquad& candidateFront = selfSquads.at(i);

		int dist = 0;
		for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
		{
			const SimSquad& base = *it;
			dist += Distances::getDistance(candidateFront.getPosition(), base.getPosition());
		}

		if(dist < minDist)
		{
			minDist = dist;
			front = i;
		}
	}

	// compute squad coherence score
	if(front >= 0)
	{
		const SimSquad& frontSquad = selfSquads.at(front);

		for(size_t i = 0; i < selfSquads.size(); ++i)
		{
			if(i == front)
				continue;

			const SimSquad& other = selfSquads.at(i);

			squadCoherenceScore -= 1000.0 * epsilon * std::pow(Distances::getDistance((other).getPosition(), (frontSquad).getPosition()), 2);
		}
	}

	for(auto it = startStateEnemyBases.begin(); it != startStateEnemyBases.end(); ++it)
	{
		const SimSquad& base = *it;

		if(base.isBase())
			baseNumDifference++;
	}

	for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
	{
		const SimSquad& base = *it;

		if(base.getGroundHP() > 0)
			baseNumDifference--;
	}

	// subtract distance from each of our own non-base squads to start-state enemy bases to reward moving
	// towards enemy bases
	for(auto it = startState.getEnemyBases().begin(); it != startState.getEnemyBases().end(); ++it)
	{
		const SimSquad& base = *it;

		for(auto it_squad = selfSquads.begin(); it_squad != selfSquads.end(); ++it_squad)
		{
			const SimSquad& squad = *it_squad;

			baseDistancesScore -= 4.0 * epsilon * Distances::getDistance(base.getPosition(), squad.getPosition());
		}
	}

	GameStateHitPoints newHP = computeHitPoints();

	double selfHPRatio = (startStateHP.totalHPSelf == 0) ? 0.0 : double(newHP.totalHPSelf) / startStateHP.totalHPSelf;
	double enemyHPRatio = (startStateHP.totalHPEnemy == 0) ? 0.0 : double(newHP.totalHPEnemy) / startStateHP.totalHPEnemy;

	if(enemyHPRatio <= 0.0)
	{
		baseNumDifference++;
	}

	for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
	{
		const SimSquad& base = *it;
		
		if(base.getTrueGroundHP() <= 0)
		{
			selfHPRatio -= 1.0;
		}
	}

	for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
	{
		const SimSquad& base = *it;
		
		if(base.getTrueGroundHP() <= 0)
		{
			enemyHPRatio -= 1.0;
		}
	}

//	LOG_MESSAGE(StringBuilder() << "Self HP Ratio = " << newHP.totalHPSelf << " / " << startStateHP.totalHPSelf << " = " << selfHPRatio)
//	LOG_MESSAGE(StringBuilder() << "Enemy HP Ratio = " << newHP.totalHPEnemy << " / " << startStateHP.totalHPEnemy << " = " << enemyHPRatio)

	return baseNumDifference						+ 
			(selfHPRatio - enemyHPRatio)			+
			baseDistancesScore						+
			squadCoherenceScore						;
}

const bool GameState::isTerminal() const
{
	if(frame >= Options::ArmyManagerUCT::MAX_SIM_TIME)
		return true;
	else if(terminate)
		return true;
	else
		return false;
}

void GameState::resolveActions()
{
	// determine the shortest duration (either action or battle)
	int duration = -1;

	for(auto it = scheduledActions.begin(); it != scheduledActions.end(); ++it)
	{
		const Action& action = *it;
		if(action.durationRemaining() > 0)
		{
			duration = action.durationRemaining();
			break;
		}
	}

	computeBattleDurations();

	for(auto it = battles.begin(); it != battles.end(); ++it)
	{
		const SimBattle& battle = *it;

		const int remainingDuration = battle.getRemainingDuration();
		if(duration < 0 || remainingDuration < duration)
		{
			duration = remainingDuration;
		}

//		LOG_MESSAGE(StringBuilder() << "Battle at pos " << battle.getGraphNode()->getPosition() << " takes " << remainingDuration << " frames")
	}

	//LOG_MESSAGE(StringBuilder() << "Simming for " << duration << " frames")

	if(duration < 0)
	{
		terminate = true;
		return;
	}

	duration = std::min(duration, Options::ArmyManagerUCT::MAX_SIM_TIME - frame);

	// execute battles
	while(!battles.empty())
	{
		SimBattle& battle = battles.back();
		battle.simulateBattle(duration, *this);
		battles.pop_back();
	}

	// now execute actions
	for(auto it = scheduledActions.begin(); it != scheduledActions.end(); ++it)
	{
		const Action& action = *it;
		action.setDurationRemaining(action.durationRemaining() - duration);

		if(action.type() == MOVE_TO || action.type() == RETREAT_TO)
		{
			int squadIndex = action.squadIndex();
			bool allied = true;
			if(squadIndex >= selfSquads.size())
			{
				squadIndex -= selfSquads.size();
				allied = false;
			}

			SimSquad& squad = allied ? selfSquads.at(squadIndex) : enemySquads.at(squadIndex);

			if(squad.isInCombat() && action.type() == MOVE_TO)
				continue;		// can't actually move if it's not a retreat action and we are in combat

			if(action.durationRemaining() <= 0)
			{
				if(action.destination() == squad.getPreviousNode())		// reverse action!
				{
					if(allied)
						selfReverseActions++;
					else
						enemyReverseActions++;
				}

				squad.setPreviousNode(squad.getGraphNode());
				squad.setGraphNode(action.destination());
				Position destPos = action.destination()->getPosition();

				int dx = squad.getPosition().x - destPos.x;
				if(dx < 0)
					dx = -1;
				if(dx > 0)
					dx = 1;

				int dy = squad.getPosition().y - destPos.y;
				if(dy < 0)
					dy = -1;
				if(dy > 0)
					dy = 1;

				// add a Position with small dx and dy so we have some idea of which direction the squad came from
				squad.setPosition(action.destination()->getPosition() + Position(dx, dy));
			}
			else
			{
				Position currentPos = squad.getPosition();
				Position destination = action.destination()->getPosition();
				float ratio = (float(duration) / (duration + action.durationRemaining()));

				Position translation = (destination - currentPos);
				translation *= ratio;

				squad.setPosition(currentPos + translation);
			}
		}
	}

	// remove actions with <= 0 duration remaining and set squads of those actions back to idle
	while(!scheduledActions.empty())
	{
		const Action& action = *(scheduledActions.begin());
		if(action.durationRemaining() <= 0)
		{			
			int squadIndex = action.squadIndex();
			bool allied = true;
			if(squadIndex >= selfSquads.size())
			{
				squadIndex -= selfSquads.size();
				allied = false;
			}

			SimSquad& squad = allied ? selfSquads.at(squadIndex) : enemySquads.at(squadIndex);

			squad.setIdle(true);

			if(action.type() == RETREAT_TO)
				squad.setRetreating(false);

			scheduledActions.erase(scheduledActions.begin());
		}
		else
		{
			break;
		}
	}

	// re-compute battle locations
	computeBattleLocations();

	frame += duration;
}

vector<SimSquad>& GameState::getSelfSquads()
{
	return selfSquads;
}

vector<SimSquad>& GameState::getEnemySquads()
{
	return enemySquads;
}

vector<SimSquad>& GameState::getSelfBases()
{
	return selfBases;
}

vector<SimSquad>& GameState::getEnemyBases()
{
	return enemyBases;
}

void GameState::reset(const vector<Unitset>& selfAirSquads, const vector<Unitset>& selfGroundSquads)
{
	OpponentTracker* opponentTracker = OpponentTracker::Instance();

	selfSquads.clear();
	enemySquads.clear();
	selfBases.clear();
	enemyBases.clear();
	battles.clear();

	// Create our own squads
	for(auto it = selfAirSquads.begin(); it != selfAirSquads.end(); ++it)
	{
		const Unitset& units = *it;

		selfSquads.push_back(SimSquad(units, true));
	}

	for(auto it = selfGroundSquads.begin(); it != selfGroundSquads.end(); ++it)
	{
		const Unitset& units = *it;

		selfSquads.push_back(SimSquad(units, false));
	}

	// Create enemy Squads
	const vector<Unitset>& opponentSquads = OpponentTracker::Instance()->getOpponentClusters();
	for(auto it = opponentSquads.begin(); it != opponentSquads.end(); ++it)
	{
		const Unitset& units = *it;

		if(units.size() == 1 && OpponentTracker::Instance()->getLastType(units.front()).isWorker())
			continue;

		enemySquads.push_back(SimSquad(*it, false));
	}

	BaseManager* baseManager = BaseManager::Instance();

	// Create self bases
	const vector<Base>& myBases = baseManager->getSelfBases();

	for(auto it = myBases.begin(); it != myBases.end(); ++it)
	{
		selfBases.push_back(SimSquad(*it));
	}

	// Create enemy bases
	const vector<Base>& opponentBases = baseManager->getOpponentBases();

	if(opponentBases.size() > 0)
	{
		for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
		{
			Base base = *it;
			const Unitset& buildings = base.getBuildings();
			bool legitBuildingFound = false;

			for(auto building = buildings.begin(); building != buildings.end(); ++building)
			{
				Unit u = *building;

				if(!u->isVisible() && !Broodwar->isVisible(PositionUtils::toTilePosition(opponentTracker->getLastPosition(u))))
				{
					legitBuildingFound = true;
					break;
				}
				else if(u->getHitPoints() > 0)
				{
					legitBuildingFound = true;
					break;
				}
			}

			if(legitBuildingFound && !base.purelyDefensive())
			{
				enemyBases.push_back(SimSquad(*it));
			}
		}

		if(enemyBases.empty())		// means the only bases we have observed are purely defensive
		{
			for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
			{
				enemyBases.push_back(SimSquad(*it));
			}
		}
	}
	else		// no bases observed, place a dummy base at each unobserved start position
	{
		const vector<BaseLocation*>& baseLocs = MapAnalyser::Instance()->getBaseLocations();

		for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
		{
			BaseLocation* loc = *it;

			if(!loc->isStartLocation())
				continue;

			const int lastObserved = loc->getLastObserved();

			if(lastObserved < 5)
			{
				enemyBases.push_back(SimSquad(Base(loc, Broodwar->enemy())));
			}
		}

		if(enemyBases.empty())		// means all start positions have been scouted already
		{
			for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
			{
				BaseLocation* loc = *it;

				const int lastObserved = loc->getLastObserved();

				if(lastObserved < 20000)
				{
					enemyBases.push_back(SimSquad(Base(loc, Broodwar->enemy())));
				}
			}
		}
	}

	computeBattleLocations();

	terminate = false;
	frame = 0;
	enemyReverseActions = 0;
	selfReverseActions = 0;
}

#ifdef MAASCRAFT_DEBUG
void GameState::dumpInfo() const
{
	LOG_MESSAGE("============= GAME STATE =============")
	LOG_MESSAGE("SELF SQUADS:")
	for(auto it = selfSquads.begin(); it != selfSquads.end(); ++it)
	{
		const SimSquad& squad = *it;

		if(squad.getAirHP() > 0 || squad.getGroundHP() > 0)
		{
			LOG_MESSAGE("Self Squad")
			LOG_MESSAGE(StringBuilder() << "Ground HP = " << squad.getGroundHP())
			LOG_MESSAGE(StringBuilder() << "Position = " << PositionUtils::toWalkPosition(squad.getPosition()))
			LOG_MESSAGE("")
		}
	}
	LOG_MESSAGE("SELF BASES:")
	for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
	{
		const SimSquad& base = *it;

		if(base.getAirHP() > 0 || base.getGroundHP() > 0)
		{
			LOG_MESSAGE("Self Base")
			LOG_MESSAGE(StringBuilder() << "Ground HP = " << base.getGroundHP())
			LOG_MESSAGE(StringBuilder() << "Position = " << PositionUtils::toWalkPosition(base.getPosition()))
			LOG_MESSAGE("")
		}
	}
	LOG_MESSAGE("ENEMY SQUADS:")
	for(auto it = enemySquads.begin(); it != enemySquads.end(); ++it)
	{
		const SimSquad& squad = *it;

		if(squad.getAirHP() > 0 || squad.getGroundHP() > 0)
		{
			LOG_MESSAGE("Enemy Squad")
			LOG_MESSAGE(StringBuilder() << "Ground HP = " << squad.getGroundHP())
			LOG_MESSAGE(StringBuilder() << "Position = " << PositionUtils::toWalkPosition(squad.getPosition()))
			LOG_MESSAGE("")
		}
	}
	LOG_MESSAGE("ENEMY BASES:")
	for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
	{
		const SimSquad& base = *it;

		if(base.getAirHP() > 0 || base.getGroundHP() > 0)
		{
			LOG_MESSAGE("Enemy Base")
			LOG_MESSAGE(StringBuilder() << "Ground HP = " << base.getGroundHP())
			LOG_MESSAGE(StringBuilder() << "Position = " << PositionUtils::toWalkPosition(base.getPosition()))
			LOG_MESSAGE("")
		}
	}
	LOG_MESSAGE("")
}

void GameState::visualize() const
{
	DebugDrawing::ColorIterator colorIterator;

	for(auto it = selfSquads.begin(); it != selfSquads.end(); ++it)
	{
		SimSquad simSquad = *it;

		Color color = colorIterator.nextColor();
		DebugDrawing::drawBoxesAroundUnits(simSquad.getUnits(), color);

		Broodwar->drawCircleMap(simSquad.getGraphNode()->getPosition(), 10, color, true);
	}

	for(auto it = enemySquads.begin(); it != enemySquads.end(); ++it)
	{
		SimSquad simSquad = *it;

		Broodwar->drawCircleMap(simSquad.getGraphNode()->getPosition() + Position(5, 0), 10, Colors::Black, true);
	}
}
#endif

#pragma warning( pop )

#endif // ARMY_MANAGER_UCT