/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

#include "CommonIncludes.h"

#include <cmath>
#include "Point2D.h"

#pragma warning( push )
#pragma warning( disable: 4244 )

/*
	Convenience functions for calculating and comparing distances in a 2-dimensional Euclidean space

	All functions are relatively short, so they will be implemented in header file to allow
	for increased optimization during compilation by inlining.
*/

namespace Distances
{
	static const double getDistance(const BWAPI::Position& p1, const BWAPI::Position& p2);
	static const double getDistance(const double x1, const double y1, const double x2, const double y2);
	static const double getDistance(const Point2D& p1, const Point2D& p2);
	static const float getDistance(const BWAPI::Unit u1, const BWAPI::Unit u2);
	static const float getDistanceF(const float x1, const float y1, const float x2, const float y2);
	static const float getDistanceF(const Point2D& p1, const Point2D& p2);

	static const double getSquaredDistance(const double x1, const double y1, const double x2, const double y2);
	static const double getSquaredDistance(const Point2D& p1, const Point2D& p2);
	static const int getSquaredDistance(const BWAPI::Position& p1, const BWAPI::Position& p2);
	static const int getSquaredDistance(const BWAPI::Position& p, const BWAPI::Unit u);
	static const int getSquaredDistance(const BWAPI::TilePosition& tp1, const BWAPI::TilePosition& tp2);
	static const int getSquaredDistance(const BWAPI::Unit u1, const BWAPI::Unit u2);
	static const float getSquaredDistanceF(const float x1, const float y1, const float x2, const float y2);
	static const float getSquaredDistanceF(const Point2D& p1, const Point2D& p2);

	static const double getManhattanDistance(const double x1, const double y1, const double x2, const double y2);
	static const double getManhattanDistance(const Point2D& p1, const Point2D& p2);
	static const int getManhattanDistance(const BWAPI::TilePosition& tp1, const BWAPI::TilePosition& tp2);
	static const float getManhattanDistanceF(const float x1, const float y1, const float x2, const float y2);
	static const float getManhattanDistanceF(const Point2D& p1, const Point2D& p2);
	static const int getManhattanDistanceInt(const int x1, const int y1, const int x2, const int y2);

	static const bool withinDistance(const double x1, const double y1,
									const double x2, const double y2,
									const double maxDistance				);

	static const bool withinDistance(const Point2D& p1, const Point2D& p2,
									const double maxDistance				);

	static const bool withinSquaredDistance(const double x1, const double y1,
											const double x2, const double y2,
											const double squaredMaxDistance		);

	static const bool withinSquaredDistanceInt(const int x1, const int y1,
												const int x2, const int y2,
												const int squaredMaxDistance		);

	static const bool withinSquaredDistance(const Point2D& p1, const Point2D& p2,
											const double squaredMaxDistance		);

	static const bool withinSquaredDistance(const BWAPI::Position& p1, const BWAPI::Position& p2,
											const int squaredMaxDistance							);
}

/*
		Implementations of the above declared functions
*/

inline const double Distances::getDistance(const BWAPI::Position& p1, const BWAPI::Position& p2)
{
	const int dx = p2.x - p1.x;
	const int dy = p2.y - p1.y;
	return std::sqrt(double(dx*dx + dy*dy));
}

inline const double Distances::getDistance(const double x1, const double y1, 
											const double x2, const double y2)
{
	const double dx = x2 - x1;
	const double dy = y2 - y1;
	return std::sqrt(dx*dx + dy*dy);
}

inline const double Distances::getDistance(const Point2D& p1, const Point2D& p2)
{
	const double dx = p2.x - p1.x;
	const double dy = p2.y - p1.y;
	return std::sqrt(dx*dx + dy*dy);
}

inline const float Distances::getDistance(const BWAPI::Unit u1, const BWAPI::Unit u2)
{
	int dx = u1->getLeft() - u2->getRight();
	if(dx < 0)
		dx = u2->getLeft() - u1->getRight();

	int dy = u1->getTop() - u2->getBottom();
	if(dy < 0)
		dy = u2->getTop() - u1->getBottom();

	return std::sqrt(float(dx*dx) + dy*dy);
}

inline const float Distances::getDistanceF(const float x1, const float y1, 
											const float x2, const float y2)
{
	const float dx = x2 - x1;
	const float dy = y2 - y1;
	return std::sqrt(dx*dx + dy*dy);
}

inline const float Distances::getDistanceF(const Point2D& p1, const Point2D& p2)
{
	const float dx = p2.x - p1.x;
	const float dy = p2.y - p1.y;
	return std::sqrt(dx*dx + dy*dy);
}

inline const double Distances::getSquaredDistance(const double x1, const double y1, const double x2, const double y2)
{
	const double dx = x2 - x1;
	const double dy = y2 - y1;
	return (dx*dx + dy*dy);
}

inline const double Distances::getSquaredDistance(const Point2D& p1, const Point2D& p2)
{
	const double dx = p2.x - p1.x;
	const double dy = p2.y - p1.y;
	return (dx*dx + dy*dy);
}

inline const int Distances::getSquaredDistance(const BWAPI::Position& p1, const BWAPI::Position& p2)
{
	const int dx = p2.x - p1.x;
	const int dy = p2.y - p1.y;
	return (dx*dx + dy*dy);
}

const int Distances::getSquaredDistance(const BWAPI::Position& p, const BWAPI::Unit u)
{
	const int px = p.x;
	const int py = p.y;

	int dx = px - u->getRight();
	if(dx < 0)
		dx = u->getLeft() - px;

	int dy = py - u->getBottom();
	if(dy < 0)
		dy = u->getTop() - py;

	return (dx*dx + dy*dy);
}

const int Distances::getSquaredDistance(const BWAPI::TilePosition& tp1, const BWAPI::TilePosition& tp2)
{
	const int dx = tp2.x - tp1.x;
	const int dy = tp2.y - tp1.y;
	return (dx*dx + dy*dy);
}

const int Distances::getSquaredDistance(const BWAPI::Unit u1, const BWAPI::Unit u2)
{
	int dx = u1->getLeft() - u2->getRight();
	if(dx < 0)
		dx = u2->getLeft() - u1->getRight();

	int dy = u1->getTop() - u2->getBottom();
	if(dy < 0)
		dy = u2->getTop() - u1->getBottom();

	return (dx*dx + dy*dy);
}

inline const float Distances::getSquaredDistanceF(const float x1, const float y1, const float x2, const float y2)
{
	const float dx = x2 - x1;
	const float dy = y2 - y1;
	return (dx*dx + dy*dy);
}

inline const float Distances::getSquaredDistanceF(const Point2D& p1, const Point2D& p2)
{
	const float dx = p2.x - p1.x;
	const float dy = p2.y - p1.y;
	return (dx*dx + dy*dy);
}

inline const double Distances::getManhattanDistance(const double x1, const double y1, const double x2, const double y2)
{
	return (std::abs(x2 - x1) + std::abs(y2 - y1));
}

inline const double Distances::getManhattanDistance(const Point2D& p1, const Point2D& p2)
{
	return (std::abs(p2.x - p1.x) + std::abs(p2.y - p1.y));
}

inline const int getManhattanDistance(const BWAPI::TilePosition& tp1, const BWAPI::TilePosition& tp2)
{
	return (std::abs(tp2.x - tp1.x) + std::abs(tp2.y - tp1.y));
}

inline const float Distances::getManhattanDistanceF(const float x1, const float y1, const float x2, const float y2)
{
	return (std::abs(x2 - x1) + std::abs(y2 - y1));
}

inline const float Distances::getManhattanDistanceF(const Point2D& p1, const Point2D& p2)
{
	return (std::abs(p2.x - p1.x) + std::abs(p2.y - p1.y));
}

inline const int Distances::getManhattanDistanceInt(const int x1, const int y1, const int x2, const int y2)
{
	return (std::abs(x2 - x1) + std::abs(y2 - y1));
}

inline const bool Distances::withinDistance(const double x1, const double y1,
											const double x2, const double y2, const double maxDistance)
{
	const double dx = x2 - x1;
	const double dy = y2 - y1;
	return ((dx*dx + dy*dy) <= (maxDistance*maxDistance));
}

inline const bool Distances::withinDistance(const Point2D& p1, const Point2D& p2, const double maxDistance)
{
	const double dx = p2.x - p1.x;
	const double dy = p2.y - p1.y;
	return ((dx*dx + dy*dy) <= (maxDistance*maxDistance));
}

inline const bool Distances::withinSquaredDistance(const double x1, const double y1,
													const double x2, const double y2, const double squaredMaxDistance)
{
	const double dx = x2 - x1;
	const double dy = y2 - y1;
	return ((dx*dx + dy*dy) <= squaredMaxDistance);
}

inline const bool Distances::withinSquaredDistanceInt(const int x1, const int y1,
														const int x2, const int y2, const int squaredMaxDistance)
{
	const int dx = x2 - x1;
	const int dy = y2 - y1;
	return ((dx*dx + dy*dy) <= squaredMaxDistance);
}

inline const bool Distances::withinSquaredDistance(const Point2D& p1, const Point2D& p2, const double squaredMaxDistance)
{
	const double dx = p2.x - p1.x;
	const double dy = p2.y - p1.y;
	return ((dx*dx + dy*dy) <= squaredMaxDistance);
}

inline const bool Distances::withinSquaredDistance(const BWAPI::Position& p1, const BWAPI::Position& p2, const int squaredMaxDistance)
{
	const int dx = p2.x - p1.x;
	const int dy = p2.y - p1.y;
	return ((dx*dx + dy*dy) <= squaredMaxDistance);
}

#pragma warning( pop )