/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of DebugDrawing.h
*/

#include "CommonIncludes.h"

#include "DebugDrawing.h"

#ifdef MAASCRAFT_DEBUG

#include "ArrayUtils.h"
#include "PathFinder.h"
#include "PositionState.h"
#include "PositionUtils.h"

using namespace BWAPI;

/*
	Drawing functions
*/
void DebugDrawing::colorPosition(const Position& pos, const Color& color)
{
	Broodwar->drawBoxMap(pos, pos, color, true);
}

void DebugDrawing::colorTilePosition(const TilePosition& pos, const Color& color)
{
	Broodwar->drawBoxMap(PositionUtils::leftTop(pos), PositionUtils::rightBot(pos), color, true);
}

void DebugDrawing::colorWalkPosition(const WalkPosition& pos, const Color& color)
{
	Broodwar->drawBoxMap(PositionUtils::leftTop(pos), PositionUtils::rightBot(pos), color, true);
}

void DebugDrawing::drawBoxAroundUnit(const Unit unit, const Color& color)
{
	Broodwar->drawBoxMap(unit->getLeft(), unit->getTop(), unit->getRight(), unit->getBottom(), color);
}

void DebugDrawing::drawBoxesAroundUnits(const Unitset& unitset, const Color& color)
{
	for(auto it = unitset.begin(); it != unitset.end(); ++it)
	{
		drawBoxAroundUnit(*it, color);
	}
}

void DebugDrawing::drawLineMap(const BWAPI::Position& pos1, const BWAPI::Position& pos2, const BWAPI::Color color)
{
	Broodwar->drawLineMap(pos1, pos2, color);
}

void DebugDrawing::drawPath(std::forward_list<PositionState>& path, const BWAPI::Color& color)
{
	if(!path.empty())
	{
		PositionState firstState = path.front();
		path.pop_front();
		Position first = PositionUtils::toPosition(firstState.getWalkPosition());

		PositionState nextState = path.front();
		path.pop_front();
		Position next = PositionUtils::toPosition(nextState.getWalkPosition());

		while(!path.empty())
		{
			Broodwar->drawLineMap(first, next, color);

			first = next;
			nextState = path.front();
			path.pop_front();
			next = PositionUtils::toPosition(nextState.getWalkPosition());
		}
	}
}

void DebugDrawing::drawUnitHealthBar(const Unit unit)
{
	if(!unit -> isVisible())
		return;

	int maxHealth = unit->getType().maxHitPoints() + unit->getType().maxShields();
	int health = unit->getHitPoints() + unit->getShields();

	Color barColor = Colors::Green;
	if (unit->isUnderAttack())
		barColor = Colors::Yellow;

	Broodwar->drawBoxMap(	unit->getPosition().x - 7 + (int) (20.0 * health / maxHealth), 
							unit->getPosition().y + 7,
							unit->getPosition().x - 7 + 20,
							unit->getPosition().y + 10,
							Colors::Orange, true);

	Broodwar->drawBoxMap(	unit->getPosition().x - 7, 
							unit->getPosition().y + 7,
							unit->getPosition().x - 7 + (int) (20.0 * health / maxHealth),
							unit->getPosition().y + 10,
							barColor, true);
}

void DebugDrawing::linkUnits(const BWAPI::Unit u1, const BWAPI::Unit u2, const BWAPI::Color& color)
{
	Broodwar->drawLineMap(u1->getPosition(), u2->getPosition(), color);
}

void DebugDrawing::linkUnitsToPos(const BWAPI::Position pos, const BWAPI::Unitset& units, const BWAPI::Color& color)
{
	for(auto it = units.begin(); it != units.end(); ++it)
	{
		Broodwar->drawLineMap((*it)->getPosition(), pos, color);
	}
}

void DebugDrawing::linkUnitsToRoot(const BWAPI::Unit root, const BWAPI::Unitset& units, const BWAPI::Color& color)
{
	for(auto it = units.begin(); it != units.end(); ++it)
	{
		linkUnits(*it, root, color);
	}
}

/*
	Color Iterator stuff
*/
const BWAPI::Color DebugDrawing::ColorIterator::colors[] = 
	{ 
		BWAPI::Colors::Black,
		BWAPI::Colors::Blue,
		BWAPI::Colors::Brown,
		BWAPI::Colors::Cyan,
		BWAPI::Colors::Green,
		BWAPI::Colors::Grey,
		BWAPI::Colors::Orange,
		BWAPI::Colors::Purple,
		BWAPI::Colors::Red,
		BWAPI::Colors::Teal,
		BWAPI::Colors::White,
		BWAPI::Colors::Yellow
	};

DebugDrawing::ColorIterator::ColorIterator() :
	index(0) {}

DebugDrawing::ColorIterator::~ColorIterator()	{}

const BWAPI::Color DebugDrawing::ColorIterator::nextColor()
{
	const BWAPI::Color return_val = colors[index];
	index++;

	if(index > ArrayUtils::getArraySize(colors))
		index = 0;

	return return_val;
}

#else

void DebugDrawing::drawLineMap(const BWAPI::Position& pos1, const BWAPI::Position& pos2, const BWAPI::Color color)
{}

#endif	// MAASCRAFT_DEBUG