/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

/*
	Interface to the Build Manager Singleton.
*/

#include "CommonIncludes.h"

#include <vector>

#include "MapAnalyser.h"
#include "MapRegion.h"
#include "UnitOwner.h"

struct TileScoreFunction
{
	virtual const int operator()(BWAPI::TilePosition& tilePos) = 0;
};

class BuildingManager : public UnitOwner
{
public:
	static BuildingManager* Instance()
	{
		static BuildingManager instance;
		static BuildingManager* instance_ptr = &instance;
		return instance_ptr;
	}

	/*
		Returns true iff a ''building'' of type ''buildingType'' can be built at location specified where ''position'' is the top left corner.
	
		If ''builder'' is not nullptr and is a pointer to a Worker, that specific unit will be ignored when determining whether or not
		he blocks the building location.

		If ''exploredOnly'' is set to true, the function will return false if the location contains any unexplored tiles
	*/
	const bool canBuildAtTile(const BWAPI::UnitType buildingType, const BWAPI::TilePosition& position, 
								const BWAPI::Unit builder = nullptr, const bool exploredOnly = false) const;

	/*
		Similar to above, but only uses static map data. Therefore also does not care about who builds and exploration.
	*/
	const bool canBuildAtTileStatic(const BWAPI::UnitType buildingType, const BWAPI::TilePosition& position) const;

	// spiral search to find a valid build location near a specified tile
	const BWAPI::TilePosition getBuildLocationNear(const BWAPI::TilePosition& position, const BWAPI::UnitType buildingType, 
													const MapRegion* preferredRegion										) const;

	const BWAPI::TilePosition getBuildLocationAtDistance(const BWAPI::TilePosition& position, const BWAPI::UnitType buildingType, 
															const MapRegion* preferredRegion, const int minDist, const int maxDist) const;

	const bool isAllowedInResourceBox(const BWAPI::UnitType buildingType) const;
	const bool isTileInResourceBox(const int x, const int y) const;
	const bool isTileInResourceBox(const BWAPI::TilePosition& tilePos) const;
	const bool isTileReserved(const int x, const int y) const;
	const bool isTileReserved(const BWAPI::TilePosition& tilePos) const;

	void freeReservedSpaceForBuilding(const BWAPI::Unit building);
	void freeReservedSpaceForBuilding(const BWAPI::UnitType buildingType, const BWAPI::TilePosition& position);
	void reserveSpaceForBuilding(const BWAPI::UnitType buildingType, const BWAPI::TilePosition& position);

	const int getNumPylonsBuilding() const;

	// EVENTS
	void onBuildingDestroy(BWAPI::Unit building);
	void onFrame();
	void onPylonFinishedBuilding();
	void onPylonStartBuilding();
	void onStart();

private:
	BuildingManager();
	~BuildingManager();

	short* reserved;
	bool* resourceBox;

	int pylonsBuilding;

	static const short NOT_RESERVED = 0;
	static const short RESERVED = 1;

	BuildingManager(BuildingManager const &);
};