/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include <deque>
#include <vector>

#include "ArrayUtils.h"
#include "BaseLocation.h"
#include "BuildingManager.h"
#include "Distances.h"
#include "MapAnalyser.h"
#include "MapRegion.h"
#include "MathConstants.h"
#include "Point2D.h"
#include "UnitTracker.h"

#pragma warning( push )
#pragma warning( disable:4244)

/*
	Implementation of the BuildingManager
*/

using namespace BWAPI;
using namespace std;

BuildingManager::BuildingManager()
{
	const int width = Broodwar->mapWidth();
	const int height = Broodwar->mapHeight();

	reserved = new short[width*height];
	ArrayUtils::initializeShortArray(reserved, width*height, NOT_RESERVED);

	resourceBox = new bool[width*height];
	ArrayUtils::initializeBoolArray(resourceBox, width*height, false);

	pylonsBuilding = 0;
}

BuildingManager::~BuildingManager()
{
	delete[] reserved;
	delete[] resourceBox;
}

void BuildingManager::onBuildingDestroy(Unit building)
{
	freeReservedSpaceForBuilding(building);
}

void BuildingManager::onFrame()
{
}

void BuildingManager::onPylonFinishedBuilding()
{
	pylonsBuilding--;
}

void BuildingManager::onPylonStartBuilding()
{
	pylonsBuilding++;
}

void BuildingManager::onStart()
{
	// Mark straight paths from depots to resource nodes as resource boxes
	const vector<BaseLocation*>& baseLocations = MapAnalyser::Instance()->getBaseLocations();
	
	for(auto it_base_loc = baseLocations.begin(); it_base_loc != baseLocations.end(); ++it_base_loc)
	{
		BaseLocation* baseLoc = *it_base_loc;
		TilePosition depotLeftTop = baseLoc->getOptimalDepotLocation();
		const int depotLeft = depotLeftTop.x;
		const int depotTop = depotLeftTop.y;
		const int depotRight = depotLeft + UnitTypes::Protoss_Nexus.tileWidth() + 1;
		const int depotBot = depotTop + UnitTypes::Protoss_Nexus.tileHeight() + 1;

		const int depotX = (depotLeft + depotRight) / 2;
		const int depotY = (depotTop + depotBot) / 2;

		const Unitset& resources = baseLoc->getResources();

		for(auto it_resource = resources.begin(); it_resource != resources.end(); ++it_resource)
		{
			Unit resource = *it_resource;

			TilePosition initialPos = resource->getInitialTilePosition();
			const int left = initialPos.x;
			const int right = initialPos.x + resource->getType().dimensionRight() / 16;
			const int top = initialPos.y;
			const int bottom = initialPos.y + resource->getType().dimensionDown() / 16;

			const int resourceX = (left + right) / 2;
			const int resourceY = (top + bottom) / 2;

			const int dx = depotX - resourceX;
			const int dy = depotY - resourceY;

			double x = resourceX;
			double y = resourceY;

			Vector2D dir = Vector2D(dx, dy);
			dir.normalize();

			while(x < depotLeft || x > depotRight || y < depotTop || y > depotBot)
			{
				x += dir.x;
				y += dir.y;

				ArrayUtils::setMatrixValue(resourceBox, y, x, Broodwar->mapWidth(), true);
			}
		}
	}

	// reserve space around starting buildings
	const Unitset& buildings = UnitTracker::Instance()->getSelfBuildings();

	for(auto it = buildings.begin(); it != buildings.end(); ++it)
	{
		Unit building = *it;
		reserveSpaceForBuilding(building->getType(), building->getTilePosition());
	}
}

/*
	Implementation of this function mostly based on implementation of the similar canBuildHere() function from
	BWAPI.

	Slightly optimized by removing some checks which I never expect to need for this bot specifically, for example;
		the bot never expects to play Zerg or Terran, always play only Protoss. This assumption allows to skip checks 
			related to creep (Zerg) and Addons (Terran)
		the bot does not expect to try building a Building of type ''Special_Start_Location''

	This method will also consider extra space requirements around buildings which need to produce units, and disallow
	building on locations which have been marked as ''reserved'' previously by the BuildingManager

*/
const bool BuildingManager::canBuildAtTile(const UnitType buildingType, const TilePosition& position, 
											const Unit builder, const bool exploredOnly					) const

{
	TilePosition leftTopExtraSpace = (position - TilePosition(1, 1)).makeValid();
	TilePosition leftTop = position;
	TilePosition rightBot = leftTop + buildingType.tileSize();
	TilePosition rightBotExtraSpace = (rightBot + TilePosition(1, 1)).makeValid();

	// Map limit check
	if ( !leftTop.isValid() || !rightBot.isValid() )
		return false;

	// If builder is not a nullptr, check if he can reach this location
	if(builder)
	{
		if(!MapAnalyser::Instance()->pathExists(PositionUtils::toWalkPosition(builder->getPosition()), 
												PositionUtils::toWalkPosition(leftTop)))
			return false;
	}

	if(buildingType.isRefinery())
	{
		// In this case, we want to build a refinery. This means we need to make sure we're building on a Geyser
		Unitset geysers = Broodwar->getGeysers();

		for(auto geyser = geysers.begin(); geyser != geysers.end(); ++geyser)
		{
			if(geyser->getTilePosition() == position)
			{
				if(geyser->isVisible() && geyser->getType() != UnitTypes::Resource_Vespene_Geyser)
					return false;

				return true;
			}
		}
		return false;
	}

	// Since we're playing Protoss, we'll have to check for Power/Pylons
	if(buildingType.requiresPsi() && !Broodwar->hasPower(leftTop, buildingType))
	{
		return false;
	}

	// In this case, we're not building a refinery. We must check collision stuff
	// I assume this bot will never play Zerg, so can also immediately incorporate creep check here
	for(int x = leftTop.x; x < rightBot.x; ++x)
	{
		for(int y = leftTop.y; y < rightBot.y; ++y)
		{
			if(	!Broodwar->isBuildable(x, y, true)					|| 
				(exploredOnly && !Broodwar->isExplored(x, y))		|| 
				Broodwar->hasCreep(x, y)							|| 
				isTileReserved(x, y)								||
				isTileInResourceBox(x, y)								)
			{
				return false;
			}
		}
	}

	for(int x = leftTopExtraSpace.x; x < rightBotExtraSpace.x; ++x)
	{
		if(!Broodwar->isBuildable(x, leftTopExtraSpace.y, true))
			return false;

		if(!Broodwar->isBuildable(x, rightBotExtraSpace.y, true))
			return false;
	}

	for(int y = leftTopExtraSpace.y; y < rightBotExtraSpace.y; ++y)
	{
		if(!Broodwar->isBuildable(leftTopExtraSpace.x, y, true))
			return false;

		if(!Broodwar->isBuildable(rightBotExtraSpace.x, y, true))
			return false;
	}

	// Check for ground units
	Position targetPos = Position(leftTop) + Position(buildingType.tileSize()) / 2;
	Unitset unitsInRect(Broodwar->getUnitsInRectangle(Position(leftTop), Position(rightBot), 
														!Filter::IsFlyer												&&
														!Filter::IsLoaded												&&
														[&builder, &buildingType](Unit u) {return u != builder;}		&&
														Filter::GetLeft <= targetPos.x + buildingType.dimensionRight()	&&
														Filter::GetTop <= targetPos.y + buildingType.dimensionDown()	&&
														Filter::GetRight >= targetPos.x - buildingType.dimensionDown()	&&
														Filter::GetBottom >= targetPos.y - buildingType.dimensionUp()		));

	if(unitsInRect.size() > 0)
		return false;

	// Resource Depots cannot be placed too close to Minerals
	if(buildingType.isResourceDepot())
	{
		Unitset minerals = Broodwar->getStaticMinerals();
		for(auto it = minerals.begin(); it != minerals.end(); ++it)
		{
			TilePosition mineralPos = it->getInitialTilePosition();

			if(		mineralPos.x > leftTop.x - 5 &&
					mineralPos.y > leftTop.y - 4 &&
					mineralPos.x < leftTop.x + 7 &&
					mineralPos.y < leftTop.y + 6	)
				return false;
		}

		Unitset geysers = Broodwar->getStaticGeysers();
		for(auto it = geysers.begin(); it != geysers.end(); ++it)
		{
			TilePosition geyserPos = it->getInitialTilePosition();
			if(		geyserPos.x > leftTop.x - 7 &&
					geyserPos.y > leftTop.y - 5 &&
					geyserPos.x < leftTop.x + 7 &&
					geyserPos.y < leftTop.y + 6		)
				return false;
		}
	}
	return true;
}

const bool BuildingManager::canBuildAtTileStatic(const UnitType buildingType, const TilePosition& position) const
{
	TilePosition leftTop = position - TilePosition(1, 0);
	TilePosition rightBot = position + buildingType.tileSize() + TilePosition(0, 1);

	// Map limit check
	if ( !leftTop.isValid() || !(TilePosition(rightBot) - TilePosition(1,1)).isValid() )
		return false;

	if(buildingType.isRefinery())
	{
		// In this case, we want to build a refinery. This means we need to make sure we're building on a Geyser
		Unitset geysers = Broodwar->getStaticGeysers();

		for(auto geyser = geysers.begin(); geyser != geysers.end(); ++geyser)
		{
			if(geyser->getInitialPosition() == position)
				return true;
		}
		return false;
	}

	// In this case, we're not building a refinery. We must check collision stuff
	for(int x = leftTop.x; x < rightBot.x; ++x)
	{
		for(int y = leftTop.y; y < rightBot.y; ++y)
		{
			if(!Broodwar->isBuildable(x, y))
				return false;
		}
	}

	// Resource Depots cannot be placed too close to Minerals
	if(buildingType.isResourceDepot())
	{
		Unitset minerals = Broodwar->getStaticMinerals();
		for(auto it = minerals.begin(); it != minerals.end(); ++it)
		{
			TilePosition mineralPos = it->getInitialTilePosition();

			if(		mineralPos.x > leftTop.x - 5 &&
					mineralPos.y > leftTop.y - 4 &&
					mineralPos.x < leftTop.x + 7 &&
					mineralPos.y < leftTop.y + 6	)
				return false;
		}

		Unitset geysers = Broodwar->getStaticGeysers();
		for(auto it = geysers.begin(); it != geysers.end(); ++it)
		{
			TilePosition geyserPos = it->getInitialTilePosition();
			if(		geyserPos.x > leftTop.x - 7 &&
					geyserPos.y > leftTop.y - 5 &&
					geyserPos.x < leftTop.x + 7 &&
					geyserPos.y < leftTop.y + 6		)
				return false;
		}
	}
	return true;
}

// Implementation largely based on UAlbertaBot
const TilePosition BuildingManager::getBuildLocationNear(const TilePosition& position, const UnitType buildingType, const MapRegion* preferredRegion) const
{
	//returns a valid build location near the specified tile position.
	//searches outward in a spiral.
	int x = position.x;
	int y = position.y;
	int length = 1;
	int j = 0;
	bool first = true;
	int dx = 0;
	int dy = 1;

	const int mapHeight = Broodwar->mapHeight();
	const int mapWidth = Broodwar->mapWidth();

	TilePosition bestSoFar = TilePositions::Invalid;
	while (length < mapWidth) //We'll ride the spiral to the end
	{
		bool inMapBounds = (ArrayUtils::boundsCheck(x, mapWidth) && ArrayUtils::boundsCheck(y, mapHeight));

		if(inMapBounds)
		{
			TilePosition potentialPos = TilePosition(x, y);

			// can we build this building at this location
			if(canBuildAtTile(buildingType, potentialPos))
			{
				if(preferredRegion)
				{
					// the region the build tile is in
					MapRegion* tileRegion = MapAnalyser::Instance()->getRegion(potentialPos);

					if(preferredRegion == tileRegion)
						return potentialPos;
					else if(bestSoFar == TilePositions::Invalid)
						bestSoFar = potentialPos;
				}
				else	// no preference for build region
					return potentialPos;
			}
		}

		// skip ahead if out of map bounds
		if(!inMapBounds)
		{
			int skip = length - j;
			if(x < 0 && dx == 1)
				skip = -x;
			else if(x >= mapWidth && dx == -1)
				skip = x - mapWidth + 1;
			else if(y < 0 && dy == 1)
				skip = -y;
			else if(y >= mapHeight && dy == -1)
				skip = y - mapHeight + 1;

			j += skip;
			x += skip * dx;
			y += skip * dy;
		}
		else
		{
			x = x + dx;
			y = y + dy;

			//count how many steps we take in this direction
			++j;
		}

		if (j == length) //if we've reached the end, its time to turn
		{
			//reset step counter
			j = 0;

			//Spiral out. Keep going.
			if (!first)
				++length; //increment step counter if needed

			//first=true for every other turn so we spiral out at the right rate
			first = !first;

			//turn counter clockwise 90 degrees:
			if (dx == 0)
			{
				dx = dy;
				dy = 0;
			}
			else
			{
				dy = -dx;
				dx = 0;
			}
		}
		//Spiral out. Keep going.
	}

	return bestSoFar;
}


const TilePosition BuildingManager::getBuildLocationAtDistance(const TilePosition& position, const UnitType buildingType, 
																const MapRegion* preferredRegion, const int minDist, const int maxDist) const
{
	//returns a valid build location near the specified tile position.
	//searches outward in a spiral.
	int x = position.x;
	int y = position.y;
	int length = 1;
	int j = 0;
	bool first = true;
	int dx = 0;
	int dy = 1;

	const int mapHeight = Broodwar->mapHeight();
	const int mapWidth = Broodwar->mapWidth();

	const int minDistSq = minDist*minDist;
	const int maxDistSq = maxDist*maxDist;

	TilePosition bestSoFar = TilePositions::Invalid;
	while (length < maxDist + 2)	// no need to spiral further than maxDist allows us (+ error margin of 2 since I don't care to think of
	{								// details like whether it should be maxDist + 0 or + 1 or something else...
		bool inMapBounds = (ArrayUtils::boundsCheck(x, mapWidth) && ArrayUtils::boundsCheck(y, mapHeight));

		if(inMapBounds)
		{
			TilePosition potentialPos = TilePosition(x, y);

			const int distSq = Distances::getSquaredDistance(position, potentialPos);
			bool withinDistanceMargin = (distSq >= minDistSq && distSq <= maxDistSq);

			// can we build this building at this location
			if(withinDistanceMargin && canBuildAtTile(buildingType, potentialPos))
			{
				if(preferredRegion)
				{
					// the region the build tile is in
					MapRegion* tileRegion = MapAnalyser::Instance()->getRegion(potentialPos);

					if(preferredRegion == tileRegion)
						return potentialPos;
					else if(bestSoFar == TilePositions::Invalid)
						bestSoFar = potentialPos;
				}
				else	// no preference for build region
					return potentialPos;
			}
		}

		// skip ahead if out of map bounds
		if(!inMapBounds)
		{
			int skip = length - j;
			if(x < 0 && dx == 1)
				skip = -x;
			else if(x >= mapWidth && dx == -1)
				skip = x - mapWidth + 1;
			else if(y < 0 && dy == 1)
				skip = -y;
			else if(y >= mapHeight && dy == -1)
				skip = y - mapHeight + 1;

			j += skip;
			x += skip * dx;
			y += skip * dy;
		}
		else
		{
			x = x + dx;
			y = y + dy;

			//count how many steps we take in this direction
			++j;
		}

		if (j == length) //if we've reached the end, its time to turn
		{
			//reset step counter
			j = 0;

			//Spiral out. Keep going.
			if (!first)
				++length; //increment step counter if needed

			//first=true for every other turn so we spiral out at the right rate
			first = !first;

			//turn counter clockwise 90 degrees:
			if (dx == 0)
			{
				dx = dy;
				dy = 0;
			}
			else
			{
				dy = -dx;
				dx = 0;
			}
		}
		//Spiral out. Keep going.
	}

	return bestSoFar;
}

const bool BuildingManager::isAllowedInResourceBox(const UnitType buildingType) const
{
	return (buildingType == UnitTypes::Protoss_Nexus			|| 
			buildingType == UnitTypes::Protoss_Photon_Cannon	||
			buildingType == UnitTypes::Protoss_Assimilator			);
}

const bool BuildingManager::isTileInResourceBox(const int x, const int y) const
{
	return ArrayUtils::getMatrixValue(resourceBox, y, x, Broodwar->mapWidth());
}

const bool BuildingManager::isTileInResourceBox(const TilePosition& tilePos) const
{
	return ArrayUtils::getMatrixValue(resourceBox, tilePos.y, tilePos.x, Broodwar->mapWidth());
}

const bool BuildingManager::isTileReserved(const int x, const int y) const
{
	return (ArrayUtils::getMatrixValue(reserved, y, x, Broodwar->mapWidth()) > NOT_RESERVED);
}

const bool BuildingManager::isTileReserved(const TilePosition& tilePos) const
{
	return (ArrayUtils::getMatrixValue(reserved, tilePos.y, tilePos.x, Broodwar->mapWidth()) > NOT_RESERVED);
}

void BuildingManager::freeReservedSpaceForBuilding(const Unit building)
{
	const int mapWidth = Broodwar->mapWidth();

	const TilePosition position = building->getTilePosition();
	const UnitType buildingType = building->getType();

	const int left = position.x - 1;
	const int right = position.x + buildingType.tileSize().x + 1;
	const int top = position.y - 1;
	const int bot = position.y + buildingType.tileSize().y + 1;

	for(int y = top; y < bot; ++y)
	{
		for(int x = left; x < right; ++x)
		{
			ArrayUtils::decrementMatrixValue(reserved, y, x, mapWidth);
		}
	}
}

void BuildingManager::freeReservedSpaceForBuilding(const UnitType buildingType, const TilePosition& position)
{
	const int mapWidth = Broodwar->mapWidth();
	const int mapHeight = Broodwar->mapHeight();

	const int left = max(position.x - 1, 0);
	const int right = min(position.x + buildingType.tileSize().x + 1, mapWidth);
	const int top = max(position.y - 1, 0);
	const int bot = min(position.y + buildingType.tileSize().y + 1, mapWidth);

	for(int y = top; y < bot; ++y)
	{
		for(int x = left; x < right; ++x)
		{
			ArrayUtils::decrementMatrixValue(reserved, y, x, mapWidth);
		}
	}
}

void BuildingManager::reserveSpaceForBuilding(const UnitType buildingType, const TilePosition& position)
{
	const int mapWidth = Broodwar->mapWidth();
	const int mapHeight = Broodwar->mapHeight();

	const int left = max(position.x - 1, 0);
	const int right = min(position.x + buildingType.tileSize().x + 1, mapWidth);
	const int top = max(position.y - 1, 0);
	const int bot = min(position.y + buildingType.tileSize().y + 1, mapWidth);

	for(int y = top; y < bot; ++y)
	{
		for(int x = left; x < right; ++x)
		{
			ArrayUtils::incrementMatrixValue(reserved, y, x, mapWidth);
		}
	}
}

const int BuildingManager::getNumPylonsBuilding() const
{
	return pylonsBuilding;
}

#pragma warning( pop )