/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include "ArmyManager.h"
#include "Base.h"
#include "BaseManager.h"
#include "Distances.h"
#include "MapAnalyser.h"
#include "MathConstants.h"
#include "OpponentTracker.h"
#include "Squad.h"
#include "WorkerManager.h"

/*
	Implementation of the BaseManager
*/

using namespace BWAPI;
using namespace std;

void BaseManager::onFrame()
{
	updateMaxNumWorkers();

	MapAnalyser* mapAnalyser = MapAnalyser::Instance();

	// update when we've last seen various base locations
	const vector<BaseLocation*>& baseLocations = mapAnalyser->getBaseLocations();
	const int currentFrame = Broodwar->getFrameCount();

	// we can update by seeing the optimal depot location of a Base Location...
	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		BaseLocation* loc = *it;

		if(Broodwar->isVisible(loc->getOptimalDepotLocation()))
		{
			loc->setLastObserved(currentFrame);
		}
	}

	// ... or we can update by seeing a building of a base which is part of this Base Location
	for(auto it = opponentBases.begin(); it != opponentBases.end(); /**/)
	{
		Base base = *it;
		const Unitset& buildings = base.getBuildings();

		for(auto it_building = buildings.begin(); it_building != buildings.end(); ++it_building)
		{
			Unit building = *it_building;

			if(building->isVisible())
			{
				base.getLocation()->setLastObserved(currentFrame);
				break;
			}
		}

		if(buildings.empty())
		{
			it = opponentBases.erase(it);
		}
		else
		{
			++it;
		}
	}

	if(selfBases.size() == 0)	// we've probably lost the game, no longer care about defense
		return;

	// Check if ArmyManager can deal with defense
	if(!ArmyManager::Instance()->shouldWorkersDefend())
	{
		WorkerManager::Instance()->noDefenseNeeded();
		return;
	}

	// We have no defensive squads, so alert WorkerManager if base is under attack by ground units
	OpponentTracker* opponentTracker = OpponentTracker::Instance();
	bool underAttack = false;
	Base attackedBase = selfBases.front();

	for(auto it_base = selfBases.begin(); it_base != selfBases.end(); ++it_base)
	{
		Base base = *it_base;
		Unitset opponents = opponentTracker->getOpponentsInRegion(base.getLocation()->getRegion());

		if(!opponents.empty())	// enemies in base; make sure they are threats which our probes can and should deal with
		{
			for(auto it = opponents.begin(); it != opponents.end(); ++it)
			{
				Unit unit = *it;
				UnitType type = unit->getType();

				if(type.isFlyer())		// probes cant attack flyers
					continue;

				// only care about defending if they actually attack us or if it's a building (cheese!)
				if(!(unit->isAttacking() || unit->isStartingAttack() || unit->isAttackFrame() || unit->isConstructing())	&& 
					(Broodwar->getFrameCount() - lastAttack) > 50															&&
					!unit->getType().isBuilding()																				)
				{
					continue;
				}

				lastAttack = Broodwar->getFrameCount();
				underAttack = true;
				attackedBase = base;
				break;
			}
		}
	}

	if(underAttack)
	{
#ifdef ARMY_MANAGER_SCRIPTED
		// Check if we have offensive squads with members still in the base
		const vector<Squad*>& groundSquads = ArmyManager::Instance()->getGroundSquads();
		const vector<Squad*>& airSquads = ArmyManager::Instance()->getAirSquads();

		for(auto it_squad = groundSquads.begin(); it_squad != groundSquads.end(); ++it_squad)
		{
			Squad* squad = *it_squad;
			const Unitset& units = squad->getOwnedUnits();

			for(auto it = units.begin(); it != units.end(); ++it)
			{
				Unit unit = *it;

				if(mapAnalyser->getRegion(unit->getPosition()) == attackedBase.getLocation()->getRegion())
				{
					WorkerManager::Instance()->noDefenseNeeded();
					return;
				}
			}
		}

		for(auto it_squad = airSquads.begin(); it_squad != airSquads.end(); ++it_squad)
		{
			Squad* squad = *it_squad;
			const Unitset& units = squad->getOwnedUnits();

			for(auto it = units.begin(); it != units.end(); ++it)
			{
				Unit unit = *it;

				if(mapAnalyser->getRegion(unit->getPosition()) == attackedBase.getLocation()->getRegion())
				{
					WorkerManager::Instance()->noDefenseNeeded();
					return;
				}
			}
		}
#endif // ARMY_MANAGER_SCRIPTED

		WorkerManager::Instance()->baseUnderAttack(attackedBase);
		return;
	}

	WorkerManager::Instance()->noDefenseNeeded();
}

void BaseManager::onOpponentBuildingDestroy(Unit building)
{
	if(building->isLifted())
		return;

	// find nearest base location to this building
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	const vector<BaseLocation*>& baseLocations = mapAnalyser->getBaseLocations();
	int minDistSq = MathConstants::MAX_INT;
	BaseLocation* nearestLoc = nullptr;
	Position pos = OpponentTracker::Instance()->getLastPosition(building);
	MapRegion* buildingRegion = mapAnalyser->getRegion(pos);

	if(building->getType() == UnitTypes::Unknown && Broodwar->enemy()->getRace() == Races::Terran)	// probably was a lifted building which flew away from us
		return;

	if(pos == Positions::Invalid || pos == Positions::Unknown)
		pos = OpponentTracker::Instance()->getLastPosition(building);

	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		BaseLocation* loc = *it;
		int distSq = Distances::getSquaredDistance(pos, PositionUtils::toPosition(loc->getOptimalDepotLocation()));

		MapRegion* baseRegion = loc->getRegion();

		if(baseRegion != buildingRegion)	// penalty for wrong region
			distSq *= 5;	

		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestLoc = loc;
		}
	}

	// find corresponding base and remove building
	for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
	{
		if((*it).getLocation() == nearestLoc)
		{
			if((*it).removeBuilding(building))
				opponentBases.erase(it);

			return;
		}
	}

#ifdef MAASCRAFT_DEBUG
	LOG_WARNING("Could not find base corresponding to destroyed opponent building!")
	LOG_MESSAGE(StringBuilder() << "Destroyed opponent building type = " << building->getType())
	if(building->getType() == UnitTypes::Unknown)
		LOG_MESSAGE(StringBuilder() << "OpponentTracker says type = " << OpponentTracker::Instance()->getLastType(building))
#endif
}

void BaseManager::onOpponentBuildingShow(Unit building)
{
	// find nearest base location to this building
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	const vector<BaseLocation*>& baseLocations = mapAnalyser->getBaseLocations();
	int minDistSq = MathConstants::MAX_INT;
	BaseLocation* nearestLoc = nullptr;
	Position pos = building->getPosition();
	MapRegion* buildingRegion = mapAnalyser->getRegion(pos);

	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		BaseLocation* loc = *it;
		int distSq = Distances::getSquaredDistance(pos, PositionUtils::toPosition(loc->getOptimalDepotLocation()));

		MapRegion* baseRegion = loc->getRegion();

		if(baseRegion != buildingRegion)	// penalty for wrong region
			distSq *= 5;	

		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestLoc = loc;
		}
	}

	// if we already know about a base at this baseLocation, add building to that base
	for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
	{
		if((*it).getLocation() == nearestLoc)
		{
			(*it).addBuilding(building);
			return;
		}
	}

	// No base found which we can add the building to, so create new base for this building
	Base base(nearestLoc, Broodwar->enemy());
	base.addBuilding(building);
	opponentBases.push_back(base);

	// Since we observed a new base, we'll want our attacking squads to re-plan
	ArmyManager::Instance()->newEnemyBaseObserved();
}

void BaseManager::onSelfBuildingComplete(Unit building)
{
	// find nearest base location to this building
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	const int DIFFERENT_REGION_PENALTY = 10000000;

	Position pos = building->getPosition();
	MapRegion* buildingRegion = mapAnalyser->getRegion(pos);

	const vector<BaseLocation*>& baseLocations = mapAnalyser->getBaseLocations();
	int minDistSq = MathConstants::MAX_INT;
	BaseLocation* nearestLoc = nullptr;

	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		BaseLocation* loc = *it;
		int distSq = Distances::getSquaredDistance(pos, PositionUtils::toPosition(loc->getOptimalDepotLocation()));

		MapRegion* baseRegion = loc->getRegion();
		if(baseRegion != buildingRegion)
			distSq += DIFFERENT_REGION_PENALTY;

		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestLoc = loc;
		}
	}

	// if we already know about a base at this baseLocation, add building to that base
	for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
	{
		if((*it).getLocation() == nearestLoc)
		{
			(*it).addBuilding(building);
			return;
		}
	}

	// No base found which we can add the building to, so create new base for this building
	selfBases.push_back(Base(nearestLoc, Broodwar->self()));
	selfBases.back().addBuilding(building);
}

void BaseManager::onSelfBuildingDestroy(Unit building)
{
	// find nearest base location to this building
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	const int DIFFERENT_REGION_PENALTY = 10000000;

	Position pos = building->getPosition();
	MapRegion* buildingRegion = mapAnalyser->getRegion(pos);

	const vector<BaseLocation*>& baseLocations = mapAnalyser->getBaseLocations();
	int minDistSq = MathConstants::MAX_INT;
	BaseLocation* nearestLoc = nullptr;

	for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
	{
		BaseLocation* loc = *it;
		int distSq = Distances::getSquaredDistance(pos, PositionUtils::toPosition(loc->getOptimalDepotLocation()));

		MapRegion* baseRegion = loc->getRegion();
		if(baseRegion != buildingRegion)
			distSq += DIFFERENT_REGION_PENALTY;

		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestLoc = loc;
		}
	}

	// find corresponding base and remove building
	for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
	{
		if((*it).getLocation() == nearestLoc)
		{
			if((*it).removeBuilding(building))
			{
				selfBases.erase(it);
			}
			return;
		}
	}

	LOG_WARNING("Could not find base corresponding to destroyed self building!")
	LOG_MESSAGE(StringBuilder() << "Destroyed self building type = " << building->getType())
}

void BaseManager::addBase(Base newBase)
{
	if(newBase.getOwner() == Broodwar->self())
	{
		selfBases.push_back(newBase);
	}
	else
	{
		opponentBases.push_back(newBase);
	}
}

const BaseLocation* BaseManager::getMainBaseLocation()
{
	return mainBase;
}

void BaseManager::setMainBaseLocation(BaseLocation* mainBaseLoc)
{
	mainBase = mainBaseLoc;
}

const bool BaseManager::hasBaseAtLocation(const Player player, const BaseLocation* baseLoc) const
{
	if(player == Broodwar->self())
	{
		for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
		{
			if((*it).getLocation() == baseLoc)
				return true;
		}
	}
	else
	{
		for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
		{
			if((*it).getLocation() == baseLoc)
				return true;
		}
	}

	return false;
}

Base BaseManager::getBaseAtLocation(const Player player, BaseLocation* baseLoc) const
{
	if(player == Broodwar->self())
	{
		for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
		{
			if((*it).getLocation() == baseLoc)
				return (*it);
		}
	}
	else
	{
		for(auto it = opponentBases.begin(); it != opponentBases.end(); ++it)
		{
			if((*it).getLocation() == baseLoc)
				return (*it);
		}
	}

	return Base(baseLoc, player);
}

void BaseManager::updateMaxNumWorkers()
{
	maxNumWorkers = 0;
	float tempMaxNumWorkers = 0.0f;

	for(auto it_base = selfBases.begin(); it_base != selfBases.end(); ++it_base)
	{
		Base base = *it_base;
		const BaseLocation* baseLoc = base.getLocation();
		const Unitset resources = baseLoc->getResources();

		for(auto it_resource = resources.begin(); it_resource != resources.end(); ++it_resource)
		{
			Unit resource = *it_resource;

			if(resource->getResources() > 0)
			{
				if(resource->getType().isMineralField())
				{
					tempMaxNumWorkers += Options::ResourceGathering::WORKERS_PER_MINERAL_PATCH;
				}
				else if(resource->getType() == UnitTypes::Protoss_Assimilator && resource->getRemainingBuildTime() <= UnitTypes::Protoss_Probe.buildTime())
				{
					tempMaxNumWorkers += Options::ResourceGathering::WORKERS_PER_VESPENE_GAS_GEYSER;
				}
			}
		}
	}

	maxNumWorkers = std::ceil(tempMaxNumWorkers);
}

vector<BaseLocation*> BaseManager::getScoutLocations()
{
	vector<BaseLocation*> scoutLocations;
	const vector<BaseLocation*>& baseLocations = MapAnalyser::Instance()->getBaseLocations();
	const int maxSqRadius = Options::Scouting::SQUARED_MAX_EXPANSION_RADIUS;

	for(auto it_base = opponentBases.begin(); it_base != opponentBases.end(); ++it_base)
	{
		Base base = *it_base;

		if(std::find(scoutLocations.begin(), scoutLocations.end(), base.getLocation()) == scoutLocations.end())
			scoutLocations.push_back(base.getLocation());

		int minDistSq = MathConstants::MAX_INT;
		BaseLocation* nearest = nullptr;

		for(auto it_loc = baseLocations.begin(); it_loc != baseLocations.end(); ++it_loc)
		{
			BaseLocation* loc = *it_loc;

			if(std::find(scoutLocations.begin(), scoutLocations.end(), loc) != scoutLocations.end())
				continue;

			const int distSq = Distances::getSquaredDistance(base.getDepotCenter(), PositionUtils::toPosition(loc->getOptimalDepotLocation()));

			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				nearest = loc;
			}
		}

		scoutLocations.push_back(nearest);
	}

	return scoutLocations;
}