/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

/*
	Utility functions for C-style arrays
*/

namespace ArrayUtils
{
	// Returns true iff 0 <= index < arraySize. 
	// Faster than checking for >= 0 and < arraySize according to: http://www.agner.org/optimize/optimizing_cpp.pdf page 137
	static const __forceinline bool boundsCheck(const int index, const size_t arraySize)
	{
		return (arraySize > ((unsigned int) index));
	}

	// Returns number of elements in passed array
	template <typename T, size_t N> 
	static size_t getArraySize(const T(&)[N]);

	// TODO: optimize the initialization functions with memset? Make copy functions with memcpy?

	// Initializes an already created array of bools, starting at arr_start, consisting of num_elements elements, with value in each position
	static void initializeBoolArray(bool* const arr_start, const size_t num_elements, const bool value);

	// Initializes an already created array of floats, starting at arr_start, consisting of num_elements elements, with value in each position
	static void initializeFloatArray(float* const arr_start, const size_t num_elements, const float value);

	// Initializes an already created array of ints, starting at arr_start, consisting of num_elements elements, with value in each position
	static void initializeIntArray(int* const arr_start, const size_t num_elements, const int value);

	// Initializes an already created array of shorts, starting at arr_start, consisting of num_elements elements, with value in each position
	static void initializeShortArray(short* const arr_start, const size_t num_elements, const short value);

	// Adds a value to an existing value in a one-dimensional array as if it were a two dimensional array
	template <typename T>
	static void addMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width, T addition);

	// Decrements a value in a one-dimensional array as if it were a two dimensional array
	template <typename T>
	static void decrementMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width);

	// Obtains a value from a one-dimensional array as if it were a two dimensional array
	template <typename T>
	static T getMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width);

	// Increments a value in a one-dimensional array as if it were a two dimensional array
	template <typename T>
	static void incrementMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width);

	// Sets a value in a one-dimensional array as if it were a two dimensional array
	template <typename T>
	static void setMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width, T value);
}

/*	==========================================================================================
			Definition of declarations above

			Implementing short functions in header file
			Allows for better compiler optimizations through inlining
	========================================================================================== */

template <typename T, size_t N>
inline size_t ArrayUtils::getArraySize(const T(&)[N])
{
	return N;
}

inline void ArrayUtils::initializeBoolArray(bool* const arr_start, const size_t num_elements, const bool value)
{
	for(size_t i = 0; i < num_elements; i++)
	{
		*(arr_start + i) = value; 
	}
}

inline void ArrayUtils::initializeFloatArray(float* const arr_start, const size_t num_elements, const float value)
{
	for(size_t i = 0; i < num_elements; i++)
	{
		*(arr_start + i) = value; 
	}
}

inline void ArrayUtils::initializeIntArray(int* const arr_start, const size_t num_elements, const int value)
{
	for(size_t i = 0; i < num_elements; i++)
	{
		*(arr_start + i) = value; 
	}
}

inline void ArrayUtils::initializeShortArray(short* const arr_start, const size_t num_elements, const short value)
{
	for(size_t i = 0; i < num_elements; i++)
	{
		*(arr_start + i) = value; 
	}
}

template <typename T>
inline void ArrayUtils::addMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width, T addition)
{
	oneDimensionalArray[col + width*row] += addition;
}

template <typename T>
inline void ArrayUtils::decrementMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width)
{
	oneDimensionalArray[col + width*row]--;
}

template <typename T>
inline T ArrayUtils::getMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width)
{
	return oneDimensionalArray[col + width*row];
}

template <typename T>
inline void ArrayUtils::incrementMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width)
{
	oneDimensionalArray[col + width*row]++;
}

template <typename T>
inline void ArrayUtils::setMatrixValue(T* oneDimensionalArray, const int row, const int col, const int width, T value)
{
	oneDimensionalArray[col + width*row] = value;
}