/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include "ArmyManager.h"
#include "BaseManager.h"
#include "Clustering.h"
#include "MapAnalyser.h"
#include "MathConstants.h"
#include "OpponentTracker.h"
#include "Priorities.h"
#include "Squad.h"
#include "StrategyManager.h"
#include "Tasks/AttackBase.hpp"
#include "Tasks/AttackUnits.hpp"
#include "Tasks/DefendChokePoint.hpp"
#include "Tasks/Retreat.hpp"
#include "UnitBehaviours/CombatBehaviourFight.h"
#include "UnitTracker.h"

/*
	Implementation of the ArmyManager
*/

using namespace BWAPI;
using namespace std;

#ifdef ARMY_MANAGER_UCT

#include "Timer.hpp"

ArmyManager::ArmyManager() : 
	searchEngine(), airSquads(), groundSquads()
{}

ArmyManager::~ArmyManager()
{}

void ArmyManager::onFrame(const float allowedMilliSeconds)
{
	Timer armyManagerTimer;
	armyManagerTimer.start();

	// Start new search if current search engine ran for long enough
	if(Broodwar->getFrameCount() - searchEngine.getStartFrame() > Options::ArmyManagerUCT::FRAMES_PER_SEARCH)
	{
		UnitTracker* unitTracker = UnitTracker::Instance();

		// cluster our military units
		const Unitset& selfUnits = unitTracker->getSelfArmyUnits();
		Unitset airUnits;
		Unitset groundUnits;

		for(auto it = selfUnits.begin(); it != selfUnits.end(); ++it)
		{
			Unit u = *it;

			if(u->getType().isFlyer())
				airUnits.push_back(u);
			else
				groundUnits.push_back(u);

			UnitOwner* previousOwner = unitTracker->getUnitOwner(u);
			
			if(previousOwner)
				previousOwner->transferOwnershipTo(u, this);
			else
				unitTracker->setOwner(u, this);
		}

		vector<Unitset> airClusters;
		vector<Unitset> groundClusters;

		// first squad we make in the game should be large enough
		//if(groundSquads.empty() && airSquads.empty())
		if(Broodwar->self()->completedUnitCount(UnitTypes::Protoss_Zealot) + Broodwar->self()->deadUnitCount(UnitTypes::Protoss_Zealot) < 5)
		{
			if(Broodwar->enemy()->getRace() == Races::Terran)
			{
				airClusters = Clustering::getClustersForUnits(airUnits, 192, 5);
				groundClusters = Clustering::getClustersForUnits(groundUnits, 192, 5);
			}
			else
			{
				airClusters = Clustering::getClustersForUnits(airUnits, 192, 4);
				groundClusters = Clustering::getClustersForUnits(groundUnits, 192, 4);
			}
		}
		else
		{
			airClusters = Clustering::getClustersForUnits(airUnits, 192, 3);
			groundClusters = Clustering::getClustersForUnits(groundUnits, 192, 3);
		}

		// clean up previous Squads
		for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
		{
			Squad* squad = *it;
			delete squad;
			squad = nullptr;
		}
		airSquads.clear();

		for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
		{
			Squad* squad = *it;
			delete squad;
			squad = nullptr;
		}
		groundSquads.clear();

		// obtain squads from search engine
		searchEngine.fillSquads(airSquads, groundSquads);

		// initialize new search
		searchEngine.init(airClusters, groundClusters);
	}

	// have current squads execute their tasks
	for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
	{
		(*it)->onFrame();
	}

	for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
	{
		(*it)->onFrame();
	}

	commandLooseUnits();

	searchEngine.onFrame(min(Options::ArmyManagerUCT::MS_PER_FRAME, int(allowedMilliSeconds - armyManagerTimer.getElapsedTimeInMilliSec())));

#ifdef MAASCRAFT_DEBUG
	if(Options::Debug::ScreenInfo::DEBUG_ARMY_MANAGER_INFO)
	{
		Broodwar->drawTextScreen(350, 20, "Num Air Squads = %d", airSquads.size());
		Broodwar->drawTextScreen(350, 40, "Num Ground Squads = %d", groundSquads.size());
	}
#endif
}

void ArmyManager::commandLooseUnits()
{
	const Unitset& looseUnits = getOwnedUnits();

	if(looseUnits.size() == 0)
		return;

	// let loose units defend our base if there are any enemies in our base
	const vector<Base>& bases = BaseManager::Instance()->getSelfBases();
	OpponentTracker* opponentTracker = OpponentTracker::Instance();
	
	for(auto it = bases.begin(); it != bases.end(); ++it)
	{
		MapRegion* region = (*it).getLocation()->getRegion();
		Unitset opponents = opponentTracker->getOpponentsInRegion(region);

		bool defend = false;

		for(auto opponent = opponents.begin(); opponent != opponents.end(); ++opponent)
		{
			Unit u = *opponent;
			UnitType type = u->getType();

			if(!type.isWorker() && !u->isLifted())
			{
				defend = true;
				break;
			}
		}

		if(defend)		// defend this base by sending all loose units to the first enemy in this set
		{
			for(auto it = looseUnits.begin(); it != looseUnits.end(); ++it)
			{
				Unit unit = *it;

				if(UnitUtils::isInCombat(unit))
				{
					if(Behaviours::combatBehaviourFight(unit))
						continue;
				}
				
				if(UnitUtils::canMove(unit))
				{
					unit->move(opponents.front()->getPosition());
				}
			}

			return;
		}
	}

	// no bases to defend, so instead move to closest squad (or base if no squad found)
	for(auto it = looseUnits.begin(); it != looseUnits.end(); ++it)
	{
		Unit unit = *it;
		UnitType type = unit->getType();

		int minDistSq = MathConstants::MAX_INT;
		Squad* nearestSquad = nullptr;

		if(UnitUtils::isInCombat(unit))
		{
			if(Behaviours::combatBehaviourFight(unit))
				continue;
		}
		else if(type.isFlyer())
		{
			for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
			{
				Squad* squad = *it;

				Unit livingSquadMember = nullptr;
				const Unitset& squadMembers = squad->getOwnedUnits();

				for(auto it_member = squadMembers.begin(); it_member != squadMembers.end(); ++it_member)
				{
					Unit member = *it_member;

					if(member && UnitUtils::isUnitValid(member))
					{
						livingSquadMember = member;
						break;
					}
				}

				if(!livingSquadMember)
					continue;

				const int distSq = Distances::getSquaredDistance(unit->getPosition(), livingSquadMember->getPosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					nearestSquad = squad;
				}
			}
		}
		else
		{
			for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
			{
				Squad* squad = *it;

				Unit livingSquadMember = nullptr;
				const Unitset& squadMembers = squad->getOwnedUnits();

				for(auto it_member = squadMembers.begin(); it_member != squadMembers.end(); ++it_member)
				{
					Unit member = *it_member;

					if(member && UnitUtils::isUnitValid(member))
					{
						livingSquadMember = member;
						break;
					}
				}

				if(!livingSquadMember)
					continue;

				const int distSq = Distances::getSquaredDistance(unit->getPosition(), livingSquadMember->getPosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					nearestSquad = squad;
				}
			}
		}

		if(nearestSquad)
		{
			Unit squadMember = nearestSquad->getOwnedUnits().front();
			Position pos = squadMember->getPosition();

			unit->move(pos);
		}
		else
		{
			if(bases.size() > 0)
				unit->move(bases.front().getDepotCenter());
		}
	}
}

const Unitset ArmyManager::getPotentialTargets(Unit defender) const
{
	Unitset potentialTargets = UnitUtils::getUnitsInRadius(defender, 960, Filter::IsEnemy && Filter::IsDetected);

	return potentialTargets;
}

void ArmyManager::newEnemyBaseObserved() const
{
	// TODO implement
}

const bool ArmyManager::shouldWorkersDefend() const
{
	for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
	{
		Squad* squad = *it;
		if(squad->getSize() > 0)
		{
			return false;
		}
	}

	return true;
}

void ArmyManager::onUnitComplete(BWAPI::Unit unit)
{
}

#endif // ARMY_MANAGER_UCT

#ifdef ARMY_MANAGER_SCRIPTED

ArmyManager::ArmyManager() :
	airSquads(), defensiveSquads(), groundSquads(), attacking(false), retreating(false), playOffensive(false)
{
	TilePosition startTile = Broodwar->self()->getStartLocation();
	MapRegion* startRegion = MapAnalyser::Instance()->getRegion(startTile);
	const vector<MapChokePoint*> chokepoints = startRegion->getChokepoints();

	for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
	{
		MapChokePoint* chokepoint = *it;
		Squad* squad = new Squad();
		defensiveSquads.push_back(squad);
		squad->addTask(new DefendChokePoint(Priorities::PRIORITY_FALL_BACK_BEHAVIOUR, squad, startRegion, chokepoint));
	}
}

ArmyManager::~ArmyManager()
{
	for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
	{
		Squad* squad = *it;
		delete squad;
	}
	airSquads.clear();

	for(auto it = defensiveSquads.begin(); it != defensiveSquads.end(); ++it)
	{
		Squad* squad = *it;
		delete squad;
	}
	defensiveSquads.clear();

	for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
	{
		Squad* squad = *it;
		delete squad;
	}
	groundSquads.clear();
}

void ArmyManager::onFrame()
{
	// First decide whether we wanna play aggressively or defensively
	ArmyManager* armyManager = ArmyManager::Instance();
	OpponentTracker* opponentTracker = OpponentTracker::Instance();
	Player self = Broodwar->self();
	Player enemy = Broodwar->enemy();

	// start DPS values at 0.1f to avoid dividing by zero.
	float enemyAirDPS = 0.1f;
	int enemyAirHP = 0;
	float enemyGroundDPS = 0.1f;
	int enemyGroundHP = 0;

	float selfAirDPS = 0.1f;
	int selfAirHP = 0;
	float selfGroundDPS = 0.1f;
	int selfGroundHP = 0;

	const unordered_map<Unit, UnitUtils::UnitData, UnitUtils::HashFunctionUnit>& unitDataMap = opponentTracker->getUnitDataMap();

	for(auto it = unitDataMap.begin(); it != unitDataMap.end(); ++it)
	{
		Unit unit = (*it).first;
		UnitUtils::UnitData data = (*it).second;

		UnitType type = opponentTracker->getLastType(unit);
		if(type == UnitTypes::Zerg_Overlord)
			continue;

		if(type.canAttack() && !type.isWorker())
		{
			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			if(airWeapon != WeaponTypes::None && airWeapon != WeaponTypes::Unknown)
			{
				enemyAirDPS += (enemy->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));

#ifdef MAASCRAFT_DEBUG
				if(airWeapon.damageCooldown() == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
			if(groundWeapon != WeaponTypes::None && groundWeapon != WeaponTypes::Unknown)
			{
				enemyGroundDPS += (enemy->damage(groundWeapon) * (24.0f / enemy->weaponDamageCooldown(type)));

#ifdef MAASCRAFT_DEBUG
				if(enemy->weaponDamageCooldown(type) == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
		}

		if(type.isFlyer())
			enemyAirHP += opponentTracker->getLastHP(unit) + unit->getShields();
		else if(!type.isWorker())
			enemyGroundHP += opponentTracker->getLastHP(unit) + unit->getShields();
	}

	const Unitset& defensiveStructures = opponentTracker->getOpponentDefensiveStructures();
	for(auto it = defensiveStructures.begin(); it != defensiveStructures.end(); ++it)
	{
		Unit structure = *it;

		UnitType type = opponentTracker->getLastType(structure);
		if(type.canAttack())
		{
			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			if(airWeapon != WeaponTypes::None && airWeapon != WeaponTypes::Unknown)
			{
				enemyAirDPS += (enemy->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));

#ifdef MAASCRAFT_DEBUG
				if(airWeapon.damageCooldown() == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
			if(groundWeapon != WeaponTypes::None && groundWeapon != WeaponTypes::Unknown)
			{
				enemyGroundDPS += (enemy->damage(groundWeapon) * (24.0f / enemy->weaponDamageCooldown(type)));

#ifdef MAASCRAFT_DEBUG
				if(enemy->weaponDamageCooldown(type) == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
		}
		else		// must be a bunker I guess?
		{
#ifdef MAASCRAFT_DEBUG
			if(type != UnitTypes::Terran_Bunker)
			{
				LOG_WARNING(StringBuilder() << "Found defensive structure which cannot attack and is not a bunker! Type = " << type)
			}
#endif
			enemyGroundDPS += 50.0f;
		}

		enemyGroundHP += opponentTracker->getLastHP(structure) + structure->getShields();
	}

	if(playOffensive)		// only use the squads which are we've sent out there into the field, not the defensive squads
	{
		const vector<Squad*>& airSquads = armyManager->getAirSquads();
		const vector<Squad*>& groundSquads = armyManager->getGroundSquads();

		for(auto it_squad = airSquads.begin(); it_squad != airSquads.end(); ++it_squad)
		{
			Squad* squad = *it_squad;
			const Unitset& members = squad->getOwnedUnits();

			for(auto it = members.begin(); it != members.end(); ++it)
			{
				Unit unit = *it;

				UnitType type = unit->getType();
				if(type.canAttack())
				{
					WeaponType airWeapon = type.airWeapon();
					WeaponType groundWeapon = type.groundWeapon();

					if(airWeapon != WeaponTypes::None)
						selfAirDPS += (self->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));
					if(groundWeapon != WeaponTypes::None)
						selfGroundDPS += (self->damage(groundWeapon) * (24.0f / self->weaponDamageCooldown(type)));
				}

				selfAirHP += unit->getHitPoints() + unit->getShields();
			}
		}

		for(auto it_squad = groundSquads.begin(); it_squad != groundSquads.end(); ++it_squad)
		{
			Squad* squad = *it_squad;
			const Unitset& members = squad->getOwnedUnits();

			for(auto it = members.begin(); it != members.end(); ++it)
			{
				Unit unit = *it;

				UnitType type = unit->getType();
				if(type.canAttack())
				{
					WeaponType airWeapon = type.airWeapon();
					WeaponType groundWeapon = type.groundWeapon();

					if(airWeapon != WeaponTypes::None)
						selfAirDPS += (self->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));
					if(groundWeapon != WeaponTypes::None)
						selfGroundDPS += (self->damage(groundWeapon) * (24.0f / self->weaponDamageCooldown(type)));
				}

				selfGroundHP += unit->getHitPoints() + unit->getShields();
			}
		}
	}
	else		// currently playing defensive, so use the defensive squads to determine our army's strength
	{
		const vector<Squad*>& defensiveSquads = armyManager->getDefensiveSquads();

		for(auto it_squad = defensiveSquads.begin(); it_squad != defensiveSquads.end(); ++it_squad)
		{
			Squad* squad = *it_squad;
			const Unitset& members = squad->getOwnedUnits();

			for(auto it = members.begin(); it != members.end(); ++it)
			{
				Unit unit = *it;

				UnitType type = unit->getType();
				if(type.canAttack())
				{
					WeaponType airWeapon = type.airWeapon();
					WeaponType groundWeapon = type.groundWeapon();

					if(airWeapon != WeaponTypes::None)
						selfAirDPS += (self->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));
					if(groundWeapon != WeaponTypes::None)
						selfGroundDPS += (self->damage(groundWeapon) * (24.0f / self->weaponDamageCooldown(type)));
				}

				if(type.isFlyer())
					selfAirHP += unit->getHitPoints() + unit->getShields();
				else
					selfGroundHP += unit->getHitPoints() + unit->getShields();
			}
		}
	}

	float timeToKillEnemyAir = enemyAirHP / selfAirDPS;
	float timeToKillEnemyGround = enemyGroundHP / selfGroundDPS;
	float timeToKillSelfAir = selfAirHP / enemyAirDPS;
	float timeToKillSelfGround = selfGroundHP / enemyGroundDPS;

	float selfTimeRequired = timeToKillEnemyAir + timeToKillEnemyGround;
	float enemyTimeRequired = timeToKillSelfAir + timeToKillSelfGround;

#ifdef MAASCRAFT_DEBUG
	if(Options::Debug::ScreenInfo::DEBUG_ARMY_MANAGER_INFO)
	{
		Broodwar->drawTextScreen(0, 30, "\x07 M A A S C R A F T");
		Broodwar->drawTextScreen(0, 40, "\x07 Air DPS = %0.3f", selfAirDPS);
		Broodwar->drawTextScreen(0, 50, "\x07 Ground DPS = %0.3f", selfGroundDPS);
		Broodwar->drawTextScreen(0, 60, "\x07 Air HP = %d", selfAirHP);
		Broodwar->drawTextScreen(0, 70, "\x07 Ground HP = %d", selfGroundHP);
		Broodwar->drawTextScreen(0, 80, "\x07 Required Time = %0.1f sec.", selfTimeRequired);

		Broodwar->drawTextScreen(500, 30, "\x08 O P P O N E N T");
		Broodwar->drawTextScreen(500, 40, "\x08 Air DPS = %0.3f", enemyAirDPS);
		Broodwar->drawTextScreen(500, 50, "\x08 Ground DPS = %0.3f", enemyGroundDPS);
		Broodwar->drawTextScreen(500, 60, "\x08 Air HP = %d", enemyAirHP);
		Broodwar->drawTextScreen(500, 70, "\x08 Ground HP = %d", enemyGroundHP);
		Broodwar->drawTextScreen(500, 80, "\x08 Required Time = %0.1f sec.", enemyTimeRequired);
	}
#endif

	if((selfGroundHP + selfAirHP >= 800 || playOffensive) && 1.75f*selfTimeRequired < enemyTimeRequired)
	{
		playOffensive = true;
	}
	else if(selfTimeRequired > enemyTimeRequired || enemyTimeRequired == 0.0f)
	{
		playOffensive = false;
	}

	for(auto it = airSquads.begin(); it != airSquads.end(); /**/)
	{
		Squad* squad = *it;

		if(squad->getSize() == 0)
		{
			delete squad;
			it = airSquads.erase(it);
		}
		else
		{
			++it;
		}
	}

	for(auto it = groundSquads.begin(); it != groundSquads.end(); /**/)
	{
		Squad* squad = *it;

		if(squad->getSize() == 0)
		{
			delete squad;
			it = groundSquads.erase(it);
		}
		else
		{
			++it;
		}
	}

	if(retreating && airSquads.empty() && groundSquads.empty())
		retreating = false;

	if(shouldPlayOffensive() && 
		((BaseManager::Instance()->getOpponentBases().size() > 0) || (OpponentTracker::Instance()->getOpponentClusters().size() > 0)))
	{
		if(!attacking && !retreating)
		{
			// split up defensive squads into air and ground squads and launch new attack
			attacking = true;

			// for now, making one squad for each unit type. May want to change that in the future?
			Squad* zealotSquad = new Squad();
			Squad* dragoonSquad = new Squad();
			Squad* highTemplarSquad = new Squad();
			Squad* darkTemplarSquad = new Squad();
			Squad* reaverSquad = new Squad();
			Squad* archonSquad = new Squad();
			Squad* darkArchonSquad = new Squad();

			Squad* observerSquad = new Squad();
			Squad* shuttleSquad = new Squad();
			Squad* scoutSquad = new Squad();
			Squad* carrierSquad = new Squad();
			Squad* arbiterSquad = new Squad();
			Squad* corsairSquad = new Squad();

			for(auto it_squad = defensiveSquads.begin(); it_squad != defensiveSquads.end(); ++it_squad)
			{
				Squad* defensiveSquad = *it_squad;
				Unitset units = defensiveSquad->getOwnedUnits();

				for(auto it = units.begin(); it != units.end(); ++it)
				{
					Unit unit = *it;
					unit->stop();
					UnitType type = unit->getType();

					if(type.isFlyer())
					{
						if(type == UnitTypes::Protoss_Observer)
							defensiveSquad->transferOwnershipTo(unit, observerSquad);
						else if(type == UnitTypes::Protoss_Shuttle)
							defensiveSquad->transferOwnershipTo(unit, shuttleSquad);
						else if(type == UnitTypes::Protoss_Scout)
							defensiveSquad->transferOwnershipTo(unit, scoutSquad);
						else if(type == UnitTypes::Protoss_Carrier)
							defensiveSquad->transferOwnershipTo(unit, carrierSquad);
						else if(type == UnitTypes::Protoss_Arbiter)
							defensiveSquad->transferOwnershipTo(unit, arbiterSquad);
						else if(type == UnitTypes::Protoss_Corsair)
							defensiveSquad->transferOwnershipTo(unit, corsairSquad);
					}
					else
					{
						if(type == UnitTypes::Protoss_Zealot)
							defensiveSquad->transferOwnershipTo(unit, zealotSquad);
						else if(type == UnitTypes::Protoss_Dragoon)
							defensiveSquad->transferOwnershipTo(unit, dragoonSquad);
						else if(type == UnitTypes::Protoss_High_Templar)
							defensiveSquad->transferOwnershipTo(unit, highTemplarSquad);
						else if(type == UnitTypes::Protoss_Dark_Templar)
							defensiveSquad->transferOwnershipTo(unit, darkTemplarSquad);
						else if(type == UnitTypes::Protoss_Reaver)
							defensiveSquad->transferOwnershipTo(unit, reaverSquad);
						else if(type == UnitTypes::Protoss_Archon)
							defensiveSquad->transferOwnershipTo(unit, archonSquad);
						else if(type == UnitTypes::Protoss_Dark_Archon)
							defensiveSquad->transferOwnershipTo(unit, darkArchonSquad);
					}
				}
			}

			airSquads.push_back(observerSquad);
			airSquads.push_back(shuttleSquad);
			airSquads.push_back(scoutSquad);
			airSquads.push_back(carrierSquad);
			airSquads.push_back(arbiterSquad);
			airSquads.push_back(corsairSquad);

			groundSquads.push_back(zealotSquad);
			groundSquads.push_back(dragoonSquad);
			groundSquads.push_back(highTemplarSquad);
			groundSquads.push_back(darkTemplarSquad);
			groundSquads.push_back(reaverSquad);
			groundSquads.push_back(archonSquad);
			groundSquads.push_back(darkArchonSquad);
		}
	}
	else if(attacking)
	{
		//start retreat
		attacking = false;
		retreating = true;

		for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
		{
			(*it)->addTask(new Retreat(Priorities::PRIORITY_COMBAT_RETREAT, (*it)));
		}

		for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
		{
			(*it)->addTask(new Retreat(Priorities::PRIORITY_COMBAT_RETREAT, (*it)));
		}
	}

	for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
	{
		(*it)->onFrame();
	}

	for(auto it = defensiveSquads.begin(); it != defensiveSquads.end(); ++it)
	{
		(*it)->onFrame();
	}

	for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
	{
		(*it)->onFrame();
	}
}

void ArmyManager::onUnitComplete(Unit unit)
{
	MapRegion* unitRegion = MapAnalyser::Instance()->getRegion(unit->getPosition());
	UnitType unitType = unit->getType();

	Squad* smallestSquad = nullptr;
	int smallestSquadSize = MathConstants::MAX_INT;

	for(auto it = defensiveSquads.begin(); it != defensiveSquads.end(); ++it)
	{
		Squad* squad = *it;
		const int size = squad->getSize();

		if(size < smallestSquadSize)
		{
			smallestSquad = squad;
			smallestSquadSize = size;
		}
	}

#ifdef MAASCRAFT_DEBUG
	if(!smallestSquad)
	{
		LOG_WARNING("ArmyManager:: No smallest defensive squad found!")
		return;
	}
#endif

	smallestSquad->receiveOwnership(unit);
}

const vector<Squad*>& ArmyManager::getAirSquads()
{
	return airSquads;
}

const vector<Squad*>& ArmyManager::getDefensiveSquads()
{
	return defensiveSquads;
}

const vector<Squad*>& ArmyManager::getGroundSquads()
{
	return groundSquads;
}

const bool ArmyManager::isAttacking() const
{
	return attacking;
}

const bool ArmyManager::isRetreating() const
{
	return retreating;
}

void ArmyManager::newEnemyBaseObserved() const
{
	// re-plan attack squads

	for(auto it = airSquads.begin(); it != airSquads.end(); ++it)
	{
		Squad* squad = *it;
		squad->clearTasks(Priorities::PRIORITY_MAX);
	}

	for(auto it = groundSquads.begin(); it != groundSquads.end(); ++it)
	{
		Squad* squad = *it;
		squad->clearTasks(Priorities::PRIORITY_COMBAT_ATTACK);
	}
}

Task* ArmyManager::requestTask(Squad* squad) const
{
	if(!attacking)
	{
#ifdef MAASCRAFT_DEBUG
		LOG_WARNING("Squad request Task from ArmyManager which was not in attacking state")
#endif
		return nullptr;
	}

	const vector<Unitset>& clusters = OpponentTracker::Instance()->getOpponentClusters();
	const BaseLocation* mainBase = BaseManager::Instance()->getMainBaseLocation();
	Position mainBasePos = PositionUtils::toPosition(mainBase->getOptimalDepotLocation());

	if(!clusters.empty())	// attack enemy cluster closest to our main base
	{
		int minDistSq = MathConstants::MAX_INT;
		Unitset nearestCluster = Unitset::none;

		for(auto it = clusters.begin(); it != clusters.end(); ++it)
		{
			Unitset cluster = *it;
			const int distSq = Distances::getSquaredDistance(mainBasePos, cluster.getPosition());

			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				nearestCluster = cluster;
			}
		}

		return new AttackUnits(Priorities::PRIORITY_COMBAT_ATTACK, squad, nearestCluster);
	}
	else					// attack enemy base closest to our main base
	{
		const vector<Base>& bases = BaseManager::Instance()->getOpponentBases();

		if(!bases.empty())
		{
			int minDistSq = MathConstants::MAX_INT;
			Base nearestBase = bases.front();

			for(auto it = bases.begin(); it != bases.end(); ++it)
			{
				Base base = *it;
				const int distSq = Distances::getSquaredDistance(mainBasePos, base.getDepotCenter());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					nearestBase = base;
				}
			}

			return new AttackBase(Priorities::PRIORITY_COMBAT_ATTACK, squad, nearestBase.getLocation());
		}
		else
		{
			LOG_MESSAGE("Returning no Task for Squad")
			return nullptr;
		}
	}
}

const bool ArmyManager::shouldWorkersDefend() const
{
	for(auto it = defensiveSquads.begin(); it != defensiveSquads.end(); ++it)
	{
		Squad* squad = *it;
		if(squad->getSize() > 0)
		{
			return false;
		}
	}

	return true;
}

void ArmyManager::transferUnitToDefensiveSquad(Squad* previousSquad, const Unit unit)
{
	Squad* smallestSquad = nullptr;
	int smallestSquadSize = MathConstants::MAX_INT;

	for(auto it = defensiveSquads.begin(); it != defensiveSquads.end(); ++it)
	{
		Squad* squad = *it;
		const int size = squad->getSize();

		if(size < smallestSquadSize)
		{
			smallestSquad = squad;
			smallestSquadSize = size;
		}
	}

#ifdef MAASCRAFT_DEBUG
	if(!smallestSquad)
	{
		LOG_WARNING("ArmyManager:: No smallest defensive squad found!")
		return;
	}
#endif

	previousSquad->transferOwnershipTo(unit, smallestSquad);
}

#endif // ARMY_MANAGER_SCRIPTED