/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A highly customizable, template-based implementation of the A* algorithm

	Implementation slightly inspired by: https://github.com/justinhj/astar-algorithm-cpp

	NOTE:
	The current implementation assumes the UserState type is cheap to copy. TODO: Maybe an alternative implementation working with pointers?
*/

#pragma once

#include <forward_list>

#include "MathConstants.h"
#include "Predicates.h"

struct G_plus_H;		// forward declaration of functor which adds up two arguments g and h

// UserState is the user's state-space type.
// Get_Cost is a functor which returns the cost of moving from the first state to the second. Will only be called on neighbouring states!
// Get_Successors is a function which generates all successors of a given state and adds them using aStarSearch.addSuccessor()
// Goal_Distance_Estimate is a function which returns the heuristic estimate distance of a node to a goal node
// F_function is the function used to compute an f-value from a given g and h value. By default, f = g + h
// Is_Same_State is a function which returns true iff two given states are equal. By default, uses ''=='' operator
template <class UserState,
			typename Get_Cost,
			typename Get_Successors,
			typename Goal_Distance_Estimate,
			typename F_function = G_plus_H,
			typename Is_Same_State = Predicates::Equality<UserState> > 
class AStar
{
private:
	class Node;				// forward declaration of Node class

	std::vector<Node*> openList;		// Heap (simple vector but used as a heap, cf. Steve Rabin's game gems article)
	std::vector<Node*> closedList;
	std::vector<Node*> successors;		// Successors is a vector filled out by the user each time successors to a node are generated

	Node* startNode;
	Node* goalNode;

	bool solutionFound;

	AStar(const AStar& other);		// AStar objects are big and should not be copied

	// Node class used internally as wrapper around UserStates
	class Node
	{
	public:
		UserState userState;

		Node* parent;
		Node* child;

		float g;
		float h;
		float f;

		Node(const UserState state) :
			userState(state), parent(nullptr), child(nullptr), g(0.0f), h(0.0f), f(0.0f) {}

	private:
		Node();
		Node(const Node& other);
	};

	// Functor which compares f values of two Nodes
	struct HeapCompare_f
	{
		bool operator() (const Node* x, const Node* y) const
		{
			return x->f > y->f;
		}
	};

	void freeSolutionNodes()
	{
		Node* n = startNode;

		if( startNode->child )
		{
			do
			{
				Node* del = n;
				n = n->child;
				delete del;

				del = nullptr;
			} while( n != goalNode );

			delete n;
		}
		else
		{
			delete startNode;
			delete goalNode;
		}
	}

	void freeUnusedNodes()
	{
		// iterate open list and delete unused nodes
		auto iterOpen = openList.begin();

		while( iterOpen != openList.end() )
		{
			Node* n = (*iterOpen);

			if( !n->child )
			{
				delete n;
				n = nullptr;
			}

			++iterOpen;
		}

		openList.clear();

		// iterate closed list and delete unused nodes
		auto iterClosed = closedList.begin();

		for( ; iterClosed != closedList.end(); ++iterClosed )
		{
			Node* n = (*iterClosed);

			if( !n->child )
			{
				delete n;
				n = nullptr;
			}
		}

		closedList.clear();
	}

public:
	// Constructor accepts some hints which, when accurate, will have a positive impact on performance.
	// The hints allows the search engine to instantiate some vectors it uses internally with appropriate sizes
	//
	// successorsHint should be the most useful hint, and should equal the expected maximum number of successors in a single step
	//	(for example, 7 for pathfinding on 8-connected tiles, since the search will not go back to parent nodes directly)
	//
	// openListHint is the initial size reserved for the openList vector
	// same for closedListHint and closedList vector
	AStar(const int successorsHint = 0, const int openListHint = 0, const int closedListHint = 0) :
		openList(), closedList(), successors(), startNode(nullptr), goalNode(nullptr), solutionFound(false)
	{
		successors.reserve(successorsHint);
		openList.reserve(openListHint);
		closedList.reserve(closedListHint);
	}

	~AStar()
	{
		freeSolutionNodes();

		if(!solutionFound)
			freeUnusedNodes();
	}

	inline const bool foundSolution()
	{
		return solutionFound;
	}

	void setStartAndGoalStates( const UserState& start, const UserState& goal )
	{
		startNode = new Node(start);
		goalNode = new Node(goal);

		// Initialise the A* specific parts of the Start Node
		// The user only needs fill out the state information
		startNode->g = 0.0f;
		startNode->h = Goal_Distance_Estimate()(startNode->userState, goalNode->userState);
		startNode->f = F_function()(startNode->g, startNode->h);
		startNode->parent = nullptr;

		// Push the start node on the Open list
		openList.push_back( startNode ); // heap now unsorted

		// Sort back element into heap
		push_heap( openList.begin(), openList.end(), HeapCompare_f() );
	}

	// Advances search one step
	// Returns false if path cannot be found. Returns true otherwise.
	const bool searchStep()
	{
		// Next I want it to be safe to do a searchstep once the search has succeeded...
		if( solutionFound )
			return true;

		// Failure is defined as emptying the open list as there is nothing left to search...
		if( openList.empty() )
		{
			solutionFound = false;
			return false;
		}

		// Pop the best node (the one with the lowest f)
		Node* n = openList.front();
		std::pop_heap( openList.begin(), openList.end(), HeapCompare_f() );
		openList.pop_back();

		// Check for the goal, once we pop that we're done
		if(	Is_Same_State()(n->userState, goalNode->userState ) )
		{
			// The user is going to use the Goal Node he passed in
			// so copy the parent pointer of n
			goalNode->parent = n->parent;
			goalNode->g = n->g;

			// A special case is that the goal was passed in as the start state
			// so handle that here
			if(	!(Is_Same_State()(n->userState, startNode->userState)) )
			{
				delete n;

				// set the child pointers in each node (except Goal which has no child)
				Node* nodeChild = goalNode;
				Node* nodeParent = goalNode->parent;

				do
				{
					nodeParent->child = nodeChild;
					nodeChild = nodeParent;
					nodeParent = nodeParent->parent;
				}
				while( nodeChild != startNode ); // Start is always the first node by definition

			}

			// delete nodes that aren't needed for the solution
			freeUnusedNodes();
			solutionFound = true;
			return true;
		}
		else // not goal
		{
			// We now need to generate the successors of this node
			// The user helps us to do this, and we keep the new nodes in successors
			successors.clear();

			// User provides this functions and uses addSuccessor to add each successor of
			// node 'n' to successors
			Get_Successors()(n->userState, n->parent ? &n->parent->userState : nullptr, this);

			// Now handle each successor to the current node ...
			for( auto successor = successors.begin(); successor != successors.end(); ++successor )
			{
				float new_g = n->g + Get_Cost()(n->userState, (*successor)->userState);

				// Now we need to find whether the node is on the open or closed lists
				// If it is but the node that is already on them is better (lower g)
				// then we can forget about this successor
				//
				// First linear search of open list to find node
				auto openlist_result = openList.begin();
				for( ; openlist_result != openList.end(); ++openlist_result )
				{
					if(	Is_Same_State()((*openlist_result)->userState, (*successor)->userState) )
						break;	
				}

				if( openlist_result != openList.end() )
				{
					// we found this state on open
					if( (*openlist_result)->g <= new_g )
					{
						// the one on Open is cheaper than this one
						delete (*successor);
						continue;
					}
				}

				auto closedlist_result = closedList.begin();
				for( ; closedlist_result != closedList.end(); ++closedlist_result )
				{
					if(	Is_Same_State()((*closedlist_result)->userState, (*successor)->userState) )
						break;	
				}

				if( closedlist_result != closedList.end() )
				{
					// we found this state on closed
					if( (*closedlist_result)->g <= new_g )
					{
						// the one on Closed is cheaper than this one
						delete (*successor);
						continue;
					}
				}

				// This node is the best node so far with this particular state
				(*successor)->parent = n;
				(*successor)->g = new_g;
				(*successor)->h = Goal_Distance_Estimate()((*successor)->userState, goalNode->userState);
				(*successor)->f = F_function()((*successor)->g, (*successor)->h);

				// Remove successor from closed if it was on it
				if( closedlist_result != closedList.end() )
				{
					delete (*closedlist_result);
					closedList.erase( closedlist_result );
				}

				// Update old version of this node
				if( openlist_result != openList.end() )
				{	
					delete (*openlist_result);
					openList.erase( openlist_result );

					// re-make the heap
					std::make_heap( openList.begin(), openList.end(), HeapCompare_f() );
				}

				// heap now unsorted
				openList.push_back( (*successor) );

				// sort back element into heap
				std::push_heap( openList.begin(), openList.end(), HeapCompare_f() );
			}

			// push n onto Closed, as we have expanded it now
			if(closedList.size() == closedList.capacity())
			{
				for(auto iter = closedList.begin(); iter != closedList.end(); ++iter)
				{
					Node* iterNode = *iter;
				}
			}
			closedList.push_back( n );
		}
		return true;
	}

	// User calls this to add a successor to a list of successors when expanding the search frontier
	const bool addSuccessor( UserState& state )
	{
		Node* node = new Node(state);
		if( node )
		{
			successors.push_back( node );
			return true;
		}
		return false;
	}

	std::forward_list<UserState> getPath() const
	{
		std::forward_list<UserState> path;

		if(!solutionFound)
			return path;

		Node* n = goalNode;
		path.push_front(n->userState);

		while(n->parent)
		{
			n = n->parent;
			path.push_front(n->userState);
		}

		return path;
	}

	// Returns MathConstants::MAX_FLOAT if goal is not defined or there is no solution
	float getSolutionCost()
	{
		if( goalNode && solutionFound )
			return goalNode->g;
		else
			return MathConstants::MAX_FLOAT;
	}
};

// Functor which computes f = g + h
struct G_plus_H
{
	float operator() (const float g, const float h) const
	{
		return g + h;
	}
};