#include "NavigationAgent.h"
#include "PFFunctions.h"
#include "AgentManager.h"
#include "MapManager.h"
#include "Commander.h"
#include "Profiler.h"
#include "WalkTile.h"
#include <math.h>

bool NavigationAgent::instanceFlag = false;
NavigationAgent* NavigationAgent::instance = NULL;

NavigationAgent::NavigationAgent()
{
	checkRange = 5;
	mapW = Broodwar->mapWidth() * 4;
	mapH = Broodwar->mapHeight() * 4;
}

NavigationAgent::~NavigationAgent()
{
	instanceFlag = false;
	instance = NULL;
}

NavigationAgent* NavigationAgent::getInstance()
{
	if (!instanceFlag)
	{
		instance = new NavigationAgent();
		instanceFlag = true;
	}
	return instance;
}

bool NavigationAgent::computeMove(BaseAgent* agent, TilePosition goal, bool defensive)
{
	bool cmd = false;
	double r = agent->getUnitType().seekRange();
	if (agent->getUnitType().sightRange() > r)
	{
		r = agent->getUnitType().sightRange();
	}
	
	bool enemyInRange = false;
	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++)
	{
		double dist = agent->getUnit()->getPosition().getDistance((*i)->getPosition());
		if (dist <= r)
		{
			enemyInRange = true;
			break;
		}
	}

	//Retreat to center of the squad if the enemy
	//is overwhelming.
	if (agent->isUnderAttack() && (agent->getUnit()->isIdle() || agent->getUnit()->isInterruptible()))
	{
		int ownI = MapManager::getInstance()->getOwnGroundInfluenceIn(agent->getUnit()->getTilePosition());
		int enI = MapManager::getInstance()->getEnemyGroundInfluenceIn(agent->getUnit()->getTilePosition());
		if (enI > ownI)
		{
			//Broodwar->printf("Retreat from (%d, %d) %d<%d", agent->getUnit()->getTilePosition().x(), agent->getUnit()->getTilePosition().y(), ownI, enI);
			Squad* sq = Commander::getInstance()->getSquad(agent->getSquadID());
			if (sq != NULL && !sq->isExplorer())
			{
				TilePosition center = sq->getCenter();
				//Broodwar->printf("Squad center is at (%d,%d)", center.x(), center.y());
				if (center.getDistance(agent->getUnit()->getTilePosition()) >= 4 && !agent->getUnit()->isCloaked())
				{
					if (agent->getUnit()->isSieged())
					{
						agent->getUnit()->unsiege();
						return true;
					}
					if (agent->getUnit()->isBurrowed())
					{
						agent->getUnit()->unburrow();
						return true;
					}
					//Broodwar->printf("Retreat from (%d, %d) %d<%d", agent->getUnit()->getTilePosition().x(), agent->getUnit()->getTilePosition().y(), ownI, enI);
					agent->getUnit()->rightClick(Position(center));
					return true;
				}
			}
		}
	}

	if (enemyInRange)
	{
		//Profiler::getInstance()->start("PFmove");
		//computePotentialFieldMove(agent, defensive);
		//Profiler::getInstance()->end("PFmove");

		Profiler::getInstance()->start("BoidsMove");
		cmd = computeBoidsMove(agent, defensive);
		Profiler::getInstance()->end("BoidsMove");
	}
	else
	{
		Profiler::getInstance()->start("NormMove");
		cmd = computePathfindingMove(agent, goal);
		Profiler::getInstance()->end("NormMove");
	}
	return cmd;
}

int NavigationAgent::getMaxUnitSize(UnitType type)
{
	int size = type.dimensionDown();
	if (type.dimensionLeft() > size) size = type.dimensionLeft();
	if (type.dimensionRight() > size) size = type.dimensionRight();
	if (type.dimensionUp() > size) size = type.dimensionUp();
	return size;
}


bool NavigationAgent::computeBoidsMove(BaseAgent* agent, bool defensive)
{
	if (!agent->getUnit()->isIdle() && !agent->getUnit()->isMoving()) return false;

	Unit* unit = agent->getUnit();
	if (unit->isSieged() || unit->isBurrowed() || unit->isLoaded())
	{
		return false;
	}

	//The difference from current position the agent
	//shall move to.
	double aDiffX = 0;
	double aDiffY = 0;

	//Apply goal
	if (agent->getGoal().x() != -1 && agent->getGoal().y() != -1)
	{
		Position goal = Position(agent->getGoal());
		double addX = ((double)goal.x() - (double)agent->getUnit()->getPosition().x()) / 100.0;
		double addY = ((double)goal.y() - (double)agent->getUnit()->getPosition().y()) / 100.0;
		//if (addX != 0 || addY != 0) Broodwar->printf("Apply goal (%f,%f)", addX, addY);
		aDiffX += addX;
		aDiffX += addY;
	}

	//Apply average position for the squad
	double totDX = 0;
	double totDY = 0;
	Squad* sq = Commander::getInstance()->getSquad(agent->getSquadID());
	if (sq != NULL)
	{
		vector<BaseAgent*> agents = sq->getMembers();
		int no = 0;
		for(int i = 0; i < (int)agents.size(); i++)
		{
			if (agents.at(i)->isAlive() && agents.at(i)->getUnitID() != agent->getUnitID())
			{
				totDX += (double)agents.at(i)->getUnit()->getPosition().x();
				totDY += (double)agents.at(i)->getUnit()->getPosition().y();
				no++;
			}
		}

		totDX = totDX / (double)(no - 1);
        totDY = totDY / (double)(no - 1);
		
		double addX = (totDX - (double)agent->getUnit()->getPosition().x()) / 100.0;
		double addY = (totDY - (double)agent->getUnit()->getPosition().y()) / 100.0;
		//if (addX != 0 || addY != 0) Broodwar->printf("Apply avg position (%f,%f)", addX, addY);
		aDiffX += addX;
		aDiffX += addY;
	}

	//Apply average heading for the squad
	totDX = 0;
	totDY = 0;
	if (sq != NULL)
	{
		vector<BaseAgent*> agents = sq->getMembers();
		int no = 0;
		for(int i = 0; i < (int)agents.size(); i++)
		{
			if (agents.at(i)->isAlive() && agents.at(i)->getUnitID() != agent->getUnitID())
			{
				totDX += cos(agents.at(i)->getUnit()->getAngle());
				totDY += sin(agents.at(i)->getUnit()->getAngle());
				no++;
			}
		}

		totDX = totDX / (double)(no - 1);
        totDY = totDY / (double)(no - 1);
		
		double addX = (totDX - cos(agent->getUnit()->getAngle())) / 5.0;
		double addY = (totDY - sin(agent->getUnit()->getAngle())) / 5.0;
		//if (addX != 0 || addY != 0) Broodwar->printf("Apply avg heading (%f,%f)", addX, addY);
		aDiffX += addX;
		aDiffX += addY;
	}

	//Apply separation from own units. Does not apply for air units
	totDX = 0;
	totDY = 0;
	double detectionLimit = 10.0;
	
	if (sq != NULL && !agent->getUnitType().isFlyer())
	{
		vector<BaseAgent*> agents = sq->getMembers();
		int no = 0;
		for(int i = 0; i < (int)agents.size(); i++)
		{
			//Set detection limit to be the radius of both units + 2
			detectionLimit = (double)(getMaxUnitSize(agent->getUnitType()) + getMaxUnitSize(agents.at(i)->getUnitType()) + 2);
			
			if (agents.at(i)->isAlive() && agents.at(i)->getUnitID() != agent->getUnitID())
			{
				double d = agent->getUnit()->getPosition().getDistance(agents.at(i)->getUnit()->getPosition());
				if (d <= detectionLimit)
				{
					totDX -= (agents.at(i)->getUnit()->getPosition().x() - agent->getUnit()->getPosition().x());
					totDY -= (agents.at(i)->getUnit()->getPosition().y() - agent->getUnit()->getPosition().y());
				}
			}
		}

		double addX = totDX / 5.0;
		double addY = totDY / 5.0;
		//if (addX != 0 || addY != 0) Broodwar->printf("Apply separation, own (%f,%f)", addX, addY);
		aDiffX += addX;
		aDiffX += addY;
	}

	//Apply separation from enemy units
	totDX = 0;
	totDY = 0;

	//Check if the agent targets ground and/or air
	//Check range of weapons
	bool targetsGround = false;
	int groundRange = 0;
	if (agent->getUnitType().groundWeapon().targetsGround())
	{
		targetsGround = true;
		if (agent->getUnitType().groundWeapon().maxRange() > groundRange)
		{
			groundRange = agent->getUnitType().groundWeapon().maxRange();
		}
	}
	if (agent->getUnitType().airWeapon().targetsGround())
	{
		targetsGround = true;
		if (agent->getUnitType().airWeapon().maxRange() > groundRange)
		{
			groundRange = agent->getUnitType().airWeapon().maxRange();
		}
	}

	bool targetsAir = false;
	int airRange = 0;
	if (agent->getUnitType().groundWeapon().targetsAir())
	{
		targetsAir = true;
		if (agent->getUnitType().groundWeapon().maxRange() > groundRange)
		{
			airRange = agent->getUnitType().groundWeapon().maxRange();
		}
	}
	if (agent->getUnitType().airWeapon().targetsAir())
	{
		targetsAir = true;
		if (agent->getUnitType().airWeapon().maxRange() > groundRange)
		{
			airRange = agent->getUnitType().airWeapon().maxRange();
		}
	}

	//Iterate through enemies
	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++)
	{
		if ((*i)->exists())
		{
			UnitType t = (*i)->getType();
			double detectionLimit = 100000;
			if (t.isFlyer() && targetsAir) detectionLimit = (double)airRange - getMaxUnitSize(agent->getUnitType())-2;
			if (!t.isFlyer() && targetsGround) detectionLimit = (double)groundRange - getMaxUnitSize(agent->getUnitType())-2;
			if (!agent->getUnitType().canAttack() && agent->getUnitType().isFlyer()) detectionLimit = (double)((*i)->getType().airWeapon().maxRange()) + 8;
			if (!agent->getUnitType().canAttack() && !agent->getUnitType().isFlyer()) detectionLimit = (double)((*i)->getType().groundWeapon().maxRange()) + 8;
			if (detectionLimit < 5) detectionLimit = 5; //Minimum separation

			double d = agent->getUnit()->getPosition().getDistance((*i)->getPosition());
			if (d <= detectionLimit)
			{
				totDX -= ((*i)->getPosition().x() - agent->getUnit()->getPosition().x());
				totDY -= ((*i)->getPosition().y() - agent->getUnit()->getPosition().y());
			}
		}

		double addX = totDX;
		double addY = totDY;
		//if (addX != 0 || addY != 0) Broodwar->printf("Apply separation, enemy (%f,%f)", addX, addY);
		aDiffX += addX;
		aDiffX += addY;
	}

	//Apply separation  from terrain
	totDX = 0;
	totDY = 0;
	WalkTile unitWT = WalkTile(agent->getUnit()->getPosition());
	detectionLimit = (double)(getMaxUnitSize(agent->getUnitType()) + 16);
	for (int tx = unitWT.X() - 2; tx <= unitWT.X() + 2; tx++)
	{
		for (int ty = unitWT.Y() - 2; ty <= unitWT.Y() + 2; ty++)
		{
			if (!Broodwar->isWalkable(tx, ty))
			{
				WalkTile terrainWT = WalkTile(tx, ty);
				double d = agent->getUnit()->getPosition().getDistance(terrainWT.getPosition());
				if (d <= detectionLimit)
				{
					totDX -= (terrainWT.getPosition().x() - agent->getUnit()->getPosition().x());
					totDY -= (terrainWT.getPosition().y() - agent->getUnit()->getPosition().y());
				}
			}
		}
	}
	double addX = totDX / 10.0; //TODO: 5.0 ok?
	double addY = totDY / 10.0;
	//if (addX != 0 || addY != 0) Broodwar->printf("Apply terrain separation (%d,%d)", (int)addX, (int)addY);
	aDiffX += addX;
	aDiffX += addY;

	//Update new position
	int newX = (int)(agent->getUnit()->getPosition().x() + aDiffX);
	int newY = (int)(agent->getUnit()->getPosition().y() + aDiffY);
	Position toMove = Position(newX, newY);

	if (agent->getUnit()->getPosition().getDistance(toMove) >= 1)
	{
		//Broodwar->printf("Boids move (%d,%d) -> (%d,%d)", agent->getUnit()->getPosition().x(), agent->getUnit()->getPosition().y(), newX, newY);
		if (defensive)
		{
			return agent->getUnit()->rightClick(toMove);
		}
		else
		{
			bool ok = agent->getUnit()->attack(toMove);
			if (!ok) Broodwar->printf("Move failed: %s", Broodwar->getLastError().c_str());
			return ok;
		}
	}

	return false;
}


bool NavigationAgent::computePotentialFieldMove(BaseAgent* agent, bool defensive)
{
	if (!agent->getUnit()->isIdle() && !agent->getUnit()->isMoving()) return false;

	Unit* unit = agent->getUnit();

	if (unit->isSieged() || unit->isBurrowed() || unit->isLoaded())
	{
		return false;
	}
	
	WalkTile unitWT = WalkTile(unit->getPosition());
	int wtX = unitWT.X();
	int wtY = unitWT.Y();

	float bestP = getAttackingUnitP(agent, unitWT, defensive);
	//bestP += PFFunctions::getGoalP(Position(unitX,unitY), goal);
	//bestP += PFFunctions::getTrailP(agent, unitX, unitY);
	bestP += PFFunctions::getTerrainP(agent, unitWT);

	float cP = 0;
	
	float startP = bestP;
    int bestX = wtX;
    int bestY = wtY;

	for (int cX = wtX - checkRange; cX <= wtX + checkRange; cX++)
	{
        for (int cY = wtY - checkRange; cY <= wtY + checkRange; cY++)
		{
			if (cX >= 0 && cY >= 0 && cX <= mapW && cY <= mapH)
			{
				WalkTile wt = WalkTile(cX, cY);
				cP = getAttackingUnitP(agent, wt, defensive);
				//cP += PFFunctions::getGoalP(Position(cX,cY), goal);
				//cP += PFFunctions::getTrailP(agent, cX, cY);
				cP += PFFunctions::getTerrainP(agent, wt);
				
                if (cP > bestP)
				{
					bestP = cP;
					bestX = cX;
					bestY = cY;
				}
            }
        }
    }
	
	if (bestX != wtX || bestY != wtY)
	{
		WalkTile wt = WalkTile(bestX, bestY);
		Position toMove = wt.getPosition();

		if (defensive)
		{
			return agent->getUnit()->rightClick(toMove);
		}
		else
		{
			return agent->getUnit()->attack(toMove);
		}
    }

	return false;
}

bool NavigationAgent::computePathfindingMove(BaseAgent* agent, TilePosition goal)
{
	TilePosition checkpoint = goal;
	if (agent->getSquadID() >= 0)
	{
		Squad* sq = Commander::getInstance()->getSquad(agent->getSquadID());
		if (sq != NULL)
		{
			checkpoint = sq->nextMovePosition();
			if (agent->isOfType(UnitTypes::Terran_SCV))
			{
				checkpoint = sq->nextFollowMovePosition();
				agent->setGoal(checkpoint);
			}
		}
	}
	
	if (goal.x() != -1)
	{
		moveToGoal(agent, checkpoint, goal);
		return true;
	}
	return false;
}

void NavigationAgent::displayPF(BaseAgent* agent)
{
	Unit* unit = agent->getUnit();
	if (unit->isBeingConstructed()) return;

	//PF
	WalkTile w = WalkTile(agent->getUnit()->getPosition());
	int tileX = w.X();
	int tileY = w.Y();
	int range = 10*3;

	for (int cTileX = tileX - range; cTileX < tileX + range; cTileX+=3)
	{
        for (int cTileY = tileY - range; cTileY < tileY + range; cTileY+=3)
		{
            if (cTileX >= 0 && cTileY >= 0 && cTileX < mapW && cTileY < mapH)
			{
				WalkTile wt = WalkTile(cTileX+1, cTileY+1);
				float p = getAttackingUnitP(agent, wt, agent->getDefensive());
				//cP += PFFunctions::getGoalP(Position(cX,cY), goal);
				//cP += PFFunctions::getTrailP(agent, cX, cY);
				p += PFFunctions::getTerrainP(agent, wt);
					
				//print box
				if (p > -950)
				{
					Position pos = wt.getPosition();
					Broodwar->drawBoxMap(pos.x()-8,pos.y()-8,pos.x()+8,pos.y()+8,getColor(p),true);
				}
            }
        }
    }
}

Color NavigationAgent::getColor(float p)
{
	if (p >= 0)
	{
		int v = (int)(p * 3);
		int halfV = (int)(p * 0.6);

		if (v > 255) v = 255;
		if (halfV > 255) halfV = 255;

		return Color(halfV, halfV, v);
	}
	else
	{
		p = -p;
		int v = (int)(p * 1.6);
		
		int v1 = 255 - v;
		if (v1 <= 0) v1 = 40;
		int halfV1 = v1 * 0.6;
		
		return Color(v1, halfV1, halfV1);
	}
}

bool NavigationAgent::moveToGoal(BaseAgent* agent,  TilePosition checkpoint, TilePosition goal)
{
	if (checkpoint.x() == -1 || goal.x() == -1) return false;
	Unit* unit = agent->getUnit();

	if (unit->isStartingAttack() || unit->isAttacking())
	{
		return false;
	}

	Position toReach = Position(checkpoint.x()*32+16, checkpoint.y()*32+16);
	double distToReach = toReach.getDistance(unit->getPosition());

	int engageDist = Broodwar->self()->groundWeaponMaxRange(unit->getType()) * 0.5;
	if (engageDist == 0) engageDist = Broodwar->self()->airWeaponMaxRange(unit->getType()) * 0.5;
	if (engageDist < 2*32) engageDist = 2*32;

	//Explorer units shall have
	//less engage dist to avoid getting
	//stuck at waypoints.
	Squad* sq = Commander::getInstance()->getSquad(agent->getSquadID());
	if (sq != NULL)
	{
		if (sq->isExplorer())
		{
			engageDist = 32;
		}
		else
		{
			//Add additional distance to avoid clogging
			//choke points.
			if (!sq->isActive())
			{
				engageDist += 4*32;
			}
		}
	}

	if (distToReach <= engageDist)
	{
		//Dont stop close to chokepoints
		TilePosition tp = unit->getTilePosition();
		Chokepoint* cp = getNearestChokepoint(tp);
		double d = tp.getDistance(TilePosition(cp->getCenter()));
		if (d > 4)
		{
			if (unit->isMoving()) unit->stop();
			return true;
		}
	}

	int squadID = agent->getSquadID();
	if (squadID != -1)
	{
		Squad* sq = Commander::getInstance()->getSquad(squadID);
		if (sq != NULL)
		{
			if (sq->isAttacking())
			{
				//Squad is attacking. Don't stop
				//at checkpoints.
				toReach = Position(goal.x()*32+16,goal.y()*32+16);
			}
		}
	}
	//Move
	if (!unit->isMoving()) return unit->attack(toReach);
	else return true;
}

float NavigationAgent::getAttackingUnitP(BaseAgent* agent, WalkTile wp, bool defensive)
{
	float p = 0;
	
	//Enemy Units
	float p_off = 0;
	float p_def = 0;
	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++)
	{
		//Enemy seen
		float dist = PFFunctions::getDistance(wp, (*i));
		float ptmp = PFFunctions::calcOffensiveUnitP(dist, agent->getUnit(), (*i));
		p_def += PFFunctions::calcDefensiveUnitP(dist, agent->getUnit(), (*i));

		if (ptmp > p_off)
		{
			p_off = ptmp;
		}
	}

	if (!defensive) p = p_off;

	if (defensive || p_off == 0)
	{
		p += p_def;
	}
	
	//Own Units
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++)
	{
		if (agents.at(i)->isAlive())
		{
			float dist = PFFunctions::getDistance(wp, agents.at(i)->getUnit());
			float ptmp = PFFunctions::calcOwnUnitP(dist, wp, agent->getUnit(), agents.at(i)->getUnit());
			p += ptmp;
		}
	}

	//Neutral Units
	for(set<Unit*>::const_iterator i=Broodwar->getNeutralUnits().begin();i!=Broodwar->getNeutralUnits().end();i++)
	{
		if ((*i)->getType().getID() == UnitTypes::Terran_Vulture_Spider_Mine.getID())
		{
			WalkTile w2 = WalkTile((*i));
			float dist = PFFunctions::getDistance(wp, (*i));
			if (dist <= 2)
			{
				p -= 50.0;
			}
		}
	}

	return p;
}

float NavigationAgent::getDefendingUnitP(BaseAgent* agent, WalkTile wp)
{
	float p = 0;
	
	p += PFFunctions::getGoalP(agent, wp);
	p += PFFunctions::getTerrainP(agent, wp);

	//Own Units
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++)
	{
		if (agents.at(i)->isAlive())
		{
			float dist = PFFunctions::getDistance(wp, agents.at(i)->getUnit());
			float ptmp = PFFunctions::calcOwnUnitP(dist, wp, agent->getUnit(), agents.at(i)->getUnit());
			p += ptmp;
		}
	}

	return p;
}
