#include "MapManager.h"
#include "AgentManager.h"
#include "ExplorationManager.h"

MapManager* MapManager::instance = NULL;

MapManager::MapManager()
{
	//Add the regions for this map
	for(set<BWTA::Region*>::const_iterator i=getRegions().begin();i!=getRegions().end();i++)
	{
		MRegion* mr = new MRegion();
		mr->region = (*i);
		mr->resetInfluence();
		map.push_back(mr);
	}
	lastCallFrame = 0;
}

const MRegion* MapManager::getMapRegion(const BWTA::Region* r)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* mr = map.at(i);
		if (mr->region->getCenter().x() == r->getCenter().x() && mr->region->getCenter().y() == r->getCenter().y())
		{
			return mr;
		}
	}
	return NULL;
}

bool MapManager::isValidChokepoint(const Chokepoint* cp)
{
	//Use this code to hard-code chokepoints that shall not be used.
	if (Broodwar->mapFileName() == "(4)Andromeda.scx")
	{
		Position c = cp->getCenter();
		if (c.x() == 2780 && c.y() == 3604) return false;
		if (c.x() == 2776 && c.y() == 448) return false;
		if (c.x() == 1292 && c.y() == 436) return false;
		if (c.x() == 1300 && c.y() == 3584) return false;
	}
	if (Broodwar->mapFileName() == "(2)Benzene.scx")
	{
		Position c = cp->getCenter();
		if (c.x() == 4044 && c.y() == 1088) return false;
		if (c.x() == 44 && c.y() == 1064) return false;
	}
	if (Broodwar->mapFileName() == "(2)Destination.scx")
	{
		Position c = cp->getCenter();
		if (c.x() == 1309 && c.y() == 3851) return false;
		if (c.x() == 1730 && c.y() == 226) return false;
	}
	if (Broodwar->mapFileName() == "(4)Fortress.scx")
	{
		Position c = cp->getCenter();
		if (c.x() == 3132 && c.y() == 912) return false;
		if (c.x() == 764 && c.y() == 3312) return false;
	}

	return true;
}

const BWTA::Chokepoint* MapManager::findGuardChokepoint(const MRegion* mr)
{
	for(set<Chokepoint*>::const_iterator c=mr->region->getChokepoints().begin();c!=mr->region->getChokepoints().end();c++)
	{
		if (isValidChokepoint((*c)))
		{
			pair<BWTA::Region*,BWTA::Region*> regions = (*c)->getRegions();
			if (regions.first->getCenter().x() == mr->region->getCenter().x() && regions.first->getCenter().y() == mr->region->getCenter().y())
			{
				const MRegion* adj = getMapRegion(regions.second);
				if (adj->inf_own_buildings == 0)
				{
					return (*c);
				}
			}

			if (regions.second->getCenter().x() == mr->region->getCenter().x() && regions.second->getCenter().y() == mr->region->getCenter().y())
			{
				const MRegion* adj = getMapRegion(regions.first);
				if (adj->inf_own_buildings == 0)
				{
					return (*c);
				}
			}
		}
	}
	return NULL;
}

const BWTA::Chokepoint* MapManager::getDefenseLocation()
{
	int bestInfluence = 0;
	const BWTA::Chokepoint* best = NULL;

	for (int i = 0; i < (int)map.size(); i++)
	{
		const MRegion* base = map.at(i);
		if (base->inf_own_buildings > bestInfluence)
		{
			const BWTA::Chokepoint* cp = findGuardChokepoint(base);
			if (cp != NULL)
			{
				if (base->inf_own_buildings > bestInfluence)
				{
					bestInfluence = base->inf_own_buildings;
					best = cp;
				}
			}
		}
	}

	return best;
}

int MapManager::getMapIndexFor(Position p)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* mr = map.at(i);
		if (mr->region->getPolygon().isInside(p))
		{
			return i;
		}
	}
	return -1;
}

bool MapManager::hasEnemyInfluence()
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		if (map.at(i)->inf_en_buildings > 0)
		{
			return true;
		}
	}
	return false;
}

void MapManager::update()
{
	//Dont call too often
	int cFrame = Broodwar->getFrameCount();
	if (cFrame - lastCallFrame < 10)
	{
		return;
	}
	lastCallFrame = cFrame;

	//Reset previous influence scores
	for (int i = 0; i < (int)map.size(); i++)
	{
		map.at(i)->resetInfluence();
	}

	//Update own influence
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++)
	{
		if (agents.at(i)->isAlive())
		{
			int index = getMapIndexFor(agents.at(i)->getUnit()->getPosition());
			if (index != -1)
			{
				MRegion* mr = map.at(index);
				BaseAgent* a = agents.at(i);
				if (a->getUnitType().isBuilding())
				{
					mr->inf_own_buildings += a->getUnitType().buildScore();
					if (a->getUnitType().canAttack())
					{
						if (a->getUnitType().groundWeapon().targetsGround() || a->getUnitType().airWeapon().targetsGround())
						{
							mr->inf_own_ground += a->getUnitType().buildScore();
						}
						if (a->getUnitType().groundWeapon().targetsAir() || a->getUnitType().airWeapon().targetsAir())
						{
							mr->inf_own_air += a->getUnitType().buildScore();
						}
					}
				}
				else if (a->getUnitType().isAddon())
				{
					//No influence from addons
				}
				else if (a->getUnitType().isWorker())
				{
					//No influence for workers
				}
				else
				{
					//Regular units
					if (a->getUnitType().canAttack())
					{
						if (a->getUnitType().groundWeapon().targetsGround() || a->getUnitType().airWeapon().targetsGround())
						{
							mr->inf_own_ground += a->getUnitType().buildScore();
						}
						if (a->getUnitType().groundWeapon().targetsAir() || a->getUnitType().airWeapon().targetsAir())
						{
							mr->inf_own_air += a->getUnitType().buildScore();
						}
					}
				}
			}
		}
	}

	//Update enemy buildings influence
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* mr = map.at(i);
		mr->inf_en_buildings = ExplorationManager::getInstance()->getSpottedInfluenceInRegion(mr->region);
	}

	//Update enemy units influence
	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++)
	{
		//Enemy seen
		if ((*i)->exists())
		{
			int index = getMapIndexFor((*i)->getPosition());
			if (index != -1)
			{
				MRegion* mr = map.at(index);
				UnitType type = (*i)->getType();
				if (!type.isWorker() && type.canAttack())
				{
					if (type.groundWeapon().targetsGround() || type.airWeapon().targetsGround())
					{
						mr->inf_en_ground += type.buildScore();
					}
					if (type.groundWeapon().targetsAir() || type.airWeapon().targetsAir())
					{
						mr->inf_en_air += type.buildScore();
					}
				}
			}
		}
	}
}

int MapManager::getOwnGroundInfluenceIn(TilePosition pos)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* cm = map.at(i);
		if (cm->region->getPolygon().isInside(Position(pos)))
		{
			return cm->inf_own_ground;
		}
	}
	return 0;
}

int MapManager::getEnemyGroundInfluenceIn(TilePosition pos)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* cm = map.at(i);
		if (cm->region->getPolygon().isInside(Position(pos)))
		{
			return cm->inf_en_ground;
		}
	}
	return 0;
}

bool MapManager::hasOwnInfluenceIn(TilePosition pos)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* cm = map.at(i);
		if (cm->inf_own_buildings > 0 && cm->region->getPolygon().isInside(Position(pos)))
		{
			return true;
		}
	}
	return false;
}

bool MapManager::hasEnemyInfluenceIn(TilePosition pos)
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* cm = map.at(i);
		if (cm->inf_en_buildings > 0 && cm->region->getPolygon().isInside(Position(pos)))
		{
			return true;
		}
	}
	return false;
}

TilePosition MapManager::findAttackPosition()
{
	MRegion* best = NULL;
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* cm = map.at(i);
		if (cm->inf_en_buildings > 0)
		{
			if (best == NULL)
			{
				best = cm;
			}
			else
			{
				//Launch an attack at the enemy controlled region with the
				//lowest influence.
				if (cm->inf_en_buildings < best->inf_en_buildings)
				{
					best = cm;
				}
			}
		}
	}
	
	if (best != NULL)
	{
		return TilePosition(best->region->getCenter());
	}
	else
	{
		return TilePosition(-1, -1);
	}
}

MapManager::~MapManager()
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		delete map.at(i);
	}
	
	instance = NULL;
}

MapManager* MapManager::getInstance()
{
	if (instance == NULL)
	{
		instance = new MapManager();
	}
	return instance;
}


void MapManager::printInfo()
{
	for (int i = 0; i < (int)map.size(); i++)
	{
		MRegion* mr = map.at(i);
		int x1 = mr->region->getCenter().x();
		int y1 = mr->region->getCenter().y();
		int x2 = x1 + 110;
		int y2 = y1 + 90;

		Broodwar->drawBoxMap(x1,y1,x2,y2,Colors::Brown,true);
		Broodwar->drawTextMap(x1 + 5, y1, "Buildings own: %d", mr->inf_own_buildings);
		Broodwar->drawTextMap(x1 + 5, y1 + 15, "Ground own: %d", mr->inf_own_ground);
		Broodwar->drawTextMap(x1 + 5, y1 + 30, "Air own: %d", mr->inf_own_air);
		Broodwar->drawTextMap(x1 + 5, y1 + 45, "Buildings en: %d", mr->inf_en_buildings);
		Broodwar->drawTextMap(x1 + 5, y1 + 60, "Ground en: %d", mr->inf_en_ground);
		Broodwar->drawTextMap(x1 + 5, y1 + 75, "Air en: %d", mr->inf_en_air);

		//Print location of each chokepoint, and also if it is blocked
		//as defense position.
		for(set<Chokepoint*>::const_iterator c=mr->region->getChokepoints().begin();c!=mr->region->getChokepoints().end();c++)
		{
			x1 = (*c)->getCenter().x();
			y1 = (*c)->getCenter().y();
			Broodwar->drawTextMap(x1, y1, "(%d,%d)", x1, y1);
			if (!isValidChokepoint((*c))) Broodwar->drawTextMap(x1, y1+12, "Blocked");
		}
		Broodwar->drawTextScreen(10, 120, "'%s'", Broodwar->mapFileName().c_str());
	}
}
