#include "ExplorationManager.h"
#include "AgentManager.h"
#include "BuildingPlacer.h"

ExplorationManager* ExplorationManager::instance = NULL;

ExplorationManager::ExplorationManager()
{
	active = true;
	
	//Add the regions for this map
	for(set<BWTA::Region*>::const_iterator i=getRegions().begin();i!=getRegions().end();i++)
	{
		exploreData.push_back(ExploreData((*i)->getCenter()));
	}

	siteSetFrame = 0;

	lastCallFrame = Broodwar->getFrameCount();

	expansionSite = TilePosition(-1, -1);
}

ExplorationManager::~ExplorationManager()
{
	instance = NULL;
}

void ExplorationManager::setInactive()
{
	active = false;
}

bool ExplorationManager::isActive()
{
	return active;
}

ExplorationManager* ExplorationManager::getInstance()
{
	if (instance == NULL)
	{
		instance = new ExplorationManager();
	}
	return instance;
}

void ExplorationManager::computeActions()
{
	//Dont call too often
	int cFrame = Broodwar->getFrameCount();
	if (cFrame - lastCallFrame < 73)
	{
		return;
	}
	lastCallFrame = cFrame;

	if (!active)
	{
		return;
	}
}

TilePosition ExplorationManager::searchExpansionSite()
{
	getExpansionSite();

	if (expansionSite.x() == -1)
	{
		expansionSite = BuildingPlacer::getInstance()->findExpansionSite();
		siteSetFrame = Broodwar->getFrameCount();
		//Broodwar->printf("Found expansion site around (%d,%d)", expansionSite.x(), expansionSite.y());
	}

	return expansionSite;
}

TilePosition ExplorationManager::getExpansionSite()
{
	if (expansionSite.x() >= 0)
	{
		if (Broodwar->getFrameCount() - siteSetFrame > 500)
		{
			expansionSite = TilePosition(-1, -1);
		}
	}

	return expansionSite;
}

void ExplorationManager::setExpansionSite(TilePosition pos)
{
	if (pos.x() >= 0)
	{
		siteSetFrame = Broodwar->getFrameCount();
		expansionSite = pos;
	}
}

TilePosition ExplorationManager::getNextToExplore(Squad* squad)
{
	TilePosition curPos = squad->getCenter();
	TilePosition goal = squad->getGoal();

	//Special case: No goal set
	if (goal.x() == -1 || goal.y() == -1)
	{
		BWTA::Region* startRegion = getRegion(curPos); 
		goal = TilePosition(startRegion->getCenter());
		return goal;
	}

	double dist = curPos.getDistance(goal);

	double acceptDist = 4;
	if (squad->isGround())
	{
		acceptDist = 6;
	}

	if (dist <= acceptDist)
	{
		//Squad is close to goal

		//1. Set region to explored
		setExplored(goal);

		//2. Find new region to explore
		BWTA::Region* startRegion = getRegion(goal);
		BWTA::Region* bestRegion = startRegion;

		if (bestRegion != NULL)
		{
			int bestLastVisitFrame = getLastVisitFrame(bestRegion);

			if (!squad->isAir())
			{
				//Ground explorers
				for(set<BWTA::Region*>::const_iterator i=startRegion->getReachableRegions().begin();i!=startRegion->getReachableRegions().end();i++)
				{
					int cLastVisitFrame = getLastVisitFrame((*i));
					TilePosition c = TilePosition((*i)->getCenter());
					if (cLastVisitFrame <= bestLastVisitFrame)
				{
						bestLastVisitFrame = cLastVisitFrame;
						bestRegion = (*i);
					}
				}
			}
			else
			{
				//Air explorers
				double bestDist = 100000;
				for(set<BWTA::Region*>::const_iterator i=getRegions().begin();i!=getRegions().end();i++)
				{
					int cLastVisitFrame = getLastVisitFrame((*i));
					TilePosition c = TilePosition((*i)->getCenter());
					double dist = c.getDistance(curPos);
					if (cLastVisitFrame < bestLastVisitFrame)
				{
						bestLastVisitFrame = cLastVisitFrame;
						bestRegion = (*i);
						bestDist = dist;
					}
					if (cLastVisitFrame == bestLastVisitFrame && dist < bestDist)
				{
						bestLastVisitFrame = cLastVisitFrame;
						bestRegion = (*i);
						bestDist = dist;
					}
				}
			}

			TilePosition newGoal = TilePosition(bestRegion->getCenter());
			return newGoal;
			//Broodwar->printf("Explorer: new goal (%d,%d) I am at (%d,%d) agentGoal (%d,%d)", newGoal.x(), newGoal.y(), curPos.x(), curPos.y(), agent->getGoal().x(), agent->getGoal().y());
		}
	}

	return TilePosition(-1, -1);
}

void ExplorationManager::setExplored(TilePosition goal)
{
	bool found = false;
	for (int i = 0; i < (int)exploreData.size(); i++)
	{
		if (exploreData.at(i).matches(goal))
		{
			exploreData.at(i).lastVisitFrame = Broodwar->getFrameCount();
			found = true;
		}
	}
}

int ExplorationManager::getLastVisitFrame(BWTA::Region* region)
{
	for (int i = 0; i < (int)exploreData.size(); i++)
	{
		if (exploreData.at(i).matches(region))
		{

			//Check if region is visible. If so, set lastVisitFrame to now
			if (Broodwar->isVisible(exploreData.at(i).center))
			{
				exploreData.at(i).lastVisitFrame = Broodwar->getFrameCount();
			}

			return exploreData.at(i).lastVisitFrame;
		}
	}
	
	//Error: No region found
	TilePosition goal = TilePosition(region->getCenter());
	Broodwar->printf("FATAL GetLastVF: Unable to find region for tile (%d,%d)", goal.x(), goal.y());
	return -1;
}

void ExplorationManager::printInfo()
{
	//Uncomment this if you want to draw a mark at detected enemy buildings.
	/*for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			int x1 = spottedBuildings.at(i)->getTilePosition().x() * 32;
			int y1 = spottedBuildings.at(i)->getTilePosition().y() * 32;
			int x2 = x1 + 32;
			int y2 = y1 + 32;

			Broodwar->drawBoxMap(x1,y1,x2,y2,Colors::Blue,true);
		}
	}*/

	//Draw a circle around detectors
}

void ExplorationManager::addSpottedUnit(Unit* unit)
{
	if (unit->getType().isBuilding())
	{
		
		//Check if we already have seen this building
		bool found = false;
		for (int i = 0; i < (int)spottedBuildings.size(); i++)
		{
			if (spottedBuildings.at(i)->getUnitID() == unit->getID())
			{
				found = true;
				break;
			}
		}

		if (!found)
		{
			spottedBuildings.push_back(new SpottedObject(unit));
		}
	}
	else
	{
		bool found = false;
		for (int i = 0; i < (int)spottedUnits.size(); i++)
		{
			if (spottedUnits.at(i)->getUnitID() == unit->getID())
			{
				found = true;
				break;
			}
		}

		if (!found)
		{
			spottedUnits.push_back(new SpottedObject(unit));
		}
	}
}

void ExplorationManager::unitDestroyed(Unit* unit)
{
	TilePosition uPos = unit->getTilePosition();
	if (unit->getType().isBuilding())
	{
		bool removed = false;
		for (int i = 0; i < (int)spottedBuildings.size(); i++)
		{
			TilePosition sPos = spottedBuildings.at(i)->getTilePosition();
			if (uPos.x() == sPos.x() && uPos.y() == sPos.y())
			{
				spottedBuildings.at(i)->setInactive();
				removed = true;
			}
		}

		if (!removed)
		{
			//Broodwar->printf("[EM]: Building %s at (%d,%d) was not removed from EM!!!", unit->getType().getName().c_str(), uPos.x(), uPos.y());
		}
	}
	else
	{
		for (int i = 0; i < (int)spottedUnits.size(); i++)
		{
			if (spottedUnits.at(i)->getUnitID() == unit->getID())
			{
				spottedUnits.at(i)->setInactive();
				//Broodwar->printf("[EM]: Remove %s at (%d,%d)", unit->getType().getName().c_str(), uPos.x(), uPos.y());
			}
		}
	}
}

void ExplorationManager::cleanup()
{
	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			if (Broodwar->isVisible(spottedBuildings.at(i)->getTilePosition()))
			{
				int id = spottedBuildings.at(i)->getUnitID();
				bool found = false;
				for(set<Unit*>::const_iterator it=Broodwar->enemy()->getUnits().begin();it!=Broodwar->enemy()->getUnits().end();it++)
				{
					if ((*it)->exists())
				{
						if ((*it)->getID() == id)
				{
							found = true;
							break;
						}
					}
				}
				if (!found)
				{
					spottedBuildings.at(i)->setInactive();
				}
			}
		}
	}
}

int ExplorationManager::getSpottedInfluenceInRegion(const BWTA::Region* region)
{
	int im = 0;
	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			if (region->getPolygon().isInside(spottedBuildings.at(i)->getPosition()))
			{
				im += spottedBuildings.at(i)->getType().buildScore();
			}
		}
	}
	return im;
}

int ExplorationManager::spottedBuildingsWithinRange(TilePosition pos, int range)
{
	cleanup();

	int eCnt = 0;
	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			if (pos.getDistance(spottedBuildings.at(i)->getTilePosition()) <= range)
			{
				eCnt++;
			}
		}
	}

	return eCnt;
}

TilePosition ExplorationManager::getClosestSpottedBuilding(TilePosition start)
{
	cleanup();

	TilePosition pos = TilePosition(-1, -1);
	double bestDist = 100000;

	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			double cDist = start.getDistance(spottedBuildings.at(i)->getTilePosition());
			if (cDist < bestDist)
			{
				bestDist = cDist;
				pos = spottedBuildings.at(i)->getTilePosition();
			}
		}
	}

	return pos;
}

const vector<SpottedObject*>& ExplorationManager::getSpottedBuildings()
{
	cleanup();
	return spottedBuildings;
}

bool ExplorationManager::buildingsSpotted()
{
	if (spottedBuildings.size() > 0)
	{
		return true;
	}
	return false;
}

bool ExplorationManager::canReach(TilePosition a, TilePosition b)
{
	int w = Broodwar->mapWidth();
	int h = Broodwar->mapHeight();
	if (a.x() < 0 || a.x() >= w || a.y() < 0 || a.y() >= h)
	{
		return false;
	}
	if (b.x() < 0 || b.x() >= w || b.y() < 0 || b.y() >= h)
	{
		return false;
	}
	bool ok = true;

	ok = a.hasPath(b);
	
	return ok;
}

bool ExplorationManager::canReach(BaseAgent* agent, TilePosition b)
{
	return agent->getUnit()->hasPath(Position(b));
}

bool ExplorationManager::enemyIsProtoss()
{
	for(set<Player*>::const_iterator i=Broodwar->getPlayers().begin();i!=Broodwar->getPlayers().end();i++)
	{
		if ((*i)->isEnemy(Broodwar->self()))
		{
			if ((*i)->getRace().getID() == Races::Protoss.getID())
			{
				return true;
			}
		}
	}
	return false;
}

bool ExplorationManager::enemyIsZerg()
{
	for(set<Player*>::const_iterator i=Broodwar->getPlayers().begin();i!=Broodwar->getPlayers().end();i++)
	{
		if ((*i)->isEnemy(Broodwar->self()))
		{
			if ((*i)->getRace().getID() == Races::Zerg.getID())
			{
				return true;
			}
		}
	}
	return false;
}

bool ExplorationManager::enemyIsTerran()
{
	for(set<Player*>::const_iterator i=Broodwar->getPlayers().begin();i!=Broodwar->getPlayers().end();i++)
	{
		if ((*i)->isEnemy(Broodwar->self()))
		{
			if ((*i)->getRace().getID() == Races::Terran.getID())
			{
				return true;
			}
		}
	}
	return false;
}

bool ExplorationManager::enemyIsUnknown()
{
	if (!enemyIsTerran() && !enemyIsProtoss() && !enemyIsZerg())
	{
		return true;
	}
	return false;
}

TilePosition ExplorationManager::scanForVulnerableBase()
{
	TilePosition spot = TilePosition(-1, -1);
	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			SpottedObject* obj = spottedBuildings.at(i);
			if (obj->getType().isResourceDepot())
			{
				if (!isDetectorCovering(obj->getTilePosition()))
				{
					//Broodwar->printf("Found probable vulnerable base at (%d,%d)", obj->getTilePosition().x(), obj->getTilePosition().y());
					spot = obj->getTilePosition();
				}
			}
		}
	}

	if (spot.x() < 0)
	{
		//Broodwar->printf("Scan: No vulnerable base found");
	}

	return spot;
}

bool ExplorationManager::isDetectorCovering(TilePosition pos)
{
	return isDetectorCovering(Position(pos));
}

bool ExplorationManager::isDetectorCovering(Position pos)
{
	for (int i = 0; i < (int)spottedBuildings.size(); i++)
	{
		if (spottedBuildings.at(i)->isActive())
		{
			SpottedObject* obj = spottedBuildings.at(i);
			if (obj->getType().isDetector())
			{
				double dist = obj->getPosition().getDistance(pos);
				if (dist <= obj->getType().sightRange())
				{
					return true;
				}
			}
		}
	}
	return false;
}
