package org.bwapi.unit;

import static org.junit.Assert.assertNotNull;

import java.awt.event.InputEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.bwapi.proxy.ProxyBot;
import org.bwapi.proxy.ProxyBotFactory;
import org.bwapi.proxy.ProxyServer;
import org.bwapi.proxy.model.Game;
import org.bwapi.proxy.model.Player;
import org.bwapi.proxy.model.Position;
import org.bwapi.proxy.model.ROUnit;
import org.bwapi.proxy.util.Pair;
import org.bwapi.unit.model.BroodwarButton;
import org.xvolks.jnative.exceptions.NativeException;
import org.xvolks.jnative.misc.basicStructures.HWND;
import org.xvolks.jnative.util.User32;

/**
 * Base test case for JUnit 4 testing
 * 
 * @author Chad Retz
 */
public class BwapiTestCase {

	private static final String BWAPI_REPLAY_FILE = "maps\\replays\\BWAPI_LastReplay.rep";
	private static final String REPLAY_FILE = "maps\\replays\\LastReplay.rep";

	/**
	 * Start a Starcraft game with the given settings
	 * 
	 * @param testInformation
	 * @param chaosProcess
	 * @throws Exception
	 */
	private static void startStarcraft(BwapiTestInformation testInformation, AtomicReference<Process> chaosProcess) {
		assertNotNull("testInformation must be set in base class", testInformation);
		// put the map folder there if not there
		BwapiTestUtils.createMapFileDirectory(new File(BwapiTestUtils.getMapDirectory(testInformation
		        .getStarcraftFolder())));
		// load up SC
		HWND scWnd = BwapiTestUtils.loadStarcraft(testInformation.getChaosLauncherFolder(), chaosProcess);
		// single player
		try {
			BwapiTestUtils.waitForAndClickButton(scWnd, BroodwarButton.SINGLE_PLAYER);

			// select expansion
			BwapiTestUtils.waitForAndClickButton(scWnd, BroodwarButton.EXPANSION_PACK);
			// select default ID
			BwapiTestUtils.waitForAndClickButton(scWnd, BroodwarButton.ID_OK);
			// play custom
			BwapiTestUtils.delay(1000); // FIXME: more delay
			BwapiTestUtils.waitForAndClickButton(scWnd, BroodwarButton.PLAY_CUSTOM);
			// select the map after waiting
			BwapiTestUtils.delay(1000);
			BwapiTestUtils.selectMap(scWnd, testInformation.getStarcraftFolder(), testInformation.getMap());
			// game type
			BwapiTestUtils.selectGameType(scWnd, testInformation.getGameType());
			// players
			BwapiTestUtils.setupPlayers(scWnd, testInformation.getPlayers());
			// click OK
			BwapiTestUtils.relativeClick(scWnd, BroodwarButton.ID_OK.getX(), BroodwarButton.ID_OK.getY(),
			        InputEvent.BUTTON1_MASK);
			// click start (wait just a sec...)
			BwapiTestUtils.delay(1000);
			BwapiTestUtils.relativeClick(scWnd, BroodwarButton.ID_OK.getX(), BroodwarButton.ID_OK.getY(),
			        InputEvent.BUTTON1_MASK);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Start a Starcraft game with the given settings
	 * 
	 * @param testInformation
	 * @param chaosProcess
	 * @throws Exception
	 */
	private static void startStarcraftFancyStyle(BwapiTestInformation testInformation, AtomicReference<Process> chaosProcess) {
		assertNotNull("testInformation must be set in base class", testInformation);
		// put the map folder there if not there
		BwapiTestUtils.createMapFileDirectory(new File(BwapiTestUtils.getMapDirectory(testInformation
		        .getStarcraftFolder())));
		try {
	    BwapiTestUtils.makeLocalMapCopy(testInformation.getStarcraftFolder(), testInformation.getMap());
    } catch (Exception e) {
    	throw new RuntimeException(e);
    }
		BwapiTestUtils.writeSinglePlayerIniFile(testInformation, BWAPI_REPLAY_FILE);
		BwapiTestUtils.loadStarcraft(testInformation.getChaosLauncherFolder(), chaosProcess);
	}
	
	/**
	 * Stop Starcraft
	 * 
	 * @param chaosProcess
	 * @throws Exception
	 */
	private static void stopStarcraft() throws Exception {
		// grab SC and close
		// HWND scWnd = BwapiTestUtils.getStarcraftWindow();
		// if (scWnd != null) {
		// BwapiTestUtils.relativeClick(scWnd, 322, 249,
		// InputEvent.BUTTON1_MASK);
		// wait a little
		// BwapiTestUtils.delay(500);
		// just kill the damned process
		BwapiTestUtils.killStarcraft();
		// }
	}
	
	protected static void executeBotVsBot(final BwapiTestInformation info1, final BwapiTestInformation info2,
			final File outputDir, boolean fast) throws Exception {
		AtomicReference<Process> chaosProcess = new AtomicReference<Process>(null);
			
		File fastFile = new File(info1.getStarcraftFolder(), "superfast");
		if (fast && !fastFile.exists()) {
			System.out.println("Creating " + fastFile.getPath());
			fastFile.createNewFile();
		}
		if (!fast && fastFile.exists()) {
			System.out.println("Deleting " + fastFile.getPath());
			fastFile.delete();
		}

		final File output1 = new File(outputDir, "bot1-heartbeat.txt");
		final File output2 = new File(outputDir, "bot2-heartbeat.txt");
		
		// start up starcraft
		Pair<Integer,Integer> pids = startDoubleStarcraft(info1, info2, chaosProcess);

		// accumulate stats and wait for game to end or be interrupted or something
		int lastUpdate1 = -1;
		int lastUpdate2 = -1;
		int numFailedChecks1 = 0;
		int numFailedChecks2 = 0;
		int numTimeSpikes1 = 0;
		int numTimeSpikes2 = 0;
		int numFrame0Spikes1 = 0;
		int numFrame0Spikes2 = 0;
		while (true) {
			BwapiTestUtils.delay(7000);
			boolean done = false;
			if (output1.exists()) {
				BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(output1)));
 				String line;
 				if ((line = bufferedReader.readLine()) != null) {
 					int newTime = Integer.parseInt(line.split(" ")[2]);
 					bufferedReader.close();
 					if (newTime <= lastUpdate1) {
 						if (newTime > 0 && ++numTimeSpikes1 >= 3) {
 							done = true;
 						} else if (newTime == 0 && ++numFrame0Spikes1 >= 10) {
 							done = true;
 						}
 					} else {
 						File copy = new File(outputDir, "bot1-frame-" + newTime + ".txt");
 						Runtime.getRuntime().exec("cmd.exe /c copy " + output1.getPath() + " " + copy.getPath());
 					}
 					lastUpdate1 = newTime;
 				} else {
 					bufferedReader.close();
 				}
			} else {
				numFailedChecks1++;
			}
			if (output2.exists()) {
				BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(output2)));
				String line;
				if ((line = bufferedReader.readLine()) != null) {
					int newTime = Integer.parseInt(line.split(" ")[2]);
					bufferedReader.close();
					if (newTime <= lastUpdate2) {
						if (newTime > 0 && ++numTimeSpikes2 >= 3) {
							done = true;
						} else if (newTime == 0 && ++numFrame0Spikes2 >= 10) {
							done = true;
						}
					} else {
						File copy = new File(outputDir, "bot2-frame-" + newTime + ".txt");
						Runtime.getRuntime().exec("cmd.exe /c copy " + output2.getPath() + " " + copy.getPath());
					}
					lastUpdate2 = newTime;
				} else {
					bufferedReader.close();
				}
			} else {
				numFailedChecks2++;
			}
			boolean shouldFinish = false;
			boolean p1Won = true;
			if (done) {
				System.out.println("Game ended!");
				p1Won = declareWinner(output1, output2);
				shouldFinish = true;
			}
			if (lastUpdate1 >= 45000) {
				System.out.println("Game is going on too long!");
				p1Won = declareWinner(output1, output2);
				shouldFinish = true;
			}
			if (numFailedChecks1 >= 20) {
				System.out.println("No hearbeat from Bot1!");
				p1Won = false;
				shouldFinish = true;
			}
			if (numFailedChecks2 >= 20) {
				System.out.println("No heartbeat from Bot2!");
				p1Won = true;
				shouldFinish = true;
			}
			if (shouldFinish) {
				BwapiTestUtils.killSpecificStarcraft(p1Won ? pids.getSecond() : pids.getFirst());
				BwapiTestUtils.delay(3000);
				stopStarcraft();
				break;
			}
		}
//		File replaySave = new File(outputDir, "match.rep");
//		String lastReplay = info1.getStarcraftFolder() + "\\" + REPLAY_FILE;
//		String cmd = "cmd.exe /c copy \"" + lastReplay + "\" \"" + replaySave.getPath() + "\"";
//		System.out.println("Running " + cmd);
//		Runtime.getRuntime().exec(cmd);
  }

	private static boolean declareWinner(final File output1, final File output2) throws IOException {
	  int supply1 = getSupply(output1);
	  int supply2 = getSupply(output2);
	  if (supply1 < 0 || supply2 < 0) {
	  	System.out.println("Error getting final supply values!");
	  	return false;
	  }
	  int better = Math.max(supply1, supply2);
	  int worse = Math.min(supply1, supply2);
	  if ((better - worse >= 100) ||
	      (better >= 100 && worse <= 40) ||
	      (better >= 20 && worse <= 4) ||
	      (better > 0 && worse == 0)) {
	  	boolean p1Won = supply1 > supply2;
			String winner = p1Won ? "bot1" : "bot2";
	  	System.out.println(winner + " is declared the winner!");
	  	return p1Won;
	  } else {
	  	System.out.println("Too close to call!");
	  	return false;
	  }
  }

	static Pattern pattern = Pattern.compile("Supply: (\\d+)/\\d+");
	private static int getSupply(File output) throws IOException {
		if (!output.exists()) {
			return -1;
		}
		BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(output)));
		String line;
		int supply = -1;
		if ((line = bufferedReader.readLine()) != null) {
			if ((line = bufferedReader.readLine()) != null) {
				Matcher m = pattern.matcher(line);
				if (m.matches()) {
					supply = Integer.parseInt(m.group(1));
				}
			}
		}
		bufferedReader.close();
	  return supply;
  }

	protected static void executeSinglePlayerGame(final BwapiTestInformation info, final File outputDir, boolean fast) throws Exception {
		AtomicReference<Process> chaosProcess = new AtomicReference<Process>(null);
			
		File fastFile = new File(info.getStarcraftFolder(), "superfast");
		if (fast && !fastFile.exists()) {
			System.out.println("Creating " + fastFile.getPath());
			fastFile.createNewFile();
		}
		if (!fast && fastFile.exists()) {
			fastFile.delete();
		}
		
		if (!outputDir.exists()) {
			outputDir.mkdir();
		}
		final File output = new File(outputDir, "heartbeat.txt");
		
		// start up starcraft
		startStarcraftFancyStyle(info, chaosProcess);

		// accumulate stats and wait for game to end or be interrupted or something
		int lastUpdate1 = -1;
		int numFailedChecks1 = 0;
		int numTimeSpikes1 = 0;
		int numFrame0Spikes1 = 0;
		while (true) {
			BwapiTestUtils.delay(7000);
			boolean done = false;
			if (output.exists()) {
				BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(output)));
 				String line;
				if ((line = bufferedReader.readLine()) != null) {
					int newTime = Integer.parseInt(line.split(" ")[2]);
					bufferedReader.close();
					if (newTime <= lastUpdate1) {
						if (newTime > 0 && ++numTimeSpikes1 >= 3) {
							done = true;
						} else if (newTime == 0 && ++numFrame0Spikes1 >= 20) {
							done = true;
						}
					} else {
						File copy = new File(outputDir, "frame-" + newTime + ".txt");
						Runtime.getRuntime().exec("cmd.exe /c copy " + output.getPath() + " " + copy.getPath());
					}
					lastUpdate1 = newTime;
				} else {
					bufferedReader.close();
				}
			} else {
				numFailedChecks1++;
			}
			boolean shouldFinish = false;
			if (done) {
				System.out.println("Game ended!");
				shouldFinish = true;
			}
			if (numFailedChecks1 >= 10) {
				System.out.println("No hearbeat from Bot1!");
				shouldFinish = true;
			}
			if (shouldFinish) {
				stopStarcraft();
				break;
			}
		}
		File replaySave = new File(outputDir, "match.rep");
		String lastReplay = info.getStarcraftFolder() + "\\" + REPLAY_FILE;
		String cmd = "cmd.exe /c copy \"" + lastReplay + "\" \"" + replaySave.getPath() + "\"";
		System.out.println("Running " + cmd);
		Runtime.getRuntime().exec(cmd);
  }

	@SuppressWarnings("unused")
  private static void dumpStreamToFile(File outFile, InputStream inputStream) throws FileNotFoundException,
      IOException {
	  PrintWriter out = new PrintWriter(outFile);
		BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
		for (String line = in.readLine(); line != null; line = in.readLine()) {
			out.println(line);
		}
		in.close();
		out.close();
  }

	private static Pair<Integer, Integer> startDoubleStarcraft(final BwapiTestInformation info1, final BwapiTestInformation info2,
      AtomicReference<Process> chaosProcess) throws FileNotFoundException, IOException, Exception, NativeException,
      IllegalAccessException {
		try {
			BwapiTestUtils.killStarcraft(); // just make sure no earlier instances are lying around
			BwapiTestUtils.delay(500);
			BwapiTestUtils.createMapFileDirectory(new File(BwapiTestUtils.getMapDirectory(info1.getStarcraftFolder())));
			BwapiTestUtils.makeLocalMapCopy(info1.getStarcraftFolder(), info1.getMap());
			BwapiTestUtils.delay(1000);
//			File dir = new File(BwapiTestUtils.getMapDirectory(info1.getStarcraftFolder()));
			File dir = new File("maps/Broodwar");
			
			String relativeMapFilename = dir.getAbsolutePath() + "\\" + info1.getMap().getName();
			BwapiTestUtils.writeUDPIniFile(info1.getStarcraftFolder(), relativeMapFilename, info1.getPlayers()[0], info1.getGameType(), BWAPI_REPLAY_FILE, true);
			HWND scWnd1 = BwapiTestUtils.loadStarcraft(info1.getChaosLauncherFolder(), chaosProcess);
			BwapiTestUtils.delay(3000);
			BwapiTestUtils.relativeClick(scWnd1, BroodwarButton.ID_OK.getX(), BroodwarButton.ID_OK.getY(),
					InputEvent.BUTTON1_MASK);
			BwapiTestUtils.delay(1000);
			int pid1 = BwapiTestUtils.getStarcraftPID();
			BwapiTestUtils.writeUDPIniFile(info2.getStarcraftFolder(), null, info2.getPlayers()[0], info2.getGameType(), BWAPI_REPLAY_FILE, false);
			BwapiTestUtils.loadStarcraft(info2.getChaosLauncherFolder(), chaosProcess);
			BwapiTestUtils.delay(5000);
			int pid2 = BwapiTestUtils.getStarcraftPID(pid1);
			User32.SetForegroundWindow(scWnd1);
			BwapiTestUtils.delay(500);
			BwapiTestUtils.relativeClick(scWnd1, BroodwarButton.ID_OK.getX(), BroodwarButton.ID_OK.getY(),
					InputEvent.BUTTON1_MASK);
			return Pair.makePair(pid1, pid2);
		} catch(Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	
	/**
	 * Execute bot with the given information
	 * 
	 * @param testInformation
	 * @throws Exception
	 */
	protected static void execute(final BwapiTestInformation testInformation) throws Exception {
		final ReentrantLock lock = new ReentrantLock();
		final Condition cond = lock.newCondition();
		final AtomicReference<Throwable> throwable = new AtomicReference<Throwable>(null);
		AtomicReference<Process> chaosProcess = new AtomicReference<Process>(null);
		final AtomicReference<Boolean> done = new AtomicReference<Boolean>(false);
		try {
			ProxyBotFactory f = new ProxyBotFactory() {

				@Override
				public ProxyBot getBot(Game g) {
					try {
						ProxyBot inner = testInformation.getFactory().getBot(g);
						return new WrapperBot(lock, cond, done, throwable, inner);
					} catch (Exception e) {
						throw new RuntimeException(e);
					}
				}

			};

			Thread t = new Thread(new ProxyServer(f,ProxyServer.extractPort("1"), null));
			t.start();

			// start up starcraft
			startStarcraft(testInformation, chaosProcess);
			lock.lock();
			try {
				while (!done.get())
					cond.await();
			} catch (InterruptedException e) {

			} finally {
				lock.unlock();
			}
			t.interrupt();
			
			if (throwable.get() != null) {
				throwable.get().printStackTrace();
				throw new RuntimeException("Failure during execution", throwable.get());
			}
		} finally {
			stopStarcraft();
		}
	}

	public static class WrapperBot implements ProxyBot {

		private final Condition cond;
		private final AtomicReference<Throwable> throwable;
		private final AtomicReference<Boolean> done;
		private final ProxyBot bot;
		private final Lock lock;

		public WrapperBot(Lock lock, Condition cond, AtomicReference<Boolean> done,
		        AtomicReference<Throwable> throwable, ProxyBot bot) {
			this.cond = cond;
			this.throwable = throwable;
			this.bot = bot;
			this.done = done;
			this.lock = lock;
			lock.lock();
		}

		@Override
		public void onEnd(boolean isWinnerFlag) {
			try {
				bot.onEnd(isWinnerFlag);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
			done.set(true);
			cond.signalAll();
			lock.unlock();

		}

		@Override
		public void onFrame() {
			try {
				bot.onFrame();
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onNukeDetect(Position position) {
			try {
				bot.onNukeDetect(position);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onPlayerLeft(Player player) {
			try {
				bot.onPlayerLeft(player);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}

		}

		@Override
		public void onReceiveText(Player player, String text) {
			try {
				bot.onReceiveText(player, text);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}

		}

		@Override
		public void onSendText(String text) {
			try {
				bot.onSendText(text);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}

		}

		@Override
		public void onStart() {
			try {
				bot.onStart();
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}

		}

		@Override
		public void onDroppedConnection() {
			done.set(true);
			cond.signalAll();
			lock.unlock();
		}

		@Override
		public void onUnitCreate(ROUnit unit) {
			try {
				bot.onUnitCreate(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onUnitDestroy(ROUnit unit) {
			try {
				bot.onUnitDestroy(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onUnitHide(ROUnit unit) {
			try {
				bot.onUnitHide(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onUnitMorph(ROUnit unit) {
			try {
				bot.onUnitMorph(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onUnitRenegade(ROUnit unit) {
			try {
				bot.onUnitRenegade(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

		@Override
		public void onUnitShow(ROUnit unit) {
			try {
				bot.onUnitShow(unit);
			} catch (Throwable e) {
				throwable.set(e);
				done.set(true);
				cond.signalAll();
				lock.unlock();
				throw new RuntimeException("Caught an exception during test: ", e);
			}
		}

	}

}
