package org.bwapi.proxy.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.bwapi.proxy.messages.Commands;
import org.bwapi.proxy.messages.GameMessages;
import org.bwapi.proxy.messages.Commands.CommandType;
import org.bwapi.proxy.messages.GameMessages.UnitId;

public class Unit implements ROUnit {
	private final Game g = Game.getInstance();

	private final int id;
	private final CommandValidator validator;
	private final GameMessages.Unit fixedMessage;

	public Unit(int id) {
		this.id = id;
		this.validator = g.getInvalidator();
		this.fixedMessage = null;
	}

	public Unit(int id, CommandValidator validator) {
		this.id = id;
		this.validator = validator;
		this.fixedMessage = null;
	}

	private Unit(int id, GameMessages.Unit fixedMessage) {
		this.id = id;
		this.validator = g.getInvalidator();
		this.fixedMessage = fixedMessage;
	}

	public Unit commandableUnit(CommandValidator validator) {
		return new Unit(id, validator);
	}

	private GameMessages.Unit getMessage() {
		if (fixedMessage != null)
			return fixedMessage;
		return g.unitManager.getMessage(id);
	}

	private GameMessages.Unit getInitialMessage() {
		return g.unitManager.getInitialMessage(id);
	}

	private GameMessages.Unit getPreviousMessage() {
		if (fixedMessage != null)
			return null;
		return g.unitManager.getPreviousMessage(id);
	}

	public UnitId toMessage() {
		UnitId.Builder b = UnitId.newBuilder();
		b.setId(id);
		return b.build();
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == null || !(obj instanceof ROUnit)) {
			return false;
		}
		ROUnit u = (ROUnit) obj;
		return id == u.getID();
	}

	@Override
	public int hashCode() {
		return id;
	}

	@Override
	public String toString() {
		return "Unit [" + getType().getName() + " " + id + "]";
	}

	public ROUnit getPreviousUnitData() {
		if (getPreviousMessage() == null)
			return null;
		return new Unit(this.id, getPreviousMessage());
	}

	private List<UnitType> buildUnitTypeList(List<GameMessages.UnitType> list) {
		List<UnitType> newList = new ArrayList<UnitType>(list.size());
		for (GameMessages.UnitType t : list) {
			newList.add(UnitType.getUnitTypeFromId(t.getId()));
		}
		return newList;
	}

	private Set<? extends ROUnit> buildUnitSet(List<GameMessages.UnitId> list) {
		Set<Unit> set = new HashSet<Unit>();
		for (GameMessages.UnitId u : list) {
			set.add(new Unit(u.getId()));
		}
		return set;
	}

	public int getID() {
		return id;
	}

	// Orders

	public void attackMove(Position position) {
		if (!validator.canAttackMove(this, position)) {
			throw new RuntimeException("Invalid attackMove command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.attackMove);
		b.setPos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}
	
	public void attack(ROUnit nearest) {
		attackUnit(nearest);
	}

	public void attackUnit(ROUnit nearest) {
		if (!validator.canAttackUnit(this, nearest)) {
			throw new RuntimeException("Invalid attackUnit command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.attackUnit);
		b.setTarget(nearest.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void build(TilePosition position, UnitType type) {
		if (!validator.canBuild(this, position, type)) {
			throw new RuntimeException("Invalid build command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.build);
		b.setTilePos(position.getMessage());
		b.setUnitType(type.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void buildAddon(UnitType type) {
		if (!validator.canBuildAddon(this, type)) {
			throw new RuntimeException("Invalid buildAddon command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.buildAddon);
		b.setUnitType(type.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void burrow() {
		if (!validator.canBurrow(this)) {
			throw new RuntimeException("Invalid burrow command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.burrow);
		g.commandManager.addCommand(b.build());
	}

	public void cancelAddon() {
		if (!validator.canCancelAddon(this)) {
			throw new RuntimeException("Invalid cancelAddon command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelAddon);
		g.commandManager.addCommand(b.build());
	}

	public void cancelConstruction() {
		if (!validator.canCancelConstruction(this)) {
			throw new RuntimeException("Invalid cancelConstruction command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelConstruction);
		g.commandManager.addCommand(b.build());
	}

	public void cancelMorph() {
		if (!validator.canCancelMorph(this)) {
			throw new RuntimeException("Invalid cancelMorph command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelMorph);
		g.commandManager.addCommand(b.build());
	}

	public void cancelResearch() {
		if (!validator.canCancelResearch(this)) {
			throw new RuntimeException("Invalid cancelResearch command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelResearch);
		g.commandManager.addCommand(b.build());
	}

	public void cancelTrain() {
		if (!validator.canCancelTrain(this)) {
			throw new RuntimeException("Invalid cancelTrain command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelTrain);
		g.commandManager.addCommand(b.build());
	}

	public void cancelTrain(int slot) {
		if (!validator.canCancelTrain(this, slot)) {
			throw new RuntimeException("Invalid cancelTrain command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelTrainSlot);
		b.setSlot(slot);
		g.commandManager.addCommand(b.build());
	}

	public void cancelUpgrade() {
		if (!validator.canCancelUpgrade(this)) {
			throw new RuntimeException("Invalid cancelUpgrade command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cancelUpgrade);
		g.commandManager.addCommand(b.build());
	}

	public void cloak() {
		if (!validator.canCloak(this)) {
			throw new RuntimeException("Invalid cloak command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.cloak);
		g.commandManager.addCommand(b.build());
	}

	public void decloak() {
		if (!validator.canDecloak(this)) {
			throw new RuntimeException("Invalid decloak command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.decloak);
		g.commandManager.addCommand(b.build());
	}

	public void follow(ROUnit target) {
		if (!validator.canFollow(this, target)) {
			throw new RuntimeException("Invalid follow command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.follow);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void haltConstruction() {
		if (!validator.canHaltConstruction(this)) {
			throw new RuntimeException("Invalid haltConstruction command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.haltConstruction);
		g.commandManager.addCommand(b.build());
	}

	public void holdPosition() {
		if (!validator.canHoldPosition(this)) {
			throw new RuntimeException("Invalid holdPosition command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.holdPosition);
		g.commandManager.addCommand(b.build());
	}

	public void land(TilePosition position) {
		if (!validator.canLand(this, position)) {
			throw new RuntimeException("Invalid land command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.land);
		b.setTilePos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void lift() {
		if (!validator.canLift(this)) {
			throw new RuntimeException("Invalid lift command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.lift);
		g.commandManager.addCommand(b.build());
	}

	public void load(ROUnit target) {
		if (!validator.canLoad(this, target)) {
			throw new RuntimeException("Invalid load command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.load);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void morph(UnitType type) {
		if (!validator.canMorph(this, type)) {
			throw new RuntimeException("Invalid morph command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.morph);
		b.setUnitType(type.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void patrol(Position position) {
		if (!validator.canPatrol(this, position)) {
			throw new RuntimeException("Invalid patrol command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.patrol);
		b.setPos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void repair(ROUnit target) {
		if (!validator.canRepair(this, target)) {
			throw new RuntimeException("Invalid repair command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.repair);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());

	}

	public void research(TechType tech) {
		if (!validator.canResearch(this, tech)) {
			throw new RuntimeException("Invalid research command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.research);
		b.setTechType(tech.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void rightClick(Position position) {
		if (!validator.canRightClick(this, position)) {
			throw new RuntimeException("Invalid rightClick command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.rightClick);
		b.setPos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void rightClick(TilePosition position) {
		rightClick(new Position(position));
	}
	
	public void move(TilePosition position) {
		move(new Position(position));
	}
	
	
	public void move(Position position) {
		rightClick(position);
	}

	public void rightClick(ROUnit target) {
		if (!validator.canRightClick(this, target)) {
			throw new RuntimeException("Invalid rightClick command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.rightClickUnit);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void setRallyPosition(Position target) {
		if (!validator.canSetRallyPosition(this, target)) {
			throw new RuntimeException("Invalid setRallyPosition command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.setRallyPosition);
		b.setPos(target.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void setRallyUnit(ROUnit target) {
		if (!validator.canSetRallyUnit(this, target)) {
			throw new RuntimeException("Invalid setRallyUnit command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.setRallyUnit);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void siege() {
		if (!validator.canSiege(this)) {
			throw new RuntimeException("Invalid siege command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.siege);
		g.commandManager.addCommand(b.build());
	}

	public void stop() {
		if (!validator.canStop(this)) {
			throw new RuntimeException("Invalid stop command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.stop);
		g.commandManager.addCommand(b.build());
	}

	public void train(UnitType type) {
		if (!validator.canTrain(this, type)) {
			throw new RuntimeException("Invalid train command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.train);
		b.setUnitType(type.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void unburrow() {
		if (!validator.canUnburrow(this)) {
			throw new RuntimeException("Invalid unburrow command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.unburrow);
		g.commandManager.addCommand(b.build());
	}

	public void unload(ROUnit target) {
		if (!validator.canUnload(this, target)) {
			throw new RuntimeException("Invalid unload command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.unload);
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	public void unloadAll() {
		if (!validator.canUnloadAll(this)) {
			throw new RuntimeException("Invalid unloadAll command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.unloadAll);
		g.commandManager.addCommand(b.build());
	}

	public void unloadAll(Position position) {
		if (!validator.canUnloadAll(this, position)) {
			throw new RuntimeException("Invalid unloadAll command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.unloadAllPosition);
		b.setPos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void unsiege() {
		if (!validator.canUnsiege(this)) {
			throw new RuntimeException("Invalid unsiege command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.unsiege);
		g.commandManager.addCommand(b.build());
	}

	public void upgrade(UpgradeType upgrade) {
		if (!validator.canUpgrade(this, upgrade)) {
			throw new RuntimeException("Invalid upgrade command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.upgrade);
		b.setUpgradeType(upgrade.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void useTech(TechType tech) {
		if (!validator.canUseTech(this, tech)) {
			throw new RuntimeException("Invalid useTech command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.useTech);
		b.setTechType(tech.getTypeMessage());
		g.commandManager.addCommand(b.build());
	}

	public void useTech(TechType tech, Position position) {
		if (!validator.canUseTech(this, tech, position)) {
			throw new RuntimeException("Invalid useTech command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.useTechPosition);
		b.setTechType(tech.getTypeMessage());
		b.setPos(position.getMessage());
		g.commandManager.addCommand(b.build());
	}

	public void useTech(TechType tech, ROUnit target) {
		if (!validator.canUseTech(this, tech, target)) {
			throw new RuntimeException("Invalid useTech command on " + this);
		}
		Commands.Command.Builder b = Commands.Command.newBuilder();
		b.setSelf(this.toMessage());
		b.setType(CommandType.useTechTarget);
		b.setTechType(tech.getTypeMessage());
		b.setTarget(target.toMessage());
		g.commandManager.addCommand(b.build());
	}

	// Getters for BWAPI data

	public boolean exists() {
		return getMessage() == null ? false : getMessage().getExists();
	}

	public ROUnit getAddon() {
		return getMessage() == null ? null : (getMessage().hasAddon() ? new Unit(getMessage().getAddon().getId()) : null);
	}

	public int getAirWeaponCooldown() {
		return getMessage() == null ? 0 : getMessage().getAirWeaponCooldown();
	}

	public int getArmor() {
		return getMessage() == null ? 0 : getType().armor() + getPlayer().getUpgradeLevel(getType().armorUpgrade());
	}

	public double getAngle() {
		return getMessage() == null ? 0 : getMessage().getAngle();
	}

	public ROUnit getBuildUnit() {
		return (getMessage() != null && getMessage().hasBuildUnit()) ? new Unit(getMessage().getBuildUnit().getId()) : null;
	}

	public ROUnit getCarrier() {
		return (getMessage() != null && getMessage().hasCarrier()) ? new Unit(getMessage().getCarrier().getId()) : null;
	}

	public int getDefenseMatrixPoints() {
		return getMessage() == null ? 0 : getMessage().getDefenseMatrixPoints();
	}

	public int getDefenseMatrixTimer() {
		return getMessage() == null ? 0 : getMessage().getDefenseMatrixTimer();
	}

	public double getDistance(Position target) {
		if (getLastKnownPosition().equals(Position.INVALID))
			return 0;
		return getLastKnownPosition().getDistance(target);
	}

	public double getCenterDistance(ROUnit target) {
		if (getLastKnownPosition().equals(Position.INVALID))
			return 0;
		return getLastKnownPosition().getDistance(target.getLastKnownPosition());
	}

	public int getDistance(ROUnit target) {
		double result;
		if (target == this)
			return 0;
		Position thisPos = this.getLastKnownPosition();
		Position targetPos = target.getLastKnownPosition();
		if (thisPos.y() - this.getType().dimensionUp() <= targetPos.y() + target.getType().dimensionDown())
			if (thisPos.y() + this.getType().dimensionDown() >= targetPos.y() - target.getType().dimensionUp())
				if (thisPos.x() > targetPos.x())
					result = thisPos.x() - this.getType().dimensionLeft() - targetPos.x() - target.getType().dimensionRight();
				else
					result = targetPos.x() - target.getType().dimensionRight() - thisPos.x() - this.getType().dimensionLeft();

		if (thisPos.x() - this.getType().dimensionLeft() <= targetPos.x() + target.getType().dimensionRight())
			if (thisPos.x() + this.getType().dimensionRight() >= targetPos.x() - target.getType().dimensionLeft())
				if (thisPos.y() > targetPos.y())
					result = thisPos.y() - this.getType().dimensionUp() - targetPos.y() - target.getType().dimensionDown();
				else
					result = targetPos.y() - target.getType().dimensionDown() - thisPos.y() - this.getType().dimensionUp();

		if (thisPos.x() > targetPos.x())
			if (thisPos.y() > targetPos.y())
				result = new Position(thisPos.x() - this.getType().dimensionLeft(), thisPos.y() - this.getType().dimensionUp())
				    .getDistance(new Position(targetPos.x() + target.getType().dimensionRight(), targetPos.y()
				        + target.getType().dimensionDown()));
			else
				result = new Position(thisPos.x() - this.getType().dimensionLeft(), thisPos.y()
				    + this.getType().dimensionDown()).getDistance(new Position(targetPos.x()
				    + target.getType().dimensionRight(), targetPos.y() - target.getType().dimensionUp()));
		else if (thisPos.y() > targetPos.y())
			result = new Position(thisPos.x() + this.getType().dimensionRight(), thisPos.y() - this.getType().dimensionUp())
			    .getDistance(new Position(targetPos.x() - target.getType().dimensionLeft(), targetPos.y()
			        + target.getType().dimensionDown()));
		else
			result = new Position(thisPos.x() + this.getType().dimensionRight(), thisPos.y() + this.getType().dimensionDown())
			    .getDistance(new Position(targetPos.x() - target.getType().dimensionLeft(), targetPos.y()
			        - target.getType().dimensionUp()));
		if (result > 0)
			return (int) result;
		else
			return 0;
	}

	public int getEnergy() {
		return getMessage() == null ? 0 : getMessage().getEnergy();
	}

	public int getEnsnareTimer() {
		return getMessage() == null ? 0 : getMessage().getEnsnareTimer();
	}

	public int getGroundWeaponCooldown() {
		return getMessage() == null ? 0 : getMessage().getGroundWeaponCooldown();
	}

	public int getHitPoints() {
		return getMessage() == null ? 0 : getMessage().getHitPoints();
	}

	public int getInitialHitPoints() {
		return getInitialMessage() == null ? 0 : getInitialMessage().getInitialHitPoints();
	}

	public Position getInitialPosition() {
		if (getInitialMessage() == null)
			return Position.INVALID;
		return new Position(getInitialMessage().getInitialPosition());
	}

	public int getInitialResources() {
		return getInitialMessage() == null ? 0 : getInitialMessage().getInitialResources();
	}

	public TilePosition getInitialTilePosition() {
		if (getInitialMessage() == null)
			return TilePosition.INVALID;
		return new TilePosition(getInitialMessage().getInitialTilePosition());
	}

	public UnitType getInitialType() {
		if (getInitialMessage() == null)
			return UnitType.UNKNOWN;
		return UnitType.getUnitTypeFromId(getInitialMessage().getInitialType().getId());
	}

	public int getInterceptorCount() {
		return getMessage() == null ? 0 : getMessage().getInterceptorCount();
	}

	public Set<? extends ROUnit> getInterceptors() {
		if (getMessage() == null)
			return new HashSet<Unit>();
		return buildUnitSet(getMessage().getInterceptorsList());
	}

	public int getIrradiateTimer() {
		return getMessage() == null ? 0 : getMessage().getIrradiateTimer();
	}

	public int getKillCount() {
		return getMessage() == null ? 0 : getMessage().getKillCount();
	}

	public Set<? extends ROUnit> getLoadedUnits() {
		if (getMessage() == null)
			return new HashSet<Unit>();
		return buildUnitSet(getMessage().getLoadedUnitsList());
	}

	public int getLockdownTimer() {
		return getMessage() == null ? 0 : getMessage().getLockdownTimer();
	}

	public int getMaelstromTimer() {
		return getMessage() == null ? 0 : getMessage().getMaelstromTimer();
	}

	public ROUnit getNydusExit() {
		return (getMessage() != null && getMessage().hasNydusExit()) ? new Unit(getMessage().getNydusExit().getId()) : null;
	}

	public Order getOrder() {
		return getMessage() == null ? Order.UNKNOWN : Order.getOrderFromId(getMessage().getOrder());
	}

	public Unit getOrderTarget() {
		return getMessage() == null ? null : (getMessage().hasOrderTarget() ? new Unit(getMessage().getOrderTarget()
		    .getId()) : null);
	}

	public int getOrderTimer() {
		return getMessage() == null ? 0 : getMessage().getOrderTimer();
	}

	public int getPlagueTimer() {
		return getMessage() == null ? 0 : getMessage().getPlagueTimer();
	}

	public Player getPlayer() {
		return getMessage() != null ? g.playerManager.getPlayer(getMessage().getPlayer())
		    : this.getPreviousMessage() == null ? null : g.playerManager.getPlayer(getPreviousMessage().getPlayer());
	}

	public Position getPosition() {
		if (getMessage() == null)
			return Position.INVALID;
		return new Position(getMessage().getPosition());
	}

	public Position getRallyPosition() {
		if (getMessage() == null)
			return Position.INVALID;
		return new Position(getMessage().getRallyPosition());
	}

	public Unit getRallyUnit() {
		if (getMessage() == null)
			return null;
		return getMessage().hasRallyUnit() ? new Unit(getMessage().getRallyUnit().getId()) : null;
	}

	public int getRemainingBuildTime() {
		return getMessage() == null ? 0 : getMessage().getRemainingBuildTime();
	}

	public int getRemainingResearchTime() {
		return getMessage() == null ? 0 : getMessage().getRemainingResearchTime();
	}

	public int getRemainingTrainTime() {
		return getMessage() == null ? 0 : getMessage().getRemainingTrainTime();
	}

	public int getRemainingUpgradeTime() {
		return getMessage() == null ? 0 : getMessage().getRemainingUpgradeTime();
	}

	public int getRemoveTimer() {
		return getMessage() == null ? 0 : getMessage().getRemoveTimer();
	}

	public int getResources() {
		return getMessage() == null ? 0 : getMessage().getResources();
	}

	public int getScarabCount() {
		return getMessage() == null ? 0 : getMessage().getScarabCount();
	}

	public Order getSecondaryOrder() {
		return getMessage() == null ? Order.UNKNOWN : Order.getOrderFromId(getMessage().getSecondaryOrder());
	}

	public int getShields() {
		return getMessage() == null ? 0 : getMessage().getShields();
	}

	public int getSpellCooldown() {
		return getMessage() == null ? 0 : getMessage().getSpellCooldown();
	}

	public int getSpiderMineCount() {
		return getMessage() == null ? 0 : getMessage().getSpiderMineCount();
	}

	public int getStasisTimer() {
		return getMessage() == null ? 0 : getMessage().getStasisTimer();
	}

	public int getStimTimer() {
		return getMessage() == null ? 0 : getMessage().getStimTimer();
	}

	public Unit getTarget() {
		return getMessage() == null ? null : (getMessage().hasTarget() ? new Unit(getMessage().getTarget().getId()) : null);
	}

	public Position getTargetPosition() {
		if (getMessage() == null)
			return Position.INVALID;
		return new Position(getMessage().getTargetPosition());
	}

	public TechType getTech() {
		return getMessage() == null ? TechType.UNKNOWN : TechType.getTechTypeFromId(getMessage().getTech());
	}

	public TilePosition getTilePosition() {
		if (getMessage() == null)
			return TilePosition.INVALID;
		return new TilePosition(getMessage().getTilePosition());
	}

	public List<UnitType> getTrainingQueue() {
		if (getMessage() == null)
			return new ArrayList<UnitType>();
		return buildUnitTypeList(getMessage().getTrainingQueueList());
	}

	public Unit getTransport() {
		return getMessage() == null ? null : (getMessage().hasTransport() ? new Unit(getMessage().getTransport().getId())
		    : null);
	}

	public UnitType getType() {
		return getMessage() == null ? (getPreviousMessage() == null ? UnitType.UNKNOWN : UnitType
		    .getUnitTypeFromId(getPreviousMessage().getType().getId())) : UnitType.getUnitTypeFromId(getMessage().getType()
		    .getId());
	}
	
	public UnitType getBuildType() {
		return getMessage() == null ? (getPreviousMessage() == null || !getPreviousMessage().hasBuildType()? UnitType.UNKNOWN : UnitType
		    .getUnitTypeFromId(getPreviousMessage().getBuildType().getId())) : getMessage().hasBuildType() ? UnitType.getUnitTypeFromId(getMessage().getType()
		    .getId()) : UnitType.UNKNOWN;
	}

	public UpgradeType getUpgrade() {
		return getMessage() == null ? UpgradeType.UNKNOWN : UpgradeType.getUpgradeTypeFromId(getMessage().getUpgrade());
	}

	public int getUpgradeLevel(UpgradeType upgrade) {
		return getMessage() == null ? 0 : getPlayer().getUpgradeLevel(upgrade);
	}

	public double getVelocityX() {
		return getMessage() == null ? 0 : getMessage().getVelocityX();
	}

	public double getVelocityY() {
		return getMessage() == null ? 0 : getMessage().getVelocityY();
	}

	public boolean isAccelerating() {
		return getMessage() == null ? false : getMessage().getAccelerating();
	}

	public boolean isAttacking() {
		return getMessage() == null ? false : getMessage().getAttacking();
	}
	
	public boolean isAttackFrame() {
		return getMessage() == null ? false : getMessage().getAttacking();
	}

	public boolean isBeingConstructed() {
		return getMessage() == null ? false : getMessage().getBeingConstructed();
	}

	public boolean isBeingGathered() {
		return getMessage() == null ? false : getMessage().getBeingGathered();
	}

	public boolean isBeingHealed() {
		return getMessage() == null ? false : getMessage().getBeingHealed();
	}

	public boolean isBlind() {
		return getMessage() == null ? false : getMessage().getBlind();
	}

	public boolean isBraking() {
		return getMessage() == null ? false : getMessage().getBraking();
	}

	public boolean isBurrowed() {
		return getMessage() == null ? false : getMessage().getBurrowed();
	}

	public boolean isCarryingGas() {
		return getMessage() == null ? false : getMessage().getCarryingGas();
	}

	public boolean isCarryingMinerals() {
		return getMessage() == null ? false : getMessage().getCarryingMinerals();
	}

	public boolean isCloaked() {
		return getMessage() == null ? false : getMessage().getCloaked();
	}

	public boolean isCompleted() {
		return getMessage() == null ? false : getMessage().getCompleted();
	}

	public boolean isConstructing() {
		return getMessage() == null ? false : getMessage().getConstructing();
	}

	public boolean isDetected() {
		return getMessage() == null ? false : getMessage().getDetected();
	}

	public boolean isDefenseMatrixed() {
		return getMessage() == null ? false : getMessage().getDefenseMatrixed();
	}

	public boolean isEnsnared() {
		return getMessage() == null ? false : getMessage().getEnsnared();
	}

	public boolean isFollowing() {
		return getMessage() == null ? false : getMessage().getFollowing();
	}

	public boolean isGatheringGas() {
		return getMessage() == null ? false : getMessage().getGatheringGas();
	}

	public boolean isGatheringMinerals() {
		return getMessage() == null ? false : getMessage().getGatheringMinerals();
	}

	public boolean isHallucination() {
		return getMessage() == null ? false : getMessage().getHallucination();
	}

	public boolean isIdle() {
		return getMessage() == null ? false : getMessage().getIdle();
	}

	public boolean isIrradiated() {
		return getMessage() == null ? false : getMessage().getIrradiated();
	}

	public boolean isLifted() {
		return getMessage() == null ? false : getMessage().getLifted();
	}

	public boolean isLoaded() {
		return getMessage() == null ? false : getMessage().getLoaded();
	}

	public boolean isLockedDown() {
		return getMessage() == null ? false : getMessage().getLockedDown();
	}

	public boolean isMaelstrommed() {
		return getMessage() == null ? false : getMessage().getMaelstrommed();
	}

	public boolean isMorphing() {
		return getMessage() == null ? false : getMessage().getMorphing();
	}

	public boolean isMoving() {
		return getMessage() == null ? false : getMessage().getMoving();
	}

	public boolean isParasited() {
		return getMessage() == null ? false : getMessage().getParasited();
	}

	public boolean isPatrolling() {
		return getMessage() == null ? false : getMessage().getPatrolling();
	}

	public boolean isPlagued() {
		return getMessage() == null ? false : getMessage().getPlagued();
	}

	public boolean isRepairing() {
		return getMessage() == null ? false : getMessage().getRepairing();
	}

	public boolean isResearching() {
		return getMessage() == null ? false : getMessage().getResearching();
	}

	public boolean isSelected() {
		return getMessage() == null ? false : getMessage().getSelected();
	}

	public boolean isSieged() {
		return getMessage() == null ? false : getMessage().getSieged();
	}

	public boolean isStartingAttack() {
		return getMessage() == null ? false : getMessage().getStartingAttack();
	}

	public boolean isStasised() {
		return getMessage() == null ? false : getMessage().getStasised();
	}

	public boolean isStimmed() {
		return getMessage() == null ? false : getMessage().getStimmed();
	}

	public boolean isStuck() {
		return getMessage() == null ? false : getMessage().getStuck();
	}

	public boolean isTraining() {
		return getMessage() == null ? false : getMessage().getTraining();
	}

	public boolean isUnderStorm() {
		return getMessage() == null ? false : getMessage().getUnderStorm();
	}
	
	public boolean isUnderAttack() {
		return getMessage() == null ? false : getMessage().getUnderAttack();
	}
	
	public boolean isUnderDarkSwarm() {
		return getMessage() == null ? false : getMessage().getUnderDarkSwarm();
	}

	public boolean isUnpowered() {
		return getMessage() == null ? false : getMessage().getUnpowered();
	}

	public boolean isUpgrading() {
		return getMessage() == null ? false : getMessage().getUpgrading();
	}

	public boolean isVisible() {
		return getMessage() == null ? false : getMessage().getVisible();
	}

	// Additional getters

	public boolean canAttack(ROUnit target) {
		if (getAttackRange(target) > 0)
			return true;
		WeaponType w = target.isFlying() ? getType().airWeapon() : getType().groundWeapon();
		return !w.equals(WeaponType.NONE);
	}

	public boolean canCloak() {
		return getPlayer().canCloak(getType());
	}

	public int getAirRange() {
		return getRange(true);
	}

	public int getAirWeaponDamage() {
		return (getPlayer() == null || getType() == null) ? 0 : getPlayer().getWeaponDamage(getType().airWeapon());
	}

	public int getAttackRange(ROUnit target) {
		return target.isFlying() ? getAirRange() : getGroundRange();
	}

	public double getDamagePerShot(ROUnit target) {
		return getPlayer().getDamagePerShot(getType().pickWeapon(target.isFlying()), target.getType(), target.getPlayer());
	}
	
	public double getDamagePerShot(boolean vsAir) {
		return getPlayer().getDamagePerShot(getType().pickWeapon(vsAir));
	}

	public double getDps(boolean vsAir) {
		if (getType().isBuilding() && (isUnpowered() || !wasCompleted())) {
			return 0;
		}
		if (getType().equals(UnitType.TERRAN_BUNKER) && getPlayer().equals(Game.getInstance().self())) {
			double total = 0;
			for (ROUnit u : getLoadedUnits()) {
				total += u.getDps(vsAir);
			}
			return total;
		}
		if (getType().equals(UnitType.PROTOSS_CARRIER)) {
			return getInterceptorCount() * getPlayer().getDps(UnitType.PROTOSS_INTERCEPTOR, vsAir);
		}
		double dps = getPlayer().getDps(getType(), vsAir);
		if (isStimmed()) {
			dps *= 2;
		}
		return dps;
	}

	public double getDps(ROUnit target) {
		if (getType().isBuilding() && (isUnpowered() || !wasCompleted())) {
			return 0;
		}
		if (getType().equals(UnitType.TERRAN_BUNKER) && getPlayer().equals(Game.getInstance().self())) {
			double total = 0;
			for (ROUnit u : getLoadedUnits()) {
				total += u.getDps(target);
			}
			return total;
		}
		if (getType().equals(UnitType.PROTOSS_CARRIER)) {
			return getInterceptorCount() * getPlayer().getDps(UnitType.PROTOSS_INTERCEPTOR, target);
		}
		double dps = getPlayer().getDps(getType(), target);
		if (isStimmed()) {
			dps *= 2;
		}
		return dps;
	}

	public int getGroundRange() {
		return getRange(false);
	}

	public int getGroundWeaponDamage() {
		return (getPlayer() == null || getType() == null) ? 0 : getPlayer().getWeaponDamage(getType().groundWeapon());
	}

	public int getLastKnownHitPoints() {
		int hp = getHitPoints();
		if (hp > 0)
			return hp;
		if (getPreviousMessage() == null)
			return 0;
		return getPreviousUnitData().getHitPoints();
	}

	public Position getLastKnownPosition() {
		Position current = getPosition();
		if (!current.equals(Position.INVALID))
			return current;
		if (getPreviousMessage() == null)
			return Position.INVALID;
		return getPreviousUnitData().getPosition();
	}

	public int getLastKnownResources() {
		int r = getResources();
		if (r > 0)
			return r;
		if (getPreviousMessage() == null)
			return 0;
		return getPreviousUnitData().getResources();
	}

	public int getLastKnownShields() {
		int hp = getShields();
		if (hp > 0)
			return hp;
		if (getPreviousMessage() == null)
			return 0;
		return getPreviousUnitData().getShields();
	}

	public TilePosition getLastKnownTilePosition() {
		TilePosition current = getTilePosition();
		if (!current.equals(TilePosition.INVALID))
			return current;
		if (getPreviousMessage() == null)
			return TilePosition.INVALID;
		return getPreviousUnitData().getTilePosition();
	}

	public int getShieldUpgradeLevel() {
		return getMessage() == null ? 0 : getPlayer().getUpgradeLevel(UpgradeType.PROTOSS_PLASMA_SHIELDS);
	}

	public boolean isFlying() {
		return getType().isFlyer()
		    || (getType().isFlyingBuilding() && ((isVisible() && isLifted()) || (!isVisible() && getPreviousUnitData()
		        .isLifted())));
	}

	public boolean isInRange(ROUnit target) {
		if (!canAttack(target))
			return false;
		return getDistance(target) <= getAttackRange(target);
	}

	public boolean isStopped() {
		int o = getOrder().getID();
		return o == Order.PLAYER_GUARD.getID() || o == Order.CARRIER_HOLD_POSITION.getID()
		    || o == Order.REAVER_HOLD_POSITION.getID() || o == Order.HOLD_POSITION.getID() || o == Order.GUARD.getID()
		    || o == Order.GUARD_POST.getID() || o == Order.NONE.getID() || o == Order.NOTHING.getID()
		    || o == Order.NOTHING_3.getID() || o == Order.STOP.getID();
	}

	public boolean isTargetable() {
		if (isVisible()) {
			return isDetected() || !(isCloaked() || isBurrowed());
		}
		return false;
	}

	public boolean wasCompleted() {
		return getMessage() == null ? (getPreviousMessage() == null ? false : getPreviousMessage().getCompleted())
		    : getMessage().getCompleted();
	}

	// private functions

	private int getBunkerRange(boolean vsAir) {
		Player p = getPlayer();
		if (!p.equals(Game.getInstance().self())) {
			return p.getWeaponRange(WeaponType.GAUSS_RIFLE) + 32;
		} else {
			return getMaxBunkerRange(getLoadedUnits(), vsAir);
		}
	}

	private int getMaxBunkerRange(Set<? extends ROUnit> loadedUnits, boolean vsAir) {
		int max = 0;
		for (ROUnit u : loadedUnits) {
			if (u.getType().equals(UnitType.TERRAN_SCV))
				continue;
			max = Math.max(max, (vsAir ? u.getAirRange() : u.getGroundRange()) + 32);
		}
		return max;
	}

	private int getRange(boolean vsAir) {
		if (getType().equals(UnitType.TERRAN_BUNKER)) {
			return getBunkerRange(vsAir);
		}
		if (getType().equals(UnitType.PROTOSS_CARRIER)) {
			return 12 * 32;
		}
		return getPlayer().getWeaponRange(getType().pickWeapon(vsAir));
	}

	public boolean canMake(UnitType type) {
		return Game.getInstance().canMake(this, type);
	}

	public boolean canBuildHere(TilePosition tilePosition, UnitType type) {
	  return Game.getInstance().canBuildHere(this, tilePosition, type);
  }

	public boolean canUpgrade(UpgradeType type) {
	  return Game.getInstance().canUpgrade(this, type);
  }
	
	/**
	 * Returns 
	 * @param unit
	 * @return
	 */
	public Set<TilePosition> getBuildTilePositions() {
		if (this.getType() == null) return Collections.emptySet();
		return TilePosition.getTilePositions(this.getLastKnownTilePosition(),
				this.getType().tileWidth(), this.getType().tileHeight());
	}
	
	public TilePosition getAddonTilePosition() {
		return getLastKnownTilePosition().add(4, 1);
	}
}
