package org.bwapi.proxy.model;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

import org.bwapi.proxy.messages.BasicTypes;
import org.bwapi.proxy.util.Pair;

public final class TilePosition implements Serializable {
	private static final long serialVersionUID = 9089209349409016589L;

	@Override
	public String toString() {
		return "TilePosition [x=" + x + ", y=" + y + "]";
	}

	public static final TilePosition INVALID = new TilePosition(-1, -1);
	
	private final int x;
	private final int y;
	
	public TilePosition(int x, int y) {
		this.x = x;
		this.y = y;
	}
	
	public TilePosition(Position p) {
		this(p.x()/32, p.y()/32);
	}
	
	TilePosition(BasicTypes.TilePosition message) {
		this(message.getX(), message.getY());
	}

	public int x() {
		return x;
	}
	
	public int y() {
		return y;
	}
	
	public double getLength() {
		if (equals(INVALID)) return 0;
		return Math.sqrt(x*x + y*y);
	}
	
	public TilePosition subtract(TilePosition pos) {
		if (equals(INVALID) || pos.equals(INVALID)) return INVALID;
		return new TilePosition(x - pos.x, y -pos.y);
	}
	
	public TilePosition subtract(int xx, int yy) {
		if (equals(INVALID)) return INVALID;
		return new TilePosition(x - xx, y - yy);
	}
	
	public TilePosition add(TilePosition pos) {
		if (equals(INVALID) || pos.equals(INVALID)) return INVALID;
		return new TilePosition(x + pos.x, y +pos.y);
	}
	
	public TilePosition add(int xx, int yy) {
		if (equals(INVALID)) return INVALID;
		return new TilePosition(x +xx, y + yy);
	}
	
	
	public double getDistance(TilePosition pos) {
		return subtract(pos).getLength();
	}
	
	public double manhattanDistance(TilePosition pos) {
		return Math.abs(this.x - pos.x) + Math.abs(this.y - pos.y);
	}
	
	public double getDistance(Position pos) {
		return getDistance(new TilePosition(pos));
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == null || !(obj instanceof TilePosition)) return false;
		TilePosition other = (TilePosition)obj;
		return other.x == x && other.y == y;
	}
	
	@Override
	public int hashCode() {
		return x * 293 + y;
	}

	BasicTypes.TilePosition getMessage() {
		BasicTypes.TilePosition.Builder b = BasicTypes.TilePosition.newBuilder();
		b.setX(x);
		b.setY(y);
		return b.build();
	}
	
	public static Set<TilePosition> getTilePositions(TilePosition upperLeftCorner, int width, int height) {
		// Return the set of tile positions occupied by a building
		// with the specified upper-left corner and dimensions.
		
		Set<TilePosition> tilePositions = new HashSet<TilePosition>();
		
		for (int y = 0; y < height; ++y) {
			for (int x = 0; x < width; ++x) {
				tilePositions.add(new TilePosition(upperLeftCorner.x() + x, upperLeftCorner.y() + y));
			}			
		}
		
		return tilePositions;
	}

	public static Pair<TilePosition, TilePosition> selectNearestPositions(Set<TilePosition> tilePositions1,
	    Set<TilePosition> tilePositions2) {
		// Returns a pair of tile positions that are closest to each other
		// between the two given sets of tile positions.
		// If there are none, returns a pair of nulls.

		double nearestDistance = Double.POSITIVE_INFINITY;
		TilePosition nearestTilePosition1 = null;
		TilePosition nearestTilePosition2 = null;

		for (TilePosition tilePosition1 : tilePositions1) {
			for (TilePosition tilePosition2 : tilePositions2) {
				double distance = tilePosition1.getDistance(tilePosition2);
				if (distance < nearestDistance) {
					nearestDistance = distance;
					nearestTilePosition1 = tilePosition1;
					nearestTilePosition2 = tilePosition2;
				}
			}
		}

		return Pair.makePair(nearestTilePosition1, nearestTilePosition2);
	}
	
}
