package org.bwapi.proxy.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.bwapi.proxy.messages.BasicTypes;
import org.bwapi.proxy.messages.TerrainMessages;

public class Polygon implements Iterable<Position> {

	private List<Polygon> holes;
	private List<Position> vertices;
	private double perimeter;
	private Position center;
	private double area;

	Polygon(TerrainMessages.Polygon polygon) {
		holes = new ArrayList<Polygon>();
		for (TerrainMessages.Polygon pos : polygon.getHolesList()) {
			holes.add(new Polygon(pos));
		}
		vertices = new ArrayList<Position>();
		for (BasicTypes.Position pos : polygon.getVerticesList()) {
			vertices.add(new Position(pos));
		}

		perimeter = polygon.getPerimeter();
		center = new Position(polygon.getCenter());
		area = polygon.getArea();
	}

	public Polygon(List<Position> vertices, List<Polygon> holes, double perimeter, Position center, double area) {
		this.holes = holes;
		this.vertices = vertices;
		this.perimeter = perimeter;
		this.center = center;
		this.area = area;
	}

	public Polygon(List<Position> vertices) {
	  this.vertices = vertices;
	  this.holes = Collections.emptyList();
	  this.perimeter = calculatePerimeter();
	  this.area = calculateArea();
	  this.center = calculateCenter(this.area);
  }

	public double getArea() {
		return area;
	}

	public Position getCenter() {
		return center;
	}

	public double getPerimeter() {
		return perimeter;
	}

	public List<Position> getVertices() {
		return vertices;
	}

	public List<Polygon> getHoles() {
		return holes;
	}

	@Override
	public Iterator<Position> iterator() {
		return vertices.iterator();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(area);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		result = prime * result + ((center == null) ? 0 : center.hashCode());
		result = prime * result + ((holes == null) ? 0 : holes.hashCode());
		temp = Double.doubleToLongBits(perimeter);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		result = prime * result + ((vertices == null) ? 0 : vertices.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Polygon other = (Polygon) obj;
		if (Double.doubleToLongBits(area) != Double.doubleToLongBits(other.area))
			return false;
		if (center == null) {
			if (other.center != null)
				return false;
		} else if (!center.equals(other.center))
			return false;
		if (holes == null) {
			if (other.holes != null)
				return false;
		} else if (!holes.equals(other.holes))
			return false;
		if (Double.doubleToLongBits(perimeter) != Double.doubleToLongBits(other.perimeter))
			return false;
		if (vertices == null) {
			if (other.vertices != null)
				return false;
		} else if (!vertices.equals(other.vertices))
			return false;
		return true;
	}

	public boolean isInside(Position pt) {
		if (!point_is_inside(pt))
			return false;
		for (Polygon p : holes) {
			if (p.point_is_inside(pt))
				return false;
		}
		return true;
	}
	
	// cribbed from bwta
	public Position getNearestPoint(Position p) {
		double x3 = p.x();
		double y3 = p.y();
		Position minp = null;
		int j = 1;
		double mind2 = -1;
		for (int i = 0; i < vertices.size(); i++) {
			j = (i + 1) % vertices.size();
			double x1 = vertices.get(i).x();
			double y1 = vertices.get(i).y();
			double x2 = vertices.get(j).x();
			double y2 = vertices.get(j).y();
			double u = ((x3 - x1) * (x2 - x1) + (y3 - y1) * (y2 - y1))
			    / ((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));
			if (u < 0)
				u = 0;
			if (u > 1)
				u = 1;
			double x = x1 + u * (x2 - x1);
			double y = y1 + u * (y2 - y1);
			double d2 = (x - x3) * (x - x3) + (y - y3) * (y - y3);
			if (mind2 < 0 || d2 < mind2) {
				mind2 = d2;
				minp = new Position((int) x, (int) y);
			}
		}
		for (Polygon hole : getHoles()) {
			Position hnp = hole.getNearestPoint(p);
			if (hnp.getDistance(p) < minp.getDistance(p))
				minp = hnp;
		}
		return minp;
	}

	// cribbed from:
	// http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html
	private boolean point_is_inside(Position pt) {
		int i, j;
		boolean c = false;
		int nvert = vertices.size();
		for (i = 0, j = nvert - 1; i < nvert; j = i++) {
			Position vi = vertices.get(i);
			Position vj = vertices.get(j);
			double viy = vi.y();
			double vix = vi.x();
			double vjx = vj.x();
			double vjy = vj.y();
			if (((viy > pt.y()) != (vjy > pt.y())) && (pt.x() < (vjx - vix) * (pt.y() - viy) / (vjy - viy) + vix))
				c = !c;
		}
		return c;
	}



	public static void main(String[] args) {
		Position[] square_v = { new Position(0, 0), new Position(100, 0), new Position(100, 100),
		    new Position(0, 100) };

		Position[] small_square_v = { new Position(25, 25), new Position(75, 25), new Position(75, 75),
		    new Position(25, 75) };

		Polygon square = new Polygon(Arrays.asList(square_v), Collections.<Polygon> emptyList(), 0, new Position(
		    0, 0), 0);
		Polygon smallSquare = new Polygon(Arrays.asList(small_square_v), Collections.<Polygon> emptyList(), 0,
		    new Position(0, 0), 0);
		Polygon holeSquare = new Polygon(Arrays.asList(square_v), Collections.singletonList(smallSquare), 0,
		    new Position(0, 0), 0);

		Position test_points[] = { new Position(50, 50), new Position(50, 80), new Position(20, 20),
		    new Position(00, 00), new Position(100, 100), new Position(25, 25), new Position(1, 50),
		    new Position(22, 74), new Position(0, 50), new Position(100, 50), new Position(-4, 10) };

		System.out.println("square");
		for (Position p : test_points) {
			System.out.println(p + " " + square.isInside(p));
		}
		System.out.println("small square");
		for (Position p : test_points) {
			System.out.println(p + " " + smallSquare.isInside(p));
		}
		System.out.println("hole square");
		for (Position p : test_points) {
			System.out.println(p + " " + holeSquare.isInside(p));
		}

	}
	
  private double calculatePerimeter() {
    if (vertices.size()<2) return 0;
    double p=0;
    for(int i=0;i+1<vertices.size();i++)
    {
      p+=vertices.get(i).getDistance(vertices.get(i+1));
    }
    p+=vertices.get(vertices.size()-1).getDistance(vertices.get(0));
    return p;
  }
  
  
  double calculateArea() {
    if (vertices.size()<3) return 0;
    double a=0;
    for(int i=0;i+1<vertices.size();i++)
    {
      a+=(double)vertices.get(i).x()*vertices.get(i+1).y()-(double)vertices.get(i+1).x()*vertices.get(i).y();
    }
    a+=vertices.get(vertices.size()-1).x()*vertices.get(0).y()-vertices.get(0).x()*vertices.get(vertices.size()-1).y();
    a/=2;
    a=Math.abs(a);
    return a;
  }
  
  Position calculateCenter(double area) {
    double a = area;
    double cx=0;
    double cy=0;
    double temp;
    for(int i=0,j=1;i<vertices.size();i++,j++)
    {
      if (j==vertices.size())
        j=0;
      temp=(double)vertices.get(i).x()*vertices.get(j).y()-(double)vertices.get(j).x()*vertices.get(i).y();
      cx+=(vertices.get(i).x()+vertices.get(j).x())*temp;
      cy+=(vertices.get(i).y()+vertices.get(j).y())*temp;
    }
    cx=cx/(6.0*a);
    cy=cy/(6.0*a);
    return new Position((int)cx,(int)cy);
  }



}
