package org.bwapi.proxy.model;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.bwapi.proxy.messages.GameMessages;

public class Player {
	private static enum PlayerStatus {
		Neutral,
		Ally,
		Enemy,
		Unknown
	}

	private final Game g = Game.getInstance();

	private final int id;
	
	private GameMessages.Player message;
	
	private Set<Unit> units;
	private Map<Player, PlayerStatus> stances;
	private TilePosition startLocation;
	private Map<UnitType, Integer> allUnitCount;
	private Map<UnitType, Integer> completedUnitCount;
	private Map<UnitType, Integer> incompleteUnitCount;
	private Map<UnitType, Integer> deadUnitCount;
	private Map<UnitType, Integer> killedUnitCount;
	private int[] upgradeLevels;
	private boolean[] researchedTechs;
	private boolean[] researching;
	private boolean[] upgrading;

	Player(int id) {
		this.id = id;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null || !(obj instanceof Player)) { return false; }
		Player p = (Player)obj;
		return id == p.id;
	}
	
	@Override
	public int hashCode() {
		return id;
	}

	void setMessage(GameMessages.Player message) {
		this.message = message;
		
		this.units = null;
		this.stances = null;
		this.startLocation = null;
		this.allUnitCount = null;
		this.completedUnitCount = null;
		this.incompleteUnitCount = null;
		this.deadUnitCount = null;
		this.killedUnitCount = null;
		this.upgradeLevels = null;
		this.researchedTechs = null;
		this.researching = null;
		this.upgrading = null;
	}
	
	private void checkUpgrading() {
		if (upgrading == null) {
			upgrading = upgradeBoolArray(message.getUpgradingList());
		}
	}
	
	private void checkResearching() {
		if (researching == null) {
			researching = techTypeBoolArray(message.getResearchingList());
		}
	}
	
	private void checkResearchedTechs() {
		if (researchedTechs == null) {
			researchedTechs = techTypeBoolArray(message.getResearchedTechsList());
		}
	}
	
	private void checkUpgradeLevels() {
		if (upgradeLevels == null) {
			upgradeLevels = upgradeIntArray(message.getUpgradeLevelsList());
		}
	}
	
	private void checkKilledUnitCount() {
		if (killedUnitCount == null) {
			killedUnitCount = buildUnitTypeCountsFromMessages(message.getKilledUnitCountList());
		}
	}
	
	private void checkDeadUnitCount() {
		if (deadUnitCount == null) {
			deadUnitCount = buildUnitTypeCountsFromMessages(message.getDeadUnitCountList());
		}
	}
	
	private void checkIncompleteUnitCount() {
		if (incompleteUnitCount == null) {
			incompleteUnitCount = buildUnitTypeCountsFromMessages(message.getIncompleteUnitCountList());
		}
	}
	
	private void checkCompletedUnitCount() {
		if (completedUnitCount == null) {
			completedUnitCount = buildUnitTypeCountsFromMessages(message.getCompletedUnitCountList());
		}
	}
	
	private void checkAllUnitCount() {
		if (allUnitCount == null) {
			allUnitCount = buildUnitTypeCountsFromMessages(message.getAllUnitCountList());
		}
	}
	
	private void checkStartLocation() {
		if (startLocation == null) {
			startLocation = new TilePosition(message.getStartLocation());
		}
	}
	
	private void checkUnits() { 
		if (units == null) {
			units = buildUnitSet(message.getUnitsList());
		}
	}
	
	private void checkStances() {
		if (stances == null) {
			stances = buildPlayerMap(message.getStancesList());
		}
	}

	private Set<Unit> buildUnitSet(List<GameMessages.UnitId> list) {
		Set<Unit> set = new HashSet<Unit>();
		for (GameMessages.UnitId u : list) {
			set.add(new Unit(u.getId()));
		}
		return set;
	}

	private Map<Player, PlayerStatus> buildPlayerMap(List<GameMessages.PlayerStatus> list) {
		Map<Player, PlayerStatus> map = new HashMap<Player, PlayerStatus>(8);
		for (GameMessages.PlayerStatus status : list) {
			map.put(g.playerManager.getPlayer(status.getId()), getStatus(status.getStance()));
		}
		return map;
	}

	private PlayerStatus getStatus(GameMessages.PlayerStance stance) {
		switch(stance) {
		case Neutral:
			return PlayerStatus.Neutral;
		case Ally:
			return PlayerStatus.Ally;
		case Enemy:
			return PlayerStatus.Enemy;
		default:
			return PlayerStatus.Unknown;
		}
	}

	private Map<UnitType, Integer> buildUnitTypeCountsFromMessages(List<GameMessages.UnitTypeIntPair> list) {
		Map<UnitType, Integer> map = new HashMap<UnitType, Integer>();
		for (GameMessages.UnitTypeIntPair pair : list) {
			UnitType key = UnitType.getUnitTypeFromId(pair.getType().getId());
			map.put(key, pair.getCount());
		}
		return map;
	}

	private int[] upgradeIntArray(List<GameMessages.UpgradeLevel> list) {
		int[] a = new int[UpgradeType.getMaxIdPlusOne()];
		for (GameMessages.UpgradeLevel ul : list) {
			a[ul.getType().getId()] = ul.getLevel();
		}
		return a;
	}

	private boolean[] upgradeBoolArray(List<GameMessages.UpgradeType> list) {
		boolean[] a = new boolean[UpgradeType.getMaxIdPlusOne()];
		for (GameMessages.UpgradeType ut : list) {
			a[ut.getId()] = true;
		}
		return a;
	}

	private boolean[] techTypeBoolArray(List<GameMessages.TechType> list) {
		boolean[] a = new boolean[TechType.getMaxIdPlusOne()];
		for (GameMessages.TechType tt : list) {
			a[tt.getId()] = true;
		}
		return a;
	}

	public int allUnitCount(UnitType unit) {
		checkAllUnitCount();
		Integer rv = allUnitCount.get(unit);
		return rv == null ? 0 : rv;
	}

	public int completedUnitCount(UnitType unit) {
		checkCompletedUnitCount();
		Integer rv = completedUnitCount.get(unit);
		return rv == null ? 0 : rv;
	}

	public int cumulativeGas() {
		return message.getCumulativeGas();
	}

	public int cumulativeMinerals() {
		return message.getCumulativeMinerals();
	}

	public int deadUnitCount(UnitType unit) {
		checkDeadUnitCount();
		Integer rv = deadUnitCount.get(unit);
		return rv == null ? 0 : rv;
	}

	public int gas() {
		return message.getGas();
	}

	public int getID() {
		return id;
	}

	public String getName() {
		return message.getName();
	}

	public Race getRace() {
		return Race.getRaceFromId(message.getRace().getNumber());
	}

	public TilePosition getStartLocation() {
		checkStartLocation();
		return startLocation;
	}

	public Set<? extends ROUnit> getUnits() {
		checkUnits();
		return units;
	}

	public int getUpgradeLevel(UpgradeType upgrade) {
		checkUpgradeLevels();
		return upgradeLevels[upgrade.getID()];
	}

	public boolean hasResearched(TechType tech) {
		checkResearchedTechs();
		return researchedTechs[tech.getID()];
	}

	public int incompleteUnitCount(UnitType unit) {
		checkIncompleteUnitCount();
		Integer count = incompleteUnitCount.get(unit);
		return count == null ? 0 : count;
	}

	public boolean isAlly(Player player) {
		checkStances();
		return stances.containsKey(player) && stances.get(player).equals(PlayerStatus.Ally);
	}

	public boolean isEnemy(Player player) {
		checkStances();
		return stances.containsKey(player) && stances.get(player).equals(PlayerStatus.Enemy);
	}

	public boolean isNeutral() {
		return message.getNeutral();
	}

	public boolean isResearching(TechType tech) {
		checkResearching();
		return researching[tech.getID()];
	}

	public boolean isUpgrading(UpgradeType upgrade) {
		checkUpgrading();
		return upgrading[upgrade.getID()];
	}

	public int killedUnitCount(UnitType unit) {
		checkKilledUnitCount();
		return killedUnitCount.get(unit);
	}

	public boolean leftGame() {
		return message.getLeftGame();
	}

	public int minerals() {
		return message.getMinerals();
	}

	public int supplyTotal() {
		return message.getSupplyTotal();
	}

	public int supplyUsed() {
		return message.getSupplyUsed();
	}

	boolean isSelf() {
		return message.getSelf();
	}

	public boolean isVictorious() {
		return message.getVictorious();
	}

	public boolean isDefeated() {
		return message.getDefeated();
	}

	public boolean canCloak(UnitType type) {
		if (type.isBuilding()) return false;
		if (type.equals(UnitType.ZERG_LURKER) || type.equals(UnitType.PROTOSS_DARK_TEMPLAR) || type.equals(UnitType.PROTOSS_OBSERVER)) return true;
		if (hasResearched(TechType.BURROWING)) {
			if (type.equals(UnitType.ZERG_DRONE) || type.equals(UnitType.ZERG_ZERGLING) || type.equals(UnitType.ZERG_HYDRALISK) || type.equals(UnitType.ZERG_DEFILER)) {
				return true;
			}
		}
		if (hasResearched(TechType.CLOAKING_FIELD)) {
			if (type.equals(UnitType.TERRAN_WRAITH)) {
				return true;
			}
		}
		if (hasResearched(TechType.PERSONNEL_CLOAKING)) {
			if (type.equals(UnitType.TERRAN_GHOST)) {
				return true;
			}
		}
		return false;
    }
	
	public double getDamagePerShot(WeaponType weapon) {
		return getDamagePerShot(weapon, null);
	}
	
	public double getDamagePerShot(WeaponType weapon, ROUnit target) {
		if (target == null) {
			return getDamagePerShot(weapon, null, 0);
		}
		double armor = target.getShields() > 0 ? target.getShieldUpgradeLevel() : target.getArmor();
		return getDamagePerShot(weapon, target.getType(), armor);
	}
	
	public double getDamagePerShot(WeaponType weapon, UnitType targetType, Player targetPlayer) {
		double armor = targetType.armor() + targetPlayer.getUpgradeLevel(targetType.armorUpgrade());
		if (targetType.getRace().equals(Race.PROTOSS)) {
			armor *= targetType.maxHitPoints();
			armor += targetType.maxShields() * targetPlayer.getUpgradeLevel(UpgradeType.PROTOSS_PLASMA_SHIELDS);
			double weight = targetType.maxShields() + targetType.maxHitPoints();
			armor /= weight;
		}
		if (targetType.equals(UnitType.ZERG_ULTRALISK) && targetPlayer.getUpgradeLevel(UpgradeType.CHITINOUS_PLATING) > 0) {
			armor += 2;
		}
		return getDamagePerShot(weapon, targetType, armor);
	}

	private double getDamagePerShot(WeaponType weapon, UnitType targetType, double armor) {
		int weaponDamage = getWeaponDamage(weapon);
		double damageTypeMultiplier = targetType == null ? 1 : getDamageMultiplier(weapon.damageType(), targetType.size());
		double damagePerShot = (weaponDamage - armor) * damageTypeMultiplier;
		return damagePerShot;
	}

	public double getDps(WeaponType weapon) {
	  return getDps(null, null, weapon);
  }

	public double getDps(UnitType shooterType, ROUnit target) {
		return getDps(shooterType, target, shooterType.pickWeapon(target.isFlying()));
	}
	
	public double getDps(UnitType shooterType, boolean vsAir) {
		return getDps(shooterType, null, shooterType.pickWeapon(vsAir));
	}

	private double getDps(UnitType shooterType, ROUnit target, WeaponType w) {
		if (shooterType != null && shooterType.equals(UnitType.TERRAN_BUNKER)) {
			// For unknown bunkers, assume they're loaded with 4 marines
			return 4 * getDps(UnitType.TERRAN_MARINE, target, w);
		}
		double damagePerShot = getDamagePerShot(w, target);
		int cooldown = w.equals(WeaponType.PULSE_CANNON) ? 37 : w.damageCooldown(); // interceptor cooldowns are just wrong
		double shotsPerSecond = 24.0 / cooldown; 
		if (shooterType != null && (shooterType.equals(UnitType.ZERG_SCOURGE) || shooterType.equals(UnitType.ZERG_INFESTED_TERRAN))) {
			// This is totally broken, but is sort of like guessing that there's a 10% chance of a suicide unit
			// pulling off an attack for every second that you're nearby -- of course this is even worse for
			// times greater than 10 seconds
			shotsPerSecond = 0.1;
		}
		return damagePerShot * shotsPerSecond;
	}

	public int getWeaponDamage(WeaponType weapon) {
		return weapon.damageAmount() + weapon.damageBonus() * getUpgradeLevel(weapon.upgradeType());
	}
	
	public int getWeaponRange(WeaponType weapon) {
		int range = weapon.maxRange();
		if (weapon.equals(WeaponType.GAUSS_RIFLE) && getUpgradeLevel(UpgradeType.U_238_SHELLS) > 0) {
			range += 32;
		}
		if (weapon.equals(WeaponType.PHASE_DISRUPTOR) && getUpgradeLevel(UpgradeType.SINGULARITY_CHARGE) > 0) {
			range += 64;
		}
		if (weapon.equals(WeaponType.NEEDLE_SPINES) && getUpgradeLevel(UpgradeType.GROOVED_SPINES) > 0) {
			range += 32;
		}
		if (weapon.equals(WeaponType.HELLFIRE_MISSILE_PACK) && getUpgradeLevel(UpgradeType.CHARON_BOOSTER) > 0) {
			range += 96;
		}
		return range;
	}

	private static double getDamageMultiplier(DamageType damageType, UnitSizeType size) {
		if (damageType.equals(DamageType.CONCUSSIVE)) {
			if (size.equals(UnitSizeType.LARGE)) return 0.25;
			if (size.equals(UnitSizeType.MEDIUM)) return 0.5;
		}
		if (damageType.equals(DamageType.EXPLOSIVE)) {
			if (size.equals(UnitSizeType.SMALL)) return 0.5;
			if (size.equals(UnitSizeType.MEDIUM)) return 0.75;
		}
		return 1.0;
	}
	
	public boolean canSeeUnitAtPosition(UnitType type, Position pos) {
		TilePosition tilePosition = new TilePosition(pos);
		// The isVisible() logic was determined by experiments in Brood War.
		// It is a little weird: a building is considered visible
		// when any of the build tiles it occupies are visible even if the building is lifted.
		// Consider the upper-left corner of the dimensions box of a unit to be the origin.
		// A non-building unit is considered visible if its origin is on a visible tile.
		// Additionally, if the dimensions box is larger than a build tile,
		// the unit is considered visible if the rectangle made of build tiles
		// that best fits into the dimensions box when the rectangle
		// is aligned to the origin overlaps a visible build tile.
		if (type.isBuilding()) {
			// Building case:
			
			// True if any build tile the building occupies is visible.
			for (int buildDeltaX = 0; buildDeltaX < type.tileWidth(); ++buildDeltaX) {
				for (int buildDeltaY = 0; buildDeltaY < type.tileHeight(); ++ buildDeltaY) {
					if (Game.getInstance().isVisible(
							tilePosition.x() + buildDeltaX, tilePosition.y() + buildDeltaY))
					{
						return true;
					}
				}
			}
			
			return false;
		}
		else {
			// Non-building case:
			
			final int dimensionOriginX = pos.x() - type.dimensionLeft();
			final int dimensionOriginY = pos.y() - type.dimensionUp();
			
			// The rectangle for determining visibility may be offset from the dimensions box of the unit.
			// (See the comments for the initialization of the offsets in the static initializer.)
			int adjustedDimensionOriginX = dimensionOriginX;
			int adjustedDimensionOriginY = dimensionOriginY;
			final Position visibilityOffset = gUnitTypesToVisibilityOffset.get(type);
			if (visibilityOffset != null) {
				adjustedDimensionOriginX += visibilityOffset.x();
				adjustedDimensionOriginY += visibilityOffset.y();
			}
			
			final int dimensionRightOfOrigin = type.dimensionLeft() + type.dimensionRight();
			final int dimensionDownOfOrigin =  type.dimensionUp() + type.dimensionDown();
			
			// Note: Has to be "floored," not rounded.
			final int dimensionRightOfOriginInBuildTiles = dimensionRightOfOrigin / 32;
			final int dimensionDownOfOriginInBuildTiles = dimensionDownOfOrigin / 32;
			
			// "Lower bounds" that the visibility rectangle touches.
			// (Note: We don't just floor using integer division because the coordinates may be negative.)
			final int buildTileLeft = (int) Math.floor(
					adjustedDimensionOriginX / 32.0);
			final int buildTileTop = (int) Math.floor(
					adjustedDimensionOriginY / 32.0);
			
			// "Upper bounds" that the visibility rectangle touches.
			// Note that we are dealing with inclusive ranges (not half-open).
			// (Note: We don't just floor using integer division because the coordinates may be negative.)
			final int buildTileRight = (int) Math.floor(
					(adjustedDimensionOriginX + dimensionRightOfOriginInBuildTiles * 32)
					/ 32.0);
			final int buildTileBottom = (int) Math.floor(
					(adjustedDimensionOriginY + dimensionDownOfOriginInBuildTiles * 32.0)
					/ 32.0);
			
			// True if the adjusted dimensions box is on a visible build tile.
			for (int buildTileX = buildTileLeft; buildTileX <= buildTileRight; ++buildTileX) {
				for (int buildTileY = buildTileTop; buildTileY <= buildTileBottom; ++buildTileY) {
					if (Game.getInstance().isVisible(buildTileX, buildTileY)) {
						return true;
					}
				}
			}
			
			return false;
		}
	}
	
	protected static Map<UnitType, Position> gUnitTypesToVisibilityOffset = new HashMap<UnitType, Position>();
	static {
		// In Brood War, the rectangle that determines whether or not a non-building unit is visible
		// is almost like the unit's dimension rectangle where the width and height
		// are floored to the nearest whole build tile unit.
		// However, for some unit types, this "visibility rectangle" has an offset.
		// The following offsets were painstakingly gathered
		// by checking when non-building units become invisible at the border of the fog of war,
		// whether it is above, below, or on either side of the unit.
		// TODO: The offset for only AIR and GROUND units
		// as they are given in the campaign editor were tested for.
		// This means that we need to determine the offset for units such as eggs and larva.
		// TODO: The offset for Interceptors was not determined
		// because they do not keep still and cannot be created in the Starcraft campaign editor.
		// However, all the other Protoss air units do not have offsets.
		// TODO: Determine the offsets of neutral non-building units.
		
		// Protoss:
		// Dark Templar: Up nine.
		gUnitTypesToVisibilityOffset.put(UnitType.PROTOSS_DARK_TEMPLAR, new Position(0, -9));
		// Dragoon: Left one, up one.
		gUnitTypesToVisibilityOffset.put(UnitType.PROTOSS_DRAGOON, new Position(-1, -1));
		// High Templar: Up four.
		gUnitTypesToVisibilityOffset.put(UnitType.PROTOSS_HIGH_TEMPLAR, new Position(0, -4));
		// Zealot: Up eight.
		gUnitTypesToVisibilityOffset.put(UnitType.PROTOSS_ZEALOT, new Position(0, -8));
		
		// Terran:
		// Battlecruiser: Left three, up three.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_BATTLECRUISER, new Position(-3, -3));
		// Dropship: Up two.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_DROPSHIP, new Position(0, -2));
		// Firebat: Up seven.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_FIREBAT, new Position(0, -7));
		// Ghost: Up one.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_GHOST, new Position(0, -1));
		// Marine: Up one.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_MARINE, new Position(0, -1));
		// Medic: Up one.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_MEDIC, new Position(0, -1));
		// Science Vessel: Down eight.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_SCIENCE_VESSEL, new Position(0, 8));
		// Spider Mine: Left one, up one.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_VULTURE_SPIDER_MINE, new Position(-1, -1));
		// Valkyrie: Left one, up nine.
		gUnitTypesToVisibilityOffset.put(UnitType.TERRAN_VALKYRIE, new Position(-1, -9));

		// Zerg:
		// Hydralisk: Up one.
		gUnitTypesToVisibilityOffset.put(UnitType.ZERG_HYDRALISK, new Position(0, -1));
		// Infested Terran: Up one.
		gUnitTypesToVisibilityOffset.put(UnitType.ZERG_INFESTED_TERRAN, new Position(0, -1));
		// Lurker: Left one, up one.
		gUnitTypesToVisibilityOffset.put(UnitType.ZERG_LURKER, new Position(-1, -1));
		// Zergling: Up four.
		gUnitTypesToVisibilityOffset.put(UnitType.ZERG_ZERGLING, new Position(0, -4));
	}
	
	


}
