#include "terrainanalysis.h"
#include <boost/foreach.hpp>
#include "ClientModule.h"

#include <boost/thread/locks.hpp>

using namespace BWTA;

void TerrainAnalysis::waitForTerrain() {
	boost::unique_lock<boost::mutex> lock(terrainMutex);
	while(!terrainFinished) {
		cond.wait(lock);
	}
}

void TerrainAnalysis::actuallyRunTerrainAnalysis(TerrainAnalysis* self)
{
	BWTA::analyze();
	{
		boost::mutex::scoped_lock lock(self->terrainMutex);
		self->terrainFinished = true;

	}
	self->cond.notify_all();
}

void TerrainAnalysis::startTerrainAnalysis() {
	BWTA::readMap();
	terrainFinished = false;
	terrainSent = false;
	rids.clear();
	cids.clear();
	bids.clear();
	terrainWorker.swap(boost::thread(&actuallyRunTerrainAnalysis,this));
}

void TerrainAnalysis::addTerrainAnalysisData(messages::StaticTerrainInfo* msg) {
	const std::set<Region*> regions = BWTA::getRegions();
	msg->mutable_regions()->Reserve(regions.size());	

	BOOST_FOREACH(Region* r, regions) {
		messages::Region* mr = msg->add_regions();
		int id = idOf(r);
		mr->set_id(id);
	
		setPolygon(r->getPolygon(), mr->mutable_polygon());
		BOOST_FOREACH(Chokepoint* c, r->getChokepoints()) {
			mr->add_chokepoints(idOf(c));
		}

		BOOST_FOREACH(BaseLocation* c, r->getBaseLocations()) {
			mr->add_baselocations(idOf(c));
		}
	}

	BOOST_FOREACH(Chokepoint* c, BWTA::getChokepoints()) {
		messages::Chokepoint* mc = msg->add_chokepoints();
		mc->set_id(idOf(c));
		mc->set_region1(idOf(c->getRegions().first));
		mc->set_region2(idOf(c->getRegions().second));

		set_position(c->getCenter(),mc->mutable_center());
		set_position(c->getSides().first,mc->mutable_side1());
		set_position(c->getSides().second, mc->mutable_side2());
	    mc->set_width(c->getWidth());
	}

	BOOST_FOREACH(BaseLocation* b, BWTA::getBaseLocations()) {
		messages::BaseLocation* mb = msg->add_baselocations();
		mb->set_id(idOf(b));
		set_position(b->getPosition(), mb->mutable_position());
		set_position(b->getTilePosition(), mb->mutable_tileposition());
		mb->set_region(idOf(b->getRegion()));

		mb->set_numminerals(b->minerals());
		mb->set_numgas(b->gas());

		set_ids(b->getMinerals(), mb->mutable_minerals());
		set_ids(b->getGeysers(), mb->mutable_geysers());
		set_ids(b->getStaticMinerals(), mb->mutable_staticminerals());

		mb->set_island(b->isIsland());
		mb->set_mineralonly(b->isMineralOnly());
		mb->set_startlocation(b->isStartLocation());
		if(b->isStartLocation()) {
			msg->add_startlocations(mb->id());
		}
	}

	BOOST_FOREACH(Polygon* p, BWTA::getUnwalkablePolygons()) {
		setPolygon(*p, msg->add_unwalkablepolygon());
	}
}

void TerrainAnalysis::setPolygon(const Polygon&p, messages::Polygon* mp) {
	for(size_t i = 0; i < p.size(); ++i) {
		set_position(p[i],mp->add_vertices());
	}
	for(size_t i = 0; i < p.holes.size(); ++i) {
		setPolygon(p.holes[i],mp->add_holes());
	}
	mp->set_area(p.getArea());
	mp->set_perimeter(p.getPerimeter());
	set_position(p.getCenter(), mp->mutable_center());
}

int TerrainAnalysis::idOf(Region* r) {
	if(rids.find(r) == rids.end()) {
		rids[r] = rids.size();
	}
	return rids[r];
}

int TerrainAnalysis::idOf(Chokepoint* r) {
	if(cids.find(r) == cids.end()) {
		cids[r] = cids.size();
	}
	return cids[r];
}

int TerrainAnalysis::idOf(BaseLocation* r) {
	if(bids.find(r) == bids.end()) {
		bids[r] = bids.size();
	}
	return bids[r];
}