#include "ClientModule.h"

using namespace BWAPI;
#include <boost/thread/thread.hpp>
#include <boost/foreach.hpp>

#include <cstdio>
#include <winsock.h>
#include <stdio.h>
#include <sstream>
#include <fstream>
#include <ctime>
#include "netstream.h"
#include "md5.h"

#include <iostream>
#include <windows.h>
#include <fstream>

/** port to connect to on the java side */
// Broodwar->getInstanceNumber() is added to this
#define PORTNUM 12345

// for winners and things
static std::string botName;
static std::string botCode;
static std::string playerName;
static bool hasBotName = false;
bool hadLowSupply = false;
bool hadSomeSupply = false;
bool battleOk = true;
bool drawOk = true;

/** functions */

struct WSAStartupper {
	WSAStartupper() {
		WORD wVersionRequested;
		WSADATA wsaData;

		wVersionRequested = MAKEWORD( 2, 2 );
		int ret = 0;
		if((ret = WSAStartup(wVersionRequested, &wsaData)) != 0) {
			char Message[1024];

			FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS |
				FORMAT_MESSAGE_MAX_WIDTH_MASK, NULL, ret,
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
				(LPSTR) Message, 1024, NULL);

		} 
	}

	~WSAStartupper() {
		WSACleanup();
	}
} WSAStartupper;

NetStream* initSocket();
void doDynamicTerrain(messages::FrameMessage& msg);
void handleCommands(const messages::FrameCommands& commands);
void exportTypeData(messages::StaticGameData& msg);

ClientModule::ClientModule():proxyBotSocket(0), log_("bwapi-data\\logs\\proxy.txt",std::ios_base::app) {

	log("Online");
}

void ClientModule::log(const boost::format& f) {
	std::time_t rawtime;
	std::time ( &rawtime );
	std::tm* info = localtime ( &rawtime );
	char buf[1024];
	std::strftime(buf, 1024, "%x %X", info);
	log_ << boost::format("[%1%] ") % buf;
	log_ << f << std::endl;
}

inline void set_bit(google::protobuf::uint32& field, int bit, bool v) {
	if (v) field |= (1 << bit); else field &= ~(1<<bit); 
}

messages::Race race2Race(Race r) {
	if(r == Races::Zerg) return messages::Zerg;
	if(r == Races::Protoss) return messages::Protoss;
	if(r == Races::Terran) return messages::Terran;
	if(r == Races::Random) return messages::Random;
	if(r == Races::None) return messages::None;
	if(r == Races::Other) return messages::Other;
	return messages::Unknown;
}

int latency(Latency::Enum lat) {
	switch(lat) {
		case Latency::SinglePlayer: return 2;
		case Latency::LanLow: return 5;
		case Latency::LanMedium: return 7;
		case Latency::LanHigh: return 9;
		case Latency::BattlenetLow: return 14;
		case Latency::BattlenetMedium: return 19;
		case Latency::BattlenetHigh: return 24;
		default: throw std::runtime_error("bad latency");
	}
}

messages::UnitSizeType size2Size(UnitSizeType sz) {
	if(!messages::UnitSizeType_IsValid(sz.getID())) { Broodwar->printf("badSize!"); throw std::runtime_error("WTF? size"); }
	return (messages::UnitSizeType)(sz.getID());
}

messages::ExplosionType ex2ex(ExplosionType sz) {
	if(!messages::ExplosionType_IsValid(sz.getID())) {Broodwar->printf("bad explosion!"); throw std::runtime_error("WTF? explosion"); }
	return (messages::ExplosionType)(sz.getID());
}

messages::DamageType dam2dam(DamageType sz) {
	if(!messages::DamageType_IsValid(sz.getID())) { Broodwar->printf("bad damage!"); throw std::runtime_error("WTF? damage"); }
	return (messages::DamageType)(sz.getID());
}



// writes a message so that readDelimited on the java side will read it.
bool ClientModule::writeMessageToStream(std::ostream& out, const google::protobuf::Message& message) {
	bool ret = false;
	{
		google::protobuf::io::OstreamOutputStream oo(&out);
		google::protobuf::io::CodedOutputStream c(&oo);
		int sz = message.ByteSize();
		c.WriteVarint32(sz);
		ret = message.SerializeToCodedStream(&c);
	}
	out.flush();
	return ret;
}

// read a message written on the Java side as <Int32><Msg>
bool ClientModule::readMessageFromStream(std::istream& in, google::protobuf::Message& message) {

	google::protobuf::uint32 sz = 0;
	in.read((char*)&sz,4);
	sz = ntohl(sz);
	char* buf = (char*)alloca(sz);
	in.read(buf, sz);
	return message.ParseFromArray(buf,sz);
}



void marshall_counts(Player* player, int(Player::*countOf)(UnitType) const, google::protobuf::RepeatedPtrField<messages::UnitTypeIntPair>* target) {
	BOOST_FOREACH(UnitType u, UnitTypes::allUnitTypes()) {
		int count = ((*player).*countOf)(u);
		if(count == 0) continue;
		messages::UnitTypeIntPair* pair = target->Add();
		pair->set_count(count);
		pair->mutable_type()->set_id(u.getID());
	}
}

static std::string inferPlayerName() {
	std::ifstream in("teamname.txt");
	if(!in) return "";
	std::string res;
	in >> res;
	return res;
}

/**
* Called at the start of a match. 
*/
void ClientModule::onStart()
{
	hasBotName = false;
	playerName = inferPlayerName();
	battleOk = true;

	try {
		// First, check if we are already connected to the Java proxy
		if (proxyBotSocket == 0) {
			Broodwar->sendText("Connecting to ProxyBot");		
			proxyBotSocket = initSocket();

			// connected failed
			if (proxyBotSocket == 0) {
				Broodwar->sendText("ProxyBot connected failed");
				return;
			}
			else {
				Broodwar->sendText("Sucessfully connected to ProxyBot");
			}
		}
		else {
			Broodwar->sendText("Already connected to ProxyBot");
		}

		unitMap.clear();
		terrain.startTerrainAnalysis();


		messages::StaticGameData start;
		set_ids(Broodwar->getStaticGeysers(),start.mutable_staticgeysers());
		set_ids(Broodwar->getStaticMinerals(),start.mutable_staticminerals());
		set_ids(Broodwar->getStaticNeutralUnits(),start.mutable_staticneutralunits());

		start.set_mapfilename(Broodwar->mapFileName());
		start.set_maphash(Broodwar->mapHash());
		start.set_mapname(Broodwar->mapName());
		start.set_mapheight(Broodwar->mapHeight());
		start.set_mapwidth(Broodwar->mapWidth());


		{
			google::protobuf::uint32 buildable = 0;

			int index = 0;
			for(int x = 0; x < Broodwar->mapWidth(); ++x) {
				for(int y = 0; y < Broodwar->mapHeight(); ++y) {
					set_bit(buildable,index,Broodwar->isBuildable(x,y));
					if(++index == 32) {
						index = 0;
						start.add_buildable(buildable);
						buildable = 0;
					}
				}
			}
			start.add_buildable(buildable);
		}

		{
			int index = 0;
			google::protobuf::uint32 walkable = 0;
			for(int x = 0; x < Broodwar->mapWidth() * 4; ++x) {
				for(int y = 0; y < Broodwar->mapHeight() * 4; ++y) {
					set_bit(walkable,index,Broodwar->isWalkable(x,y));
					if(++index == 32) {
						index = 0;
						start.add_walkable(walkable);
						walkable = 0;
					}
				}
			}
			start.add_walkable(walkable);
		}


		for(int x = 0; x < Broodwar->mapWidth() * 4; ++x) {
			for(int y = 0; y < Broodwar->mapHeight() * 4; ++y) {
				start.add_groundheight(Broodwar->getGroundHeight(x,y));
			}
		}

		BOOST_FOREACH(TilePosition pos, Broodwar->getStartLocations()) {
			set_position(pos,start.add_startlocations());
		}


		// Colors
		for(int i = 0; i < 256; ++i) {
			Color c(i);
			messages::Color* mc = start.add_colors();
			mc->set_id(i);
			mc->set_blue(c.blue());
			mc->set_red(c.red());
			mc->set_green(c.green());
		}
		exportTypeData(start);

		if(!writeMessageToStream(*proxyBotSocket, start)) { Broodwar->printf("screwed up initial write!"); delete proxyBotSocket; proxyBotSocket = 0;}
		bool speedExists = (std::ifstream("superfast"));
		if(speedExists) {
			Broodwar->printf("Vroom vroom");
			Broodwar->setLocalSpeed(0);
		}

		drawOk = !(std::ifstream("nodrawing"));
		

		messages::BoxedBoolean b;
		if(!readMessageFromStream(*proxyBotSocket, b)) {
			Broodwar->printf("Didn't get an ACK from host!");
			delete proxyBotSocket; proxyBotSocket = 0; 
		}  
	} catch(const std::runtime_error& err) {
		proxyBotSocket = 0;
		Broodwar->printf("Can't connect to socket! %s",err.what());
		this->log(err.what());
	} catch(...) {
		proxyBotSocket = 0;
		Broodwar->printf("Can't connect to socket!");
		this->log("uncaught exception?");
	}
}


int badTimes = 0;

void ClientModule::registerUnit(messages::FrameMessage& frameMessage, Unit * u) {
	if(!registeredUnits.insert(u).second) return;

	messages::Unit* mu = frameMessage.add_units();

	unitMap[u->getID()] = u;

	mu->set_player(u->getPlayer()->getID());
	mu->mutable_theid()->set_id(u->getID());
	SET_ID(&u->getType(),mu->mutable_type());
	SET_ID(&u->getInitialType(),mu->mutable_initialtype());
	mu->set_hitpoints(u->getHitPoints());
	mu->set_initialhitpoints(u->getInitialHitPoints());
	mu->set_shields(u->getShields());
	mu->set_energy(u->getEnergy());
	mu->set_resources(u->getResources());
	mu->set_initialresources(u->getInitialResources());
	mu->set_killcount(u->getKillCount());
	mu->set_groundweaponcooldown(u->getGroundWeaponCooldown());
	mu->set_airweaponcooldown(u->getAirWeaponCooldown());
	mu->set_spellcooldown(u->getSpellCooldown());
	mu->set_defensematrixpoints(u->getDefenseMatrixPoints());

	mu->set_defensematrixtimer(u->getDefenseMatrixTimer());
	mu->set_ensnaretimer(u->getEnsnareTimer());
	mu->set_irradiatetimer(u->getIrradiateTimer());
	mu->set_lockdowntimer(u->getLockdownTimer());
	mu->set_maelstromtimer(u->getMaelstromTimer());
	mu->set_plaguetimer(u->getPlagueTimer());
	mu->set_removetimer(u->getRemoveTimer());
	mu->set_stasistimer(u->getStasisTimer());
	mu->set_stimtimer(u->getStimTimer());
	mu->set_resourcegroup(u->getResourceGroup());


	set_position(u->getPosition(),mu->mutable_position());
	set_position(u->getInitialPosition(),mu->mutable_initialposition());
	set_position(u->getTilePosition(),mu->mutable_tileposition());
	set_position(u->getInitialTilePosition(),mu->mutable_initialtileposition());
	mu->set_angle(u->getAngle());
	mu->set_velocityx(u->getVelocityX());
	mu->set_velocityy(u->getVelocityY());

	BOOST_FOREACH(Unit* u2, u->getInterceptors()) {
		SET_ID(u2, mu->add_interceptors());
	}
	SET_ID(u->getCarrier(), mu->mutable_carrier());

	BOOST_FOREACH(Unit* u2, u->getLarva()) {
		SET_ID(u2, mu->add_larvae());
	}

	SET_ID(u->getHatchery(), mu->mutable_hatchery());
	SET_ID(u->getNydusExit(), mu->mutable_nydusexit());

	SET_ID(u->getTarget(),mu->mutable_target());
	set_position(u->getTargetPosition(),mu->mutable_targetposition());
	mu->set_order(u->getOrder().getID());
	SET_ID(u->getOrderTarget(),mu->mutable_ordertarget());
	mu->set_ordertimer(u->getOrderTimer());
	mu->set_secondaryorder(u->getSecondaryOrder().getID());
	SET_ID(u->getBuildUnit(),mu->mutable_buildunit());
	mu->set_remainingbuildtime(u->getRemainingBuildTime());
	mu->set_remainingtraintime(u->getRemainingTrainTime());
	BOOST_FOREACH(UnitType t, u->getTrainingQueue()) {
		mu->add_trainingqueue()->set_id(t.getID());
	}
	mu->mutable_buildtype()->set_id(u->getBuildType().getID());
	SET_ID(u->getTransport(),mu->mutable_transport());
	set_ids(u->getLoadedUnits(),mu->mutable_loadedunits());
	mu->set_interceptorcount(u->getInterceptorCount());
	mu->set_scarabcount(u->getScarabCount());
	mu->set_spiderminecount(u->getSpiderMineCount());
	mu->set_tech(u->getTech().getID());
	mu->set_upgrade(u->getUpgrade().getID());
	mu->set_remainingresearchtime(u->getRemainingResearchTime());
	mu->set_remainingupgradetime(u->getRemainingUpgradeTime());
	set_position(u->getRallyPosition(), mu->mutable_rallyposition());
	SET_ID(u->getRallyUnit(),mu->mutable_rallyunit());
	SET_ID(u->getAddon(),mu->mutable_addon());

	mu->set_exists(u->exists());
	mu->set_hasnuke(u->hasNuke());
	mu->set_accelerating(u->isAccelerating());
	mu->set_beingconstructed(u->isBeingConstructed());
	mu->set_beinghealed(u->isBeingHealed());
	mu->set_blind(u->isBlind());
	mu->set_braking(u->isBraking());
	mu->set_burrowed(u->isBurrowed());
	mu->set_carryinggas(u->isCarryingGas());
	mu->set_carryingminerals(u->isCarryingMinerals());
	mu->set_cloaked(u->isCloaked());
	mu->set_completed(u->isCompleted());
	mu->set_constructing(u->isConstructing());
	mu->set_defensematrixed(u->isDefenseMatrixed());
	mu->set_ensnared(u->isEnsnared());
	mu->set_following(u->isFollowing());
	mu->set_gatheringgas(u->isGatheringGas());
	mu->set_gatheringminerals(u->isGatheringMinerals());
	mu->set_hallucination(u->isHallucination());
	mu->set_idle(u->isIdle());
	mu->set_irradiated(u->isIrradiated());
	mu->set_lifted(u->isLifted());
	mu->set_loaded(u->isLoaded());
	mu->set_lockeddown(u->isLockedDown());
	mu->set_maelstrommed(u->isMaelstrommed());
	mu->set_morphing(u->isMorphing());
	mu->set_moving(u->isMoving());
	mu->set_parasited(u->isParasited());
	mu->set_patrolling(u->isPatrolling());
	mu->set_plagued(u->isPlagued());
	mu->set_repairing(u->isRepairing());
	mu->set_researching(u->isResearching());
	mu->set_selected(u->isSelected());
	mu->set_sieged(u->isSieged());
	mu->set_startingattack(u->isStartingAttack());
	mu->set_stasised(u->isStasised());
	mu->set_stimmed(u->isStimmed());
	mu->set_training(u->isTraining());
	mu->set_understorm(u->isUnderStorm());
	mu->set_underattack(u->isUnderAttack());
	mu->set_underdarkswarm(u->isUnderDarkSwarm());
	mu->set_unpowered(u->isUnpowered());
	mu->set_upgrading(u->isUpgrading());
	mu->set_visible(u->isVisible());
	mu->set_beinggathered(u->isBeingGathered());
	mu->set_detected(u->isDetected());
	mu->set_attacking(u->isAttacking());
	mu->set_attackframe(u->isAttackFrame());
	mu->set_stuck(u->isStuck());
}

static bool bulletMatters(Bullet* b) {
	return true;
}

long totalTime = 0;
long totalFrames = 0;


bool checkIfBattleOk() {
	bool playersOk = Broodwar->getPlayers().size() == 2;
	//if(!playersOk) {
	//	Broodwar->printf("Too many players (%d) in this game. Boss battle doesn't count!", Broodwar->getPlayers().size());
	//	return false;
	//}
	if(!Broodwar->isMultiplayer()) {
		Broodwar->printf("Not a multiplayer game. Boss battle doesn't count!");
	}
	std::string mapName = Broodwar->mapName();
	if(mapName.find("eartbrea") == std::string::npos && mapName.find("ython") == std::string::npos && mapName.find("ross") == std::string::npos) {
		Broodwar->printf("Map %s is not an acceptable map for a boss battle!",mapName.c_str());
		return false;
	}
	return true;
}

static int getNumBuildings() {
	int numBuildings = 0;
	BOOST_FOREACH(Unit * u,Broodwar->self()->getUnits()) {
		if(u->getType().isBuilding()) {
			numBuildings+=1;
		}
	}
	return numBuildings;
}


/**
* Runs every frame 
*
* Sends the unit status to the ProxyBot, then waits for a list of command messages.
*/
void ClientModule::onFrame()
{
	// check if the Proxy Bot is connected
	if (proxyBotSocket == 0) {
		return;
	}

	frameMessage.set_framecount(Broodwar->getFrameCount());
	frameMessage.set_mousex(Broodwar->getMousePosition().x());
	frameMessage.set_mousey(Broodwar->getMousePosition().y());
	frameMessage.set_screenx(Broodwar->getScreenPosition().x());
	frameMessage.set_screeny(Broodwar->getScreenPosition().y());


	BOOST_FOREACH(Unit* u, Broodwar->getAllUnits()) {
		registerUnit(frameMessage,u);
	}
	registeredUnits.clear();


	BOOST_FOREACH(Player* player, Broodwar->getPlayers()) {
		messages::Player* mp = frameMessage.add_players();

		set_ids(player->getUnits(), mp->mutable_units());
		mp->set_id(player->getID());
		mp->set_name(player->getName());
		mp->set_race(race2Race(player->getRace()));
		//mu->set_playertype(u->getPlayerType())();
		//required Force* getForce();
		mp->set_neutral(player->isNeutral());
		set_position(player->getStartLocation(),mp->mutable_startlocation());
		mp->set_leftgame(player->leftGame());
		mp->set_victorious(player->isVictorious());
		mp->set_defeated(player->isDefeated());

		mp->set_minerals(player->minerals());
		mp->set_gas(player->gas());
		mp->set_cumulativeminerals(player->cumulativeMinerals());
		mp->set_cumulativegas(player->cumulativeGas());

		mp->set_supplytotal(player->supplyTotal());
		mp->set_supplyused(player->supplyUsed());

		marshall_counts(player, &Player::allUnitCount, mp->mutable_allunitcount());
		marshall_counts(player, &Player::completedUnitCount, mp->mutable_completedunitcount());
		marshall_counts(player, &Player::incompleteUnitCount, mp->mutable_incompleteunitcount());
		marshall_counts(player, &Player::deadUnitCount, mp->mutable_deadunitcount());
		marshall_counts(player, &Player::killedUnitCount, mp->mutable_killedunitcount());

		BOOST_FOREACH(UpgradeType ug, UpgradeTypes::allUpgradeTypes()) {
			if(player->getUpgradeLevel(ug) > 0) {
				messages::UpgradeLevel* lvl = mp->add_upgradelevels();
				lvl->set_level(player->getUpgradeLevel(ug));
				lvl->mutable_type()->set_id(ug.getID());
			}

			if(player->isUpgrading(ug)) {
				mp->add_upgrading()->set_id(ug.getID());
			}
		}


		BOOST_FOREACH(TechType t, TechTypes::allTechTypes()) {
			if(player->hasResearched(t)) {
				mp->add_researchedtechs()->set_id(t.getID());
			} else if(player->isResearching(t)) {
				mp->add_researching()->set_id(t.getID());
			}
		}

		if(player == Broodwar->self()) {
			mp->set_self(true);
		}

		BOOST_FOREACH(Player* p2, Broodwar->getPlayers()) {
			messages::PlayerStatus* stance = mp->add_stances();
			stance->set_id(p2->getID());
			stance->set_ownerid(player->getID());

			if(player->isAlly(p2)) {
				stance->set_stance(messages::Ally);
			} else if(player->isEnemy(p2)) {
				stance->set_stance(messages::Enemy);
			} else {
				stance->set_stance(messages::Neutral);
			}
		}

	}

	frameMessage.set_multiplayer(Broodwar->isMultiplayer());
	frameMessage.set_paused(Broodwar->isPaused());
	frameMessage.set_replay(Broodwar->isReplay());
	frameMessage.set_latency(Broodwar->getLatency())	;
	set_position(Broodwar->getScreenPosition(),frameMessage.mutable_screenposition());
	set_ids(Broodwar->getSelectedUnits(), frameMessage.mutable_selectedunits());

	BOOST_FOREACH(Bullet* b, Broodwar->getBullets()) {
		if(bulletMatters(b)) {
			messages::Bullet* mb = frameMessage.add_bullets();
			mb->set_id(b->getID());
			if(b->getPlayer() != 0) mb->set_player(b->getPlayer()->getID());
			mb->set_type( (messages::BulletType)(b->getType().getID()));
			SET_ID(b->getSource(), mb->mutable_source());
			set_position(b->getPosition(), mb->mutable_position());
			mb->set_angle(b->getAngle());
			mb->set_velocityx(b->getVelocityX());
			mb->set_velocityy(b->getVelocityY());
			SET_ID(b->getTarget(), mb->mutable_target());
			set_position(b->getTargetPosition(), mb->mutable_targetposition());
			mb->set_removetimer(b->getRemoveTimer());
			mb->set_exists(b->exists());
			mb->set_isvisible(b->isVisible());
		}
	}

	doDynamicTerrain(frameMessage);

	
	if(!terrain.isTerrainSent()) {
		if(!terrain.isTerrainFinished()) {
			terrain.waitForTerrain();
		}
		terrain.setTerrainSent();
		Broodwar->printf("Terrain Analysis complete");
		terrain.addTerrainAnalysisData(frameMessage.mutable_terraininfo());
	}
	

	if(!writeMessageToStream(*proxyBotSocket, frameMessage)) {
		Broodwar->printf("Couldn't send message for frame %d",Broodwar->getFrameCount());
		delete proxyBotSocket;
		proxyBotSocket = 0;

		frameMessage.Clear();
		return;
	}

	


	bool isGameOver = frameMessage.gameover();

	frameMessage.Clear();

	if(isGameOver) {
		delete proxyBotSocket;
		proxyBotSocket = 0;
		return;
	}

	long timeIn = GetTickCount();
	messages::FrameCommands response;
	if(!readMessageFromStream(*proxyBotSocket, response)) {
		Broodwar->printf("Received message was bad for frame %d", Broodwar->getFrameCount());
		return;
	}
	if(response.has_botname() && battleOk && !hasBotName) {
		botName = response.botname();
		botCode = response.botcode();
		hasBotName= true;
		//if(Broodwar->isFlagEnabled(USER_INPUT) {
		//	Broodwar->printf("Disabling user input for boss battle.");
		//	Broodwar->s
		//}
		battleOk = checkIfBattleOk();
		if(!battleOk) {
			Broodwar->printf("Boss battle disabled!");
		}
		if(playerName.empty()) {
			Broodwar->printf("No player name! Please make a file called c:\\program files\\starcraft\\teamname.txt! Boss Battle disabled.");
			battleOk = false;
		} else {
			Broodwar->printf("Boss battle enabled. Player is %s",playerName.c_str());
		}

	}

	handleCommands(response);
	long timeOut = GetTickCount();
	totalTime += timeOut - timeIn;
	if(timeOut - timeIn > 150) badTimes += 1;
	totalFrames += 1;

	int numBuildings = getNumBuildings();
	hadSomeSupply = hadSomeSupply || (totalFrames > (40 * 60 * 3) && numBuildings > 1);
	hadLowSupply = totalFrames > (40 * 60 * 3) && (Broodwar->self()->supplyUsed() < 10 || numBuildings < 2);

	double timePerFrame = totalTime * 1.0 / totalFrames;
	Broodwar->drawTextScreen(100,10,"%cTime per frame: %f", (timePerFrame > 43 ? '\x06' : timePerFrame > 39 ? '\x03' : '\x07'), timePerFrame); 
	Broodwar->drawTextScreen(100,20,"%cBad times: %d", (badTimes > 1 ? '\x06' : timePerFrame > 0 ? '\x03' : '\x07'), badTimes); 


}

void ClientModule::handleBotDefeated(const std::string& bossName, const std::string& bossCode,
					   const std::string& playerName) {
	char buf[4096];
	std::string salt = "qqq";
	std::string code = salt + ":" + bossCode + ":" + playerName;
	MD5 mm(code);
	_snprintf(buf,4096,"You have defeated me!\nConfirmation code: %s %s %s\nUse ctrl-c to copy me to clipboard, or look in Starcraft\\bwapi-data\\logs\\proxy.txt .", playerName.c_str(), botName.c_str(), mm.hexdigest().c_str());
	log(buf);
	//log(code);
	log(mm.hexdigest());
	int msgboxID = MessageBox(
        NULL,
		(LPCSTR)buf,
        (LPCSTR)"Victory!",
        MB_ICONWARNING | MB_OK
    );

}

/**
* Called at the end of a game. This is where we shut down sockets and clean
* up any data structures we have created.
*/
void ClientModule::onEnd(bool isWinner) 
{

	
	if(isWinner) {
		if(std::ifstream("lose")) {
			std::remove("lose");
		}
		if(!(std::ifstream("win"))) {
			std::ofstream("win");
		}
	} else {
		if(std::ifstream("win")) {
			std::remove("win");
		}
		if(!std::ifstream("lose")) {
			std::ofstream("lose");
		}
	}

	if(!isWinner && hasBotName && hadLowSupply && battleOk && hadSomeSupply) {
		handleBotDefeated(botName,botCode,playerName);
	}

	if (proxyBotSocket == 0) {
		return;
	}
    Broodwar->setGUI(true);

	frameMessage.set_iswinner(isWinner);
	frameMessage.set_gameover(true);
}





void ClientModule::onUnitCreate(Unit* unit)
{
	registerUnit(frameMessage,unit);
	frameMessage.add_createdunits()->set_id(unit->getID());
}

void ClientModule::onUnitRenegade(Unit* unit)
{
	registerUnit(frameMessage,unit);
	frameMessage.add_renegadedunits()->set_id(unit->getID());
}

void ClientModule::onUnitHide(Unit* unit)
{
	registerUnit(frameMessage,unit);
	frameMessage.add_hiddenunits()->set_id(unit->getID());
}

void ClientModule::onUnitShow(Unit* unit)
{
	registerUnit(frameMessage,unit);
	frameMessage.add_shownunits()->set_id(unit->getID());
}

void ClientModule::onUnitMorph(Unit* unit)
{
	registerUnit(frameMessage,unit);
	frameMessage.add_morphedunits()->set_id(unit->getID());
}

void ClientModule::onNukeDetect(Position pos) {
	set_position(pos,frameMessage.mutable_nukedetect());
}

void ClientModule::onPlayerLeft(Player* player)
{
	frameMessage.add_leftplayers(player->getID());
}


/**
* Removes the unit from the ID->unit mapping
*/
void ClientModule::onUnitDestroy(BWAPI::Unit* unit)
{
	frameMessage.add_destroyedunits()->set_id(unit->getID());
}

void ClientModule::onSendText(std::string text)
{
	frameMessage.add_senttext(text);
}

void ClientModule::onSaveGame(std::string text)
{
	// TODO ?
}

void ClientModule::onReceiveText(Player* p, std::string text)
{
	messages::PlayerText* t = frameMessage.add_receivedtexts();
	t->set_player(p->getID());
	t->set_text(text);
}


/**
* Establishes a connection with the ProxyBot.
*
* throws if the connection fails
*/
NetStream* initSocket() 
{
	// get data from cfg
	using namespace std;
	char path[256];
	_snprintf(path,250,"bwapi-data/AI/cfg%d.txt",Broodwar->getInstanceNumber());
	ifstream fin(path); // cfg.txt should be stored in same
	//  directory as ClientModule.dll under the starcraft parent folder. This is optional

	string host_name;
	int port;
	if (fin.fail()) { // no config file. connect to localhost
		host_name = "127.0.0.1";
		port = PORTNUM + Broodwar->getInstanceNumber();
	} else { // config file. connect to ip/port
		fin >> host_name;
		fin >> port;
		fin.close();
	}

	return new NetStream(host_name, port);
}



void doDynamicTerrain(messages::FrameMessage& msg) {
	messages::DynamicTerrainInfo* data = msg.mutable_dynamicterraininfo();

	google::protobuf::uint32 visible = 0;
	google::protobuf::uint32 explored = 0;
	google::protobuf::uint32 hascreep = 0;

	int index = 0;
	for(int x = 0; x < Broodwar->mapWidth(); ++x) {
		for(int y = 0; y < Broodwar->mapHeight(); ++y) {
			set_bit(visible,index,Broodwar->isVisible(x,y));
			set_bit(explored,index,Broodwar->isExplored(x,y));
			set_bit(hascreep,index,Broodwar->hasCreep(x,y));
			if(++index == 32) {
				index = 0;
				data->add_hascreep(hascreep);
				data->add_isexplored(explored);
				data->add_isvisible(visible);
				visible = explored = hascreep = 0;
			}
		}
	}
	data->add_hascreep(hascreep);
	data->add_isexplored(explored);
	data->add_isvisible(visible);
}



static Position pos2pos(const messages::Position& pos) {
	return Position(pos.x(), pos.y());
}

static TilePosition pos2pos(const messages::TilePosition& pos) {
	return TilePosition(pos.x(), pos.y());
}


static UnitType u2u(const messages::UnitType& type) {
	return UnitType(type.id());
}

static TechType t2t(const messages::TechType& type) {
	return TechType(type.id());
}

static UpgradeType ug2ug(const messages::UpgradeType& type) {
	return UpgradeType(type.id());
}

static BWAPI::CoordinateType::Enum ctype2ctype(const messages::CoordinateType& type) {
	return (CoordinateType::Enum)(type);
}

static Color col2col(const messages::Color& color) {
	if(color.has_id()) return Color(color.id());
	else return Color(color.red(), color.green(), color.blue());
}

void ClientModule::handleCommands(const messages::FrameCommands& commands) {
	for(int i = 0; i < commands.commands_size(); ++i) {
		const messages::Command& c = commands.commands(i);
		Unit* u = unitMap[c.self().id()];
		switch(c.type()) {
		case messages::none:
			break;
		case messages::attackMove: 
			assert(c.has_pos());
			u->attack(pos2pos(c.pos()));
			break;
		case messages::attackUnit:
			assert(c.has_target());
			u->attack(unitMap[c.target().id()]);
			break;
		case messages::rightClick:
			assert(c.has_pos());
			u->rightClick(pos2pos(c.pos()));
			break;
		case messages::rightClickUnit:
			assert(c.has_target());
			u->rightClick(unitMap[c.target().id()]);
			break;
		case messages::train:
			assert(c.has_unittype());
			u->train(u2u(c.unittype()));
			break;
		case messages::build:
			assert(c.has_unittype());
			assert(c.has_tilepos());
			u->build(pos2pos(c.tilepos()),u2u(c.unittype()));
			break;
		case messages::buildAddon:
			assert(c.has_unittype());
			u->buildAddon(u2u(c.unittype()));
			break;
		case messages::research:
			assert(c.has_techtype());
			u->research(t2t(c.techtype()));
			break;
		case messages::upgrade:
			assert(c.has_upgradetype());
			u->upgrade(ug2ug(c.upgradetype()));
			break; 
		case messages::stop:
			u->stop();
			break; 
		case messages::holdPosition:
			u->holdPosition();
			break; 
		case messages::patrol:
			assert(c.has_pos());
			u->patrol(pos2pos(c.pos()));
			break;
		case messages::follow:
			assert(c.has_target());
			u->follow(unitMap[c.target().id()]);
			break; 
		case messages::setRallyPosition:
			assert(c.has_pos());
			u->setRallyPoint(pos2pos(c.pos()));
			break; 
		case messages::setRallyUnit:
			assert(c.has_target());
			Broodwar->printf("Cannot respond to setRallyUnit any more!");
			//u->setRa(unitMap[c.target().id()]);
			break; 
		case messages::repair:
			assert(c.has_target());
			u->repair(unitMap[c.target().id()]);
			break; 
		case messages::morph:
			assert(c.has_unittype());
			u->morph(u2u(c.unittype()));
			break;
		case messages::burrow:
			u->burrow();
			break; 
		case messages::unburrow:
			u->unburrow();
			break; 
		case messages::siege:
			u->siege();
			break; 
		case messages::unsiege:
			u->unsiege();
			break; 
		case messages::cloak:
			u->cloak();
			break; 
		case messages::decloak:
			u->decloak();
			break; 
		case messages::lift:
			u->lift();
			break; 
		case messages::land:
			assert(c.has_tilepos());
			u->land(pos2pos(c.tilepos()));
			break;
		case messages::load:
			assert(c.has_target());
			u->load(unitMap[c.target().id()]);
			break;
		case messages::unload:
			assert(c.has_target());
			u->unload(unitMap[c.target().id()]);
			break;
		case messages::unloadAll:
			u->unloadAll();
			break; 
		case messages::unloadAllPosition:
			assert(c.has_pos());
			u->unloadAll(pos2pos(c.pos()));
			break; 
		case messages::cancelConstruction:
			u->cancelConstruction();
			break; 
		case messages::haltConstruction:
			u->haltConstruction();
			break; 
		case messages::cancelMorph:
			u->cancelMorph();
			break; 
		case messages::cancelTrain:
			u->cancelTrain();
			break; 
		case messages::cancelTrainSlot:
			assert(c.has_slot());
			u->cancelTrain(c.slot());
			break;
		case messages::cancelAddon:
			u->cancelAddon();
			break; 
		case messages::cancelResearch:
			u->cancelResearch();
			break; 
		case messages::cancelUpgrade:
			u->cancelUpgrade();
			break; 
		case messages::useTech:
			assert(c.has_techtype());
			u->useTech(t2t(c.techtype()));
			break; 
		case messages::useTechPosition:
			assert(c.has_techtype());
			assert(c.has_pos());
			u->useTech(t2t(c.techtype()),pos2pos(c.pos()));
			break; 
		case messages::useTechTarget:
			assert(c.has_techtype());
			assert(c.has_target());
			u->useTech(t2t(c.techtype()),unitMap[c.target().id()]);
			break; 		
		}



	}

	BOOST_FOREACH(int flag, commands.flags()) {
		if(flag != BWAPI::Flag::UserInput || hasBotName == false)
			Broodwar->enableFlag(flag);
	}

	BOOST_FOREACH(const std::string& msg, commands.sendtext()) {
		Broodwar->sendText(msg.c_str());
	}

	BOOST_FOREACH(const std::string& msg, commands.printf()) {
		Broodwar->printf(msg.c_str());
	}

	if(commands.has_gamespeed()) { Broodwar->printf("Ignoring gamespeed for tournament."); } //Broodwar->setLocalSpeed(commands.gamespeed());
	if(commands.has_screenposition()) Broodwar->setScreenPosition(commands.screenposition().x(),commands.screenposition().y());


	for(int i = 0; drawOk && i < commands.drawcommands_size(); ++i) {
		try {
			const messages::DrawCommand& c = commands.drawcommands(i);
			CoordinateType::Enum ctype = ctype2ctype(c.coordinatetype());
			switch(c.shape()) {
			case messages::text: {
				assert(c.has_text());
				int x = c.coordinates(0);
				int y = c.coordinates(1);
				Broodwar->drawText(ctype, x, y, c.text().c_str());
				break;
								 } 
			case messages::box: {
				int left = c.coordinates(0);
				int top = c.coordinates(1);
				int right = c.coordinates(2);
				int bottom = c.coordinates(3);
				assert(c.has_color());
				assert(c.has_issolid());
				Color color = col2col(c.color());
				Broodwar->drawBox(ctype, left, top, right, bottom, color, c.issolid());
				break;
								}
			case messages::circle: {
				int x = c.coordinates(0);
				int y = c.coordinates(1);
				int radius = c.coordinates(2);
				assert(c.has_color());
				assert(c.has_issolid());
				Color color = col2col(c.color());
				Broodwar->drawCircle(ctype, x, y, radius, color, c.issolid());
				break;
								   }
			case messages::ellipse: {
				int x = c.coordinates(0);
				int y = c.coordinates(1);
				int xrad = c.coordinates(2);
				int yrad = c.coordinates(3);
				assert(c.has_color());
				assert(c.has_issolid());
				Color color = col2col(c.color());
				Broodwar->drawEllipse(ctype, x, y, xrad, yrad, color, c.issolid());
				break;
									}
			case messages::dot: {
				int x = c.coordinates(0);
				int y = c.coordinates(1);
				assert(c.has_color());
				assert(c.has_issolid());
				Color color = col2col(c.color());
				Broodwar->drawDot(ctype, x, y, color);
				break;
								}
			case messages::line: {
				int x1 = c.coordinates(0);
				int y1 = c.coordinates(1);
				int x2 = c.coordinates(2);
				int y2 = c.coordinates(3);
				assert(c.has_color());
				Color color = col2col(c.color());
				Broodwar->drawLine(ctype, x1, y1, x2, y2, color);
				break;
								 }
			case messages::triangle:
				{
					int ax = c.coordinates(0);
					int ay = c.coordinates(1);
					int bx = c.coordinates(2);
					int by = c.coordinates(3);
					int cx = c.coordinates(4);
					int cy = c.coordinates(5);
					assert(c.has_color());
					assert(c.has_issolid());
					Color color = col2col(c.color());
					Broodwar->drawTriangle(ctype, ax, ay, bx, by, cx, cy, color, c.issolid());
					break;
				}
			}
		} catch(std::exception* e) {
			Broodwar->printf("Exception! %s", e->what());
		} catch(std::exception& e) {
			Broodwar->printf("Exception! %s", e.what());
		} catch(...) {
			Broodwar->printf("Drawing exception!");
		}

	}
}

void exportTypeData(messages::StaticGameData& msg) {
	std::ofstream out("c:\\unit_types.txt");

	// Unit Types:
	BOOST_FOREACH(UnitType& u, UnitTypes::allUnitTypes()) {
		messages::UnitTypeData* mu = msg.add_unittypes();

		mu->set_id(u.getID());
		out << u.getID() << " " << u.getName() << std::endl;
		mu->set_name(u.getName());
		mu->set_sublabel("");
		mu->set_race(race2Race(u.getRace()));

		mu->mutable_whatbuildstype()->set_id(u.whatBuilds().first.getID());
		mu->set_whatbuildsnumber(u.whatBuilds().second);
		typedef const std::pair<UnitType,int> upair;
		BOOST_FOREACH( upair& pair, u.requiredUnits()) {
			messages::UnitTypeIntPair* p = mu->add_requiredunits();
			p->set_count(pair.second);
			p->mutable_type()->set_id(pair.first.getID());
		}
		mu->mutable_requiredtech()->set_id(u.requiredTech().getID());
		BOOST_FOREACH(TechType pair, u.abilities()) {
			messages::TechType* p = mu->add_abilities();
			p->set_id(pair.getID());
		}
		BOOST_FOREACH(UpgradeType pair, u.upgrades()) {
			messages::UpgradeType* p = mu->add_upgrades();
			p->set_id(pair.getID());
		}
		mu->mutable_armorupgrade()->set_id(u.armorUpgrade().getID());

		mu->set_maxhitpoints(u.maxHitPoints());
		mu->set_maxshields(u.maxShields());
		mu->set_maxenergy(u.maxEnergy());
		mu->set_armor(u.armor());

		mu->set_mineralprice(u.mineralPrice());
		mu->set_gasprice(u.gasPrice());
		mu->set_buildtime(u.buildTime());

		mu->set_supplyrequired(u.supplyRequired());
		mu->set_supplyprovided(u.supplyProvided());
		mu->set_spacerequired(u.spaceRequired());
		mu->set_spaceprovided(u.spaceProvided());
		mu->set_buildscore(u.buildScore());
		mu->set_destroyscore(u.destroyScore());

		mu->set_size(size2Size(u.size()));
		mu->set_tilewidth(u.tileWidth());
		mu->set_tileheight(u.tileHeight());

		mu->set_dimensionleft(u.dimensionLeft());
		mu->set_dimensionup(u.dimensionUp());
		mu->set_dimensionright(u.dimensionRight());
		mu->set_dimensiondown(u.dimensionDown());

		mu->set_seekrange(u.seekRange());
		mu->set_sightrange(u.sightRange());
		mu->mutable_groundweapon()->set_id(u.groundWeapon().getID());
		mu->set_maxgroundhits(u.maxGroundHits());
		mu->mutable_airweapon()->set_id(u.airWeapon().getID());
		mu->set_maxairhits(u.maxAirHits());

		mu->set_topspeed(u.topSpeed());
		mu->set_acceleration(u.acceleration());
		mu->set_haltdistance(u.haltDistance());
		mu->set_turnradius(u.turnRadius());

		mu->set_canproduce(u.canProduce());
		mu->set_canattack(u.canAttack());
		mu->set_canmove(u.canMove());
		mu->set_isflyer(u.isFlyer());
		mu->set_regenerateshp(u.regeneratesHP());
		mu->set_isspellcaster(u.isSpellcaster());
		mu->set_haspermanentcloak(u.hasPermanentCloak());
		mu->set_isinvincible(u.isInvincible());
		mu->set_isorganic(u.isOrganic());
		mu->set_ismechanical(u.isMechanical());
		mu->set_isrobotic(u.isRobotic());
		mu->set_isdetector(u.isDetector());
		mu->set_isresourcecontainer(u.isResourceContainer());
		mu->set_isresourcedepot(u.isResourceDepot());
		mu->set_isrefinery(u.isRefinery());
		mu->set_isworker(u.isWorker());
		mu->set_requirespsi(u.requiresPsi());
		mu->set_requirescreep(u.requiresCreep());
		mu->set_istwounitsinoneegg(u.isTwoUnitsInOneEgg());
		mu->set_isburrowable(u.isBurrowable());
		mu->set_iscloakable(u.isCloakable());
		mu->set_isbuilding(u.isBuilding());
		mu->set_isaddon(u.isAddon());
		mu->set_isflyingbuilding(u.isFlyingBuilding());
		mu->set_isneutral(u.isNeutral());
	}

	// Tech Types
	BOOST_FOREACH(TechType& u, TechTypes::allTechTypes()) {
		messages::TechTypeData* mu = msg.add_techs();

		mu->set_id(u.getID());
		mu->set_name(u.getName());
		mu->set_race(race2Race(u.getRace()));

		mu->set_mineralprice(u.mineralPrice());
		mu->set_gasprice(u.gasPrice());
		mu->set_researchtime(u.researchTime());
		mu->set_energyused(u.energyUsed());
		mu->mutable_whatresearches()->set_id(u.whatResearches().getID());
		mu->mutable_weapon()->set_id(u.getWeapon().getID());

		BOOST_FOREACH(UnitType pair, u.whatUses()) {
			messages::UnitType* p = mu->add_whatuses();
			p->set_id(pair.getID());
		}

	}

	// Upgrade Types
	BOOST_FOREACH(UpgradeType& u, UpgradeTypes::allUpgradeTypes()) {
		messages::UpgradeTypeData* mu = msg.add_upgrades();

		mu->set_id(u.getID());
		mu->set_name(u.getName());
		mu->set_race(race2Race(u.getRace()));

		mu->set_mineralpricebase(u.mineralPrice(1));
		mu->set_mineralpricefactor(u.mineralPriceFactor());
		mu->set_gaspricebase(u.gasPrice(1));
		mu->set_gaspricefactor(u.gasPriceFactor());
		mu->set_upgradetimebase(u.upgradeTime(1));
		mu->set_upgradetimefactor(u.upgradeTimeFactor());
		mu->set_maxrepeats(u.maxRepeats());
		mu->mutable_whatupgrades()->set_id(u.whatUpgrades().getID());


		BOOST_FOREACH(UnitType pair, u.whatUses()) {
			messages::UnitType* p = mu->add_whatuses();
			p->set_id(pair.getID());
		}
	}

	// Weapon Types
	BOOST_FOREACH(WeaponType& u, WeaponTypes::allWeaponTypes()) {
		messages::WeaponTypeData* mu = msg.add_weapons();

		mu->set_id(u.getID());
		mu->set_name(u.getName());
		mu->mutable_tech()->set_id(u.getTech().getID());
		mu->mutable_whatuses()->set_id(u.whatUses().getID());
		mu->set_damageamount(u.damageAmount());
		mu->set_damagebonus(u.damageBonus());
		mu->set_damagecooldown(u.damageCooldown());
		mu->set_damagefactor(u.damageFactor());
		mu->mutable_upgradetype()->set_id(u.upgradeType().getID());
		mu->set_damagetype(dam2dam(u.damageType()));
		mu->set_explosiontype(ex2ex(u.explosionType()));
		mu->set_minrange(u.minRange());
		mu->set_maxrange(u.maxRange());
		mu->set_innersplashradius(u.innerSplashRadius());
		mu->set_mediansplashradius(u.medianSplashRadius());
		mu->set_outersplashradius(u.outerSplashRadius());
		mu->set_targetsair(u.targetsAir());
		mu->set_targetsground(u.targetsGround());
		mu->set_targetsmechanical(u.targetsMechanical());
		mu->set_targetsorganic(u.targetsOrganic());
		mu->set_targetsnonbuilding(u.targetsNonBuilding());
		mu->set_targetsnonrobotic(u.targetsNonRobotic());
		mu->set_targetsterrain(u.targetsTerrain());
		mu->set_targetsorgormech(u.targetsOrgOrMech());
		mu->set_targetsown(u.targetsOwn());

	}

	// Weapon Types
	BOOST_FOREACH(Race& u, Races::allRaces()) {
		messages::RaceData* mu = msg.add_races();

		mu->set_id(race2Race(u.getID()));
		mu->set_name(u.getName());
		mu->mutable_worker()->set_id(u.getWorker().getID());
		mu->mutable_center()->set_id(u.getCenter().getID());
		mu->mutable_refinery()->set_id(u.getRefinery().getID());
		mu->mutable_supplyprovider()->set_id(u.getSupplyProvider().getID());
		mu->mutable_transport()->set_id(u.getTransport().getID());
	}


}