#include <algorithm>
#include "InformationManager.h"
#include "BuildManager.h"
#include "SquadAgent.h"
#include "SquadManager.h"
using namespace BWAPI;

InformationManager::InformationManager()
	: mapAnalyzed(false),
	_mineralsReserved(0),
	_gasReserved(0),
	_choke(NULL),
	_minSquadSize(4),
	_proxyPosition(TilePositions::None),
	_useProxyPosition(false),
	_expandType(Natural),
	_turretDefenses(false),
	_scienceVesselDetector(false),
	_autoVehicleUpgrade(false),
	_autoShipUpgrade(false),
	_autoInfanteryUpgrade(false),
	_priorCommandCenters(true),
	_retreatDisabled(false)
{
	//init train order
	_trainOrder[UnitTypes::Terran_Barracks] = UnitTypes::None;
	_trainOrder[UnitTypes::Terran_Factory]	= UnitTypes::None;
	_trainOrder[UnitTypes::Terran_Starport] = UnitTypes::None;

	//init addon order
	_addonOrder[UnitTypes::Terran_Command_Center].clear();
	_addonOrder[UnitTypes::Terran_Factory].clear();
	_addonOrder[UnitTypes::Terran_Starport].clear();
	_addonOrder[UnitTypes::Terran_Science_Facility].clear();

	//init research order
	//_researchOrder[UnitTypes::Terran_Academy];

	//init upgrade order
	//_upgradeOrder[UnitTypes::Terran_Academy];
	_upgradeOrder[UnitTypes::Terran_Engineering_Bay];

	_ourBases.clear();
	_emptyBases.clear();
	_enemyBases.clear();
	_ignoreBases.clear();
	_percentList.clear();
	_autoBuildSuplies = false;

	// Enemy DPS map:
	_last_cycle_dps_map_was_recomputed = -1;
	//_ground_enemy_dps_map = 0;
	//_air_enemy_dps_map = 0;

	int dx = BWAPI::Broodwar->mapWidth();
	int dy = BWAPI::Broodwar->mapHeight();

	_ground_enemy_dps_map = new double[dx*dy];
	_air_enemy_dps_map = new double[dx*dy];

	// clear the map:
	for(int i = 0;i<dx*dy;i++) _ground_enemy_dps_map[i] = _air_enemy_dps_map[i] = 0;

	// init scouting flags
	_lastBaseScan = _emptyBases.end();
	_lastIgnoreBaseScan = _ignoreBases.end();
};

InformationManager::~InformationManager()
{
	if (_ground_enemy_dps_map!=0) delete _ground_enemy_dps_map;
	if (_air_enemy_dps_map!=0) delete _air_enemy_dps_map;
}


void InformationManager::analyzeMap()
{
	if (!mapAnalyzed) {
		//read map information into BWTA so terrain analysis can be done in another thread
		BWTA::readMap();
		CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)AnalyzeMapThread, NULL, 0, NULL);
	}
	
}

DWORD WINAPI AnalyzeMapThread()
{
	BWTA::analyze();

	informationManager->mapAnalyzed = true;

	// --- self start location only available if the map has base locations
	if (BWTA::getStartLocation(Broodwar->self())!=NULL) {
		BWTA::BaseLocation* base = BWTA::getStartLocation(Broodwar->self());
		TilePosition geyserLocation = TilePositions::None;
		if ( !base->isMineralOnly() ) {
			UnitSet::const_iterator geyser = base->getGeysers().begin();
			geyserLocation = (*geyser)->getTilePosition();
		}
		informationManager->_ourBases[base] = geyserLocation;
		informationManager->home = base->getRegion();
	}

	// --- enemy start location
	informationManager->startLocationCouldContainEnemy = BWTA::getStartLocations();
	informationManager->startLocationCouldContainEnemy.erase(BWTA::getStartLocation(Broodwar->self()));
	informationManager->_scoutedAnEnemyBase = false;
	if (informationManager->startLocationCouldContainEnemy.size()==1) {
		informationManager->_enemyBases.insert(*informationManager->startLocationCouldContainEnemy.begin());
		informationManager->_scoutedAnEnemyBase = true;
		BWTA::BaseLocation* test = *informationManager->_enemyBases.begin();
		informationManager->_enemyStartPosition = test->getPosition();
	}

	// --- empty base locations
	informationManager->_emptyBases = BWTA::getBaseLocations();
	informationManager->_emptyBases.erase(BWTA::getStartLocation(Broodwar->self()));
	if (informationManager->startLocationCouldContainEnemy.size()==1) {
		informationManager->_emptyBases.erase(*informationManager->startLocationCouldContainEnemy.begin());
	}

	// --- ignore island bases or special ones (The Fortress)
	for(std::set<BWTA::BaseLocation*>::iterator emptyBase=informationManager->_emptyBases.begin();emptyBase!=informationManager->_emptyBases.end();) {
		if ( (*emptyBase)->isIsland() ) {
			informationManager->_ignoreBases.insert((*emptyBase)->getTilePosition());
			informationManager->_emptyBases.erase(emptyBase++);
		} else if ( Broodwar->mapHash() == "83320e505f35c65324e93510ce2eafbaa71c9aa1" && 
			((*emptyBase)->getTilePosition() == TilePosition(7,7) || (*emptyBase)->getTilePosition() == TilePosition(7,118) ||
			 (*emptyBase)->getTilePosition() == TilePosition(117,118) || (*emptyBase)->getTilePosition() == TilePosition(117,7) ) ) {
				 informationManager->_ignoreBases.insert((*emptyBase)->getTilePosition());
				 informationManager->_emptyBases.erase(emptyBase++);
		} else {
			++emptyBase;
		}
	}

	buildManager->reserveBaseLocations();

	// last sweep scan
	informationManager->_lastBaseScan = informationManager->_emptyBases.begin();

	// --- find best proxy location
	if (informationManager->_scoutedAnEnemyBase) {
		BWTA::Region* enemyStartRegion = BWTA::getRegion(TilePosition(informationManager->_enemyStartPosition));
		std::set<BWTA::Chokepoint*> enemyStartChokepoint = enemyStartRegion->getChokepoints();
		for(std::set<BWTA::Chokepoint*>::iterator c=enemyStartChokepoint.begin();c!=enemyStartChokepoint.end();++c) {
			std::pair<BWTA::Region*,BWTA::Region*> sides = (*c)->getRegions();
			if (sides.first == enemyStartRegion) {
				informationManager->_proxyPosition = TilePosition(sides.second->getCenter());
			} else {
				informationManager->_proxyPosition = TilePosition(sides.first->getCenter());
			}
			if (buildManager->getBuildLocationNear(informationManager->_proxyPosition,UnitTypes::Terran_Barracks) != TilePositions::None)
				break;
		}
	}
	
	// debug
// 	informationManager->_topLeft = Position(informationManager->_proxyPosition);
// 	informationManager->_bottomRight = Position(informationManager->_topLeft.x()+TILE_SIZE, informationManager->_topLeft.y()+TILE_SIZE);

	return 0;
}

void InformationManager::naturalExpandRequest(bool critical)
{
	_expandType = InformationManager::Natural;
	if (critical) {
		criticalBuildRequest(UnitTypes::Terran_Command_Center);
	} else {
		buildRequest(UnitTypes::Terran_Command_Center);
	}
}

void InformationManager::gasExpandRequest(bool critical)
{
	_expandType = InformationManager::Gas;
	if (critical) {
		criticalBuildRequest(UnitTypes::Terran_Command_Center);
	} else {
		buildRequest(UnitTypes::Terran_Command_Center);
	}
}

void InformationManager::onCommandCenterShow(Unit* unit)
{
	if (informationManager->_ourBases.empty()) return; //skip initial instance

	for(std::set<BWTA::BaseLocation*>::iterator emptyBase=_emptyBases.begin();emptyBase!=_emptyBases.end();++emptyBase) {
		TilePosition baseLocation = (*emptyBase)->getTilePosition();
		if (baseLocation == unit->getTilePosition()) {
			TilePosition geyserLocation = TilePositions::None;
			if ( !(*emptyBase)->isMineralOnly() ) {
				UnitSet::const_iterator geyser = (*emptyBase)->getGeysers().begin();
				geyserLocation = (*geyser)->getTilePosition();
				criticalBuildRequest(UnitTypes::Terran_Refinery);
			}
			informationManager->_ourBases[*emptyBase] = geyserLocation;
			informationManager->_basesUnderConstruction[*emptyBase] = unit;
			informationManager->_emptyBases.erase(emptyBase);
			informationManager->_lastBaseScan = informationManager->_emptyBases.begin();
			break;
		}
	}

	// update Proxy position (base closest to the enemy)
	int distance = 9999999;
	int newDistance;
	TilePosition bestLocation;
	for(std::map<BWTA::BaseLocation*, BWAPI::TilePosition>::iterator ourBase=_ourBases.begin();ourBase!=_ourBases.end();++ourBase) {
		newDistance = ourBase->first->getPosition().getApproxDistance(informationManager->_enemyStartPosition);
		if (newDistance < distance) {
			distance = newDistance;
			bestLocation = ourBase->first->getTilePosition();
		}
	}
	_proxyPosition = bestLocation;
	_useProxyPosition = true;
}

void InformationManager::onEnemyResourceDepotShow(Unit* unit)
{
	for(std::set<BWTA::BaseLocation*>::iterator emptyBase=_emptyBases.begin();emptyBase!=_emptyBases.end();++emptyBase) {
		TilePosition baseLocation = (*emptyBase)->getTilePosition();
		if (baseLocation == unit->getTilePosition()) {
			informationManager->_enemyBases.insert(*emptyBase);
			informationManager->_emptyBases.erase(emptyBase);
			informationManager->_lastBaseScan = informationManager->_emptyBases.begin();
			return;
		}
	}
}

void InformationManager::onCommandCenterDestroy(Unit* unit)
{
	for(std::map<BWTA::BaseLocation*, BWAPI::TilePosition>::const_iterator ourBase=_ourBases.begin();ourBase!=_ourBases.end();++ourBase) {
		TilePosition baseLocation = ourBase->first->getTilePosition();
		if (baseLocation == unit->getTilePosition()) {
			// remove minerals spots
			for(UnitSet::const_iterator i=ourBase->first->getMinerals().begin();i!=ourBase->first->getMinerals().end();++i) {
				workerManager->_mineralsExploitation.erase(*i);
			}
			workerManager->reassignRevomedMinerals();
			informationManager->_emptyBases.insert(ourBase->first);
			informationManager->_ourBases.erase(ourBase);
			informationManager->_basesUnderConstruction.erase(ourBase->first);
			informationManager->_lastBaseScan = informationManager->_emptyBases.begin();
			return;
		}
	}
}

void InformationManager::onEnemyResourceDepotDestroy(Unit* unit)
{
	for(std::set<BWTA::BaseLocation*>::iterator enemyBase=_enemyBases.begin();enemyBase!=_enemyBases.end();++enemyBase) {
		TilePosition baseLocation = (*enemyBase)->getTilePosition();
		if (baseLocation == unit->getTilePosition()) {
			informationManager->_emptyBases.insert(*enemyBase);
			informationManager->_enemyBases.erase(enemyBase);
			informationManager->_lastBaseScan = informationManager->_emptyBases.begin();
			return;
		}
	}
}

TilePosition InformationManager::getNaturalExpandPosition(TilePosition ignorePosition)
{
	int distance = 9999999;
	int newDistance;
	TilePosition bestLocation = TilePositions::None;
	for(std::set<BWTA::BaseLocation*>::iterator emptyBase=_emptyBases.begin();emptyBase!=_emptyBases.end();++emptyBase) {
		if ( _ignoreBases.find( (*emptyBase)->getTilePosition() ) != _ignoreBases.end() ) continue;
		//if ((*emptyBase)->getTilePosition() == ignorePosition) continue;
		for(std::map<BWTA::BaseLocation*, BWAPI::TilePosition>::iterator base=_ourBases.begin();base!=_ourBases.end();++base) {
			Position baseLocation = base->first->getPosition();
			if ( (*emptyBase)->getPosition().hasPath(baseLocation) ) {
				newDistance = (*emptyBase)->getPosition().getApproxDistance(baseLocation); // we want a base close to another
				newDistance -= (*emptyBase)->getPosition().getApproxDistance(informationManager->_enemyStartPosition);// but far from the enemy
				if (newDistance < distance) {
					distance = newDistance;
					bestLocation = (*emptyBase)->getTilePosition();
				}
			}
		}
	}

	return bestLocation;
}

TilePosition InformationManager::getGasExpandPosition(TilePosition ignorePosition)
{
	int distance = 9999999;
	int newDistance;
	TilePosition bestLocation = TilePositions::None;
	for(std::set<BWTA::BaseLocation*>::iterator emptyBase=_emptyBases.begin();emptyBase!=_emptyBases.end();++emptyBase) {
		if ( _ignoreBases.find( (*emptyBase)->getTilePosition() ) != _ignoreBases.end() ) continue;
		//if ((*emptyBase)->getTilePosition() == ignorePosition) continue;
		for(std::map<BWTA::BaseLocation*, BWAPI::TilePosition>::iterator base=_ourBases.begin();base!=_ourBases.end();++base) {
			Position baseLocation = base->first->getPosition();
			if ( (*emptyBase)->getPosition().hasPath(baseLocation) && !(*emptyBase)->isMineralOnly() ) {
				newDistance = (*emptyBase)->getPosition().getApproxDistance(baseLocation); // we want a base close to another
				newDistance -= (*emptyBase)->getPosition().getApproxDistance(informationManager->_enemyStartPosition);// but far from the enemy
				if (newDistance < distance) {
					distance = newDistance;
					bestLocation = (*emptyBase)->getTilePosition();
				}
			}
		}
	}

	return bestLocation;
}

void InformationManager::reserveMinerals(int minerals) {
	_mineralsReserved += minerals;
}

void InformationManager::removeReservedMinerals(int minerals) {
	_mineralsReserved -= minerals;
	if (_mineralsReserved < 0) {
		//DEBUG("[ERROR] Removing more minerals than we have: " << _mineralsReserved);
		_mineralsReserved = 0;
	}
}

void InformationManager::reserveGas(int gas) {
	_gasReserved += gas;
}

void InformationManager::removeReservedGas(int gas) {
	_gasReserved -= gas;
	if (_gasReserved < 0) { 
		//DEBUG("[ERROR] Removing more gas than we have: " << _gasReserved);
		_gasReserved = 0;
	}
}

int InformationManager::minerals() {
	return Broodwar->self()->minerals() - _mineralsReserved - _frameMineralSpend;
}

int InformationManager::gas() {
	return Broodwar->self()->gas() - _gasReserved - _frameGasSpend;
}

bool InformationManager::haveResources(UnitType type) {
	return ( minerals() >= type.mineralPrice() && gas() >= type.gasPrice() );
}
bool InformationManager::haveResources(TechType type) {
	return ( minerals() >= type.mineralPrice() && gas() >= type.gasPrice() );
}
bool InformationManager::haveResources(UpgradeType type) {
	return ( minerals() >= type.mineralPrice(Broodwar->self()->getUpgradeLevel(type)+1) && gas() >= type.gasPrice(Broodwar->self()->getUpgradeLevel(type)+1) );
}

void InformationManager::frameSpend(UnitType type) {
	_frameMineralSpend += type.mineralPrice();
	_frameGasSpend += type.gasPrice();
}
void InformationManager::frameSpend(TechType type) {
	_frameMineralSpend += type.mineralPrice();
	_frameGasSpend += type.gasPrice();
}
void InformationManager::frameSpend(UpgradeType type) {
	_frameMineralSpend += type.mineralPrice();
	_frameGasSpend += type.gasPrice();
}

void InformationManager::buildRequest(UnitType type) {
	buildRequest(type, false);
}

void InformationManager::buildRequest(UnitType type, bool checkUnic)
{
	if (checkUnic) {
		if ( Broodwar->self()->completedUnitCount(type) > 0 || // if we already build it
			 std::find(_buildRequest.begin(), _buildRequest.end(), type) != _buildRequest.end() || // if we didn't request before
			 buildManager->alreadyBuilding(type) ) { // if we aren't building it
			return;	 
		}
	}
	
	if (type == UnitTypes::Terran_Comsat_Station) {
		buildRequest(UnitTypes::Terran_Barracks, true);
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Academy, true);
	} else if (type == UnitTypes::Terran_Nuclear_Silo) {
		buildRequest(UnitTypes::Terran_Barracks, true);
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
		buildRequest(UnitTypes::Terran_Science_Facility, true);
		buildRequest(UnitTypes::Terran_Covert_Ops, true);
	}

	_buildRequest.push_back(type);
}

void InformationManager::criticalBuildRequest(UnitType type) {
	criticalBuildRequest(type, false);
}

void InformationManager::criticalBuildRequest(UnitType type, bool first)
{
	if ( std::find(_criticalBuildRequest.begin(), _criticalBuildRequest.end(), type) == _criticalBuildRequest.end() ) { // if we didn't request before
		if (first && Broodwar->self()->completedUnitCount(type) > 0) return; // if we already build it
		if ( Broodwar->self()->incompleteUnitCount(type) == 0 ) // if we aren't building it
			if (type == UnitTypes::Terran_Comsat_Station) {
				//Broodwar->printf("URGENTE COMSAT STATION ################################################");
				criticalBuildRequest(UnitTypes::Terran_Academy, true);
				criticalBuildRequest(UnitTypes::Terran_Refinery, true);
				criticalBuildRequest(UnitTypes::Terran_Barracks, true);
				
			}
			_criticalBuildRequest.push_back(type);
	}
}

void InformationManager::addonRequest(UnitType addonType) {
	_addonOrder[addonType.whatBuilds().first].push_back(addonType);
}

void InformationManager::researchRequest(TechType techType)
{
	if ( std::find(_researchOrder[techType.whatResearches()].begin(), _researchOrder[techType.whatResearches()].end(), techType) == _researchOrder[techType.whatResearches()].end() ) { // if we didn't request before
		if (!Broodwar->self()->hasResearched(techType) && !Broodwar->self()->isResearching(techType)) {
			_researchOrder[techType.whatResearches()].push_back(techType);
			// check dependencies
			if (techType == TechTypes::Stim_Packs || techType == TechTypes::Restoration || techType == TechTypes::Optical_Flare) {
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Academy, true);
			} else if (techType == TechTypes::Spider_Mines || techType == TechTypes::Tank_Siege_Mode) {
				// Machine Shop
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Machine_Shop, true);
			} else if (techType == TechTypes::Cloaking_Field) {
				// Control Tower
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Control_Tower, true);
			} else if (techType == TechTypes::EMP_Shockwave || techType == TechTypes::Irradiate) {
				// Science Facility
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
			} else if (techType == TechTypes::Yamato_Gun) {
				// Physics Lab
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
				buildRequest(UnitTypes::Terran_Physics_Lab, true);
			} else if (techType == TechTypes::Lockdown || techType == TechTypes::Personnel_Cloaking) {
				// CovertOps
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
				buildRequest(UnitTypes::Terran_Covert_Ops, true);
			}
		}
	}
}
void InformationManager::upgradeRequest(UpgradeType upgradeType) {
	upgradeRequest(upgradeType, 1);
}

void InformationManager::upgradeRequest(UpgradeType upgradeType, int level)
{
	if ( std::find(_upgradeOrder[upgradeType.whatUpgrades()].begin(), _upgradeOrder[upgradeType.whatUpgrades()].end(), upgradeType) == _upgradeOrder[upgradeType.whatUpgrades()].end() ) { // if we didn't request before
		int actualLevel = Broodwar->self()->getUpgradeLevel(upgradeType);
		if (Broodwar->self()->isUpgrading(upgradeType)) actualLevel++;
		if (actualLevel < level) {
			if ( upgradeType != UpgradeTypes::Terran_Vehicle_Weapons && upgradeType != UpgradeTypes::Terran_Vehicle_Plating &&
				upgradeType != UpgradeTypes::Terran_Ship_Weapons && upgradeType != UpgradeTypes::Terran_Ship_Plating &&
				upgradeType != UpgradeTypes::Terran_Infantry_Weapons && upgradeType != UpgradeTypes::Terran_Infantry_Armor ) {
				for (int i = actualLevel; i < level; ++i) {
					_upgradeOrder[upgradeType.whatUpgrades()].push_back(upgradeType);
				}
			}
			// check dependencies
			if (upgradeType == UpgradeTypes::U_238_Shells || upgradeType == UpgradeTypes::Caduceus_Reactor) {
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Academy, true);
			} else if (upgradeType == UpgradeTypes::Terran_Infantry_Weapons || upgradeType == UpgradeTypes::Terran_Infantry_Armor) {
				buildRequest(UnitTypes::Terran_Engineering_Bay, true);
				if (level > 1) {
					buildRequest(UnitTypes::Terran_Engineering_Bay, true);
					buildRequest(UnitTypes::Terran_Barracks, true);
					buildRequest(UnitTypes::Terran_Refinery, true);
					buildRequest(UnitTypes::Terran_Factory, true);
					buildRequest(UnitTypes::Terran_Starport, true);
					buildRequest(UnitTypes::Terran_Science_Facility, true);
				}
			} else if (upgradeType == UpgradeTypes::Ion_Thrusters || upgradeType == UpgradeTypes::Charon_Boosters) {
				//Machine Shop
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Machine_Shop, true);
				if (upgradeType == UpgradeTypes::Charon_Boosters)
					buildRequest(UnitTypes::Terran_Armory, true);
			} else if ( upgradeType == UpgradeTypes::Terran_Vehicle_Weapons || upgradeType == UpgradeTypes::Terran_Vehicle_Plating ||
						upgradeType == UpgradeTypes::Terran_Ship_Weapons || upgradeType == UpgradeTypes::Terran_Ship_Plating) {
				//Armory
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Armory, true);
				if (level > 1) {
					buildRequest(UnitTypes::Terran_Starport, true);
					buildRequest(UnitTypes::Terran_Science_Facility, true);
				}
			} else if (upgradeType == UpgradeTypes::Apollo_Reactor) {
				//Control Tower
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Control_Tower, true);
			} else if (upgradeType == UpgradeTypes::Titan_Reactor) {
				//Science Facility
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
			} else if (upgradeType == UpgradeTypes::Colossus_Reactor) {
				//Physics Lab
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
				buildRequest(UnitTypes::Terran_Physics_Lab, true);
			} else if (upgradeType == UpgradeTypes::Ocular_Implants || upgradeType == UpgradeTypes::Moebius_Reactor) {
				//Covert Ops
				buildRequest(UnitTypes::Terran_Barracks, true);
				buildRequest(UnitTypes::Terran_Refinery, true);
				buildRequest(UnitTypes::Terran_Factory, true);
				buildRequest(UnitTypes::Terran_Starport, true);
				buildRequest(UnitTypes::Terran_Science_Facility, true);
				buildRequest(UnitTypes::Terran_Covert_Ops, true);
			}
		}
	}
}

void InformationManager::checkRequirements(UnitType type)
{
	buildRequest(UnitTypes::Terran_Barracks, true);
	if (type == UnitTypes::Terran_Medic || type == UnitTypes::Terran_Firebat) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Academy, true);
	} else if (type == UnitTypes::Terran_Vulture) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
	}  else if (type == UnitTypes::Terran_Goliath) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Armory, true);
	} else if (type == UnitTypes::Terran_Siege_Tank_Tank_Mode) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
	} else if (type == UnitTypes::Terran_Wraith) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
	} else if (type == UnitTypes::Terran_Dropship) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
		//informationManager->buildRequest(UnitTypes::Terran_Control_Tower, true);
	} else if (type == UnitTypes::Terran_Science_Vessel) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
		//informationManager->buildRequest(UnitTypes::Terran_Control_Tower, true);
		buildRequest(UnitTypes::Terran_Science_Facility, true);
	} else if (type == UnitTypes::Terran_Ghost) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Academy, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
		buildRequest(UnitTypes::Terran_Science_Facility, true);
		buildRequest(UnitTypes::Terran_Covert_Ops, true);
	} else if (type == UnitTypes::Terran_Battlecruiser) {
		buildRequest(UnitTypes::Terran_Refinery, true);
		buildRequest(UnitTypes::Terran_Factory, true);
		buildRequest(UnitTypes::Terran_Starport, true);
		//informationManager->buildRequest(UnitTypes::Terran_Control_Tower, true);
		buildRequest(UnitTypes::Terran_Science_Facility, true);
		buildRequest(UnitTypes::Terran_Physics_Lab, true);
	}
}

void InformationManager::seenCloakedEnemy(TilePosition pos)
{
	if ( std::find(_cloakedEnemyPositions.begin(), _cloakedEnemyPositions.end(), pos) == _cloakedEnemyPositions.end() ) { // if we didn't request before
		_cloakedEnemyPositions.insert(pos);
	}
}


double InformationManager::get_enemy_ground_dps(BWAPI::TilePosition pos, int cycle)
{
	int dx = BWAPI::Broodwar->mapWidth();

	if (cycle>_last_cycle_dps_map_was_recomputed) recompute_enemy_dps(cycle);
	return _ground_enemy_dps_map[pos.x() + pos.y()*dx];
}


double InformationManager::get_enemy_air_dps(BWAPI::TilePosition pos, int cycle)
{
	int dx = BWAPI::Broodwar->mapWidth();
	if (cycle>_last_cycle_dps_map_was_recomputed) recompute_enemy_dps(cycle);
	return _air_enemy_dps_map[pos.x() + pos.y()*dx];
}


double InformationManager::get_enemy_ground_dps(int x, int y, int cycle)
{
	int dx = BWAPI::Broodwar->mapWidth();
	//DEBUG("[TEST] Cycle " << cycle << " Last cycle " << _last_cycle_dps_map_was_recomputed);
	if (cycle>_last_cycle_dps_map_was_recomputed) recompute_enemy_dps(cycle);
	return _ground_enemy_dps_map[x + y*dx];
}


double InformationManager::get_enemy_air_dps(int x, int y, int cycle)
{
	int dx = BWAPI::Broodwar->mapWidth();
	if (cycle>_last_cycle_dps_map_was_recomputed) recompute_enemy_dps(cycle);
	return _air_enemy_dps_map[x + y*dx];
}


void InformationManager::recompute_enemy_dps(int cycle)
{
	int dx = BWAPI::Broodwar->mapWidth();
	int dy = BWAPI::Broodwar->mapHeight();

	if (_ground_enemy_dps_map==0) _ground_enemy_dps_map = new double[dx*dy];
	if (_air_enemy_dps_map==0) _air_enemy_dps_map = new double[dx*dy];

	// clear the map:
	for(int i = 0;i<dx*dy;i++) _ground_enemy_dps_map[i] = _air_enemy_dps_map[i] = 0;

	UnitSet allEnemiesSeen;
	for(UnitSet::const_iterator eunit = _seenEnemies.begin();eunit!=_seenEnemies.end();eunit++) allEnemiesSeen.insert(*eunit);
	//DEBUG("seenEnemies size: " << _seenEnemies.size());
	for(SquadSet::const_iterator squadIter=squadManager->_squads.begin();squadIter!=squadManager->_squads.end();++squadIter) {
		//DEBUG("enemies assigned to a squad size: " << (*squadIter)->_enemies.size());
		for(UnitSet::const_iterator eunit = (*squadIter)->_enemies.begin();eunit!=(*squadIter)->_enemies.end();eunit++) allEnemiesSeen.insert(*eunit);
	}
	//DEBUG("allEnemiesSeen size: " << allEnemiesSeen.size());

	for(UnitSet::const_iterator eunit = allEnemiesSeen.begin();eunit!=allEnemiesSeen.end();eunit++) {
		BWAPI::TilePosition pos = (*eunit)->getTilePosition();
		BWAPI::UnitType unit_t = (*eunit)->getType();
		double air_dps = 0;
		double ground_dps = 0;
		if (unit_t == UnitTypes::Terran_Firebat ||
			unit_t == UnitTypes::Protoss_Zealot) {
			ground_dps = unit_t.groundWeapon().damageAmount()*2*(24.0/unit_t.groundWeapon().damageCooldown());
		} else {
			ground_dps = unit_t.groundWeapon().damageAmount()*(24.0/unit_t.groundWeapon().damageCooldown());
		}
		air_dps = unit_t.airWeapon().damageAmount()*(24.0/unit_t.airWeapon().damageCooldown());

		//int gmax_range = unit_t.groundWeapon().maxRange()/32;
		int gmax_range = Broodwar->enemy()->groundWeaponMaxRange(unit_t)/32;
		// simulate potential fields
		if (unit_t == UnitTypes::Protoss_Zealot || unit_t == UnitTypes::Zerg_Zergling) gmax_range = 3;
		int gmin_range = unit_t.groundWeapon().minRange()/32;
		//int amax_range = unit_t.airWeapon().maxRange()/32;
		int amax_range = Broodwar->enemy()->airWeaponMaxRange(unit_t)/32;
		int amin_range = unit_t.airWeapon().minRange()/32;
		if (ground_dps>0)  { 
			gmax_range++;	// To ensure rounding up
			if (unit_t != UnitTypes::Terran_Marine) gmax_range++;
		}
		if (air_dps>0) amax_range += 4;	// To ensure rounding up
		int max_range = (gmax_range>amax_range ? gmax_range:amax_range);
		int min_range = (gmin_range<amin_range ? gmin_range:amin_range);
	
		int gmax_range_sq = gmax_range*gmax_range;
		int gmin_range_sq = gmin_range*gmin_range;
		int amax_range_sq = amax_range*amax_range;
		int amin_range_sq = amin_range*amin_range;

//		DEBUG("Adding " << unit_t.getName() << " " << ground_dps << " with ranges " << gmin_range << "-" << gmax_range << " to ground dps map");

		for(int ix = -max_range;ix<=max_range;ix++) {
			if (pos.x()+ix>=0 && pos.x()+ix<dx) {
				for(int iy = -max_range;iy<=max_range;iy++) {
					if (pos.y()+iy>=0 && pos.y()+iy<dy) {
						int d = ix*ix + iy*iy;
						if (ground_dps>0 && d>=gmin_range_sq && d<=gmax_range_sq) 
							_ground_enemy_dps_map[pos.x()+ix + (pos.y()+iy)*dx] += ground_dps;
						if (air_dps>0 && d>=amin_range_sq && d<=amax_range_sq) 
							_air_enemy_dps_map[pos.x()+ix + (pos.y()+iy)*dx] += air_dps;
					}
				}
			}
		}
	}

/*
	// show it:
	{
		double max_dps = 0;
		for(int i = 0;i<dx*dy;i++) if (_ground_enemy_dps_map[i]>max_dps) max_dps = _ground_enemy_dps_map[i];
		if (max_dps>0) {
			char *line;
			line = new char[dx+1];
			DEBUG("ground dps map, max = " << max_dps);
			for(int y = 0;y<dy;y++) {
				
				for(int x = 0;x<dx;x++) {
					line[x] = '0' + int((_ground_enemy_dps_map[y*dx + x]*9)/max_dps);
				}
				line[dx]=0;
				DEBUG(line);
			}
			delete line;
		}
	}
*/
	_last_cycle_dps_map_was_recomputed = cycle;
}

void InformationManager::drawAirDPSMap()
{
	for(int x=0; x < Broodwar->mapWidth(); ++x) {
		for(int y=0; y < Broodwar->mapHeight(); ++y) {
			if (_air_enemy_dps_map[x + y*BWAPI::Broodwar->mapWidth()] != 0)
				Broodwar->drawTextMap(x*TILE_SIZE+16,y*TILE_SIZE+16,"%0.2f",_air_enemy_dps_map[x + y*BWAPI::Broodwar->mapWidth()]);
		}
	}
}

void InformationManager::drawGroundDPSMap()
{
	for(int x=0; x < Broodwar->mapWidth(); ++x) {
		for(int y=0; y < Broodwar->mapHeight(); ++y) {
			if (_ground_enemy_dps_map[x + y*BWAPI::Broodwar->mapWidth()] != 0)
				Broodwar->drawTextMap(x*TILE_SIZE+16,y*TILE_SIZE+16,"%0.2f",_ground_enemy_dps_map[x + y*BWAPI::Broodwar->mapWidth()]);
		}
	}
}

void InformationManager::scanBases()
{
	if (_cloakedEnemyPositions.empty()) {
		if (!_emptyBases.empty()) {
			if (_lastBaseScan == _emptyBases.end()) {
				_lastBaseScan = _emptyBases.begin();
			}
			for (UnitToTimeMap::iterator comsat = _comsatStation.begin(); comsat != _comsatStation.end(); ++comsat) {
				if ( (comsat->first->getEnergy() >= 100 && !informationManager->_turretDefenses) ||
					 (comsat->first->getEnergy() >= 150 && informationManager->_turretDefenses) ) {
					Position pos = (*_lastBaseScan)->getPosition();
					bool scanned = useScanner(comsat->first, pos);
					if (scanned) {
						_lastBaseScan++;
						if (_lastBaseScan == _emptyBases.end()) {
							_lastBaseScan = _emptyBases.begin();
						}
					}
				}
			}
		} else if (!_ignoreBases.empty()) {
			if (_lastIgnoreBaseScan == _ignoreBases.end()) {
				_lastIgnoreBaseScan = _ignoreBases.begin();
			}
			for (UnitToTimeMap::iterator comsat = _comsatStation.begin(); comsat != _comsatStation.end(); ++comsat) {
				if (comsat->first->getEnergy() >= 100) {
					Position pos = Position(*_lastIgnoreBaseScan);
					bool scanned = useScanner(comsat->first, pos);
					if (scanned) {
						_lastIgnoreBaseScan++;
						if (_lastIgnoreBaseScan == _ignoreBases.end()) {
							_lastIgnoreBaseScan = _ignoreBases.begin();
						}
					}
				}
			}
		}
	}
}

bool InformationManager::useScanner(Unit* comsat, Position pos) 
{
	if (Broodwar->getFrameCount() - _comsatStation[comsat] > SCANNER_SWEEP_FREQUENCY) {
		//Broodwar->printf("Using Scanner Sweep on base (%d,%d)", pos.x(), pos.y());
		comsat->useTech(TechTypes::Scanner_Sweep, pos);
		_comsatStation[comsat] = Broodwar->getFrameCount();
		return true;
	}
	return false;
}

void InformationManager::onMissileTurretShow(Unit* unit)
{
	_missileTurrets[unit] = BWTA::getRegion(unit->getTilePosition());
}

void InformationManager::onMissileTurretDestroy(Unit* unit)
{
	_missileTurrets.erase(unit);
}