#include "CombatAgent.h"
#include "SquadAgent.h"
#include "SquadManager.h"
#include "BuildManager.h"

using namespace BWAPI;

#define NON_ATTACKABLE_UNIT	-10000
#define MICRO_VULTURE_DEGREES 15

#define WEIGHT_AGGRO 10000
#define WEIGHT_DISTANCE 0.1
#define WEIGHT_TACTIC 1

// make sure medics stay with the group:
#define MAX_DISTNACE_TO_NONMEDIC	64
// go close to a medic or backwards when low health:
#define MAX_DISTANCE_TO_MEDIC	64

CombatAgent::CombatAgent(Unit* unit)
	: _unit(unit), 
	_lastTarget(0),
	_inCooldown(true),
	_lastPosition(Positions::None),
	loadedSquad(0),
	doneLoading(false),
	unloadTimer(0)
{

}

void CombatAgent::inCombat(const UnitSet &enemies, SquadAgent *squad)
{
	// special micromanagement for units that do not require target selection:
	if (_unit->getType() == UnitTypes::Terran_Medic) {
		inCombatMedic(enemies,squad);
		return;
	} else if (_unit->getType() == UnitTypes::Terran_Dropship) {
		// Dropships micro is called directly from the squad agent, so that it-s always executed, and not only during combat
		return;
	}

	double bestScore = 0;
	Unit* bestTarget = 0;

	for (UnitSet::const_iterator target = enemies.begin(); target != enemies.end(); ++target) {
		if (_unit->getType().isFlyer() && (*target)->getType() == UnitTypes::Terran_Missile_Turret) continue; //ignore missile turret
		if ((*target)->getType().isFlyer() && _unit->getType().airWeapon().damageAmount() == 0) continue;
		double score = computeTargetScore(*target);
		if (bestTarget==0 || score > bestScore) {
			bestScore = score;
			bestTarget = *target;
		}
	}
	
	if (bestScore == NON_ATTACKABLE_UNIT) { // ignore non attackable units
		if (_unit->isSieged()) unsiegeRequest();
		else _unit->attack(squad->_positionTarget); 
	} else if (bestTarget != _lastTarget && bestTarget != 0) {
		if (_unit->getType() ==  UnitTypes::Terran_Vulture) {
			if (bestTarget->getType() != UnitTypes::Protoss_Photon_Cannon && bestTarget->getType() != UnitTypes::Protoss_Dragoon)	{
				int x = _unit->getTilePosition().x();
				int y = _unit->getTilePosition().y();
				if (informationManager->get_enemy_ground_dps(x, y,BWAPI::Broodwar->getFrameCount())==0)  _unit->attack(bestTarget);
			} else isTankNear(bestTarget,squad);
			//else Do nothing (patrol inCombatVulture()
		} else _unit->attack(bestTarget);
		_lastTarget = bestTarget;

		// Sanitize target
		if (bestTarget->getType() == UnitTypes::Resource_Vespene_Geyser) {
			squad->_enemies.erase(bestTarget);
			squad->_enemiesType.erase(bestTarget);
		}
	}

	if (bestTarget != NULL && bestTarget->isVisible() && !bestTarget->isDetected() ) {
		//Broodwar->printf("Target %s is cloaked/Burrowed", _lastTarget->getType().getName().c_str());
		informationManager->seenCloakedEnemy(bestTarget->getTilePosition());
		// TODO if we have detectors, call it, else inform to informationManager
	}

	// special micromanagement for unit that require target selection:
	if (_unit->getType() == UnitTypes::Terran_Marine) inCombatMarine(bestTarget,enemies,squad);
	if (_unit->getType() == UnitTypes::Terran_Marine || 
		_unit->getType() == UnitTypes::Terran_Ghost ||
		_unit->getType() == UnitTypes::Terran_Firebat) inCombatBiological(bestTarget,enemies,squad);
	if (_unit->getType() == UnitTypes::Terran_Vulture) inCombatVulture(bestTarget,enemies,squad);
	if (_unit->getType() == UnitTypes::Terran_Wraith) inCombatWraith(bestTarget,enemies,squad);
	if (_unit->getType() == UnitTypes::Terran_Siege_Tank_Tank_Mode ||
		_unit->getType() == UnitTypes::Terran_Siege_Tank_Siege_Mode) inCombatTank(bestTarget,enemies,squad);
	if (_unit->getType() == UnitTypes::Terran_Ghost) inCombatGhost(bestTarget,enemies,squad);

//  if (_unit->getGroundWeaponCooldown() > 0 && getEnemiesInRange() > 3 && _unit->isUnderAttack()) { //retreat
// 		_inCooldown = true;
// 		_unit->move(Position(6*TILE_SIZE,36*TILE_SIZE));
// 	} else if (_inCooldown || bestTarget != _lastTarget) { //attack
// 		_inCooldown = false;
// 		if (bestScore > 0) _unit->attack(bestTarget);
// 		_lastTarget = bestTarget;
//  }
}



void CombatAgent::inCombatBiological(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	double health_level = double(_unit->getHitPoints())/double(_unit->getType().maxHitPoints());

	if (!_unit->isAttacking()) {
		// find the nearest medic and the unit with the minimum health level:
		double minimum_hl = 1, hl;
		CombatAgent *minimum_hl_unit = 0;
		int distance = 0, newDistance = 0;
		CombatAgent *closestUnit = 0;
		for(CombatUnitSet::const_iterator i=squad->_squadUnits.begin();i!=squad->_squadUnits.end();++i) {
			if ((*i)->_unit!=0) {
				if ((*i)->_unit->getType() == UnitTypes::Terran_Medic) {
					newDistance = (*i)->_unit->getPosition().getApproxDistance(_unit->getPosition());
					if (closestUnit == 0 || newDistance < distance) {
						distance = newDistance;
						closestUnit = *i;
					}
				}
				hl = double((*i)->_unit->getHitPoints())/double((*i)->_unit->getType().maxHitPoints());
				if (hl<minimum_hl) {
					minimum_hl = hl;
					minimum_hl_unit = *i;
				}
			}
		}

//		if (health_level<0.5 || minimum_hl_unit == this) {
		if (health_level<0.5) {
			if (closestUnit!=0 && distance>MAX_DISTANCE_TO_MEDIC) {
				_unit->move(closestUnit->_unit->getPosition());
			}
		}
	}
}


// Use stimpacks when there are medics around:
void CombatAgent::inCombatMarine(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	if (Broodwar->self()->hasResearched(TechTypes::Stim_Packs) && squad->hasUnitOfType(UnitTypes::Terran_Medic)) {
 		if (bestTarget!=0 && !_unit->isStimmed() && _unit->getHitPoints() > 20 && _unit->isAttacking()) {
 			_unit->useTech(TechTypes::Stim_Packs);
 		}
 	}
}


// Attack when cooldown = 0, otherwise, run to a safe position
void CombatAgent::inCombatVulture(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	Broodwar->drawTextMap(_unit->getPosition().x(), _unit->getPosition().y()-15,"%d", _unit->getSpiderMineCount() );
	if (_unit->getOrder() == Orders::PlaceMine) return;
	if (_unit->getGroundWeaponCooldown()==0 && bestTarget!=0) {
		if ( !(bestTarget->getType() == UnitTypes::Protoss_Zealot && _unit->getPosition().getDistance(bestTarget->getPosition())<120) && // if we are too close, keep running
			!(bestTarget->getType() == UnitTypes::Protoss_Dragoon && _unit->getPosition().getDistance(bestTarget->getPosition())<192 && _unit->getSpiderMineCount() > 0) ) { 
			if (_unit->getSpiderMineCount() > 0 && BWTA::getRegion( _unit->getTilePosition() ) != informationManager->home && 
				bestTarget->getType() == UnitTypes::Protoss_Dragoon && !isSpiderMineNear()) {
				_unit->useTech(TechTypes::Spider_Mines, _unit->getPosition()); return;
			}
			if (_unit->getOrder() != Orders::Patrol && _unit->getOrder() != Orders::AttackUnit) {
				// calculate 15 degree near point
				Position rotated = rotatePosition(MICRO_VULTURE_DEGREES, bestTarget->getPosition(), _unit->getPosition());
				_unit->patrol(rotated);
				_lastPosition = Positions::None;
			}
			return;
		}
	}

	// Safest spot is: minimum dps, furthest from the target enemy, but keeping it in range
	int x = _unit->getTilePosition().x();
	int y = _unit->getTilePosition().y();
	if (informationManager->get_enemy_ground_dps(x, y,BWAPI::Broodwar->getFrameCount())>0) {
		// Find best position in spiral
		Position returnPosition;
		int x      = _unit->getTilePosition().x();
		int y      = _unit->getTilePosition().y();
		int length = 1;
		int j      = 0;
		bool first = true;
		int dx     = 0;
		int dy     = 1;	
		while (length < Broodwar->mapWidth()) {
			returnPosition = Position(x*TILE_SIZE, y*TILE_SIZE);

			if (x >= 0 && x < Broodwar->mapWidth() && y >= 0 && y < Broodwar->mapHeight() && Broodwar->hasPath(_unit->getPosition(),returnPosition) ) {
				double dps = informationManager->get_enemy_ground_dps(x, y,BWAPI::Broodwar->getFrameCount());
				if (dps == 0 && !buildManager->wallNear(TilePosition(x,y))) {
					_unit->move(returnPosition);
					_lastPosition = returnPosition;
					return;
				}
			}

			//otherwise, move to another position
			x = x + dx;
			y = y + dy;
			//count how many steps we take in this direction
			j++;
			if (j == length) { //if we've reached the end, its time to turn
				j = 0;	//reset step counter

				//Spiral out. Keep going.
				if (!first)
					length++; //increment step counter if needed

				first =! first; //first=true for every other turn so we spiral out at the right rate

				//turn counter clockwise 90 degrees:
				if (dx == 0) {
					dx = dy;
					dy = 0;
				} else {
					dy = -dx;
					dx = 0;
				}
			}
			//Spiral out. Keep going.
		}
	}
}

bool CombatAgent::isSpiderMineNear()
{
	UnitSet UnitsInRange = _unit->getUnitsInRadius(50);
	for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
		if ( (*i)->getType() == UnitTypes::Terran_Vulture_Spider_Mine ) {
			return true;
		}
	}
	return false;
}

void CombatAgent::isTankNear(Unit *bestTarget, SquadAgent *squad)
{
	// get closest tank
	CombatAgent* tank = squad->getClosestUnitTo(_unit->getPosition(), UnitTypes::Terran_Siege_Tank_Siege_Mode);
	if (tank == 0) tank = squad->getClosestUnitTo(_unit->getPosition(), UnitTypes::Terran_Siege_Tank_Tank_Mode);
	if (tank == 0) {
		// no tank in squad, keep attacking
		int x = _unit->getTilePosition().x();
		int y = _unit->getTilePosition().y();
		if (informationManager->get_enemy_ground_dps(x, y,BWAPI::Broodwar->getFrameCount())==0)  _unit->attack(bestTarget);
	} else {
		// retread to tank
		//Broodwar->printf("Vulture moving next to tank");
		_unit->move(tank->_unit->getPosition());
	}
}

void CombatAgent::inCombatWraith(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	if (bestTarget == NULL) return;

	if (_unit->getGroundWeaponCooldown()==0 && bestTarget!=0) {
		_unit->attack(bestTarget);
		_lastPosition = Positions::None;
		return;
	}

	// if target has better attack range than us, don't flee
	int unitMaxRange = 0;
	if (bestTarget->getType().isFlyer()) unitMaxRange = bestTarget->getType().airWeapon().maxRange();
	else unitMaxRange = bestTarget->getType().groundWeapon().maxRange();
	if (bestTarget->getType().airWeapon().maxRange() > unitMaxRange) return;

	// Safest spot is: minimum dps, furthest from the target enemy, but keeping it in range
	int x = _unit->getTilePosition().x();
	int y = _unit->getTilePosition().y();
	if (!_unit->isCloaked() && informationManager->get_enemy_air_dps(x, y,BWAPI::Broodwar->getFrameCount())>0) {
		// Find best position in spiral
		Position returnPosition;
		int x      = _unit->getTilePosition().x();
		int y      = _unit->getTilePosition().y();
		int length = 1;
		int j      = 0;
		bool first = true;
		int dx     = 0;
		int dy     = 1;	
		while (length < Broodwar->mapWidth()) {
			returnPosition = Position(x*TILE_SIZE, y*TILE_SIZE);

			if (x >= 0 && x < Broodwar->mapWidth() && y >= 0 && y < Broodwar->mapHeight() ) {
				double dps = informationManager->get_enemy_air_dps(x, y,BWAPI::Broodwar->getFrameCount());
				if (dps == 0) {
					_unit->move(returnPosition);
					_lastPosition = returnPosition;
					return;
				}
			}

			//otherwise, move to another position
			x = x + dx;
			y = y + dy;
			//count how many steps we take in this direction
			j++;
			if (j == length) { //if we've reached the end, its time to turn
				j = 0;	//reset step counter

				//Spiral out. Keep going.
				if (!first)
					length++; //increment step counter if needed

				first =! first; //first=true for every other turn so we spiral out at the right rate

				//turn counter clockwise 90 degrees:
				if (dx == 0) {
					dx = dy;
					dy = 0;
				} else {
					dy = -dx;
					dx = 0;
				}
			}
			//Spiral out. Keep going.
		}
	}

	if (Broodwar->self()->hasResearched(TechTypes::Cloaking_Field)) {
		if (!_unit->isCloaked() && needCloakWraith()) {
			// TODO energy ok?
			_unit->useTech(TechTypes::Cloaking_Field);
		}

		if (_unit->isCloaked() && !needCloakWraith()) {
			_unit->useTech(TechTypes::Cloaking_Field);
		}
	}
}



void CombatAgent::inCombatDropship()
{
	//DEBUG("DropshipMicro start");
	std::set<Unit*> loadedUnits = _unit->getLoadedUnits();

	if (loadedUnits.size() == 0 && loadedSquad == 0) {
//		DEBUG("DropshipMicro empty dropship");
		// state 0: empty dropship
		// Look for a set of units to load: store them in some local variable
		
		// If the dropship is injured, head to a base for repair
		if (_unit->getHitPoints()<_unit->getInitialHitPoints()/2) {
			// Where do dropships go to repair?!?!?!
			// ...
			return;
		}

		double longestDistance = 0;
		SquadAgent *furthestSquad = 0;
		for(SquadSet::const_iterator sq=squadManager->_squads.begin();sq!=squadManager->_squads.end();++sq) {
			// Make sure it has enough non flying units:
			int nonFlyerSize = 0;
			for(CombatUnitSet::const_iterator i=(*sq)->_squadUnits.begin();i!=(*sq)->_squadUnits.end();++i) {
				if (!(*i)->_unit->getType().isFlyer()) {
					if ((*i)->_unit->getType().size() == UnitSizeTypes::Small) nonFlyerSize++;
					if ((*i)->_unit->getType().size() == UnitSizeTypes::Medium) nonFlyerSize+=2;
					if ((*i)->_unit->getType().size() == UnitSizeTypes::Large) nonFlyerSize+=4;
				}
			}
			if (nonFlyerSize>=8) {	// Make sure we can fully load the dropship
				BWAPI::Position target = (*sq)->_positionTarget;
				BWAPI::Position current = (*sq)->_center;
				double distance = target.getDistance(current);
				if (furthestSquad == 0 || longestDistance<distance) {
					furthestSquad = *sq;
					longestDistance = distance;
				}
			}
		}

		if (furthestSquad!=0) {
			if (longestDistance>1000) {	// If units are reasonably far
				//Broodwar->printf("Dropship going for squad at distance %g",longestDistance);
				loadedSquad = furthestSquad;
				doneLoading = false;
			}
		}
		
	} else if (!doneLoading && loadedSquad != 0) {
//		DEBUG("DropshipMicro waiting for load");
		// state 1: half-full dropship
		// check that the squad seill exists:
		bool found = false;
		for(SquadSet::const_iterator sq=squadManager->_squads.begin();sq!=squadManager->_squads.end();++sq) {
			if (*sq == loadedSquad) {
				found = true;
				break;
			}
		}
		
		if (found) {
			int spaceLeft = 8;
			for(UnitSet::const_iterator i=loadedUnits.begin();i!=loadedUnits.end();++i) {
				if ((*i)->getType().size() == UnitSizeTypes::Small) spaceLeft--;
				if ((*i)->getType().size() == UnitSizeTypes::Medium) spaceLeft-=2;
				if ((*i)->getType().size() == UnitSizeTypes::Large) spaceLeft-=4;
			}
			bool sentAnyCommand = false;
			for(CombatUnitSet::const_iterator i=loadedSquad->_squadUnits.begin();i!=loadedSquad->_squadUnits.end();++i) {
				int uSize = 1;
				if ((*i)->_unit->getType().size() == UnitSizeTypes::Medium) uSize = 2;
				if ((*i)->_unit->getType().size() == UnitSizeTypes::Large) uSize = 4;
				if (!(*i)->_unit->getType().isFlyer() && !(*i)->_unit->isLoaded() && spaceLeft>=uSize) {
					spaceLeft-=uSize;
					if ((*i)->_unit->getType() == UnitTypes::Terran_Siege_Tank_Siege_Mode) {
						(*i)->_unit->unsiege();
					} else {
						(*i)->_unit->load(_unit);
					}
					sentAnyCommand = true;
					if (spaceLeft<=0) break;
				}
			}
			if (!sentAnyCommand) {
				doneLoading = true;
				unloadTimer = 0;
			}
		} else {
			//Broodwar->printf("Dropship: Squad being loadded was destroyed!!");
			loadedSquad = 0;
		}

	} else {
//		DEBUG("DropshipMicro full dropship");
		// state 2: full dropship
		// If we are closeby, or injured, unload!
		if (loadedSquad==0) {
			//Broodwar->printf("Error! Squad in Dropship disappeared!");
			_unit->unloadAll();
		} else {
			if (_unit->getPosition().getDistance(loadedSquad->_positionTarget)<1000 || _unit->getHitPoints()<_unit->getInitialHitPoints()/2) {
				// find a position to land:
				if (loadedUnits.size()>0) {
					if (unloadTimer==0) {
						_unit->unloadAll();
						unloadTimer = 300;	// give some time for the unload action to take place...
					} else {
						unloadTimer--;
					}
				} else {
					loadedSquad = 0;
				}
			} else {
	//			Broodwar->printf("Dropship: Full dropship at distance %g",_unit->getPosition().getDistance(loadedSquad->_positionTarget));
				_unit->move(loadedSquad->_positionTarget);
			}
		}
	}
	//DEBUG("DropshipMicro end");
}


void CombatAgent::microScienceVessel(SquadAgent *squad) {
	// ************ check if we are in troubles
	int x = _unit->getTilePosition().x();
	int y = _unit->getTilePosition().y();
	if (informationManager->get_enemy_air_dps(x, y,BWAPI::Broodwar->getFrameCount())>0) {
		// Find best position in spiral
		Position returnPosition;
		int x      = _unit->getTilePosition().x();
		int y      = _unit->getTilePosition().y();
		int length = 1;
		int j      = 0;
		bool first = true;
		int dx     = 0;
		int dy     = 1;	
		while (length < Broodwar->mapWidth()) {
			returnPosition = Position(x*TILE_SIZE, y*TILE_SIZE);

			if (x >= 0 && x < Broodwar->mapWidth() && y >= 0 && y < Broodwar->mapHeight() ) {
				double dps = informationManager->get_enemy_air_dps(x, y,BWAPI::Broodwar->getFrameCount());
				if (dps == 0) {
					_unit->move(returnPosition);
					_lastPosition = returnPosition;
					return;
				}
			}

			//otherwise, move to another position
			x = x + dx;
			y = y + dy;
			//count how many steps we take in this direction
			j++;
			if (j == length) { //if we've reached the end, its time to turn
				j = 0;	//reset step counter

				//Spiral out. Keep going.
				if (!first)
					length++; //increment step counter if needed

				first =! first; //first=true for every other turn so we spiral out at the right rate

				//turn counter clockwise 90 degrees:
				if (dx == 0) {
					dx = dy;
					dy = 0;
				} else {
					dy = -dx;
					dx = 0;
				}
			}
			//Spiral out. Keep going.
		}
	} else {  //  *********** We are in a save location
		CombatAgent* bestUnit = squad->getClosestUnitTo(squad->_positionTarget, UnitTypes::None, true);
		if (bestUnit != 0) {
			Position location = bestUnit->_unit->getPosition(); // get unit closest to target
			Position scoutPosition = getPositionInDirection(location, squad->_positionTarget, 70 ); // distance approx. 2*TILE_SIZE
			TilePosition scoutTilePosition = TilePosition(scoutPosition);
			if (scoutPosition.isValid() && scoutTilePosition.isValid()) {
				if (_lastPosition != scoutPosition && informationManager->get_enemy_air_dps(scoutTilePosition.x(), scoutTilePosition.y(),BWAPI::Broodwar->getFrameCount())==0) {
					_unit->move(scoutPosition);
					_lastPosition = scoutPosition;
				}
			} else {
				//DEBUG("[MICRO ERROR] location not valid");
			}
		} else {
			// we need to find another squad
			if (!informationManager->_retreatDisabled) {
				squadManager->requestRetreat(squad);
			}
		}

	}

	// Debug info
	if (_lastPosition!=Positions::None) {
		Broodwar->drawLineMap(_unit->getPosition().x(), _unit->getPosition().y(), _lastPosition.x(), _lastPosition.y(), Colors::Cyan);
		Broodwar->drawCircleMap(_lastPosition.x(), _lastPosition.y(),3,Colors::Cyan,true);
	}
}

bool CombatAgent::needCloakWraith()
{
	bool antiAirDPS = false;
	bool detector = false;
	UnitSet UnitsInRange = _unit->getUnitsInRadius(Broodwar->self()->sightRange(_unit->getType()));
	for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
		if ( Broodwar->self()->isEnemy((*i)->getPlayer()) && (*i)->getType().airWeapon().damageAmount() != 0 && !(*i)->isLockedDown() ) {
			antiAirDPS = true;
		}
		if ( (*i)->getType().isDetector() ) detector = true;
	}

	if (antiAirDPS && !detector) return true;
	else return false;
}

void CombatAgent::inCombatTank(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	// sanitize _lastTarget
	if ( informationManager->_seenEnemies.find(_lastTarget) != informationManager->_seenEnemies.end() ) {
		_lastTarget = NULL;
	}

	if (Broodwar->self()->hasResearched(TechTypes::Tank_Siege_Mode)) {
		if (!_unit->isSieged() && (_unit->isStartingAttack() || _unit->isUnderAttack()) ) _unit->siege();
		if (!_unit->isSieged() && enemyInSiegeRange()) {
			_unit->siege();
			informationManager->_tankSiege[_unit] = Broodwar->getFrameCount();
		}
		if (_unit->isSieged()) {
			if ( !enemyInSiegeRange(false) || // if no enemy in siege range
				(bestTarget != NULL && _unit->getPosition().getDistance(bestTarget->getPosition()) > 15*TILE_SIZE) ) { // or best target too far
				unsiegeRequest();
			}

		}
	}

	if (!_unit->isSieged() && _unit->getOrder() == Orders::PlayerGuard) {
		if (squad->_positionTarget.isValid()) _unit->attack(squad->_positionTarget);
	}
	
}

bool CombatAgent::enemyInSiegeRange(bool closeBuildings)
{
	if (_lastTarget == NULL) return false;
	if ( informationManager->_seenEnemies.find(_lastTarget) != informationManager->_seenEnemies.end() ) {
		_lastTarget = NULL;
	}
	UnitSet UnitsInRange = _unit->getUnitsInWeaponRange(WeaponTypes::Arclite_Shock_Cannon);
	for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
		if ( Broodwar->self()->isEnemy((*i)->getPlayer()) && !(*i)->getType().isFlyer()) {
			// get closes to units that cannot attack
			if (closeBuildings && !(*i)->getType().canAttack() && (*i)->getType() != UnitTypes::Terran_Bunker) {
				if (_unit->getPosition().getDistance((*i)->getPosition()) > 7*TILE_SIZE) {
					continue;
				}
			}
			return true;
		}
	}

	return false;
}

bool CombatAgent::onlyBuildingEnemies(const UnitSet &enemies)
{
	for (UnitSet::const_iterator it = enemies.begin(); it != enemies.end(); ++it) {
		if ( !(*it)->getType().isBuilding() )
			return false;
	}
	return true;
}

void CombatAgent::unsiegeRequest()
{
	//DEBUG("UnsiegeRequest");
	if (!enemyInSiegeRange(false)) {
		//DEBUG("UnsiegeRequest last siege time " << informationManager->_tankSiege[_unit] << " now it is " << (Broodwar->getFrameCount()));
		if (Broodwar->getFrameCount()-informationManager->_tankSiege[_unit] > 4*24) {
// 			DEBUG("Avoid needWait dance");
// 			UnitSet UnitsInRange = _unit->getUnitsInRadius(300);
// 			for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();) {
// 				if ( !(*i)->getType().canAttack() || (*i)->getType().isWorker() ) {
// 					UnitsInRange.erase(i++);
// 				} else {
// 					++i;
// 				}
// 			}
// 			if (UnitsInRange.size() < informationManager->_minSquadSize) return;
			//DEBUG("UnsiegeRequest accepted");
			_unit->unsiege();
		}
	} //else {
		//DEBUG("UnsiegeRequest rejected, enemies in siege range");
	//}
}

void CombatAgent::inCombatGhost(Unit *bestTarget, const UnitSet &enemies, SquadAgent *squad)
{
	UnitSet UnitsInRange;
	if (Broodwar->self()->hasResearched(TechTypes::Lockdown)) {
		UnitsInRange = _unit->getUnitsInRadius(WeaponTypes::Lockdown.maxRange());
		for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
			if ( Broodwar->self()->isEnemy((*i)->getPlayer()) && (*i)->getType().isMechanical() && !(*i)->isLockedDown() && !allreadyFired(*i)) {
				// TODO energy ok?
				_unit->useTech(TechTypes::Lockdown, *i);
			}
		}
	}

	if (Broodwar->self()->hasResearched(TechTypes::Personnel_Cloaking)) {
		if (!_unit->isCloaked() && needCloak()) {
			// TODO energy ok?
			_unit->useTech(TechTypes::Personnel_Cloaking);
		}

		if (_unit->isCloaked() && !needCloak()) {
			_unit->useTech(TechTypes::Personnel_Cloaking);
		}
	}


}

bool CombatAgent::needCloak()
{
	UnitSet UnitsInRange = _unit->getUnitsInRadius(Broodwar->self()->sightRange(_unit->getType()));
	for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
		if ( Broodwar->self()->isEnemy((*i)->getPlayer()) && (*i)->getType().canAttack() && !(*i)->isLockedDown() ) {
			return true;
		}
	}
	return false;
}

bool CombatAgent::allreadyFired(Unit* enemy)
{
	std::set<Bullet*> bullets = Broodwar->getBullets();
	for(std::set<Bullet*>::const_iterator i=bullets.begin();i!=bullets.end();++i) {
		if ( (*i)->getSource() == _unit && (*i)->getTarget() == enemy) {
			return true;
		}
	}
	return false;
}

void CombatAgent::inCombatMedic(const UnitSet &enemies, SquadAgent *squad)
{
	// First, use optical flare:
	UnitSet UnitsInRange;
	if (Broodwar->self()->hasResearched(TechTypes::Optical_Flare) && _unit->getEnergy()>75) {
		UnitsInRange = _unit->getUnitsInRadius(WeaponTypes::Lockdown.maxRange());
		for(UnitSet::const_iterator i=UnitsInRange.begin();i!=UnitsInRange.end();++i) {
			if ( Broodwar->self()->isEnemy((*i)->getPlayer()) && !(*i)->isBlind() && !allreadyFired(*i)) {
				_unit->useTech(TechTypes::Optical_Flare, *i);
			}
		}
	}


	// compute distance to nearest healable non-medic unit (for now, ignore the fact that medics can heal each other):
	int distance = 0, newDistance = 0;
	CombatAgent *closestUnit = 0;
	for(CombatUnitSet::const_iterator i=squad->_squadUnits.begin();i!=squad->_squadUnits.end();++i) {
		if ((*i)->_unit!=0 &&
			((*i)->_unit->getType() == UnitTypes::Terran_Marine ||
			 (*i)->_unit->getType() == UnitTypes::Terran_Firebat ||
			 (*i)->_unit->getType() == UnitTypes::Terran_Ghost ||
			 (*i)->_unit->getType() == UnitTypes::Terran_SCV)) {
			newDistance = (*i)->_unit->getPosition().getApproxDistance(_unit->getPosition());
			if (closestUnit == 0 || newDistance < distance) {
				distance = newDistance;
				closestUnit = *i;
			}
		}
	}

	// if it-s beyond the threshold, find closest unit, and send there:
	if (closestUnit!=0 && distance>MAX_DISTNACE_TO_NONMEDIC) {
		_unit->move(closestUnit->_unit->getPosition());
	}
}


// DPS of unit to target:
double CombatAgent::dps(BWAPI::Unit *unit, BWAPI::Unit *target)
{
	if (unit==0 || target==0) {
		//DEBUG("[ERROR] No wrong unit in DPS");
		return 0;
	}

	UnitType unit_t = unit->getType();
	UnitType target_t = target->getType();
	double res_dps = 0.0;

	if (target_t.isFlyer()) {
		if (unit_t.airWeapon().damageAmount() != 0) {
//			DEBUG("DAMAGE AIR: " << unit_t.getName() << " " << unit_t.groundWeapon().damageAmount() << " " << unit_t.groundWeapon().damageCooldown()); 
			res_dps = unit_t.airWeapon().damageAmount()*(24.0/unit_t.airWeapon().damageCooldown());
			if (unit_t.airWeapon().damageType() == DamageTypes::Explosive) {
				if (target_t.size() == UnitSizeTypes::Small) res_dps *= 0.5;
				if (target_t.size() == UnitSizeTypes::Medium) res_dps *= 0.75;
			}
			if (unit_t.airWeapon().damageType() == DamageTypes::Concussive) {
				if (target_t.size() == UnitSizeTypes::Large) res_dps *= 0.25;
				if (target_t.size() == UnitSizeTypes::Medium) res_dps *= 0.5;
			}
		}
	} else {
		if (unit_t.groundWeapon().damageAmount() != 0) {
//			DEBUG("DAMAGE GROUND: " << unit_t.getName() << " " << unit_t.groundWeapon().damageAmount() << " " << unit_t.groundWeapon().damageCooldown()); 
			// In the case of Firebats and Zealots, the damage returned by BWAPI is not right, since they have two weapons:
			if (unit_t == UnitTypes::Terran_Firebat ||
				unit_t == UnitTypes::Protoss_Zealot) {
				res_dps = unit_t.groundWeapon().damageAmount()*2*(24.0/unit_t.groundWeapon().damageCooldown());
			} else {
				res_dps = unit_t.groundWeapon().damageAmount()*(24.0/unit_t.groundWeapon().damageCooldown());
			}
			if (unit_t.groundWeapon().damageType() == DamageTypes::Explosive) {
				if (target_t.size() == UnitSizeTypes::Small) res_dps *= 0.5;
				if (target_t.size() == UnitSizeTypes::Medium) res_dps *= 0.75;
			}
			if (unit_t.groundWeapon().damageType() == DamageTypes::Concussive) {
				if (target_t.size() == UnitSizeTypes::Large) res_dps *= 0.25;
				if (target_t.size() == UnitSizeTypes::Medium) res_dps *= 0.5;
			}
		}
	}


/*
	// If it's a transport/bunker/etc, count the DPS of the units inside:
	// This does not work, since we can't see the units inside enemy buners or transports
	std::set<Unit *> loadedUnits = unit->getLoadedUnits();
	if (loadedUnits.size()!=0) {
		for (std::set<Unit *>::const_iterator it = loadedUnits.begin(); it != loadedUnits.end(); ++it) {
			res_dps += dps(*it,target);
		}	
		return res_dps;
	}
*/

	// Bunkers are a special case, and we assume there are 4 marines inside:
	if (unit_t == UnitTypes::Terran_Bunker) {
		res_dps+= 4 * 8*(24.0/15);	// 4 units, 6 damage, 15 cooldown (i.e. 4 marines)
	}

	// Ignore non-threatening workers
	if (target_t.isWorker() && !(target->isRepairing() || target->getOrder()==Orders::AttackUnit || target->isConstructing() )) {
		res_dps = 1.0/WEIGHT_AGGRO;
	}

	return res_dps;
}


double CombatAgent::tacticalThreat(Unit *unit, Unit *target)
{
	if (unit==0 || target==0) return 0;

	UnitType ut = unit->getType();
	UnitType tt = target->getType();

	// Healers
	if (tt == UnitTypes::Terran_Medic) return 100;	
	if (tt == UnitTypes::Terran_SCV) {
		if (target->isRepairing()) return 400;
		else return 250;
	}

	// Workers
	if (tt.isWorker()) {
		return 250;
	}

	// Detectors
	if (tt == UnitTypes::Terran_Science_Vessel) return 100;
	if (tt == UnitTypes::Protoss_Observer) return 100;
	// Overlord has more threat as carrier

	// Carriers of other units:
	if (tt == UnitTypes::Terran_Dropship) return 400;
	if (tt == UnitTypes::Protoss_Shuttle) return 400;
	if (tt == UnitTypes::Zerg_Overlord) return 400;

	if (tt == UnitTypes::Protoss_Pylon) return 50;
	if (tt == UnitTypes::Protoss_Shield_Battery) return 50;
	if (tt == UnitTypes::Zerg_Queen) return 100;

	// Bunkers are taken into account in the DPS:
//	if (tt == UnitTypes::Terran_Bunker) return 400;

	// Buildings that can attack: (they are already taked into account in the DPS)
//	if (ut.isFlyer() && tt == UnitTypes::Terran_Missile_Turret) return 300;
//	if (tt == UnitTypes::Protoss_Photon_Cannon) return 300;
//	if (ut.isFlyer() && tt == UnitTypes::Zerg_Spore_Colony) return 300;
//	if (!ut.isFlyer() && tt == UnitTypes::Zerg_Sunken_Colony) return 300;

	// Buildings that can produce units:
	if (tt == UnitTypes::Terran_Barracks) return 200;
	if (tt == UnitTypes::Terran_Factory) return 200;
	if (tt == UnitTypes::Terran_Starport) return 200;
	if (tt == UnitTypes::Protoss_Gateway) return 200;
	if (tt == UnitTypes::Protoss_Robotics_Facility) return 200;
	if (tt == UnitTypes::Protoss_Stargate) return 200;
	if (tt == UnitTypes::Zerg_Hatchery) return 200;
	if (tt == UnitTypes::Zerg_Lair) return 200;
	if (tt == UnitTypes::Zerg_Hive) return 200;

	// Command centers:
	if (tt.isResourceDepot()) return 100;

	if (tt.isRefinery()) return 75;
	if (tt.canProduce()) return 200;

	// if we are a flyer and target can attack
	if (ut.isFlyer() && tt.canAttack()) return 2500;

	// if we are a goliath and target is a flyer
	if (ut == UnitTypes::Terran_Goliath && tt.isFlyer()) return 2500;

	return 0.0;
}

double CombatAgent::computeTargetScore(Unit* target)
{
	double score = 0;
	UnitType target_t = target->getType();
	UnitType unit_t = _unit->getType();

	// Compute distance only for units that can attack:
	int currentRange = 0;
	//if (!target_t.isBuilding()) currentRange = (int) _unit->getDistance(target->getPosition());
	if (target_t.canAttack()) currentRange = (int) _unit->getDistance(target->getPosition());

	// aggro is proportional to the dps_target_to_unit and inversely proportional to the time it will take to kill the target (hp/dps_unit_to_target)
	double hp = target->getHitPoints() + target->getShields();
	// We don't know the hp of a hidden unit
	//if (target_t == UnitTypes::Zerg_Lurker) hp = target_t.maxHitPoints(); //assume worst case
	if (target->isBurrowed() || target->isCloaked()) hp = target_t.maxHitPoints(); //assume worst case
	double dps_target_to_unit = dps(target,_unit);
	double dps_unit_to_target = dps(_unit,target);

//	Broodwar->drawTextMap(target->getPosition().x(), target->getPosition().y()-15,"%c%s %0.2f", 0x03, target_t.getName().c_str(), dps_unit_to_target );

	if (dps_unit_to_target == 0.0) return NON_ATTACKABLE_UNIT;	// if we cannot damage the unit, do not attack it!

	double time_to_kill = hp/dps_unit_to_target;

	double aggro = (time_to_kill == 0 ? 0 : dps_target_to_unit/time_to_kill);
	double tactical = tacticalThreat(_unit,target);
	score += aggro * WEIGHT_AGGRO + tactical * WEIGHT_TACTIC - currentRange * WEIGHT_DISTANCE;

	if (unit_t == UnitTypes::Terran_Vulture) {
		score -= currentRange * 1000;
		score -= tactical * (WEIGHT_TACTIC/2);
	}
// 	Broodwar->drawTextMap(target->getPosition().x(), target->getPosition().y()-5,"%d", currentRange);
//	Broodwar->drawTextMap(target->getPosition().x(), target->getPosition().y()-15,"%0.2f", score);
//	Broodwar->drawTextMap(target->getPosition().x(), target->getPosition().y()-15,"%s %0.2f", target_t.getName().c_str(), score);
//	DEBUG("Score (" << target_t.getName() << "): " << aggro << " " << tactical << " " << currentRange << "(dpss " << dps_target_to_unit << "," << dps_unit_to_target << ")");

	return score;
}

int CombatAgent::getEnemiesInRange(const UnitSet &enemies)
{
	int enemiesInRange = 0;
	for (UnitSet::const_iterator it = enemies.begin(); it != enemies.end(); ++it)
	{
		Unit* target = *it;
		if (target->isInWeaponRange(_unit))
			++enemiesInRange;
	}
	return enemiesInRange;
}