// Working memory debugger interface

package wm;

import java.util.*;
import java.lang.reflect.*;
import javax.swing.*;
import javax.swing.tree.*; 
import java.awt.event.*;
import java.awt.Dimension;
import java.awt.Component;
import java.awt.Color;

public class WorkingMemoryDebugger extends JPanel
{
    private WorkingMemory wmBeingDebugged = null; // the working memory being debugged
    private JTree wmTree = null;  // Tree display for working memory. 
    private JCheckBox wmUpdateTreeButton = null; 
    private JPopupMenu classNodePopup; // Popup menu for manipulating WME classes
    private JPopupMenu wmeNodePopup; // Popup menu for manipulating WMEs
    private JPopupMenu noNodePopup;  // Popup menu for maniuplating working memory
    private WME selectedWME; // The WME associated with the currently selected WME tree node. 
    private String selectedClass; // The class name associated with the currently selected class tree node.

    private static final int wmeEditButtonContainerHeight = 47;
    private static final int labelAndFieldHeight = 21;
    private static final int minimumEditFieldWidth = 20;
    private static final int labelFieldHSeparation = 20; // number of pixels separating the label and field container

    class PopupMenuActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    String action = e.getActionCommand();
	    if (action.equals("Delete")) {
		wmBeingDebugged.deleteWME(selectedWME);
	    }
	    else if (action.equals("Delete all")) {
		wmBeingDebugged.deleteAllWMEClass(selectedClass);
	    }
	    else if (action.equals("Add...")) {
		// Use the selectedClass to grab an example WME out of working memory.
		WME sampleWME = (WME)wmBeingDebugged.lookupWME(selectedClass).get(0);
		Class tempClass = sampleWME.getClass();
		new AddWMEDialog(tempClass).setVisible(true);
	    }
	    else if (action.equals("Modify...")) {
		new ModifyWMEDialog(selectedWME).setVisible(true);
	    }
	    else if (action.equals("Add WME...")) {
		new ChooseWMEDialog().setVisible(true);
	    }

	    // Update the debugger display if continuous monitoring isn't already doing it.
	    if (!wmUpdateTreeButton.isSelected())
		update();
	}
    }

    protected class WMCellRenderer extends JTextPane implements TreeCellRenderer {
	private final DefaultTreeCellRenderer defaultRenderer = new DefaultTreeCellRenderer();

	public WMCellRenderer()
	{
	    setContentType("text/html");
	}

	// overriden for performance reasons
	// fixme: look at DefaultTreeCellRenderer to see exactly how to override these so that something still gets drawn
	/*
	public void validate() {}
	public void revalidate() {}
	public void repaint(long tm, int x, int y, int width, int height) {}
	public void repaint(Rectangle r) {}
	public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {}
	public void firePropertyChange(String propertyName, byte oldValue, byte newValue) {}
	public void firePropertyChange(String propertyName, char oldValue, char newValue) {}
	public void firePropertyChange(String propertyName, short oldValue, short newValue) {}
	public void firePropertyChange(String propertyName, int oldValue, int newValue) {}
	public void firePropertyChange(String propertyName, long oldValue, long newValue) {}
	public void firePropertyChange(String propertyName, float oldValue, float newValue) {}
	public void firePropertyChange(String propertyName, double oldValue, double newValue) {}
	public void firePropertyChange(String propertyName, boolean oldValue, boolean newValue) {}
	*/

	public Component getTreeCellRendererComponent(JTree tree,
						      Object value,
						      boolean sel,
						      boolean expanded,
						      boolean leaf,
						      int row,
						      boolean hasFocus) 
	{
	    // Determine the icon to use
	    Icon nodeIcon;
	    if (expanded)
		nodeIcon = defaultRenderer.getOpenIcon();
	    else if (leaf) 
		nodeIcon = defaultRenderer.getLeafIcon();
	    else
		nodeIcon = defaultRenderer.getClosedIcon();

	    // Determine the background color to use
	    Color backgroundColor;
	    if (sel)
		backgroundColor = defaultRenderer.getBackgroundSelectionColor();
	    else
		backgroundColor = defaultRenderer.getBackgroundNonSelectionColor();

	    DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
	    if (node.getUserObject() != null) {
		this.setDocument(this.getEditorKit().createDefaultDocument());
		if (node.getUserObject().getClass().getName().equals("java.lang.String")) {
		    this.setText("<font size = \"-1\" face=\"Helvetica, Arial, sans-serif\">" + (String)node.getUserObject());
		}
		else {
		    this.setText(((WME)node.getUserObject()).toString_HTML());
		}
	    }

	    /*
	    // Create the JEditorPane that will be used to draw this node
	    DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
	    JEditorPane renderer;
	    if (node.getUserObject().getClass().getName().equals("java.lang.String"))
		renderer = new JEditorPane("text/html", (String)node.getUserObject());
	    else 
		renderer = new JEditorPane("text/html", ((WME)node.getUserObject()).toString_HTML());
	    */

	    this.setBackground(backgroundColor);

	    // fixme: insert icon doesn't work for text/html documents (get some kind of array index error)
	    // To make this work I'll probably have to dump the icons I want to gif files and embed them directly
	    // in the html. The other approach is to not use the html editor kit and use some alternate way to
	    // encode bolding.
	    // renderer.setCaretPosition(0);
	    // renderer.insertIcon(nodeIcon); 

	    return this;
	}
    }

    // Create a new working memory debugger displaying the current contents of the working memory.
    public WorkingMemoryDebugger(WorkingMemory wm)
    {
	wmBeingDebugged = wm;
	wm.setWMDebugger(this);
    	wmTree = new JTree(wmBeingDebugged.getWMTreeModel());
	wmTree.setCellRenderer(new WMCellRenderer());
	wmTree.setRootVisible(false);
	wmUpdateTreeButton = new JCheckBox("Continuously update working memory");
	wmUpdateTreeButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		if (wmUpdateTreeButton.isSelected())
		    update();
	    }
	});

	JScrollPane WMTreeView = new JScrollPane(wmTree);
	WMTreeView.getViewport().setScrollMode(JViewport.BACKINGSTORE_SCROLL_MODE);
	this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
	this.add(wmUpdateTreeButton);
	this.add(WMTreeView);
	this.setBorder(BorderFactory.createTitledBorder("Working Memory"));

	// Action listener for popup menu commands. 
	ActionListener popupMenuActionListener = new PopupMenuActionListener();

        // Create the class node popup menu.
	JMenuItem menuItem;
        classNodePopup = new JPopupMenu();
        menuItem = new JMenuItem("Add...");
        menuItem.addActionListener(popupMenuActionListener);
        classNodePopup.add(menuItem);
        menuItem = new JMenuItem("Delete all");
        menuItem.addActionListener(popupMenuActionListener);
        classNodePopup.add(menuItem);

	// Create the WME node popup menu.
        wmeNodePopup = new JPopupMenu();
        menuItem = new JMenuItem("Modify...");
        menuItem.addActionListener(popupMenuActionListener);
        wmeNodePopup.add(menuItem);
        menuItem = new JMenuItem("Delete");
        menuItem.addActionListener(popupMenuActionListener);
        wmeNodePopup.add(menuItem);
	
	// Create the no node popup menu
	noNodePopup = new JPopupMenu();
	menuItem = new JMenuItem("Add WME...");
	menuItem.addActionListener(popupMenuActionListener);
	noNodePopup.add(menuItem);

	// Add mouse listener to tree
	
	MouseListener ml = new MouseAdapter() {
	    public void mouseClicked(MouseEvent e) {
		if ((e.getModifiers() & MouseEvent.BUTTON3_MASK) == MouseEvent.BUTTON3_MASK) {
		    TreePath selectedNodePath = wmTree.getPathForLocation(e.getX(), e.getY());
		    if(selectedNodePath != null) {
			// Show a node popup menu
			
			WorkingMemory.WMTreeNode clickedNode = (WorkingMemory.WMTreeNode)selectedNodePath.getLastPathComponent();
			if (clickedNode.getIsClassNode()) {
			    selectedClass = (String)clickedNode.getUserObject();
			    classNodePopup.show(e.getComponent(), e.getX(), e.getY());
			}
			else {
			    selectedWME = (WME)clickedNode.getUserObject();
			    wmeNodePopup.show(e.getComponent(), e.getX(), e.getY());
			}
		    }
		    else 
			// Show the no-node popup menu
			noNodePopup.show(e.getComponent(), e.getX(), e.getY());
		}
	    }
	};

	wmTree.addMouseListener(ml);
    }

    // fixme: consolidate this code with the identical routine found in hap.runtime.Debugger. 
    private LinkedList getAllLeafPaths(TreeNode node, DefaultTreeModel model) {
	LinkedList pathList = new LinkedList();
	int childCount = node.getChildCount();
	if (childCount == 0) {
	    // Leaf node; return a LinkedList containing the single path to this child.
	    pathList.add(new TreePath(model.getPathToRoot(node)));
	    return pathList;
	}
	else {
	    // Not a leaf; return a LinkedList containing the appended
	    // paths of all leafs nodes in the child subtrees
	    for (int i = 0; i < childCount; i++)
		pathList.addAll(getAllLeafPaths(node.getChildAt(i), model));
	    return pathList;
	}
    }

    // Updates the debugger view. 
    public void update()
    {
	DefaultTreeModel wmTreeModel = wmBeingDebugged.getWMTreeModel();
	wmTree.setModel(wmTreeModel);

	// fixme: opening all the paths fucks up the rendering.
	/*
	LinkedList paths = getAllLeafPaths((TreeNode)wmTreeModel.getRoot(), wmTreeModel);
	ListIterator pathIter = paths.listIterator();
	while(pathIter.hasNext())
	    wmTree.makeVisible((TreePath)pathIter.next());
	*/
    }

    // Update the debugger view if wmUpdateTreeButton is selected. 
    public void updateIfMonitoring()
    {
	if (wmUpdateTreeButton.isSelected())
	    update();
    }

    // Clears the debugger view
    public void clearWMView()
    {
	wmTree.setModel(new DefaultTreeModel(new DefaultMutableTreeNode()));
    }

    class WMEActionListener implements ActionListener
    {
	static final int ADD_WME = 0;
	static final int MODIFY_WME = 1;
	
	private int actionToPerform; // the action (add or modify) to perform
	private WME wmeToModify; // the wme to modify (instance)
	private EditWMEDialog dialog; // dialog creating the action listener

	WMEActionListener(EditWMEDialog d, WME wme) 
	{
	    actionToPerform = MODIFY_WME;
	    wmeToModify = wme;
	    dialog = d;
	    
	}
	
	WMEActionListener(EditWMEDialog d, Class wmeClass)
	{
	    actionToPerform = ADD_WME;
	    dialog = d;
	    try {
		Constructor c = (wmeClass.getConstructor((Class[])null));
		wmeToModify = (WME)c.newInstance((Object[])null);
	    } 
	    catch (NoSuchMethodException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (InstantiationException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (IllegalAccessException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (InvocationTargetException exception) { throw new WmeReflectionError(exception.getMessage()); } 
	}

	class IllegalWMEFieldValueException extends Exception 
	{
	    public String fieldValue;
	    public Class fieldType;
	    
	    public IllegalWMEFieldValueException(String fieldValue, Class fieldType)
	    {
		this.fieldValue = fieldValue;
		this.fieldType = fieldType;
	    }
	}

	private void modifyWME()
	{
	    WME newWME;
	    try {
		newWME = (WME)wmeToModify.getClass().getConstructor((Class[])null).newInstance((Object[])null);
		newWME.assign(wmeToModify);
	    } catch (Exception e) { throw new WmeReflectionError(e); }

	    boolean errorOccurred = false; 
	    try {
		for(int i = 0; i < dialog.wmeFieldTypes.length; i++) {
		    // iterate through all the fields testing and setting values
		    String editFieldString = ((WMEFieldEditor)dialog.fieldContainer.getComponent(i)).getFieldValue();
		    String fieldName = ((JLabel)dialog.labelContainer.getComponent(i)).getText();
		    try { 
			Method wmeSetMethod = wmeToModify._getSetMethod(fieldName, dialog.wmeFieldTypes[i]);
			if (dialog.wmeFieldTypes[i] == Boolean.TYPE) {
			    if (editFieldString.equals("true") || editFieldString.equals("false")) {
				Object[] setArgs = { new Boolean(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    else { throw new IllegalWMEFieldValueException(editFieldString, dialog.wmeFieldTypes[i]); }
			}
			try {
			    if (dialog.wmeFieldTypes[i] == Byte.TYPE) {
				Object[] setArgs = { new Byte(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    if (dialog.wmeFieldTypes[i] == Integer.TYPE) {
				Object[] setArgs = { new Integer(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    if (dialog.wmeFieldTypes[i] == Long.TYPE) {
				Object[] setArgs = { new Long(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    if (dialog.wmeFieldTypes[i] == Short.TYPE) {
				Object[] setArgs = { new Short(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    if (dialog.wmeFieldTypes[i] == Float.TYPE) {
				Object[] setArgs = { new Float(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    if (dialog.wmeFieldTypes[i] == Double.TYPE) {
				Object[] setArgs = { new Double(editFieldString) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			} catch (NumberFormatException exception) { throw new IllegalWMEFieldValueException(editFieldString, dialog.wmeFieldTypes[i]); }
			if (dialog.wmeFieldTypes[i] == Character.TYPE) {
			    char[] charArray = editFieldString.toCharArray();
			    if (charArray.length == 1) {
				Object[] setArgs = { new Character(charArray[0]) };
				wmeSetMethod.invoke(newWME, setArgs);
			    }
			    else { throw new IllegalWMEFieldValueException(editFieldString, dialog.wmeFieldTypes[i]); }
			}
			if (((Class)dialog.wmeFieldTypes[i]).getName().equals("java.lang.String")) {
			    Object[] setArgs = { editFieldString };
			    wmeSetMethod.invoke(newWME, setArgs);
			}
		    } catch (NoSuchFieldException e) { } // If _getSetMethod throws a NoSuchFieldException, ignore it. The newly created WME will have a the default value for this field that is assigned by the constructor. 
		}
		if (actionToPerform == MODIFY_WME)
		    wmBeingDebugged.modifyWME(wmeToModify, newWME);
		else if (actionToPerform == ADD_WME)
		    wmBeingDebugged.addWME(newWME);
		dialog.dispose();
	    }
	    catch (InvocationTargetException exception) { throw new WmeReflectionError(exception); }
	    catch (IllegalAccessException exception) { throw new WmeReflectionError(exception); }
	    catch (WorkingMemoryException exception) { throw new WorkingMemoryError(exception.getMessage()); }
	    catch (IllegalWMEFieldValueException exception) { 
		JOptionPane.showMessageDialog(dialog, "Field value " + exception.fieldValue + 
					      " is not a valid " + exception.fieldType);
	    }
	}

	/* private void addWME()
	{
	    try {
		Constructor c = (wmeClass.getConstructor((Class[])null));
		wmeToModify = (WME)c.newInstance((Object[])null);
		modifyWME();
	    } 
	    catch (NoSuchMethodException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (InstantiationException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (IllegalAccessException exception) { throw new WmeReflectionError(exception.getMessage()); }
	    catch (InvocationTargetException exception) { throw new WmeReflectionError(exception.getMessage()); } 
	    } */
	    
	/* public void actionPerformed(ActionEvent e) {
	    if (actionToPerform == MODIFY_WME)
		modifyWME();
	    else if (actionToPerform == ADD_WME)
		addWME();
		} */

	public void actionPerformed(ActionEvent e) 
	{
	    modifyWME();
	}
    }
    
    class DefaultFieldEditor extends JTextField implements WMEFieldEditor {
	public String getFieldValue() { return getText(); }
	public void setFieldValue(String s) { setText(s); }
    }

    class EditWMEDialog extends JDialog
    {
	JButton okButton;
	JButton cancelButton; 
	JPanel contentPane;
	JPanel labelsAndFields;
	JScrollPane labelsAndFieldsView; 
	JPanel labelContainer;
	JPanel fieldContainer;
	JPanel buttonContainer;
	String[] wmeFieldNames; // array of wme field names
	Class[] wmeFieldTypes; // array of wme field types
	final EditWMEDialog me; // Used by enclosed classes

	EditWMEDialog(Class wmeClass)
	{
	    me = this;
	    setModal(true);

	    try {
		Constructor c = wmeClass.getConstructor((Class[])null);
		WME wme = (WME)c.newInstance((Object[])null);
		initializeDialog(wme);
	    }
	    catch (Exception e) { throw new WmeReflectionError(e); }
	}

	EditWMEDialog(WME wme) 
	{
	    me = this;
	    setModal(true);
	    initializeDialog(wme);
	}
	
	protected void initializeDialog(WME wme) 
	{
	    // Set the title
	    setTitle("Modify WME of class " + wme.getClass().getName());
	    
	    // Set the content pane 
	    contentPane = new JPanel();
	    setContentPane(contentPane);
	    contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));
	
	    labelsAndFields = new JPanel();
	    labelsAndFields.setLayout(new BoxLayout(labelsAndFields, BoxLayout.X_AXIS));
	    labelsAndFields.setBorder(BorderFactory.createEtchedBorder());

	    labelsAndFieldsView = new JScrollPane(labelsAndFields);
	    labelsAndFieldsView.getViewport().setScrollMode(JViewport.BACKINGSTORE_SCROLL_MODE);
	    
	    labelContainer = new JPanel();
	    labelContainer.setLayout(new BoxLayout(labelContainer, BoxLayout.Y_AXIS));
	    labelContainer.setAlignmentY(Component.TOP_ALIGNMENT);

	    fieldContainer = new JPanel();
	    fieldContainer.setLayout(new BoxLayout(fieldContainer, BoxLayout.Y_AXIS));
	    fieldContainer.setAlignmentY(Component.TOP_ALIGNMENT);
	    
	    buttonContainer = new JPanel();
	    buttonContainer.setLayout(new BoxLayout(buttonContainer, BoxLayout.X_AXIS));
	    buttonContainer.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

	    // Add the containers
	    getContentPane().add(labelsAndFieldsView);
	    labelsAndFields.add(labelContainer);
	    labelsAndFields.add(Box.createHorizontalStrut(labelFieldHSeparation));
	    labelsAndFields.add(fieldContainer);
	    getContentPane().add(buttonContainer);
	
	    // Create the buttons
	    okButton = new JButton("OK");
	    cancelButton = new JButton("Cancel");

	    // Add the cancel listener
	    cancelButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			me.dispose();
		    }
		});
	    
	    // The OK listener is added by the subclasses

	    // Add the buttons to the dialog
	    buttonContainer.add(Box.createHorizontalGlue());
	    buttonContainer.add(okButton);
	    buttonContainer.add(Box.createHorizontalStrut(10));
	    buttonContainer.add(cancelButton);
	    
	    // Set min, max and preferred size for button container.
	    int buttonContainerWidth; 
	    buttonContainerWidth = buttonContainer.getMinimumSize().width;
	    buttonContainer.setMinimumSize(new Dimension(buttonContainerWidth, wmeEditButtonContainerHeight));
	    buttonContainerWidth = buttonContainer.getMaximumSize().width; 
	    buttonContainer.setMaximumSize(new Dimension(buttonContainerWidth, wmeEditButtonContainerHeight));
	    buttonContainerWidth = buttonContainer.getPreferredSize().width; 
	    buttonContainer.setPreferredSize(new Dimension(buttonContainerWidth, wmeEditButtonContainerHeight)); 
	    
	    // Fill in the label container
	    String labelText;
	    JLabel dialogLabel;
	    JComponent fieldEditor; 
	    int labelContainerHeight = 0;
	    int labelContainerWidth = 0; 
	    int fieldContainerHeight = 0;
	    int fieldContainerWidth = 0;
	    Dimension labelSize;
	    Dimension fieldEditorSize;
	    wmeFieldTypes = wme._getFieldTypes();
	    wmeFieldNames = wme._getFieldNames();
	    for(int i = 0; i < wmeFieldNames.length; i++) {
		dialogLabel = new JLabel(wmeFieldNames[i]);
			
		// Set the min, max and preferred label size
		labelSize = dialogLabel.getPreferredSize();
		labelSize.height = labelAndFieldHeight;
		dialogLabel.setMinimumSize(labelSize);
		dialogLabel.setMaximumSize(labelSize);
		dialogLabel.setPreferredSize(labelSize); 
			
		// Accumulate label container size
		labelContainerHeight += dialogLabel.getPreferredSize().height;
		 if (dialogLabel.getPreferredSize().width > labelContainerWidth)
		     labelContainerWidth = dialogLabel.getPreferredSize().width;
			
		dialogLabel.setAlignmentY(Component.TOP_ALIGNMENT);
		labelContainer.add(dialogLabel);

		// get the editor for the field 
		try {
		    String getEditorMethodName = "_get" + WME.uppercaseFirstCharacter(wmeFieldNames[i]) + "Editor";
		    Method fieldEditorMethod = wme.getClass().getMethod(getEditorMethodName, (Class[])null);
		    fieldEditor = (JComponent)fieldEditorMethod.invoke(wme, (Object[])null);
		} catch (Exception e) {
		    fieldEditor = new DefaultFieldEditor();
		}
		
		// Set the preferred and maximum size of the edit fields
		fieldEditorSize = fieldEditor.getPreferredSize();
		fieldEditorSize.height = labelAndFieldHeight;
		fieldEditor.setPreferredSize(fieldEditorSize); 
		fieldEditorSize = fieldEditor.getMaximumSize();
		fieldEditorSize.height = labelAndFieldHeight;
		fieldEditor.setMaximumSize(fieldEditorSize); 

		
		fieldEditor.setAlignmentY(Component.TOP_ALIGNMENT);
		
		// Accumulate field container size
		fieldContainerHeight += dialogLabel.getPreferredSize().height;
		if (fieldEditor.getPreferredSize().width > fieldContainerWidth)
		    fieldContainerWidth = fieldEditor.getPreferredSize().width;

		fieldContainer.add(fieldEditor);		    
	    }
	    
	    // Set the minimum and preferred sizes of the label and field containers.
	    labelContainer.setPreferredSize(new Dimension(labelContainerWidth, labelContainerHeight));
	    labelContainer.setMinimumSize(new Dimension(labelContainerWidth, labelContainerHeight));
	    fieldContainer.setPreferredSize(new Dimension(fieldContainerWidth, fieldContainerHeight));
	    fieldContainer.setMinimumSize(new Dimension(fieldContainerWidth, fieldContainerHeight));
	
	    // Set the minimum size of the labelsAndFieldsView (scroll pane)
	    final int minimumViewWidth = labelContainer.getMinimumSize().width + 
		fieldContainer.getMinimumSize().width + 
		labelFieldHSeparation; 
	    final int minimumViewHeight = 3 * labelAndFieldHeight;
	    labelsAndFieldsView.setMinimumSize(new Dimension(minimumViewWidth, minimumViewHeight));

	    // fixme: setting this minimum size for the content pane
	    // doesn't seem to restrict the minimum size to which the
	    // dialog can be resized
	    // Set a minimum size for the content pane.  
	    // contentPane.setMinimumSize(new Dimension(contentPane.getMinimumSize().width, 100));

	    pack();
	}	    
    }

    class ModifyWMEDialog extends EditWMEDialog
    {
	ModifyWMEDialog(WME wmeToModify)
	{
	    super(wmeToModify.getClass());
	    okButton.addActionListener(new WMEActionListener(this, wmeToModify));
	    try {
		for(int i = 0; i < wmeFieldTypes.length; i++) {
		    // fixme: recurse to handle non-primitive fields. 
		
		    Method getMethod = wmeToModify._getGetMethod(wmeFieldNames[i]);
		    if (wmeFieldTypes[i] == Boolean.TYPE) {
			Boolean tempBoolean = (Boolean)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempBoolean.toString());
		    }
		    else if (wmeFieldTypes[i] == Byte.TYPE) {
			Byte tempByte = (Byte)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempByte.toString());
		    }
		    else if (wmeFieldTypes[i] == Integer.TYPE) {
			Integer tempInteger = (Integer)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempInteger.toString());
		    }
		    else if (wmeFieldTypes[i] == Long.TYPE) {
			Long tempLong = (Long)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempLong.toString());
		    }
		    else if (wmeFieldTypes[i] == Short.TYPE) {
			Short tempShort = (Short)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempShort.toString());
		    }
		    else if (wmeFieldTypes[i] == Float.TYPE) {
			Float tempFloat = (Float)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempFloat.toString());
		    }
		    else if (wmeFieldTypes[i] == Double.TYPE) {
			Double tempDouble = (Double)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempDouble.toString());
		    }
		    else if (wmeFieldTypes[i] == Character.TYPE) {
			Character tempCharacter = (Character)getMethod.invoke(wmeToModify, (Object[])null);

			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempCharacter.toString());
		    }
		    else if (((Class)wmeFieldTypes[i]).getName().equals("java.lang.String")) {
			String tempString = (String)getMethod.invoke(wmeToModify, (Object[])null);
			((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempString);
		    }
		    else {
			// Can't edit reference types
			Object o = getMethod.invoke(wmeToModify, (Object[])null); 
			if (o != null) { // If the Object field has a value, set the field to the string representation of the value.
			    String tempString = o.toString();
			    ((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue(tempString);
			}
			else { // else set the field to the string "null"
			    ((WMEFieldEditor)fieldContainer.getComponent(i)).setFieldValue("null");
			}
			((JTextField)fieldContainer.getComponent(i)).setEditable(false);
		    }
		}
	    } 
	    catch (IllegalAccessException e) {throw new WmeReflectionError(e); } 
	    catch (InvocationTargetException e) { throw new WmeReflectionError(e); }
	    catch (NoSuchFieldException e) { throw new WmeReflectionError(e); }
	}
    }

    class AddWMEDialog extends EditWMEDialog
    {
	AddWMEDialog(Class wmeClass)
	{
	    super(wmeClass);
	    okButton.addActionListener(new WMEActionListener(this, wmeClass));
	}
    }

    class ChooseWMEDialog extends JDialog
    {
	ChooseWMEDialog me; 

	ChooseWMEDialog()
	{
	    me = this;
	    setModal(true);

	    final JPanel contentPane = new JPanel();
	    contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));
	    setContentPane(contentPane);
	    final JLabel chooseWMELabel = new JLabel("Enter the fully qualified name of a WME");

	    // fixme: Even though I'm specifying an x alignment of left, the label is centered
	    chooseWMELabel.setAlignmentX(Component.LEFT_ALIGNMENT);
	    chooseWMELabel.setHorizontalAlignment(SwingConstants.LEFT);

	    // menu of frequently edited wmes - add to this over time
	    final String[] menuItems = { "facade.characters.wmedef.DAWME", "facade.characters.wmedef.PlayerGestureWME" };

	    final JComboBox wmeNameChooser = new JComboBox(menuItems);
	    wmeNameChooser.setEditable(true);
	    wmeNameChooser.setSelectedItem(""); // initially clear the name chooser
	    // clear enter from the keymap so that it passes to the root pane of the dialog so as to hit the 
	    // default button
	    // fixme: this isn't working - enter is still getting consumed somewhere.
	    // follow up with the JTextField documentation to get this working.
	    ((JTextField)wmeNameChooser.getEditor().getEditorComponent()).getKeymap().removeKeyStrokeBinding(KeyStroke.getKeyStroke("ENTER"));
	    
	    final JPanel buttonContainer = new JPanel();
	    buttonContainer.setLayout(new BoxLayout(buttonContainer, BoxLayout.X_AXIS));
	    buttonContainer.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

	    // Create the buttons
	    final JButton okButton = new JButton("OK");
	    okButton.setDefaultCapable(true); 
	    this.getRootPane().setDefaultButton(okButton);
	    final JButton cancelButton = new JButton("Cancel");

	    // Add the OK listner
	    okButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    try {
			Class requestedWMEClass = Class.forName((String)wmeNameChooser.getSelectedItem());
			Class wmeSuperclass = Class.forName("wm.WME");
			if (wmeSuperclass.isAssignableFrom(requestedWMEClass)) {
			    me.dispose();
			    new AddWMEDialog(requestedWMEClass).setVisible(true);
			}
			else
			    JOptionPane.showMessageDialog(me, "Class " + wmeNameChooser.getSelectedItem() + " is not a WME");
		    } catch (ClassNotFoundException exception) {
			JOptionPane.showMessageDialog(me, "Class file for WME " + wmeNameChooser.getSelectedItem() + " not found");
		    }
		}
	    });

	    // Add the cancel listener
	    cancelButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    me.dispose();
		}
	    });	    

	    buttonContainer.add(okButton);
	    buttonContainer.add(cancelButton);
	    contentPane.add(chooseWMELabel);
	    contentPane.add(wmeNameChooser);
	    contentPane.add(buttonContainer);
	    
	    pack();
	}
    }
}


