// Working memory. 

package wm;

import java.util.*;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import abl.runtime.AblNamedPropertySupport;
import abl.runtime.BehaviorWME;
import abl.runtime.FailStepWME;
import abl.runtime.GoalStepWME;
import abl.runtime.MentalStepWME;
import abl.runtime.PrimitiveStepWME;
import abl.runtime.SucceedStepWME;
import abl.runtime.WaitStepWME;

// fixme: will need these imports when extending WorkingMemory for network support.
// import java.rmi.server.UnicastRemoteObject;
// import java.rmi.RemoteException;
// import java.rmi.Naming;
// import java.net.MalformedURLException;

public class WorkingMemory
{
    protected Hashtable memory = new Hashtable();
    protected WorkingMemoryDebugger workingMemoryDebugger = null;
    private static Hashtable workingMemoryRegistry = new Hashtable();

    // The no-arg constructor doesn't register the working memory - this makes the memory private
    public WorkingMemory() {}

    // Register the working memory for access by other threads
    public WorkingMemory(String name)
    {
        workingMemoryRegistry.put(name, this);
    }

    public static WorkingMemory lookupRegisteredMemory(String name)
    {
        return (WorkingMemory) workingMemoryRegistry.get(name);
    }

    protected class ReflectionWMEEntry
    {
        // List of all reflection WMEs of a given type (e.g. GoalStepWME)
        private final LinkedList wmeList = new LinkedList();

        // Table of lists of reflection WMEs by signature (only GoalStepWME and subclasses of BehaviorWME have this)
        private final Hashtable signatureTable = new Hashtable();

        // Table of lists of reflection WMEs by property
        private final Hashtable propertyTable = new Hashtable();

        public ReflectionWMEEntry() {}

        // dispatcher
        public void addWME(WME w) {
            String name = wmeShortName(w);
            if (name.equals("GoalStepWME"))
                addGoalStepWME((GoalStepWME)w);
            else if (name.equals("WaitStepWME"))
                addWaitStepWME((WaitStepWME)w);
            else if (name.equals("MentalStepWME"))
                addMentalStepWME((MentalStepWME)w);
            else if (name.equals("PrimitiveStepWME"))
                addPrimitiveStepWME((PrimitiveStepWME)w);
            else if (name.equals("FailStepWME"))
                addFailStepWME((FailStepWME)w);
            else if (name.equals("SucceedStepWME"))
                addSucceedStepWME((SucceedStepWME)w);
            else if (name.equals("CollectionBehaviorWME") ||
                    name.equals("ParallelBehaviorWME") ||
                    name.equals("SequentialBehaviorWME"))
                addBehaviorWME((BehaviorWME)w);
        }

        // dispatcher
        public void deleteWME(WME w) {
            String name = wmeShortName(w);
            if (name.equals("GoalStepWME"))
                deleteGoalStepWME((GoalStepWME)w);
            else if (name.equals("WaitStepWME"))
                deleteWaitStepWME((WaitStepWME)w);
            else if (name.equals("MentalStepWME"))
                deleteMentalStepWME((MentalStepWME)w);
            else if (name.equals("PrimitiveStepWME"))
                deletePrimitiveStepWME((PrimitiveStepWME)w);
            else if (name.equals("FailStepWME"))
                deleteFailStepWME((FailStepWME)w);
            else if (name.equals("SucceedStepWME"))
                deleteSucceedStepWME((SucceedStepWME)w);
            else if (name.equals("CollectionBehaviorWME") ||
                    name.equals("ParallelBehaviorWME") ||
                    name.equals("SequentialBehaviorWME"))
                deleteBehaviorWME((BehaviorWME)w);
        }

        public void addGoalStepWME(GoalStepWME w)
        {
            wmeList.add(w);

            indexBySignature(w, w.getSignature());
            indexByUserProperties(w, w.getAllDefinedProperties());

            // fixme: remove
            /*System.out.println("");
           System.out.println("SIGNATURE");
           List l = (List)signatureTable.get("BodyResources_CleanupDemon(int)");
           if (l != null) {
           Iterator i = l.iterator();
           while(i.hasNext())
               System.out.println(i.next());
               }*/
        }

        public void addWaitStepWME(WaitStepWME w)
        {
            wmeList.add(w);
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void addMentalStepWME(MentalStepWME w)
        {
            wmeList.add(w);
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void addPrimitiveStepWME(PrimitiveStepWME w)
        {
            wmeList.add(w);
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void addFailStepWME(FailStepWME w)
        {
            wmeList.add(w);
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void addSucceedStepWME(SucceedStepWME w)
        {
            wmeList.add(w);
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void addBehaviorWME(BehaviorWME w) {
            wmeList.add(w);
            indexBySignature(w, w.getSignature());
            indexByUserProperties(w, w.getAllDefinedProperties());
        }

        public void deleteGoalStepWME(GoalStepWME w)
        {
            wmeList.remove(w);
            deleteFromSignatureTable(w, w.getSignature());
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deleteWaitStepWME(WaitStepWME w)
        {
            wmeList.remove(w);
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deleteMentalStepWME(MentalStepWME w)
        {
            wmeList.remove(w);
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deletePrimitiveStepWME(PrimitiveStepWME w)
        {
            wmeList.remove(w);
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deleteFailStepWME(FailStepWME w)
        {
            wmeList.remove(w);
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deleteSucceedStepWME(SucceedStepWME w)
        {
            wmeList.remove(w);
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        public void deleteBehaviorWME(BehaviorWME w)
        {
            wmeList.remove(w);
            deleteFromSignatureTable(w, w.getSignature());
            deleteFromUserPropertiesTable(w, w.getAllDefinedProperties());
        }

        private void indexBySignature(WME w, String signature)
        {
            List sigList = (List)signatureTable.get(signature);
            if (sigList == null) {
                sigList = new LinkedList();
                sigList.add(w);
                signatureTable.put(signature, sigList);
            }
            else {
                sigList.add(w);
            }
        }

        private void indexByUserProperties(WME w, List propertyList) {
            Iterator iter = propertyList.iterator();
            while(iter.hasNext()) {
                AblNamedPropertySupport.UserProperty property = (AblNamedPropertySupport.UserProperty) iter.next();
                String propertyName = property.getName();
                List indexedPropList = (List)propertyTable.get(propertyName);
                if (indexedPropList == null) {
                    indexedPropList = new LinkedList();
                    indexedPropList.add(w);
                    propertyTable.put(propertyName, indexedPropList);
                }
                else {
                    indexedPropList.add(w);
                }
            }

            // fixme: remove
            /* List tempList = (List)propertyTable.get("resourceOwner");
           if (tempList != null) {
           iter = tempList.iterator();
           System.out.println("$$$$$$$$$$$$$$$$ resourceOwner");
           while(iter.hasNext())
               System.out.println(iter.next());
               } */
        }

        private void deleteFromSignatureTable(WME w, String signature)
        {
            List sigList = (List)signatureTable.get(signature);

            assert (sigList != null && sigList.size() > 0 && sigList.contains(w));

            sigList.remove(w);
            if (sigList.size() == 0)
                signatureTable.remove(signature);
        }

        private void deleteFromUserPropertiesTable(WME w, List propertyList)
        {
            Iterator iter = propertyList.iterator();

            while(iter.hasNext()) {
                AblNamedPropertySupport.UserProperty property = (AblNamedPropertySupport.UserProperty)iter.next();
                String propertyName = property.getName();
                List indexedPropList = (List)propertyTable.get(propertyName);

                assert (indexedPropList != null && indexedPropList.size() > 0 && indexedPropList.contains(w));

                indexedPropList.remove(w);
            }
        }

        public LinkedList lookupWMEBySignature(String signature)
        {
            LinkedList list = (LinkedList)signatureTable.get(signature);
            if (list == null) {
                list = new LinkedList();
            }
            return list;
        }

        public LinkedList lookupWMEByProperty(String propertyName)
        {
            LinkedList list = (LinkedList)propertyTable.get(propertyName);
            if (list == null) {
                list = new LinkedList();
            }
            return list;
        }

        public LinkedList getWMEList() { return wmeList; }
    }

    /* Given a WME, returns the short name (rather than fully
       qualified name) for use in hashing WMEs */
    public static String wmeShortName(WME w) {
        String longName = w.getClass().getName();
        return longName.substring(longName.lastIndexOf('.') + 1);
    }

    /* Given a WME name, returns the short name (rather than fully
       qualified name) for use in hashing WMEs */
    public static String wmeShortName(String wmeName) {
        if (wmeName.indexOf('.') != -1)
            return wmeName.substring(wmeName.lastIndexOf('.') + 1);
        else
            return wmeName;
    }

    /**
     * Returns a list of all the short names for the provided WME, plus all its superclasses.
     * List is guaranteed to be ordered from most specific -> most abstract. Stops before WME.
     *
     * Note: Gillian Smith, added 5-26-10
     */
    public static List<String> wmeShortNames(WME w) {
        List<String> classNameList = new LinkedList<String>();
        Class currentClass = w.getClass();
        String longName = currentClass.getName();
        while(!wmeShortName(longName).equals("WME")) {
            classNameList.add(wmeShortName(longName));
            currentClass = currentClass.getSuperclass();
            longName = currentClass.getName();
        }
        return classNameList;
    }

    /**
     * Returns the number of WMEs for each WME class in memory.
     *
     * Note: Ben Weber, added 1-6-10
     */
    public TreeMap<String, Integer> getMemoryUsage() {
        TreeMap<String, Integer> map = new TreeMap<String, Integer>();

        for (Object keyObject : new ArrayList(memory.keySet())) {
            String key = (String)keyObject;
            Object value = memory.get(key);

            if (value != null) {
                if (value instanceof LinkedList) {
                    map.put(key, ((LinkedList)value).size());
                }
                else if (value instanceof ReflectionWMEEntry) {
                    map.put(key, ((ReflectionWMEEntry)value).wmeList.size());
                }
            }
        }

        return map;
    }

    // Adds WME to working memory.
    public synchronized void addWME(WME w) {
        assert (w != null);

        if (isReflectionWME(w)) {
            addReflectionWME(w);
        }
        else {
            // If the WME is a TimeStampedWME and the timestamp = 0 (was not manually set), set the timestamp to the current time.
            try {
                if ((w instanceof TimeStampedWME) && ((TimeStampedWME)w).getTimestamp() == 0)
                    ((TimeStampedWME)w).setTimestamp(System.currentTimeMillis());
            } catch (Exception e) { throw new WmeReflectionError(e); }

            /*
             * Modified 5-26-10 by Gillian Smith. Instead of hashing WMEs by just their short name, we now hash
             * by their short names + the short names of all their parent classes.
             */

            //String wmeClassName = wmeShortName(w);
            List<String> classNameList = wmeShortNames(w);
            for (String wmeClassName : classNameList) {
                if (!memory.containsKey(wmeClassName)) {
                    /* No wme with this class name has yet been added to
                   working memory. Create a new linked list and add it to
                   working memory. */
                    LinkedList newList = new LinkedList();
                    newList.add(w);
                    memory.put(wmeClassName, newList);
                }
                else {
                    // This wme class has been added to working memory. Add wme to the list.
                    LinkedList oldList = (LinkedList)memory.get(wmeClassName);
                    oldList.add(w);
                }
            }
        }
        if (workingMemoryDebugger != null)
            workingMemoryDebugger.updateIfMonitoring();
    }

    // Atomically add multiple wmes to working memory.
    public synchronized void addWMEs(List wmeList)
    {
        assert (wmeList != null);

        Iterator iter = wmeList.iterator();
        while(iter.hasNext())
            addWME((WME)iter.next());
    }

    // Adds WME to a registered working memory.
    public static void addWME(String memoryName, WME w)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        wm.addWME(w);
    }

    // Atomically add wmes to a registered working memory.
    public static void addWMEs(String memoryName, List wmeList)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        wm.addWMEs(wmeList);
    }

    // Modifies a WME in working memory.
    public synchronized void modifyWME(WME wmeToModify, WME wmeWithNewValues)
            throws IncompatibleWmeClassException, NonexistentWmeException
    {
        assert (!isReflectionWME(wmeShortName(wmeToModify)));

        if (!wmeShortName(wmeToModify).equals(wmeShortName(wmeWithNewValues)))
            throw new IncompatibleWmeClassException();

        String wmeClass = wmeShortName(wmeToModify);

        if (!memory.containsKey(wmeClass))
            throw new NonexistentWmeException();

        LinkedList wmeList = (LinkedList)memory.get(wmeClass);
        if (!wmeList.contains(wmeToModify))
            throw new NonexistentWmeException();

        wmeToModify.assign(wmeWithNewValues);

        if (workingMemoryDebugger != null)
            workingMemoryDebugger.updateIfMonitoring();
    }

    // Modifies a WME is a registered working memory.
    public static void modifyWME(String memoryName, WME wmeToModify, WME wmeWithNewValues)
            throws IncompatibleWmeClassException, NonexistentWmeException
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        wm.modifyWME(wmeToModify, wmeWithNewValues);
    }

    /* Removes a WME from working memory. If the wme is not in working
       memory does nothing (doesn't throw an error). */
    public synchronized void deleteWME(WME wmeToDelete) {
        String wmeClass = wmeShortName(wmeToDelete);
        if (isReflectionWME(wmeClass)) {
            deleteReflectionWME(wmeToDelete);
        }
        else {
            /*
             * Modified 5-26-10 by Gillian Smith. Must delete WMEs not only from their short name's
             * list, but also from their superclasses' lists.
             */
            List<String> classNameList = wmeShortNames(wmeToDelete);
            for (String wmeClassName : classNameList) {
                if (memory.containsKey(wmeClassName)) {
                    // wme class has been added to working memory; continue.
                    LinkedList wmeList = (LinkedList)memory.get(wmeClassName);
                    if (wmeList.contains(wmeToDelete))
                        // The wme is in working memory; continue.
                        wmeList.remove(wmeToDelete);

                    // If there are no WME's of this class left in working memory, remove the key.
                    if (wmeList.size() == 0)
                        memory.remove(wmeClassName);
                }
            }
        }
        if (workingMemoryDebugger != null)
            workingMemoryDebugger.updateIfMonitoring();
    }

    // Deletes a WME from a a registered working memory.
    public static void deleteWME(String memoryName, WME wmeToDelete)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        wm.deleteWME(wmeToDelete);
    }

    /* Deletes all WMEs with a given class name from working
       memory. If the WME class is not found in working memory, does
       nothing. */
    public synchronized void deleteAllWMEClass(String wmeClassName) {
        if (isReflectionWME(wmeClassName)) {
            deleteAllReflectionWMEClass(wmeClassName);
        }
        else {
            /*
             * Modified 6-8-10 by Gillian Smith. Must delete all WMEs from this class not only from their
             * short name's list, but also from their superclasses' lists.
             *
             * Do this by iterating over all the WMEs in this class. Remove them from the list, but also remove
             * them from all their other lists. Optimized so that it finds all the other lists that *aren't* a
             * superclass and removes the entire list at the end instead.
             */
            if (memory.containsKey(wmeClassName)) {
                LinkedList wmeList = (LinkedList)memory.get(wmeClassName);

                ListIterator wmeListIter = wmeList.listIterator();
                HashSet<String> listsToClear = new HashSet<String>();
                while (wmeListIter.hasNext()) {
                    WME currWME = (WME)wmeListIter.next();
                    List<String> classNames = wmeShortNames(currWME);
                    boolean isSuperclass = false;
                    for (String className : classNames) {
                        if (isSuperclass == false && className.equals(wmeClassName))
                            isSuperclass = true;

                        if (!isSuperclass) {
                            listsToClear.add(className);
                        }
                        else {
                            if (!className.equals(wmeClassName) && (memory.containsKey(className))) {
                                ((LinkedList)memory.get(className)).remove(currWME);
                            }
                        }
                    }
                    wmeListIter.remove();
                }
                for (String clearList : listsToClear) {
                    if (memory.containsKey(clearList)) ((LinkedList)memory.get(clearList)).clear();
                }



                /*ListIterator wmeListIter = wmeList.listIterator();
                while (wmeListIter.hasNext()) {
                    wmeListIter.next();
                    wmeListIter.remove();
                }*/

                /*ListIterator wmeListIter = wmeList.listIterator();
                while(wmeListIter.hasNext()) {
                    WME currWME = (WME)wmeListIter.next();
                    List<String> classNames = wmeShortNames(currWME);
                    for (String className : classNames) {
                        if (memory.containsKey(className)) {
                            ((LinkedList)memory.get(className)).remove(currWME);
                        }
                    }
                    //wmeListIter.remove();
                }
                wmeList.clear();*/
            }
        }
        if (workingMemoryDebugger != null)
            workingMemoryDebugger.updateIfMonitoring();
    }

    // Deletetes all WMEs with a given class name from a registered working memory.
    public static void deleteAllWMEClass(String memoryName, String wmeClassName)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        wm.deleteAllWMEClass(wmeClassName);
    }

    protected synchronized void deleteReflectionWME(WME w)
    {
        assert (isReflectionWME(w));

        String wmeClassName = wmeShortName(w);
        ReflectionWMEEntry entry = (ReflectionWMEEntry)memory.get(wmeClassName);

        assert entry != null: wmeClassName;

        entry.deleteWME(w);
        if (entry.getWMEList().size() == 0)
            memory.remove(wmeClassName);
    }

    protected synchronized void deleteAllReflectionWMEClass(String wmeClassName)
    {
        assert (memory.get(wmeClassName) != null);
        memory.remove(wmeClassName);
    }

    // Given a WME class name, returns the list of WMEs in working memory with that class name. 
    public synchronized LinkedList lookupWME(String wmeClassName) {
        LinkedList wmeList;
        if (isReflectionWME(wmeClassName)) {
            ReflectionWMEEntry entry = (ReflectionWMEEntry)memory.get(wmeClassName);
            if (entry == null)
                return new LinkedList();
            else
                wmeList = entry.getWMEList();

            // fixme: remove
            // System.out.println("size == " + wmeList.size());
        }
        else {
            wmeList = (LinkedList)memory.get(wmeClassName);
        }
        if (wmeList == null)
            return new LinkedList(); // No wmes with requested class name in working memory. Return an empty list.
        else
            return (LinkedList)wmeList.clone();
    }

    // Looks up a WME class in a registered working memory
    public static LinkedList lookupWME(String memoryName, String wmeClassName)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        return wm.lookupWME(wmeClassName);
    }

    // ###############################################
    // Optimization methods for Reflection wmes

    // Look up a reflection WME using the signature to optimize
    public synchronized List lookupReflectionWMEBySignature(String wmeClassName, String signature)
    {
        assert (wmeClassName.equals("GoalStepWME") ||
                wmeClassName.equals("SequentialBehaviorWME") ||
                wmeClassName.equals("ParallelBehaviorWME") ||
                wmeClassName.equals("CollectionBehaviorWME"));

        ReflectionWMEEntry entry = (ReflectionWMEEntry)memory.get(wmeClassName);
        if (entry == null) {
            return new LinkedList();
        } else {
            return (LinkedList)entry.lookupWMEBySignature(signature).clone();
        }
    }

    public static List lookupReflectionWMEBySignature(String memoryName, String wmeClassName, String signature)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        return wm.lookupReflectionWMEBySignature(wmeClassName, signature);
    }

    // Look up a reflection wme using a user property to optimize
    public synchronized List lookupReflectionWMEByUserProperty(String wmeClassName, String userPropertyName)
    {
        assert (isReflectionWME(wmeClassName));

        ReflectionWMEEntry entry = (ReflectionWMEEntry)memory.get(wmeClassName);
        if (entry == null) {
            return new LinkedList();
        } else {
            return (LinkedList)entry.lookupWMEByProperty(userPropertyName).clone();
        }
    }

    public static List lookupReflectionWMEByUserProperty(String memoryName, String wmeClassName, String userPropertyName)
    {
        WorkingMemory wm = lookupRegisteredMemory(memoryName);
        return wm.lookupReflectionWMEByUserProperty(wmeClassName, userPropertyName);
    }

    protected synchronized void addReflectionWME(WME w)
    {
        assert (isReflectionWME(w));

        ReflectionWMEEntry entry;
        String wmeClassName = wmeShortName(w);
        if (!memory.containsKey(wmeClassName)) {
            /* No reflection wme with this class name has yet been added to
              working memory. Create a new ReflectionWMEEntry and add the reflection wme to working memory. */
            entry = new ReflectionWMEEntry();
            entry.addWME(w);
            memory.put(wmeClassName, entry);
        }
        else {
            // This wme class has been added to working memory. Add wme to the reflection entry list.
            entry = (ReflectionWMEEntry)memory.get(wmeClassName);
            entry.addWME(w);
        }
    }

    private final static Class goalStepWME = abl.runtime.GoalStepWME.class;
    private final static Class primitiveStepWME = abl.runtime.PrimitiveStepWME.class;
    private final static Class mentalStepWME = abl.runtime.MentalStepWME.class;
    private final static Class waitStepWME = abl.runtime.WaitStepWME.class;
    private final static Class failStepWME = abl.runtime.FailStepWME.class;
    private final static Class succeedStepWME = abl.runtime.SucceedStepWME.class;
    private final static Class collectionBehaviorWME = abl.runtime.CollectionBehaviorWME.class;
    private final static Class parallelBehaviorWME = abl.runtime.ParallelBehaviorWME.class;
    private final static Class sequentialBehaviorWME = abl.runtime.SequentialBehaviorWME.class;
    private final static Class stepWME = abl.runtime.StepWME.class;
    private final static Class behaviorWME = abl.runtime.BehaviorWME.class;

    // Returns true if the argument is the short name of a reflection WME, false otherwise
    public static boolean isReflectionWME(String wmeClassName)
    {
        Class wmeClass;

        if (wmeClassName.equals("GoalStepWME") ||
                wmeClassName.equals("PrimitiveStepWME") ||
                wmeClassName.equals("MentalStepWME") ||
                wmeClassName.equals("WaitStepWME") ||
                wmeClassName.equals("FailStepWME") ||
                wmeClassName.equals("SucceedStepWME") ||
                wmeClassName.equals("CollectionBehaviorWME") ||
                wmeClassName.equals("ParallelBehaviorWME") ||
                wmeClassName.equals("SequentialBehaviorWME")) {
            return true;
        }
        else
            return false;

    }

    // Returns true if the argument is a reflection WME, false otherwise
    public static boolean isReflectionWME(WME wme)
    {
        Class wmeClass = wme.getClass();

        return isReflectionWME(wmeShortName(wmeClass.getName()));
    }

    public static boolean isReflectionWME(Class wmeClass)
    {
        return isReflectionWME(wmeShortName(wmeClass.getName()));
    }

    // ###############################################
    // Episodic memory methods

    // Returns the next wme of class wmeClassName after timestamp.
    public synchronized WME findNext(String wmeClassName, long timestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return null;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            for(int i = 0; i < wmeArray.length; i++) {
                if (wmeArray[i].getTimestamp() > timestamp)
                    return wmeArray[i];
            }
            return null; // no wme found with getTimestamp() > timestamp
        }
    }

    public synchronized WME findPrev(String wmeClassName, long timestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return null;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            for(int i = wmeArray.length - 1; i >= 0; i--) {
                if (wmeArray[i].getTimestamp() < timestamp)
                    return wmeArray[i];
            }
            return null; // no wme found with getTimestamp() < timestamp
        }
    }

    public synchronized WME findFirst(String wmeClassName)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return null;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            return wmeArray[0];
        }
    }

    public synchronized WME findLast(String wmeClassName)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return null;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            return wmeArray[wmeArray.length - 1];
        }
    }

    public synchronized List findAll(String wmeClassName, long beginTimestamp, long endTimestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return new Vector();
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            List returnList = new Vector();
            for(int i = 0; i < wmeArray.length; i++) {
                if ((wmeArray[i].getTimestamp() >= beginTimestamp) &&
                        (wmeArray[i].getTimestamp() <= endTimestamp))
                    returnList.add(wmeArray[i]);
                if (wmeArray[i].getTimestamp() > endTimestamp)
                    // Beyond endTimestamp; no reason to keep looking
                    break;
            }
            return returnList;
        }
    }

    public synchronized int countWMEBefore(String wmeClassName, long timestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return 0;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            int count = 0;
            for(int i = 0; i < wmeArray.length; i++) {
                if (wmeArray[i].getTimestamp() < timestamp) count++;
                if (wmeArray[i].getTimestamp() >= timestamp) break;
            }
            return count; // no wme found with getTimestamp() < timestamp
        }
    }

    public synchronized int countWMEAfter(String wmeClassName, long timestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return 0;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            int count = 0;
            for(int i = wmeArray.length - 1; i >= 0; i--) {
                if (wmeArray[i].getTimestamp() > timestamp) count++;
                if (wmeArray[i].getTimestamp() <= timestamp) break;
            }
            return count; // no wme found with getTimestamp() > timestamp
        }
    }

    public synchronized int countWMEBetween(String wmeClassName, long beginTimestamp, long endTimestamp)
    {
        try {
            assert (Class.forName("wm.TimeStampedWME").isAssignableFrom(Class.forName(wmeClassName)));
        } catch (Exception e) { throw new WmeReflectionError(e); }
        List wmeList = lookupWME(wmeShortName(wmeClassName));
        if (wmeList.size() == 0)
            return 0;
        else {
            TimeStampedWME[] wmeArray = (TimeStampedWME[])wmeList.toArray(new TimeStampedWME[wmeList.size()]);
            Arrays.sort(wmeArray); // fixme: should maintain TimeStampedWMEs in sorted order in wm
            int count = 0;
            for(int i = 0; i < wmeArray.length; i++) {
                if ((wmeArray[i].getTimestamp() >= beginTimestamp) &&
                        (wmeArray[i].getTimestamp() <= endTimestamp))
                    count++;
                if (wmeArray[i].getTimestamp() > endTimestamp)
                    // Beyond endTimestamp; no reason to keep looking
                    break;
            }
            return count;
        }
    }

    class WMTreeNode extends DefaultMutableTreeNode
    {
        private boolean isClassNode = false; // true if the node represents a class rather than a particular wme

        WMTreeNode(Object nodeObject, boolean cn)
        {
            super(nodeObject);
            isClassNode = cn;
        }

        WMTreeNode(String nodeObject)
        {
            super(nodeObject);
        }

        boolean getIsClassNode() { return isClassNode; }
    }

    // Return a DefaultTreeModel containing a representation of working memory. Used by the debugger. 
    public synchronized DefaultTreeModel getWMTreeModel() {
        Enumeration keyEnumerator = memory.keys();
        LinkedList wmeList;
        ListIterator wmeListIter;
        String wmeClassName;
        DefaultTreeModel WMTreeModel = new DefaultTreeModel(new DefaultMutableTreeNode("working memory root"));
        DefaultMutableTreeNode WMRoot = (DefaultMutableTreeNode)WMTreeModel.getRoot();
        WMTreeNode wmeClassNode;
        WMTreeNode wmeNode;
        while(keyEnumerator.hasMoreElements()) {
            wmeClassName = (String)keyEnumerator.nextElement();
            if (isReflectionWME(wmeClassName)) {
                wmeList = ((ReflectionWMEEntry)memory.get(wmeClassName)).getWMEList();
            }
            else {
                wmeList = (LinkedList)memory.get(wmeClassName);
            }
            if (!wmeList.isEmpty()) {
                wmeListIter = wmeList.listIterator();
                wmeClassNode = new WMTreeNode(wmeClassName, true);
                WMRoot.add(wmeClassNode);
                while(wmeListIter.hasNext()) {
                    wmeNode = new WMTreeNode(wmeListIter.next(), false);
                    wmeClassNode.add(wmeNode);
                }
            }
        }
        return WMTreeModel;
    }

    // Returns a working memory debugger interface.
    public WorkingMemoryDebugger getWMDebugInterface()
    {
        return new WorkingMemoryDebugger(this);
    }

    void setWMDebugger(WorkingMemoryDebugger wmd)
    {
        // A debugger for this working memory should not have been previously created.
        assert (workingMemoryDebugger == null);

        workingMemoryDebugger = wmd;
    }

    public synchronized void markTransientWMEs() {
        Enumeration keys = memory.keys();
        while(keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            if (!isReflectionWME(key)) {
                // Reflection WMEs aren't transient
                List wmes = (List)memory.get(key);
                if (!wmes.isEmpty())
                    if (((WME)wmes.get(0)).isTransient()) {
                        // The wmes in the list are transient - mark them
                        Iterator iter = wmes.iterator();
                        while(iter.hasNext())
                            ((TransientWME)iter.next()).mark();
                    }
            }
        }
    }

    public synchronized void deleteMarkedTransientWMEs()
    {
        Enumeration keys = memory.keys();
        while(keys.hasMoreElements()) {
            String key = (String)keys.nextElement();
            if (!isReflectionWME(key)) {
                // Reflection WMEs aren't transient
                List wmes = new Vector((List)memory.get(key)); // make a copy of the list to avoid concurrent mod errors
                if (!wmes.isEmpty())
                    if (((WME)wmes.get(0)).isTransient()) {
                        // The wmes in the list are transient - see if any are marked for deletion
                        Iterator iter = wmes.iterator();
                        while(iter.hasNext()) {
                            TransientWME tWME = (TransientWME)iter.next();
                            if (tWME.getMarked())
                                deleteWME(tWME);
                        }
                    }
            }
        }
    }

}
