package macro;

import java.io.*;

public class SimpleMacro {
    public final int maxArguments = 20;
    private final static char macroAppendChar = '@';
    private boolean macroDefined = false;
    private String[] arguments = new String[maxArguments];
    private int numberOfArguments = 0;
    private String macroName;
    private String macroBody;

    private final static boolean slashSlashComments = true;
    private final static boolean slashStarComments = true; 

    private final static char singleLineCommentChar = '#';

    public SimpleMacro(String macroDefnString) {
	StringReader stringToRead = new StringReader(macroDefnString);
	
	try {
	    StreamTokenizer macroDefnTokens = new StreamTokenizer(stringToRead);
	    macroDefnTokens.slashSlashComments(slashSlashComments);
	    macroDefnTokens.slashStarComments(slashStarComments);
	    macroDefnTokens.eolIsSignificant(true);
	    macroDefnTokens.nextToken();
	    if (!macroDefnTokens.sval.equals("definemacro")) {
		throw new SimpleMacroError("Macro " + macroDefnString + " doesn't start with 'definemacro'");
	    }
	    
	    macroDefnTokens.nextToken();
	    macroName = macroDefnTokens.sval;
	    
	    macroDefnTokens.nextToken();
	    if (macroDefnTokens.ttype != '(') {
		throw new SimpleMacroError("Error finding beginning of argument list in " + macroDefnString);
	    }
	    
	    int previousTokenType = '(';
	    for(macroDefnTokens.nextToken();
		macroDefnTokens.ttype != ')';
		macroDefnTokens.nextToken()) {
		
		if (macroDefnTokens.ttype != ',' &&
		    macroDefnTokens.ttype == StreamTokenizer.TT_WORD && 
		    ( previousTokenType == ',' || previousTokenType == '(')) {
		    
		    arguments[numberOfArguments++] = macroDefnTokens.sval;
		    previousTokenType = StreamTokenizer.TT_WORD;
		}
		else if (macroDefnTokens.ttype == ',' &&
			 previousTokenType == StreamTokenizer.TT_WORD) {
		    previousTokenType = ',';
		}
		else if (macroDefnTokens.ttype != ')') {
		    throw new SimpleMacroError("Error parsing the argument list of macro " + macroDefnString);
		}
	    }
	    
	    StringBuffer macroRead = new StringBuffer();
	    
	    for(int charRead = stringToRead.read(); charRead != -1; charRead = stringToRead.read()) {
		macroRead.append((char)charRead);
	    }
	    macroBody = macroRead.toString();
	    macroDefined = true;
	} catch (IOException e) {
	    throw new SimpleMacroError("IO error occurred while reading macro " 
				       + macroDefnString + ". Error: " + e.getMessage());
	}
    }

    public SimpleMacro(File macroDefnFile) {
	FileReader fileToRead = null;

	try {
	    fileToRead = new FileReader(macroDefnFile);
	    StreamTokenizer macroDefnTokens = new StreamTokenizer(fileToRead);
	    macroDefnTokens.slashSlashComments(slashSlashComments);
	    macroDefnTokens.slashStarComments(slashStarComments);
	    macroDefnTokens.eolIsSignificant(true);
	    macroDefnTokens.nextToken();
	    if (!macroDefnTokens.sval.equals("definemacro")) {
		throw new SimpleMacroError("Macro definition file " + macroDefnFile.getAbsolutePath() + 
					       " doesn't start with 'definemacro'");
	    }
	    
	    macroDefnTokens.nextToken();
	    macroName = macroDefnTokens.sval;
	    
	    macroDefnTokens.nextToken();
	    if (macroDefnTokens.ttype != '(') {
		throw new SimpleMacroError("Error finding beginning of argument list in " + macroDefnFile.getAbsolutePath());
	    }
	    
	    int previousTokenType = '(';
	    for(macroDefnTokens.nextToken();
		macroDefnTokens.ttype != ')';
		macroDefnTokens.nextToken()) {
		
		if (macroDefnTokens.ttype != ',' &&
		    macroDefnTokens.ttype == StreamTokenizer.TT_WORD && 
		    ( previousTokenType == ',' || previousTokenType == '(')) {
		    
		    arguments[numberOfArguments++] = macroDefnTokens.sval;
		    previousTokenType = StreamTokenizer.TT_WORD;
		}
		else if (macroDefnTokens.ttype == ',' &&
			 previousTokenType == StreamTokenizer.TT_WORD) {
		    previousTokenType = ',';
		}
		else if (macroDefnTokens.ttype != ')') {
		    throw new SimpleMacroError("Error parsing the argument list of macro definition file " + 
					       macroDefnFile.getAbsolutePath());
		}
	    }

	    StringBuffer macroRead = new StringBuffer();

	    /* Throw away intial newline characters. Doesn't work
               because parsing of EOL characters doesn't seem to work
               properly on PC platforms. Seems like it is still
               leaving the second character of the EOL token in the
               stream. Ignore this for now. */
	    /* macroDefnTokens.nextToken();
	       if (!(macroDefnTokens.ttype == StreamTokenizer.TT_EOL)) 
	       macroRead.append(macroDefnTokens.sval); */
	    
	    for(int charRead = fileToRead.read(); charRead != -1; charRead = fileToRead.read()) {
		macroRead.append((char)charRead);
	    }
	    macroBody = macroRead.toString();
	    macroDefined = true;
	}	
	catch (FileNotFoundException e) {
	    throw new SimpleMacroError("Macro definition file " + macroDefnFile.getAbsolutePath() + " not found.");
	}
	catch (IOException e) {
	    throw new SimpleMacroError("IO error occurred while reading macro definition file " 
				       + macroDefnFile.getAbsolutePath() + ". Error: " + e.getMessage());
	}
	finally {
	    try {
		if (fileToRead != null) {
		    fileToRead.close();
		}
	    }
	    catch (IOException e) {
		throw new SimpleMacroError("Error closing macro definition file " + macroDefnFile.getAbsolutePath());
	    }
	}
    }

    private int argLookup(String s) {
	for(int i = 0; i < arguments.length; i++) {
	    if (s.equals(arguments[i])) {
		return i;
	    }
	}
	return -1; // The string s is not an argument. 
    }

    public String expand(String[] expansionArgs) throws SimpleMacroException {
	StringReader macroReader = null;

	try {
	    if (expansionArgs == null) {
		if (numberOfArguments != 0) {
		    throw new SimpleMacroException("The number of expansion arguments for macro " + macroName + 
						   " does not equal the number of macro arguments.");
		}
	    }
	    else if (expansionArgs.length != numberOfArguments) {
		throw new SimpleMacroException("The number of expansion arguments for macro " + macroName + 
					       " does not equal the number of macro arguments.");
	    }
	    if (!macroDefined) {
		throw new SimpleMacroException("Macro not defined.");
	    }

	    final String LINE_SEP = System.getProperty("line.separator");

	    StringBuffer expandedMacro = new StringBuffer();
	    macroReader = new StringReader(macroBody);
	    StreamTokenizer macroTokens = new StreamTokenizer(macroReader);
	    macroTokens.slashSlashComments(slashSlashComments);
	    macroTokens.slashStarComments(slashStarComments);
	    macroTokens.eolIsSignificant(true);
	    macroTokens.ordinaryChar((int)' '); // Don't skip spaces
	    macroTokens.ordinaryChar((int)'\t'); // Don't skip tabs
	    macroTokens.ordinaryChar((int)'"'); // Don't parse strings
	    macroTokens.ordinaryChar((int)'\''); // Don't parse strings
	    macroTokens.ordinaryChar((int)'.');
	    macroTokens.wordChars((int)'0', (int)'9'); // Treat digits as word characters (don't parse as a number).

	    // First substitute the macro arguments.
	    int argIndex = -1; // Initialize argIndex to no argument found.
	    for(int tokenType = macroTokens.nextToken(); 
		tokenType != StreamTokenizer.TT_EOF;
		tokenType = macroTokens.nextToken()) {
		
		if (tokenType == StreamTokenizer.TT_WORD) {
		    argIndex = argLookup(macroTokens.sval);
		    if (argIndex != -1) {
			expandedMacro.append(expansionArgs[argIndex]);
		    }
		    else {
			expandedMacro.append(macroTokens.sval);
		    }
		    
		}
		else if (tokenType == StreamTokenizer.TT_EOL) {
		    expandedMacro.append(LINE_SEP);
		}
		/* Now see if the token begins a single line comment
		   (begun with "#"). I need to do this BS because
		   setting slashSlashComments and slashStarComments on
		   StreamTokenizers doesn't seem to work. The
		   tokenizer always skips comments regardless of these
		   settings. */
		else if (tokenType == singleLineCommentChar) {
			expandedMacro.append("// ");
		}
		else {
		    expandedMacro.append((char)macroTokens.ttype);
		}
	    }
	    
	    // Next remove the macro append operators. 
	    for(int i = 0; i < expandedMacro.length(); i++) {
		if (expandedMacro.charAt(i) == macroAppendChar) {
		    expandedMacro.deleteCharAt(i);
		}
	    }
	    
	    /* Return the expanded macro, removing the initial newline. */
	    String eol = System.getProperty("line.separator");
	    if (expandedMacro.substring(0, eol.length()) == eol)
		return expandedMacro.substring(eol.length(), expandedMacro.length());
	    else
		return expandedMacro.toString();
	}
	catch (IOException e) {
	    throw new SimpleMacroException("Error occurred while reading tokens during expand of macro " + macroName
					   + ". Error: " + e.getMessage());
	}
	finally {
		if (macroReader != null) {
		    macroReader.close();
		}
	}
    }
    
    // Override the default toString method from Object
    public String toString() {
	StringWriter tempBuf = new StringWriter();
	PrintWriter tempPrint = new PrintWriter(tempBuf);

	tempPrint.println("Macro name: " + macroName);
	tempPrint.print("Arguments: ");
	for(int i = 0; i < numberOfArguments; i++) {
	    tempPrint.print(arguments[i] + " ");
	}
	tempPrint.println("");
	tempPrint.println("Macro body:");
	tempPrint.print(macroBody);
	
	return tempBuf.toString();
    }
	
}

