

package abl.runtime;

import java.lang.reflect.*;
import java.util.*;
import wm.*;

/**
 * Superclass for Step classes. Instances of children of Step represent steps in
 * the ABT.
 */
public abstract class Step implements ABTNode {

	public static final short PRIMITIVE = 0;
	public static final short GOAL = PRIMITIVE + 1;
	public static final short MENTAL = GOAL + 1;
	public static final short WAIT = MENTAL + 1;
	public static final short FAIL = WAIT + 1;
	public static final short SUCCEED = FAIL + 1;
	public static final short EXECUTE_POLICY = SUCCEED + 1;
	public static final short MODIFY = EXECUTE_POLICY + 1;

	// These will be set by constructors on abstract classes.

	protected Behavior parent; // Reference to parent behavior of this step.

	// These should be set appropriately in the constructor of each concrete behavior class.
	private boolean persistent = false; // True if the Step is marked persistent.
	private boolean persistentWhenSucceeds = false; // True if the step is marked persistent_when_succeeds.
	private boolean persistentWhenFails = false; // True if the step is marked persistent_when_fails.
	private boolean ignoreFailure = false; // True if the step is marked ignore_failure.
	private boolean effectOnly = false; // True if the step is effect only.
	private boolean teamEffectOnly = false; // True if the step is team_effect_only.
	private short priority = Short.MIN_VALUE;
	private short priorityModifier = 0;
	private boolean isAtomic = false; // True if the step is atomic (sensing turned off).
	private boolean post = false; // True if this step should post to working memory when the step completes.
	private String postMemory = null; // A non-null string if this step should post to the named memory when the step completes.
	protected final Method execute; // Method for step execute
	private final Method successTest; // Method for success test
	private final Method successTestSensorFactory; // Method for success test factory

	private final short stepType;

	protected final int stepID;

	// LinkedList of ExecutableSteps suspending this step.
	protected List stepsSuspendingMe = new LinkedList();

	// Linked list of Steps I have suspended.
	protected List stepsIHaveSuspended = new LinkedList();

	// Counter for meta-suspenders.
	private int metaSuspenderCount = 0;

	// Counter for joint-suspenders.
	private int jointSuspenderCount = 0;

	// named property support
	final AblNamedPropertySupport propertyTable;

	// reference to the reflection wme wrapping this step
	private StepWME stepWME = null;

	// Constructor sets the parent behavior.
	public Step(
		int arg_stepID,
		Behavior arg_parent,
		boolean arg_persistent,
		boolean arg_persistentWhenSucceeds,
		boolean arg_persistentWhenFails,
		boolean arg_ignoreFailure,
		boolean arg_effectOnly,
		boolean arg_teamEffectOnly,
		short arg_priority,
		short arg_priorityModifier,
		boolean arg_post,
		String arg_postMemory,
		Method arg_execute,
		Method arg_successTest,
		Method arg_successTestSensorFactory,
		AblNamedPropertySupport arg_propertyTable,
		short arg_stepType) {
		stepID = arg_stepID;
		parent = arg_parent;
		isAtomic = parent.isAtomic; // Steps inherit the parent behavior's atomicity
		persistent = arg_persistent;
		persistentWhenSucceeds = arg_persistentWhenSucceeds;
		persistentWhenFails = arg_persistentWhenFails;
		ignoreFailure = arg_ignoreFailure;
		effectOnly = arg_effectOnly;
		teamEffectOnly = arg_teamEffectOnly;
		priority = arg_priority;
		priorityModifier = arg_priorityModifier;
		post = arg_post;
		postMemory = arg_postMemory;
		execute = arg_execute;
		successTest = arg_successTest;
		successTestSensorFactory = arg_successTestSensorFactory;
		if (arg_propertyTable != null)
			propertyTable = arg_propertyTable;
		else
			propertyTable = new AblNamedPropertySupport();

		stepType = arg_stepType;
	}

	// fixme: push the toString cases down into the individual steps
	public String toString() {
		String behaviorTypeString;
		switch (getStepType()) {
			case GOAL :
				if (((GoalStep) this).getGoalExecutionType() != GoalStep.SUBGOAL && !((GoalStep) this).getRerooted())
					behaviorTypeString = "spawngoal " + ((GoalStep) this).getSignature();
				else
					behaviorTypeString = "subgoal " + ((GoalStep) this).getSignature();
				if (((GoalStep) this).isJointGoal())
					behaviorTypeString = "joint " + behaviorTypeString;
				break;
			case PRIMITIVE :
				behaviorTypeString = "act " + ((PrimitiveStep) this).name;
				break;
			case WAIT :
				behaviorTypeString = "wait";
				break;
			case MENTAL :
				behaviorTypeString = "mental";
				break;
			case FAIL :
				behaviorTypeString = "fail_step";
				break;
			case SUCCEED :
				behaviorTypeString = "succeed_step";
				break;
			case EXECUTE_POLICY :
			    behaviorTypeString = "execute_policy";
			    break;
			default :
				return "UNEXPECTED STEP TYPE";
		}
		// return behaviorTypeString + " priority: " + priority;
		return behaviorTypeString;
	}

	// ### Get Accessors ###

	// Public accessor for stepType. Since each step type defines a
	// static storing its type, getStepType has to be defined
	// separately on each step type subclass.
	final short getStepType() {
		return stepType;
	}
	final boolean getPersistent() {
		return persistent;
	}
	final boolean getPersistentWhenSucceeds() {
		return persistentWhenSucceeds;
	}
	final boolean getPersistentWhenFails() {
		return persistentWhenFails;
	}
	final boolean getIgnoreFailure() {
		return ignoreFailure;
	}
	final boolean getHasSuccessTest() {
		if (successTest != null)
			return true;
		else
			return false;
	}
	final boolean getEffectOnly() {
		return effectOnly;
	}
	final boolean getTeamEffectOnly() {
		return teamEffectOnly;
	}
	final boolean getEffectOnlyOrTeamEffectOnly() {
		return effectOnly || teamEffectOnly;
	}
	final short getPriority() {
		if (priority == Short.MIN_VALUE) {
			// Priority should be inherited from the parent and modified by priorityModifier
			priority = (short) (parent.getPriority() + priorityModifier);
			return priority;
		} else
			// Priority explicitly declared
			return priority;
	}

	final short getPriorityModifier() {
		return priorityModifier;
	}
	final boolean getIsAtomic() {
		return isAtomic;
	}
	final List getStepsSuspendingMe() {
		return new Vector(stepsSuspendingMe);
	}
	final List getStepsIHaveSuspended() {
		return new Vector(stepsIHaveSuspended);
	}
	/** Note: Ben Weber, changed from default to public accessible for debugging purposes */
	public final Behavior getParent() {
		return parent;
	}
	final int getMetaSuspenderCount() {
		return metaSuspenderCount;
	}

	final int getStepID() { return stepID; }
	
	// ### Set Accessors ###
	// Package accessors can be accessed through the public interface on reflection WMEs
	final void setPersistent(boolean b) {
		persistent = b;
	}
	final void setPersistentWhenSucceeds(boolean b) {
		persistentWhenSucceeds = b;
	}
	final void setPersistentWhenFails(boolean b) {
		persistentWhenFails = b;
	}
	final void setIgnoreFailure(boolean b) {
		ignoreFailure = b;
	}
	final void setEffectOnly(boolean b) {
		effectOnly = b;
	}
	final void setTeamEffectOnly(boolean b) {
		teamEffectOnly = b;
	}
	final void setPriority(short i) {
		priority = i;
	}
	final void setPriorityModifier(short i) {
		priority = i;
	}

	// Returns false if no success test is defined for this step, otherwise calls the success test.
	boolean successTest() {
		if (successTest == null)
			// No success test - always false
			return false;
		else {
			Object[] args = { new Integer(stepID), parent.getBehaviorVariableFrame(), BehavingEntity.getBehavingEntity()};
			try {
				Boolean b = (Boolean) successTest.invoke(null, args);
				return b.booleanValue();
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking success test", e);
			}
		}
	}

	// For those Steps with success tests referencing sensed wmes, returns a list of
	// the SensorActivations needed by the success test. Really
	// only makes sense for ExecutableSteps (steps that take time to
	// execute) but it does no harm to associate success tests and
	// sensor activations with non-ExecutableSteps
	// (e.g. MentalSteps). Eventually getSuccessTestSensorActivations() should be
	// associated with the ExecutableAct interface, but this would create type headaches right now.
	final SensorActivation[] getSuccessTestSensorActivations() {
		if (successTestSensorFactory == null)
			return null; // no sensor factory
		else {
			Object[] args = { new Integer(stepID)};
			try {
				return (SensorActivation[]) successTestSensorFactory.invoke(null, args);
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking success test sensor factory", e);
			}
		}
	}

	// Executes the reflected execute method.
	abstract void execute();

	// Performs bookkeeping when a step is executed.
	// Subclasses of Step perform class-specific bookeeping then call
	// super.executeBookkeeping().
	protected void executeBookkeeping() {
		BehavingEntity.getBehavingEntity().executeStep(this);
	}

	private void postCompletionWME(int success, String memoryName) {
		if (memoryName != null)
			assert WorkingMemory.lookupRegisteredMemory(memoryName) != null : "memory " + memoryName + " does not exist.";

		CompletedStepWME newWME;
		switch ((int) getStepType()) {
			case PRIMITIVE :
				newWME = new CompletedActWME(((PrimitiveStep) this).name, success, parent.getSignature(), BehavingEntity.getBehavingEntity().getBehavingEntityShortName());
				break;
			case GOAL :
				newWME = new CompletedGoalWME(((GoalStep) this).getSignature(), success, parent.getSignature(), BehavingEntity.getBehavingEntity().getBehavingEntityShortName());
				break;
			case MENTAL :
				newWME = new CompletedMentalActWME(parent.getSignature(), BehavingEntity.getBehavingEntity().getBehavingEntityShortName());
				break;
			case WAIT :
				newWME = new CompletedWaitWME(parent.getSignature(), BehavingEntity.getBehavingEntity().getBehavingEntityShortName());
				break;
			default :
				throw new AblRuntimeError("Unexpected step type " + getStepType());
		}

		WorkingMemory memory;
		if (memoryName == null)
			memory = BehavingEntity.getBehavingEntity().getWorkingMemory();
		else
			memory = WorkingMemory.lookupRegisteredMemory(memoryName);

		memory.addWME(newWME);
	}

	// Succeeds a step. Examines both step and parent behavior annotations to do the right thing.
	void succeedStep() {
		if (post)
			// Post a step completion wme to the working memory.
			postCompletionWME(CompletedExecutableStepWME.SUCCESS, null);
		if (postMemory != null)
			// Post a step completion wme to a named memory.
			postCompletionWME(CompletedExecutableStepWME.SUCCESS, postMemory);

		if (getPersistent() || getPersistentWhenSucceeds())
			resetStep();
		else {
			// Remove from parent behavior. Perform appropriate mods on BehavingEntity.
			parent.removeChild(this, true); // recursively removes tree rooted here
			if (parent.getBehaviorType() == Behavior.SEQUENTIAL) {
				Step nextStep = ((SequentialBehavior) parent).getNextStep();
				if (nextStep == null)
					parent.succeedBehavior();
				else
					// Add the step to the ABT (child of sequential behavior) and perform appropriate mods on BehavingEntity.
					 ((SequentialBehavior) parent).addChild(nextStep);
			} else if (parent.getBehaviorType() == Behavior.ADAPTIVE) {
			    // Reinforce the parent behavior's policy when the step succeeds
			    final AdaptiveBehavior parentBeh = (AdaptiveBehavior)parent;
			    parentBeh.updatePolicy();
			    parentBeh.addChildren(); // Add an ExecutePolicyStep back into the child
			} else if (((MultiStepBehavior) parent).isSuccessful())
				/* isSuccessful tests whether enough of the steps
				   have occurred to satisfy the behavior. */

				parent.succeedBehavior();
		}
	}

	// Fails a step. Examines both step and parent behavior annotations to do the right thing.
	void failStep() {

		if (post)
			// Post a step completion wme to the working memory.
			postCompletionWME(CompletedExecutableStepWME.FAILURE, null);
		if (postMemory != null)
			// Post a step completion wme to a named memory.
			postCompletionWME(CompletedExecutableStepWME.FAILURE, postMemory);

		if (getIgnoreFailure())
			succeedStep();
		else if (getPersistent() || getPersistentWhenFails())
			resetStep();
		else {
			parent.removeChild(this, true); // Parent behavior removes this failed step.
			if (parent.getBehaviorType() == Behavior.COLLECTION) {
				if (((MultiStepBehavior) parent).isSuccessful())
					parent.succeedBehavior();
			} else if (parent.getBehaviorType() == Behavior.ADAPTIVE) {
			    // Reinforce the parent behavior's policy when the step fails 
			    final AdaptiveBehavior parentBeh = (AdaptiveBehavior)parent;
			    parentBeh.updatePolicy();
			    parentBeh.addChildren(); // Add an ExecutePolicyStep back into the child
			} else if ((!getEffectOnly()) && (!getTeamEffectOnly())){
			    parent.failBehavior();
			}
		}
	}

	// Resets the step. Called when one wants to leave a step in the ABT but make it "fresh."
	abstract void resetStep();

	// Returns true if this step is in the line of expansion of the most recently pursued goal, false otherwise.
	boolean currentLineOfExpansion(GoalStep currentGoal) {
		/* If this step is not a goal step, ask the parent if the
		       current goal is the last pursued goal. GoalStep overrides
		       currentLineOfExpansion to set the current goal to the goal
		       step. */
		return parent.currentLineOfExpansion(currentGoal);
	}

	// Adds a step to the list of suspending steps.
	void suspend(ExecutableStep step) {
		stepsSuspendingMe.add(step);
		step.suspenderFor(this);
	}

	// Removes a step from the list of suspending steps.
	void unsuspend(ExecutableStep step) {
		stepsSuspendingMe.remove(step);
		step.unsuspenderFor(this);
	}

	// Meta control for suspend.
	// Once a step has been suspended with metaSuspend, it will never unsuspend until metaUnsuspend is explicitly
	// called.
	void metaSuspend() {
		assert(metaSuspenderCount >= 0);
		metaSuspenderCount++;
	}

	// Meta control for unsuspend.
	// When a step has been metaUnsuspended, it will actually unsuspend if the count
	// meta-unsuspenders is 0 and there are no regular suspenders (stepsSuspendingMe is empty).
	void metaUnsuspend() {
		assert(metaSuspenderCount >= 0);
		// metaUnsuspend does nothing if the metaSuspenderCount is already 0
		if (metaSuspenderCount > 0)
			metaSuspenderCount--;
	}

	// Suspend initated by a joint commitment.
	void jointSuspend() {
		assert(jointSuspenderCount >= 0);
		jointSuspenderCount++;
	}

	// Unsuspend initiated by a joint commitment.
	void jointUnsuspend() {
		assert(jointSuspenderCount > 0);
		jointSuspenderCount--;
	}

	// Returns true if there are any steps which have suspended this step, false otherwise.
	boolean isSuspended() {
		if (!stepsSuspendingMe.isEmpty() || (metaSuspenderCount > 0) || (jointSuspenderCount > 0))
			return true;
		else
			return false;
	}

	// Remove this step from the stepsIHaveSuspended set of every suspending step.
	void updateStepsSuspendingMe() {
		ExecutableStep[] stepsSuspendingMeArray = (ExecutableStep[]) stepsSuspendingMe.toArray(new ExecutableStep[stepsSuspendingMe.size()]);
		for (int i = 0; i < stepsSuspendingMeArray.length; i++) {
			stepsSuspendingMeArray[i].unsuspenderFor(this);
		}
	}

	// only classes in abl.runtime can set the reflection wme
	void setReflectionWME(StepWME stepWME) {
		assert stepWME != null;
		this.stepWME = stepWME;
	}

	// concrete subclasses of Step can get the reflection wme
    public StepWME getReflectionWME() {
	if (BehavingEntity.getBehavingEntity().reflectionEnabled() && stepWME == null) {
	    System.err.println("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$ UNEXPECTED NULL REFLECTION WME in " + BehavingEntity.getBehavingEntity());
	    BehavingEntity.printABTBranchUpwards(this);
	    throw new AblRuntimeError("Unexpected NULL Reflection WME in " + BehavingEntity.getBehavingEntity());
	    // BehavingEntity.getBehavingEntity().breakNow();
	}
	else {
	    return stepWME;
	}
    }
}
