// Class for primitive steps (steps consisting of a primitive act). 

package abl.runtime;

import java.lang.reflect.Method;

public class PrimitiveStep extends ExecutableStep
{
    // Reference to the primitive act implementing this primitive step.
    private PrimitiveAction act;

    // True if the primitive step is executing. Initialized to false. 
    private boolean executing = false;

    /* When a PrimitiveStep is constructed, needs to be passed an
       instantiated PrimitiveAction object to associate with the
       PrimitiveStep. */
    public PrimitiveStep(int stepID, Behavior arg_parent, boolean arg_persistent, boolean arg_persistentWhenSucceeds, 
			 boolean arg_persistentWhenFails, boolean arg_ignoreFailure, boolean arg_effectOnly, 
			 boolean arg_teamEffectOnly, short arg_priority, short arg_priorityModifier, boolean arg_post, 
			 String arg_postMemory, Method arg_execute, Method arg_successTest, 
			 Method arg_successTestSensorFactory, AblNamedPropertySupport arg_propertyTable, 
			 PrimitiveAction arg_act, String[] arg_stepsIConflictWith, String arg_actName)
    {
	super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds, arg_persistentWhenFails, arg_ignoreFailure, 
	      arg_effectOnly, arg_teamEffectOnly, arg_priority, arg_priorityModifier, arg_post, arg_postMemory, 
	      arg_execute, arg_successTest, arg_successTestSensorFactory, arg_propertyTable, PRIMITIVE, 
	      arg_stepsIConflictWith);
	act = arg_act;
	name = arg_actName;
    }

    /* Returns true if the primitive step is currently executing. A
       primitive step is executing if (executing == true) and
       getCompletionStatus() == NOT_COMPLETE. */
    final boolean isExecuting() {
	if (executing) {
	    if (getCompletionStatus() == PrimitiveAction.NOT_COMPLETE)
		return true;
	    else
		executing = false;
	}
	return false;
    }

    // Public get accessor for completionStatus (called by the concrete primitive action). 
    final int getCompletionStatus() {
	return act.getCompletionStatus();
    }

    // Called when a primitive step is reset. 
    final void resetStep() {
	if (isExecuting()) {
	    act.abort();
	}
	BehavingEntity.getBehavingEntity().resetStep(this);
	try {
	    Class actionClass = act.getClass();
	    act = (PrimitiveAction) actionClass.newInstance();
	} catch (Exception e) { throw new AblRuntimeError("Reflection error in PrimitiveStep.resetStep()", e); }
	executing = false;
    }

    private final Object[] bindArgs()
    {
	if (execute != null) {
	    // The primitive step has args, so a bind args case has been defined in PrimitiveStepExecute
	    final Object[] exeArgs = { new Integer(stepID), parent.getBehaviorVariableFrame(), 
				       BehavingEntity.getBehavingEntity() };
	    try { 
		return (Object[])execute.invoke(null, exeArgs);
	    } catch (Exception e) {  throw new AblRuntimeError("Error invoking execute", e); }
	}
	else
	    return new Object[0];
    }

    void execute() 
    {
	checkForConflictsOnExecution();
	if (!isSuspended()) {
	    executeBookkeeping();
	    Object[] args = bindArgs();
	    act.execute(args);
	}
    }

    // Called when the primitive action is executed.
    protected final void executeBookkeeping() {
	executing = true;
	super.executeBookkeeping();
    }

    // Abort a running primitive action.
    final void abort() {
	if (isExecuting()) {
	    act.abort();
	}
	BehavingEntity.getBehavingEntity().abortStep(this);
	executing = false;
    }

    // Suspends a PrimitiveStep
    void suspend(ExecutableStep step)
    {
	super.suspend(step);
	BehavingEntity.getBehavingEntity().suspendStep(this);
	if (isExecuting()) {
	    // If the step is executing, create a new act instance and modify state
	    act.abort();
	    try {
		Class actionClass = act.getClass();
		act = (PrimitiveAction) actionClass.newInstance();
	    } catch (Exception e) { throw new AblRuntimeError("Reflection error in PrimitiveStep.suspend()", e); }
	    executing = false;
	}
    }

    // Meta control for suspend
    public void metaSuspend()
    {
	super.metaSuspend();
	BehavingEntity.getBehavingEntity().suspendStep(this);
	if (isExecuting()) {
	    // If the step is executing, create a new act instance and modify state
	    act.abort();
	    try {
		Class actionClass = act.getClass();
		act = (PrimitiveAction) actionClass.newInstance();
	    } catch (Exception e) { throw new AblRuntimeError("Reflection error in PrimitiveStep.suspend()", e); }
	    executing = false;
	}
    }

    // Joint suspend.
    void jointSuspend()
    {
	super.jointSuspend();
	BehavingEntity.getBehavingEntity().suspendStep(this);
	if (isExecuting()) {
	    // If the step is executing, create a new act instance and modify state
	    act.abort();
	    try {
		Class actionClass = act.getClass();
		act = (PrimitiveAction) actionClass.newInstance();
	    } catch (Exception e) { throw new AblRuntimeError("Reflection error in PrimitiveStep.suspend()", e); }
	    executing = false;
	}
    }

    // Unsuspends a primitive step
    void unsuspend(ExecutableStep step) 
    {
	super.unsuspend(step);
	if (!isSuspended()) {
	    BehavingEntity.getBehavingEntity().unsuspendStep(this);
	}
    }

    // Meta control for unsuspend
    public void metaUnsuspend()
    {
	super.metaUnsuspend();
	if (!isSuspended()) {
	    BehavingEntity.getBehavingEntity().unsuspendStep(this);
	}
    }

    void jointUnsuspend()
    {
	super.jointUnsuspend();
	if (!isSuspended()) {
	    BehavingEntity.getBehavingEntity().unsuspendStep(this);
	}	
    }
    
}



