/* All primitive actions are a concrete subclass of PrimitveAction. A
   concrete primitive act class is responsible for actually
   implementing the code which causes the appropriate changes in the
   world. 
   
   Each unique primitive step signature within an ABL program requires
   a unique concrete class definition implementing that step. If an ABL
   program makes use of several closely related signatures (e.g. a
   action with default parameter values), code sharing for the
   implementation of these acts can be accomplished by relating the
   concrete action objects in a class hierarchy.
*/

package abl.runtime;

public abstract class PrimitiveAction {

    // Enumerated type for action completion status.
    public static final int NOT_COMPLETE = 0;
    public static final int SUCCESS = 1;
    public static final int FAILURE = 2;

    /**
     * When the primitive action completes execution, this is set to the
     * appropriate completion status ({@link PrimitiveAction#SUCCESS},
     * {@link PrimitiveAction#FAILURE}).
     */
    protected int completionStatus = NOT_COMPLETE;

    /** Aborts a running primitive act. The default implementation does nothing.  */
    public void abort() {
    }
    
    /** Public get accessor for completionStatus.  */
    public synchronized int getCompletionStatus() {
	return completionStatus;
    }

    /**
     * For actions which take significant time to execute, the action execution
     * system (e.g. virtual world, robotic body, etc.), can be written to
     * callback the PrimitiveAction when the activity completes (with success or
     * failure). The completion callback accepts a single boolean argument: true
     * for completion with success, false for completion with failure. The
     * default completion callback updates the completion status of the action.
     */ 
    public synchronized void completionCallback(boolean status) {
	if (status) 
	    completionStatus = SUCCESS;
	else 
	    completionStatus = FAILURE;
    }

    /**
     * Executes the primitive action. The execute() method defined on concrete
     * PrimitiveSteps calls execute() on the PrimitiveAction. The ABL execution
     * loop blocks on executing a primitive action. Thus primitive actions
     * should return as quickly as possible. The executing action is also
     * responsible for setting the completion status. Concrete physical actions
     * may implement execute using one of the three following strategies:
     * <p>
     * 1. For actions which execute very quickly in the underlying execution
     * system (e.g. setting a state flag), execute can directly make the
     * appropriate RPC call and set completionStatus.
     * <p>
     * 2. For longer actions, may fork another method in a separate thread and
     * immediately return the current thread of execution. The spawn thread is
     * responsible for setting completionStatus.
     * <p>
     * 3. For longer actions, the execution system (e.g. world, robotic system)
     * may be written so as to accept callbacks. The exeuction system then calls
     * the callback when the action finishes executing. The current thread of
     * execution initiates the action, registers the callback, and returns
     * immediately. The default callback routine (defined above) appropriately
     * updates the completion status.
     */
    public abstract void execute(Object[] args);

    
    /**
     * Called every decision cycle. The default implementation does nothing. Can
     * be overridden on concrete action classes if the specific sensory-motor
     * system needs some processing to be performed every decision cycle.
     */
    public void decisionCycleCallback() {}
}       






