/* Abstract superclass for parallel and collection behaviors. */

package abl.runtime;

import java.util.*;
import java.lang.reflect.Method;

public abstract class MultiStepBehavior extends Behavior {

	// Set of steps. For MultiStepBehaviors, all steps are added as leaf steps at once.
	protected HashSet<Step> children = new HashSet<Step>();

	// The number of non-effect-only steps which must succeed for this
	// behavior to succeed. A value of -1 means that the number needed
	// for success is equal to the total number of children (ie. the
	// standard criterion for behavior success).
	private int numberNeededForSuccess = -1;

	// The number of steps which have completed for this behavior.
	private int numberOfCompletedSteps = 0;

	// Last pursued goal.
	private GoalStep lastPursuedGoal = null;

    // fixme: currently passing null success condition to super constructor. Eventually require success condition to be explicitly set when we thread the success condition through the compiler. 
	public MultiStepBehavior(GoalStep arg_parent, Method arg_contextCondition, Method arg_contextConditionSensorFactory, boolean arg_isAtomic, String arg_signature, short arg_specificity, int arg_behaviorID, Object[] arg_behaviorVariableFrame, __StepDesc[] arg_stepDescs, int arg_numberNeededForSuccess) {
		super(arg_parent, arg_contextCondition, arg_contextConditionSensorFactory, null, null, arg_isAtomic, arg_signature, arg_specificity, arg_behaviorID, arg_behaviorVariableFrame, arg_stepDescs);
		// fixme: this should be computed at compile time. At runtime it may vary because of the addition of a step
		// (created by spawnGoal). Looks like I'm already setting this at compile time.
		if (arg_numberNeededForSuccess == -1) {
			// number_needed_for_success not specified.
			// set to number of non-effect-only children.
			if (this.getBehaviorType() != Behavior.ADAPTIVE){
               numberNeededForSuccess = 0;
			   Step[] tempSteps = getAllSteps();
			   for (int i = 0; i < tempSteps.length; i++)
				   if (!tempSteps[i].getEffectOnlyOrTeamEffectOnly())
					   numberNeededForSuccess++;
            }
		} else {
			numberNeededForSuccess = arg_numberNeededForSuccess;
		}
	}

    // Version of MultiStepBehavior with explicit success conditions set. Currently used only by AdaptiveBehavior, but it will eventually become the main constructor. 

    public MultiStepBehavior(
			     GoalStep arg_parent, 
			     Method arg_contextCondition, 
			     Method arg_contextConditionSensorFactory, 
			     Method arg_successCondition,
			     Method arg_successConditionSensorFactory,
			     boolean arg_isAtomic, 
			     String arg_signature, 
			     short arg_specificity, 
			     int arg_behaviorID, 
			     Object[] arg_behaviorVariableFrame, 
			     __StepDesc[] arg_stepDescs, 
			     int arg_numberNeededForSuccess) {
	super(arg_parent, arg_contextCondition, arg_contextConditionSensorFactory, arg_successCondition, arg_successConditionSensorFactory, arg_isAtomic, arg_signature, arg_specificity, arg_behaviorID, arg_behaviorVariableFrame, arg_stepDescs);
	// fixme: this should be computed at compile time. At runtime it may vary because of the addition of a step
	// (created by spawnGoal). Looks like I'm already setting this at compile time.
	if (arg_numberNeededForSuccess == -1) {
	    // number_needed_for_success not specified.
	    // set to number of non-effect-only children.
	    if (this.getBehaviorType() != Behavior.ADAPTIVE){
		numberNeededForSuccess = 0;
		Step[] tempSteps = getAllSteps();
		for (int i = 0; i < tempSteps.length; i++)
		    if (!tempSteps[i].getEffectOnlyOrTeamEffectOnly())
			numberNeededForSuccess++;
            }
	} else {
	    numberNeededForSuccess = arg_numberNeededForSuccess;
	}
    }

    public MultiStepBehavior(
			     GoalStep arg_parent, 
				 Method arg_contextCondition, 
				 Method arg_contextConditionSensorFactory, 
				 Method arg_successCondition,
				 Method arg_successConditionSensorFactory,
				 boolean arg_isAtomic, 
				 String arg_signature, 
				 short arg_specificity, 
				 int arg_behaviorID, 
				 Object[] arg_behaviorVariableFrame, 
				 __StepDesc[] arg_stepDescs, 
				 int arg_numberNeededForSuccess,
				 BehavingEntity[] arg_teamMembers) {
		super(arg_parent, arg_contextCondition, arg_contextConditionSensorFactory, arg_successCondition, arg_successConditionSensorFactory, arg_isAtomic, arg_signature, arg_specificity, arg_behaviorID, arg_behaviorVariableFrame, arg_stepDescs, arg_teamMembers);
		// fixme: this should be computed at compile time. At runtime it may vary because of the addition of a step
		// (created by spawnGoal). Looks like I'm already setting this at compile time.
		if (arg_numberNeededForSuccess == -1) {
			// number_needed_for_success not specified.
			// set to number of non-effect-only children.
			if (this.getBehaviorType() != Behavior.ADAPTIVE){
				numberNeededForSuccess = 0;
				Step[] tempSteps = getAllSteps();
				for (int i = 0; i < tempSteps.length; i++)
					if (!tempSteps[i].getEffectOnlyOrTeamEffectOnly())
						numberNeededForSuccess++;
            }
		} else {
			numberNeededForSuccess = arg_numberNeededForSuccess;
		}
    }


    // fixme: currently passing null success condition to super constructor. Eventually require success condition to be explicitly set when we thread the success condition through the compiler. 
	public MultiStepBehavior(GoalStep arg_parent, Method arg_contextCondition, Method arg_contextConditionSensorFactory, boolean arg_isAtomic, String arg_signature, short arg_specificity, int arg_behaviorID, Object[] arg_behaviorVariableFrame, __StepDesc[] arg_stepDescs, int arg_numberNeededForSuccess, BehavingEntity[] arg_teamMembers) {
		super(arg_parent, arg_contextCondition, arg_contextConditionSensorFactory, null, null, arg_isAtomic, arg_signature, arg_specificity, arg_behaviorID, arg_behaviorVariableFrame, arg_stepDescs, arg_teamMembers);
		// fixme: this should be computed at compile time. At runtime it may vary because of the addition of a step
		// (created by spawnGoal).
		if (arg_numberNeededForSuccess == -1) {
			// number_needed_for_success not specified.
			// set to number of non-effect-only children.
			Step[] tempSteps = getAllSteps();
			numberNeededForSuccess = 0;
			for (int i = 0; i < tempSteps.length; i++)
				if (!tempSteps[i].getEffectOnlyOrTeamEffectOnly())
					numberNeededForSuccess++;
		} else {
			numberNeededForSuccess = arg_numberNeededForSuccess;
		}
	}

	// ### get accessors ###

	int getNumberNeededForSuccess() {
		return numberNeededForSuccess;
	}

	final int getNumberOfCompletedSteps() {
		return numberOfCompletedSteps;
	}

	final GoalStep getLastPursuedGoal() {
		return lastPursuedGoal;
	}

	final int getNumberOfSteps() {
		return stepDescs.length;
	}

	final Step[] getChildren() {
		return (Step[]) children.toArray(new Step[children.size()]);
	}

	final void pursueGoal(GoalStep g) {
		lastPursuedGoal = g;
	}

	final void clearLastPursuedGoal() {
		lastPursuedGoal = null;
	}

	// Recursively removes all children of this behavior.
	protected void removeChildren() {
		/* Iterate over an array copy of children rather than using an
		   iterator because removing steps from children is a state
		   violation for an interator. */
		final Step[] s = (Step[]) children.toArray(new Step[children.size()]);
		for (int i = 0; i < s.length; i++)
			// Iterate through the children, removing them.
			removeChild(s[i], false);
	}

	// Adds a single child to the behavior. Called when a step is spawned at a new behavior.
	void addChild(Step s) {
		// If the behavior is suspended when the child is added, suspend the child
		if (parent != null) { // not the root behavior
			if (parent.isSuspended()) {
				List suspenders = parent.getStepsSuspendingMe();
				int metaSuspenderCount = parent.getMetaSuspenderCount();
				assert(suspenders.size() != 0 || metaSuspenderCount > 0);
				if (suspenders.size() != 0) {
					Iterator iter = suspenders.iterator();
					while (iter.hasNext())
						s.suspend((ExecutableStep) iter.next());
				}
				if (metaSuspenderCount > 0) {
					for (int i = 0; i < metaSuspenderCount; i++)
						s.metaSuspend();
				}
			}
		}

		children.add(s);
		if (!s.getEffectOnlyOrTeamEffectOnly())
			numberNeededForSuccess++; // If adding a non-effect-only child, increase the number needed for success
		BehavingEntity.getBehavingEntity().addStep(s);
		changes.firePropertyChange("child", null, s); // Fire property change for any listeners
	}

	// Adds the children for this behavior.
	void addChildren() {
		Step[] newChildren = getAllSteps();
		for (int i = 0; i < newChildren.length; i++) {
			children.add(newChildren[i]);
			BehavingEntity.getBehavingEntity().addStep(newChildren[i]);
			changes.firePropertyChange("child", null, newChildren[i]); // Fire property change for any listeners
		}
	}

	/* Removes a child from a multi-step behavior. First calls
	  super.removeChild() which takes care of appropriate bookeeping
	  depending on the type of the child and performs bookeeping on
	  BehavingEntity. Then this more concrete method removes the
	  behavior from the HashSet. */
	protected void removeChild(Step child, boolean isCallerChild) {
		// Trace.ablTrace("Enter (MultiStepBehavior)" + this.getSignature() + ".removeChild(" + child + ")");

		// The step argument is not a child of this behavior; throw an error.
		assert children.contains(child) : "child == " + child + " current children == " + children;

		super.removeChild(child, isCallerChild);
		children.remove(child);

		/* If the child being removed is a GoalStep, setting
		       lastPursuedGoal to null indicates that the behavior is no
		       longer pursuing this goal. If the child being removed is
		       not a GoalStep, lastPursuedGoal would already be null;
		       setting it to null again does nothing. */
		lastPursuedGoal = null;

		/* Increment the numberOfCompletedSteps counter if numberNeededForSuccess has been specified and if the step
		*  is not effect only.
		*  Note that numberNeededForSuccess is incremented in the event of step success or *failure*. 
		*  Technically one shouldn't increment numberOfCompletedSteps in the event of failure (unless the step annotation is ignoreFailure). 
		*  However, a non-ignoreFailure failing step will cause the enclosing behavior to fail, so the fact that
		*  numberOfCompletedSteps will be incremented to numberNeededForSuccess (since one failing step will recusively cause all remaining steps
		*  to be removed) doesn't matter, since the behavior will have been failed away. However, in the event of attempting to spawn a goal 
		*  at a failed behavior, the resulting error message is confusing since the behavior details spit out in the exception show that 
		*  numberOfCompletedSteps = numberNeededForSuccess (because removeChild was called on all steps as the behavior failed). */
		if (numberNeededForSuccess != -1 && !child.getEffectOnlyOrTeamEffectOnly())
			numberOfCompletedSteps++;
	}

	// Uses the steps to call the step factory.
	protected Step[] getAllSteps() {
		final Step[] steps = new Step[stepDescs.length];
		for (int i = 0; i < steps.length; i++) {
			try {
				final Object[] factoryArgs = { new Integer(stepDescs[i].stepID), this, getBehaviorVariableFrame()};
				steps[i] = (Step) stepDescs[i].factory.invoke(null, factoryArgs);
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking step factory", e);
			}
		}
		return steps;
	}

	// isSuccessful() returns true if the success conditions have been
	// met on this behavior, false otherwise.
	boolean isSuccessful() {
		assert(numberNeededForSuccess != -1);
		return numberOfCompletedSteps >= numberNeededForSuccess;
	}

	/* Returns true if the current goal is in the current line of
	   expansion, false otherwise. */
	boolean currentLineOfExpansion(GoalStep currentGoal) {
		if (parent != null) {
			// This is not the root collection behavior.

			if ((currentGoal == null) || (lastPursuedGoal == null) || (currentGoal == lastPursuedGoal))
				return parent.currentLineOfExpansion(currentGoal);
			else
				return false;
		} else {
			// This is the root collection behavior

			if ((currentGoal == null) || (lastPursuedGoal == null) || (currentGoal == lastPursuedGoal))
				return true;
			else
				return false;
		}
	}

	// Suspends the steps of the MultiStepBehavior
	final void suspend(ExecutableStep step) {
		final Iterator iter = children.iterator();
		Step child;
		while (iter.hasNext()) {
			child = (Step) iter.next();
			child.suspend(step);
		}
	}

	// Meta control for suspending the steps of the MultiStepBehavior
	final void metaSuspend() {
		final Iterator iter = children.iterator();
		Step child;
		while (iter.hasNext()) {
			child = (Step) iter.next();
			child.metaSuspend();
		}
	}

	// Unsuspends the steps of the MultiStepBehavior
	final void unsuspend(ExecutableStep step) {
		final Iterator iter = children.iterator();
		Step child;
		while (iter.hasNext()) {
			child = (Step) iter.next();
			child.unsuspend(step);
		}
	}

	// Meta control for unsuspending the steps of the MultiStepBehavior
	final void metaUnsuspend() {
		final Iterator iter = children.iterator();
		Step child;
		while (iter.hasNext()) {
			child = (Step) iter.next();
			child.metaUnsuspend();
		}
	}

	// Joint unsuspend.
	final void jointUnsuspend() {
		final Iterator iter = children.iterator();
		while (iter.hasNext())
			 ((Step) iter.next()).jointUnsuspend();
	}

	final void suspendSkipJointGoals(ExecutableStep step) {
		final Iterator iter = children.iterator();
		while (iter.hasNext()) {
			Step s = (Step) iter.next();
			if (!((s.getStepType() == Step.GOAL) && ((GoalStep) s).isJointGoal())) {
				if (s.getStepType() == Step.GOAL)
					 ((GoalStep) s).suspendSkipJointGoals(step);
				else
					s.suspend(step);
			}
		}
	}

	final void metaSuspendSkipJointGoals() {
		final Iterator iter = children.iterator();
		while (iter.hasNext()) {
			Step s = (Step) iter.next();
			if (!((s.getStepType() == Step.GOAL) && ((GoalStep) s).isJointGoal())) {
				if (s.getStepType() == Step.GOAL)
					 ((GoalStep) s).metaSuspendSkipJointGoals();
				else
					s.metaSuspend();
			}
		}
	}

	final List freezeSubtreeAndNegotiateRemoval() {
		if (hasTeamEffectOnlySteps())
			return freezeNonTeamEffectOnlySubtreeAndNegotiateRemoval();
		else {
			BehavingEntity.getBehavingEntity().removeBehavior(this); // remove the context condition
			final Iterator iter = children.iterator();
			final Vector<Step> v = new Vector<Step>();
			while (iter.hasNext())
				v.addAll(freezeChild((Step) iter.next()));
			return v;
		}
	}

	final protected List freezeNonTeamEffectOnlySubtreeAndNegotiateRemoval() {
		// Trace.ablTrace("Enter (MultiStepBehavior)" + this.getSignature() + ".freezeNonTeamEffectOnlySubtreeAndNegotiateRemoval()");

		BehavingEntity.getBehavingEntity().removeBehavior(this); // remove the context condition
		final Iterator iter = children.iterator();
		final Vector v = new Vector();
		while (iter.hasNext()) {
			Step s = (Step) iter.next();
			// Trace.ablTrace(s + ".getTeamEffectOnly() == " + s.getTeamEffectOnly());
			if (!s.getTeamEffectOnly()) // not team_effect_only - freeze the child
				v.addAll(freezeChild(s));
		}

		// Trace.ablTrace("Leaving MultiStepBehavior.freezeNonTeamEffectOnlySubtreeAndNegotiateRemoval()");

		return v;
	}

	// Return true if the behavior has teamEffectOnly steps.
	final protected boolean hasTeamEffectOnlySteps() {
		final Iterator iter = children.iterator();
		while (iter.hasNext())
			if (((Step) iter.next()).getTeamEffectOnly())
				return true;

		return false;
	}

	final List negotiateSuspensionOfSubtree() {
		final Iterator iter = children.iterator();
		Vector v = new Vector();
		while (iter.hasNext())
			v.addAll(negotiateSuspensionOfChild((Step) iter.next()));

		return v;
	}

	final List negotiateSuspensionOfSubtree(ExecutableStep s) {
		final Iterator iter = children.iterator();
		Vector v = new Vector();
		while (iter.hasNext())
			v.addAll(negotiateSuspensionOfChild((Step) iter.next(), s));

		return v;
	}
}
