// Class for joint goal steps. 

package abl.runtime;

import java.util.*;
import debug.Trace;
import java.lang.reflect.Method;

public class JointGoalStep extends GoalStep {
    // negotiator for this goal
    JointGoalNegotiator negotiator = null;

    // true when the goal is in the middle of negotiating
    private boolean isNegotiating = false;

    // true when all the goals in the team need to individually succeed in order for the team to negotiate success
    protected boolean teamNeededForSuccess; 
    
    // Constructor for JointGoalStep calls the super constructor. 
    public JointGoalStep(int stepID, Behavior arg_parent, boolean arg_persistent, boolean arg_persistentWhenSucceeds, 
			 boolean arg_persistentWhenFails, boolean arg_ignoreFailure, boolean arg_effectOnly, 
			 boolean arg_teamEffectOnly, short arg_priority, short arg_priorityModifier, boolean arg_post, 
			 String arg_postMemory, Method arg_execute, Method arg_successTest, 
			 Method arg_successTestSensorFactory, AblNamedPropertySupport arg_propertyTable, 
			 String arg_signature, String[] arg_stepsIConflictWith, boolean arg_teamNeededForSuccess)
    {
	super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds, arg_persistentWhenFails, arg_ignoreFailure, 
	      arg_effectOnly, arg_teamEffectOnly, arg_priority, arg_priorityModifier, arg_post, arg_postMemory, 
	      arg_execute, arg_successTest, arg_successTestSensorFactory, arg_propertyTable, arg_signature, 
	      arg_stepsIConflictWith);
	teamNeededForSuccess = arg_teamNeededForSuccess;
    }

    public JointGoalStep(int stepID, Behavior arg_parent, boolean arg_persistent, boolean arg_persistentWhenSucceeds, 
			 boolean arg_persistentWhenFails, boolean arg_ignoreFailure, boolean arg_effectOnly, 
			 boolean arg_teamEffectOnly, short arg_priority, short arg_priorityModifier, boolean arg_post, 
			 String arg_postMemory, Method arg_execute, Method arg_successTest, 
			 Method arg_successTestSensorFactory, AblNamedPropertySupport arg_propertyTable, 
			 String arg_signature, String[] arg_stepsIConflictWith, short arg_goalType, 
			 boolean arg_teamNeededForSuccess)
    {
	super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds, arg_persistentWhenFails, arg_ignoreFailure, 
	      arg_effectOnly, arg_teamEffectOnly, arg_priority, arg_priorityModifier, arg_post, arg_postMemory, 
	      arg_execute, arg_successTest, arg_successTestSensorFactory, arg_propertyTable, arg_signature, 
	      arg_stepsIConflictWith, arg_goalType);
	teamNeededForSuccess = arg_teamNeededForSuccess;
    }

    protected JointGoalNegotiator getNewJointGoalNegotiator(Hashtable commitSet) 
    {
	return new JointGoalNegotiator(commitSet);
    }

    protected JointGoalNegotiator getNewJointGoalNegotiator(Set teamMembers)
    {
	return new JointGoalNegotiator(teamMembers, this);
    }

    protected JointGoalNegotiator getNewJointGoalNegotiator(Hashtable commitSet, int state)
    {
	return new JointGoalNegotiator(commitSet, state);
    }

    protected JointGoalNegotiator getNewJointGoalNegotiator(Set teamMembers, boolean newEntryNegotiation)
    {
	return new JointGoalNegotiator(teamMembers, newEntryNegotiation, this);
    }

    // returns true for all concrete joint goal steps
    boolean isJointGoal() { return true; }

    // Public accessor for team members (BehavingEntities) participating in this joint goal.
    Set getTeamMembers() 
    { 
	if (negotiator == null) 
	    return null;
	else 
	    return negotiator.getTeamMembers(); 
    }

    synchronized boolean getIsNegotiating() { return isNegotiating; }
    synchronized void setIsNegotiating(boolean b) { isNegotiating = b; }

    boolean getTeamNeededForSuccess() { return teamNeededForSuccess; }

    void resetNegotiator(Hashtable commitSet, int state)
    {
	negotiator = getNewJointGoalNegotiator(commitSet, state);
    }

    // Version of chooseBehavior for joint goals.
    void chooseBehavior(final Object[] args) 
    {
	final Behavior beh = BehavingEntity.getBehavingEntity().chooseJointBehavior(args, failedBehaviors, this);
	final JointGoalNegotiationThread negotiateEntryThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateEntryThread") {
		    public void run() 
		    {
			// blocks on negotiatingEntry
			// the negotiator stores the current team members
			negotiator = getNewJointGoalNegotiator(beh.getTeamMembers(), true);
			Hashtable commitSet = BehavingEntity.getBehavingEntity().negotiateEntry(JointGoalStep.this, args);
			if (commitSet != null) {
			    // entry successfully negotiated

			    addChild((Behavior)beh);
			    executeBookkeeping();
			}
			else {
			    // wasn't able to enter - update goal state
			    addFailedBehavior((Behavior)beh);
			    negotiator = null;
			}
			setIsNegotiating(false);
		    }
		};

	final JointGoalNegotiationThread negotiateFailureOfOtherGoalsThread =
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateFailureOfOtherGoalsThread") {
			public void run() 
			{
			    // blocks on negotiating failure			    
			    if (negotiator != null) { // step might have been removed before this got a chance 
				negotiator.negotiateFailure();
			    
				// Once we're at this point other entities in the team have failed their joint goal.
				// We don't fail our goal though, we just remove the subtree rooted at it.
			    
				// Remove the tree rooted at this goal. 
				removeChild(true); 
				
				// Even though negotiation is finished, other negotiations associated with this thread
				// need the negotiator to queue negotiations on BehavingEntity.
				// negotiator = null;
				
				// negotiate entry with the new team members
				// wait to start until the next decision cycle
				BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateEntryThread);
			    }
			}
	    };
	
	assert (!getIsNegotiating());
	if (beh != null) {
	    /* If a behavior was found, add it as a child of this
               goal. addChild() also adds the children (steps) of the
               behavior as naked behaviors (with no steps) are never
               found in the ABT. */

	    if (negotiator == null) {
		// first time a behavior has been chosen for this goal step - initiate negotiation on BehavingEntity

		setIsNegotiating(true);
		BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateEntryThread);
	    }
	    else if (!negotiator.getTeamMembers().equals(beh.getTeamMembers())) {
		// The selected behavior has a different set of team members than the team members participating in this
		// goal. Negotiate exit (remove) from the current goal team, then negotiate entry.
		
		setIsNegotiating(true);
		BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateFailureOfOtherGoalsThread);
	    }
	    else {
		// The choosen behavior has the same team members as the joint goal.
		// Continue normally.
		addChild((Behavior)beh);
		executeBookkeeping();		
	    }
	}
	else 
	    // No behavior found. 
	    failStep();
    }

    // Given a list of joint goals, blocks until all the goals have commited to exit.
    // Must be called from within a JointGoalNegotiationThread.
    static void blockOnSubtreeNegotiation(List jointGoals)
    {
	assert (Thread.currentThread() != BehavingEntity.getBehavingEntity().getDecisionCycleThread());

	Iterator iter = jointGoals.iterator();
	while (iter.hasNext()) {
	    JointGoalStep js = (JointGoalStep)iter.next();
	    if ((js.negotiator != null) &&
		(js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_SUCCEED) &&
		(js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_FAIL) &&
		(js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_REMOVE) &&
		(js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_SUSPEND)) {
		
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    }

	    // fixme: The only situation in which the negotiator is nulled now is failure to find a behavior. May be able
	    // to revert to the old code and assert that the negotiator should not be null. 
	}
    }

    // Negotiate success of JointGoalSteps
    void succeedStep()
    {
	if (negotiator == null)
	    // Goal was never entered jointly. Do a normal step succeed.
	    super.succeedStep();
	else if (negotiator.getState() == JointGoalNegotiator.COMMIT_TO_SUCCEED)
	    // Already negotiated success - succeedStep() is being called by the negotiator from processIntentionToSucceed
	    super.succeedStep();
	else {
	    // Goal was entered jointly. Negotiate success.

	    final List jointGoals = freezeSubtreeAndNegotiateRemovalEntry();

	    final JointGoalNegotiationThread negotiateSuccessThread = 
		new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateSuccessThread") {
			public void run()
			{
			    setIsNegotiating(true);
			    
			    // blocks on subtree negotiation
			    blockOnSubtreeNegotiation(jointGoals);
			    
			    if (negotiator != null) { // step might have been removed before this got a chance 
				// blocks on negotiating success
				if (negotiator.getState() == JointGoalNegotiator.WAIT_FOR_SUCCESS)
				    negotiator.negotiateSuccess(); // another thread will actually succeed the step 
				else if (negotiator.negotiateSuccess()) {
				    succeedStepSuper();
				    				    
				    // Even though the step is no longer valid - set it's status to false anyway.
				    // Since the Debugger uses this value, if it is not set it is possible in the event of 
				    // a negotiation bug for the Debugger to falsely indicate that a step is still 
				    // negotiating
				    setIsNegotiating(false); 
				}
			    }
			}
		    };
	    
	    BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateSuccessThread);
	}
    }
    
    void failStep() {
	Trace.ablTrace("(JointGoalStep)" + this.getSignature() + "failStep()");

	if (negotiator == null)
	    // Goal was never entered jointly. Do a normal step fail.
	    super.failStep(); 
	else if (negotiator.getState() == JointGoalNegotiator.COMMIT_TO_FAIL)
	    // Already negotiated failure - failStep() is being called by the negotiator from processIntentionToFail.
	    super.failStep();
	else {
	    // Goal was entered jointly. Negotiate failure.
	    final List jointGoals = freezeSubtreeAndNegotiateRemovalEntry();

	    final JointGoalNegotiationThread negotiateFailureThread =
		new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateFailureThread") {		
			public void run() 
			{
			    setIsNegotiating(true);

			    // blocks on subtree negotiation
			    blockOnSubtreeNegotiation(jointGoals);

			    if (negotiator != null) { // step might have been removed before this got a chance 
				// blocks on negotiating failure
				if (negotiator.negotiateFailure()) {
				    failStepSuper();
				    
				    // Even though the step is no longer valid - set it's status to false anyway.
				    // Since the Debugger uses this value, if it is not set it is possible in the event of 
				    // a negotiation bug for the Debugger to falsely indicate that a step is still 
				    // negotiating
				    setIsNegotiating(false);
				}
			    }
			}
		    };

	    BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateFailureThread);
	}
    }

    private void negotiateSuspend(final ExecutableStep step)
    {
	final List jointGoals = negotiateSuspensionOfSubtreeEntry(step);

	final JointGoalNegotiationThread negotiateSuspensionThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateSuspensionThread") {	    
		    public void run() 
		    {
			setIsNegotiating(true);

			// blocks on subtree negotiation
			blockOnSubtreeNegotiation(jointGoals);

			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating suspension
			    if (negotiator.negotiateSuspend()) {
				if (step != null)
				    suspendSuper(step, false);
				else
				    metaSuspendSuper(false);
				// fixme remove?
				/* if (step != null)
				   suspendSkipJointGoals(step); // since joint goals have already negotiated this suspend, skip them
				   else
				   metaSuspendSkipJointGoals(); // since joint goals have already negotiated this metasuspend, skip them */
				negotiator.setState(JointGoalNegotiator.SUSPENDED);
				setIsNegotiating(false);
			    }
			}
		    }
		};

	BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateSuspensionThread);
    }

    // Negotiate suspend of JointGoalSteps
    void suspend(final ExecutableStep step) 
    { 
	if (negotiator == null)
	    // Goal was never entered jointly. Do a normal step suspend.
	    super.suspend(step);
	else 
	    // Goal was entered jointly. Negotiate suspension.
	    negotiateSuspend(step);	    
    }

    // Negotiated metaSuspend of JointGoalSteps
    void metaSuspend()
    {
	if (negotiator == null)
	    // Goal was never entered jointly. Do a normal step suspend.
	    super.metaSuspend();
	else 
	    // Goal was entered jointly. Negotiate suspension.
	    negotiateSuspend(null);
    }

    // negotiateUnsuspend() *doesn't* first negotiate the unsuspension of the subtree. Unsuspension is not a goal exit (like suspension or succeed)
    // nor is it a pure enter. 
    private void negotiateUnsuspend(final ExecutableStep step)
    {
	final JointGoalNegotiationThread negotiateUnsuspensionThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateUnsuspensionThread") {
		    public void run() 
		    {
			setIsNegotiating(true);

			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating unsuspension
			    if (negotiator.negotiateUnsuspend()) {
				if (step != null)
				    unsuspendSuper(step); 
				else
				    metaUnsuspendSuper(); 
				
				setIsNegotiating(false);
				
				if (isSuspended())
				    negotiator.setState(JointGoalNegotiator.SUSPENDED);
				else
				    negotiator.setState(JointGoalNegotiator.RUNNING);
			    }
			}
		    }
		};

	BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateUnsuspensionThread);
    }
  
    // fixme: defined because JointGoalStep.super<method> doesn't seem to work inside an inner class.
    private void unsuspendSuper(ExecutableStep step) { super.unsuspend(step); }
    private void metaUnsuspendSuper() { super.metaUnsuspend(); }
    private void jointUnsuspendSuper() { super.jointUnsuspend(); }
    private void suspendSuper(ExecutableStep step, boolean recurse) { super.suspend(step, recurse); }
    private void metaSuspendSuper(boolean recurse) { super.metaSuspend(recurse); }
    private void jointSuspendSuper() { super.jointSuspend(); }
    private void failStepSuper() { super.failStep(); }
    private void succeedStepSuper() { super.succeedStep(); }

    // Negotiate unsuspend of JointGoalSteps
    void unsuspend(ExecutableStep step) 
    { 
	if (negotiator != null) {
	    // Block on negotiation of unsuspend.
	    negotiateUnsuspend(step);
	}
	else
	    super.unsuspend(step);
    }

    // Negotiated metaUnsuspend of JointGoalSteps
    void metaUnsuspend()
    {
	if (negotiator != null)
	    negotiateUnsuspend(null);
	else
	    super.metaUnsuspend();
    }

    // Called by negotiator which has already negotiated unsuspend for this goal.
    void jointUnsuspendEntry() { super.jointUnsuspend(); }

    // Called during recursion of jointUnsuspend().
    void jointUnsuspend()
    {
	final JointGoalNegotiationThread jointUnsuspendThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " jointUnsuspendThread") {
		    public void run()
		    {
			setIsNegotiating(true);
			
			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating unsuspension
			    if (negotiator.negotiateUnsuspend()) {
				jointUnsuspendSuper();
				
				setIsNegotiating(false);
				
				if (isSuspended())
				    negotiator.setState(JointGoalNegotiator.SUSPENDED);
				else
				    negotiator.setState(JointGoalNegotiator.RUNNING);
			    }
			}
		    }
		};
			
	if (negotiator != null) {
	    BehavingEntity.getBehavingEntity().registerNegotiationThread(jointUnsuspendThread);
	}
	else
	    super.jointUnsuspend();
    }

    // Negotiated reset of JointGoalSteps.
    void resetStep()
    {
	// fixme: for now just doing a normal (unnegotiated) reset
	super.resetStep();
    }

    // fixme: remove
    // Since the negotiator is no longer set to null when the goal is removed, calls to removeGoal can directly
    // go to super.removeGoal().
    /* void removeGoal()
    {
	Trace.ablTrace("(JointGoalStep)" + this.getSignature() + "removeGoal()");

	// fixme: don't set negotiator to null
	// negotiator = null;
	super.removeGoal();
	}*/

    void cleanupParentIfRoot()
    {
	// If the parent is the root collection behavior, then take care of removing this step
	// since the root behavior never succeeds, fails, or is removed. 
	if (parent.isRootBehavior()) 
	    parent.removeChild(this, true);
    }

    // Entry point for returning a list of joint goals in the subtree. Doesn't return this joint goal in the list.
    List freezeSubtreeAndNegotiateRemovalEntry()
    {
	return super.freezeSubtreeAndNegotiateRemoval();
    }

    // Returns a list of the joint goals in the subtree.
    List freezeSubtreeAndNegotiateRemoval()
    {
	final JointGoalNegotiationThread negotiateRemovalThread =
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateRemovalThread") {
		    public void run() 
		    {
			setIsNegotiating(true);

			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating removal
			    if (negotiator.negotiateRemoval()) {
				removeGoal();
				
				// Even though the step is no longer valid - set it's status to false anyway.
				// Since the Debugger uses this value, if it is not set it is possible in the event of 
				// a negotiation bug for the Debugger to falsely indicate that a step is still 
				// negotiating
				setIsNegotiating(false);
			    }
			}
		    }
		};

	// No need to assert (negotiator != null && child != null) || (negotiator == null && child == null)
	// negotiator != null && child == null can be true if a subtree containing a joint goal in the middle of entry negotiation is frozen. 
	// However, it is the case that negotiator == null and child != null can never be true. 

	assert child == null || negotiator != null: "step = " + this + " negotiator = " + negotiator + ", child = " + child;

	if (negotiator != null) {
	    // Entry into a joint behavior has been negotiated
	    BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateRemovalThread);

	    // return a list consisting of the joint goals in the subtree plus this joint goal
	    Vector v = new Vector(super.freezeSubtreeAndNegotiateRemoval());
	    v.add(this);
	    return v;
	}
	else {
	    return new Vector(0); // negotiator == null meaning entry has not been negotiated
	}
    }

    // Called during suspension initiated by team member.
    List negotiateSuspensionOfSubtreeEntry() 
    {
	if (child != null)
	    return child.negotiateSuspensionOfSubtree();
	else
	    return new Vector(0); // return an empty list
    }

    // Called during suspension initiated by team member.
    List negotiateSuspensionOfSubtree()
    {
	final JointGoalNegotiationThread negotiateSuspendThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateSuspendThread") {
		    public void run()
		    {
			setIsNegotiating(true);
			
			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating suspend
			    if (negotiator.negotiateSuspend()) {
				jointSuspendSuper();
			    }
			    setIsNegotiating(false);
			}
		    }
		};

	// negotiate suspend of this JointGoalStep
	BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateSuspendThread);

	// return a list consisting of the joint goals in the subtree plus this joint goal
	Vector v = new Vector(negotiateSuspensionOfSubtreeEntry());
	v.add(this);
	return v;
    }


    // Called during suspension initiated by team member.
    List negotiateSuspensionOfSubtreeEntry(final ExecutableStep step) 
    { 
	if (child != null)
	    return child.negotiateSuspensionOfSubtree(step);
	else
	    return new Vector(0); // return an empty list
    }

    // Called during suspension initiated by team member.
    List negotiateSuspensionOfSubtree(final ExecutableStep step)
    {
	final JointGoalNegotiationThread negotiateSuspendThread = 
	    new JointGoalNegotiationThread(JointGoalStep.this.negotiator, JointGoalStep.this + " negotiateSuspendThread") {
		    public void run()
		    {
			setIsNegotiating(true);
			
			if (negotiator != null) { // step might have been removed before this got a chance 
			    // blocks on negotiating suspend
			    if (negotiator.negotiateSuspend()) {
				if (step != null)
				    suspendSuper(step, false);
				else 
				    metaSuspendSuper(false);
				
				setIsNegotiating(false);
			    }
			}
		    }
		};

	// negotiate suspend of this JointGoalStep
	BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateSuspendThread);

	// return a list consisting of the joint goals in the subtree plus this joint goal
	Vector v = new Vector(negotiateSuspensionOfSubtreeEntry(step));
	v.add(this);
	return v;
    }

}

