// Provides debug negotiation support for JointBehaviors.

package abl.runtime;

import abl.compiler.AblDebuggerConstants;
import java.util.*;
import java.io.PrintStream;

class JointGoalNegotiatorDebug extends JointGoalNegotiator implements AblDebuggerConstants
{
    private final byte debugLevel;
    private final List negotiationEvents = new ArrayList(20);

    class JointGoalNegotiationInfo
    {
	final JointGoalStep g;
	final BehavingEntity behavingEntity;
	final int intention;

	JointGoalNegotiationInfo(JointGoalStep g, BehavingEntity behavingEntity, int intention)
	{
	    this.g = g;
	    this.behavingEntity = behavingEntity;
	    this.intention = intention;
	}
    }

    class JointGoalNegotiationEvent
    {
	String negotiation; 
	JointGoalNegotiationInfo info;
	int currentState;

	JointGoalNegotiationEvent(String negotiation, JointGoalNegotiationInfo info, int currentState)
	{
	    this.negotiation = negotiation;
	    this.info = info;
	    this.currentState = currentState;
	}

	public String toString() 
	{
	    if (info != null) {
	    String entity;
	    if (info.behavingEntity != null)
		entity = info.behavingEntity.getBehavingEntityShortName();
	    else 
		entity = "self";
	    return negotiation + " from: " + entity + ", intention: " + formatIntention(info.intention) +
		", state: " + formatState(currentState);
	    }
	    else {
		return "state changed by " + negotiation + " to: " + formatState(currentState);
	    }
		
	}
    }
    
    JointGoalNegotiatorDebug(Set teamMembers, boolean isNewEntryNegotiation, JointGoalStep g, byte arg_debugLevel)
    {
	super(teamMembers, isNewEntryNegotiation, g);
	debugLevel = arg_debugLevel;
    }

    JointGoalNegotiatorDebug(Hashtable commitSet, byte arg_debugLevel)
    {
	super(commitSet);
	debugLevel = arg_debugLevel;
    }

    JointGoalNegotiatorDebug(Hashtable commitSet, int currentState, byte arg_debugLevel)
    {
	super(commitSet, currentState);
	debugLevel = arg_debugLevel;
    }

    JointGoalNegotiatorDebug(Set teamMembers, JointGoalStep g, byte arg_debugLevel)
    { 
	super(teamMembers, g);
	debugLevel = arg_debugLevel;
    }

    protected synchronized void setState(int state) 
    {
	JointGoalNegotiationThread t = (JointGoalNegotiationThread)Thread.currentThread();
	negotiationEvents.add(new JointGoalNegotiationEvent(t.descriptorString, null, state));
	super.setState(state);
    }

    // fixme: the negotiating goal could be an initiated joint goal step - need to check
    // Debug action for initiation of negotiation.
    final protected void initiateNegotiationAction(JointGoalStep g, int negotiation)
    {
	JointGoalNegotiationInfo info = new JointGoalNegotiationInfo(g, null, negotiation);
	if (debugLevel == GUI_DEBUGGER)
	    ((DebuggableJointGoalStep)negotiatingGoal).traceAblNegotiationEvent(AblEvent.INITIATE_NEGOTIATION, info);
	negotiationEvents.add(new JointGoalNegotiationEvent("INITIATE_NEGOTIATION", info, getState()));
    }

    // Debug action for completion of negotiation.
    final protected void completeNegotiationAction(JointGoalStep g, int negotiation)
    {
	JointGoalNegotiationInfo info = new JointGoalNegotiationInfo(g, null, negotiation);
	if (debugLevel == GUI_DEBUGGER)
	    ((DebuggableJointGoalStep)negotiatingGoal).traceAblNegotiationEvent(AblEvent.COMPLETE_NEGOTIATION, info);
	negotiationEvents.add(new JointGoalNegotiationEvent("COMPLETE_NEGOTIATION", info, getState()));
    }

    // Debug action for initiaton of intention.
    final protected void initiateIntentionAction(BehavingEntity sender, JointGoalStep receivingGoal, int intention) 
    {
	JointGoalNegotiationInfo info = new JointGoalNegotiationInfo(receivingGoal, sender, intention);
	if (debugLevel == GUI_DEBUGGER)
	    ((DebuggableJointGoalStep)negotiatingGoal).traceAblNegotiationEvent(AblEvent.INITIATE_INTENTION, info);
	negotiationEvents.add(new JointGoalNegotiationEvent("INITIATE_INTENTION", info, getState()));
    }

    // Debug action for commitment to intention.
    final protected void commitToIntentionAction(JointGoalStep g, int intention)
    {
	JointGoalNegotiationInfo info = new JointGoalNegotiationInfo(g, null, intention);
	if (debugLevel == GUI_DEBUGGER)
	    ((DebuggableJointGoalStep)negotiatingGoal).traceAblNegotiationEvent(AblEvent.COMMIT_TO_INTENTION, info);
	negotiationEvents.add(new JointGoalNegotiationEvent("COMMIT_TO_INTENTION", info, getState()));
    }

    // Debug action for processing an intention.
    final protected void processIntentionAction(BehavingEntity sender, JointGoalStep receivingGoal, int intention) 
    {
	JointGoalNegotiationInfo info = new JointGoalNegotiationInfo(receivingGoal, sender, intention);
	if (debugLevel == GUI_DEBUGGER)
	    ((DebuggableJointGoalStep)negotiatingGoal).traceAblNegotiationEvent(AblEvent.PROCESS_INTENTION, info);
	negotiationEvents.add(new JointGoalNegotiationEvent("PROCESS_INTENTION", info, getState()));
    }
   
    void printNegotiationHistory(PrintStream s)
    {
	Iterator iter = negotiationEvents.iterator();
	s.println("Negotiation history: " + " for " + negotiatingGoal + ":");
	s.println("   Current state: " + formatState(getState()));
	while(iter.hasNext()) {
	    JointGoalNegotiationEvent evt = (JointGoalNegotiationEvent)iter.next();
	    s.println("   " + evt);
	}
    }

    final void printNegotiationHistory()
    {
	printNegotiationHistory(System.err);
    }

}


