// Provides negotiation support for JointBehaivors.

package abl.runtime;

import java.util.*;
import debug.Trace;
import java.io.PrintStream;

public class JointGoalNegotiator {

    // current state machine state
    protected int currentState;

    protected final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
    protected JointGoalStep negotiatingGoal;

    // invalid state - used by JointNodeNegotiationThread to mark an invalid continuation condition
    static final int INVALID = -3;

    static final int FIRST_STATE = -1;
    static final int START = FIRST_STATE;
    static final int INTENTION_TO_ENTER = 0;
    static final int INTENTION_TO_REFUSE_ENTRY = INTENTION_TO_ENTER + 1;
    static final int COMMIT_TO_ENTER = INTENTION_TO_REFUSE_ENTRY + 1;
    static final int COMMIT_TO_REFUSE_ENTRY = COMMIT_TO_ENTER + 1;
    static final int RUNNING = COMMIT_TO_REFUSE_ENTRY + 1;
    static final int INTENTION_TO_SUCCEED = RUNNING + 1;
    static final int WAIT_FOR_SUCCESS = INTENTION_TO_SUCCEED + 1;
    static final int COMMIT_TO_SUCCEED = WAIT_FOR_SUCCESS + 1;
    static final int INTENTION_TO_FAIL = COMMIT_TO_SUCCEED + 1;
    static final int COMMIT_TO_FAIL = INTENTION_TO_FAIL + 1;
    static final int INTENTION_TO_SUSPEND = COMMIT_TO_FAIL + 1;
    static final int COMMIT_TO_SUSPEND = INTENTION_TO_SUSPEND + 1;
    static final int SUSPENDED = COMMIT_TO_SUSPEND + 1;
    static final int INTENTION_TO_UNSUSPEND = SUSPENDED + 1;
    static final int COMMIT_TO_UNSUSPEND = INTENTION_TO_UNSUSPEND + 1;
    static final int INTENTION_TO_REMOVE = COMMIT_TO_UNSUSPEND + 1;
    static final int COMMIT_TO_REMOVE = INTENTION_TO_REMOVE + 1;
    static final int LAST_STATE = COMMIT_TO_REMOVE;

    // constants indicating entry intentions
    static final int FIRST_ENTRY_INTENTION = LAST_STATE + 1;
    static final int ENTER = FIRST_ENTRY_INTENTION;
    static final int REFUSE_ENTRY = ENTER + 1;
    static final int LAST_ENTRY_INTENTION = REFUSE_ENTRY;

    // constants indicating exit intentions to broadcast
    static final int FIRST_EXIT_INTENTION = LAST_ENTRY_INTENTION + 1;
    static final int SUCCEED = FIRST_EXIT_INTENTION;
    static final int FAIL = SUCCEED + 1;
    static final int REMOVE = FAIL + 1;
    static final int SUSPEND = REMOVE + 1;
    static final int UNSUSPEND = SUSPEND + 1;
    static final int LAST_EXIT_INTENTION = UNSUSPEND;

    // table of commited team members - 
    // keys are team member names, values are the joint behavior associated with the team member
    protected Hashtable runningCommitSet = new Hashtable();

    // Set containing the entities currently intending the negotiated state change.
    // Entry negotiation uses runningCommitSet as its negotiating commit set.
    protected Set negotiatingCommitSet = new HashSet();

    final protected Set teamMembers;
    
    // counter used to generate a unique number for each entry negotiation instance
    private static long uniqueEntryNegotiationID = 0;
    
    // reference to the goal step generated by an intention to enter joint behavior
    // when a commitment to enter the behavior is achieved, the escroed goal step is attached to the ABT
    protected JointGoalStep escroedGoalStep = null;

    private boolean teamMembersAreBehavingEntities(Set teamMembers)
    {
	Iterator iter = teamMembers.iterator();
	while(iter.hasNext())
	    try {
		BehavingEntity entity = (BehavingEntity)iter.next();
	    } catch (ClassCastException e) {
		return false;
	    }

	return true;
    }

    JointGoalNegotiator(Set teamMembers, JointGoalStep negotiatingGoal)
    { 
	assert (teamMembersAreBehavingEntities(teamMembers));

	this.teamMembers = new HashSet(teamMembers);
	currentState = START;
	this.negotiatingGoal = negotiatingGoal;
    }

    JointGoalNegotiator(Set teamMembers, boolean isNewEntryNegotiation, JointGoalStep negotiatingGoal)
    {
	assert (teamMembersAreBehavingEntities(teamMembers));

	this.teamMembers = new HashSet(teamMembers);
	currentState = START;
	if (isNewEntryNegotiation)
	    uniqueEntryNegotiationID++;
	this.negotiatingGoal = negotiatingGoal;
    }

    JointGoalNegotiator(Hashtable commitSet)
    {
	teamMembers = new HashSet(commitSet.keySet());
	assert (teamMembersAreBehavingEntities(teamMembers));
	this.runningCommitSet = new Hashtable(commitSet);
	currentState = START;
	negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);
	assert (negotiatingGoal != null);
    }

    JointGoalNegotiator(Hashtable commitSet, int currentState)
    {
	teamMembers = new HashSet(commitSet.keySet());
	assert (teamMembersAreBehavingEntities(teamMembers));
	this.runningCommitSet = new Hashtable(commitSet);
	assert isLegalState(currentState): Integer.toString(currentState);
	this.currentState = currentState;
	negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);
	assert (negotiatingGoal != null);
    }

    public static String formatIntention(int intention) 
    {
	assert (intention >= FIRST_ENTRY_INTENTION && intention <= LAST_EXIT_INTENTION);
	switch(intention) 
	    {
	    case ENTER: return "ENTER";
	    case REFUSE_ENTRY: return "REFUSE_ENTRY";
	    case SUCCEED: return "SUCCEED";
	    case FAIL: return "FAIL";
	    case REMOVE: return "REMOVE";
	    case SUSPEND: return "SUSPEND";
	    case UNSUSPEND: return "UNSUSPEND";
	    default: return "NOT_INTENTION";
	    }
    }

    public static String formatState(int state)
    {
	assert (state >= FIRST_STATE && state <= LAST_STATE);
	switch(state) {
	case START: return "START";
	case INTENTION_TO_ENTER: return "INTENTION_TO_ENTER";
	case INTENTION_TO_REFUSE_ENTRY: return "INTENTION_TO_REFUSE_ENTRY";
	case COMMIT_TO_ENTER: return "COMMIT_TO_ENTER";
	case COMMIT_TO_REFUSE_ENTRY: return "COMMIT_TO_REFUSE_ENTRY";
	case RUNNING: return "RUNNING";
	case INTENTION_TO_SUCCEED: return "INTENTION_TO_SUCCEED";
	case WAIT_FOR_SUCCESS: return "WAIT_FOR_SUCCESS";
	case COMMIT_TO_SUCCEED: return "COMMIT_TO_SUCCEED";
	case INTENTION_TO_FAIL: return "INTENTION_TO_FAIL";
	case COMMIT_TO_FAIL: return "COMMIT_TO_FAIL";
	case INTENTION_TO_SUSPEND: return "INTENTION_TO_SUSPEND";
	case COMMIT_TO_SUSPEND: return "COMMIT_TO_SUSPEND";
	case SUSPENDED: return "SUSPENDED";
	case INTENTION_TO_UNSUSPEND: return "INTENTION_TO_UNSUSPEND";
	case COMMIT_TO_UNSUSPEND: return "COMMIT_TO_UNSUSPEND";
	case INTENTION_TO_REMOVE: return "INTENTION_TO_REMOVE";
	case COMMIT_TO_REMOVE: return "COMMIT_TO_REMOVE";
	default: return "INVALID";
	}
    }

    Long getUniqueEntryNegotiationID() { return new Long(uniqueEntryNegotiationID); }

    synchronized int getState() { return currentState; }
    
    protected synchronized void setState(int state)
    {
	assert (state >= FIRST_STATE && state <= LAST_STATE);
	currentState = state;

	if (state == COMMIT_TO_ENTER ||
	    state == COMMIT_TO_REFUSE_ENTRY ||
	    state == COMMIT_TO_SUCCEED ||
	    state == COMMIT_TO_FAIL ||
	    state == COMMIT_TO_SUSPEND ||
	    state == COMMIT_TO_UNSUSPEND ||
	    state == COMMIT_TO_REMOVE) {
	    // entering a commit state - let the BehavingEntity know
	    
	    BehavingEntity.getBehavingEntity().jointGoalNegotiatorCommitted();
	}
    }

    Set getTeamMembers() { return new HashSet(teamMembers); }

    
    void setCommitSet(Hashtable commitSet) 
    { 
	this.runningCommitSet = new Hashtable(commitSet); 
	negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);
	assert (negotiatingGoal != null);
    } 

    Hashtable getCommitSet() { return new Hashtable(runningCommitSet); }

    // returns true if the argument is a legal negotitation state, false otherwise
    static boolean isLegalState(int state)
    {
	if ( (state >= FIRST_STATE) &&
	     (state <= LAST_STATE) ) 
	    return true;
	else
	    return false;
    }

    // Overriden on subclasses to take an action on initiation of entry negotiation.
    // fixme: provide special callbacks for initiating entry
    // protected void initiateEntryNegotiationAction(JointGoalStep g, Long uniqueNegotiationID);

    // Overriden on subclasses to take an action on initiation of negotiation.
    protected void initiateNegotiationAction(JointGoalStep g, int negotiation) {}

    // Overridden on subclasses to take an action on commitment of entry negotiation.
    // fixme: provide special callbacks for initiating entry
    // protected void completeEntryNegotiationAction(JointGoalStep g, Long uniqueNegotiationID);

    // Overridden on subclasses to take an action on commitment of negotiation.
    protected void completeNegotiationAction(JointGoalStep g, int negotiation) {}

    // Overridden on subclasses to take an action when initiating an intention.
    protected void initiateIntentionAction(BehavingEntity sender, JointGoalStep negotiatingGoal, int intention) {}

    // Overridden on subclasses to take an action when commiting to an intention.
    protected void commitToIntentionAction(JointGoalStep g, int intention) {}

    // Overridden on subclasses to take an action when processing an intention.
    protected void processIntentionAction(BehavingEntity sender, JointGoalStep negotiatingGoal, int intention) {}

    // Overridden on subclasses (used for debug) to print negotiation history
    void printNegotiationHistory() {}

    // Overriden on subclasses (used for debug) to get the negotiation history as a string
    void printNegotiationHistory(PrintStream p) {}

    private int convertCommitStateToIntention(int commitState)
    {
	switch (commitState) {
	case COMMIT_TO_ENTER:
	    return ENTER;
	case COMMIT_TO_REFUSE_ENTRY:
	    return REFUSE_ENTRY;
	case COMMIT_TO_SUCCEED:
	    return SUCCEED;
	case COMMIT_TO_FAIL:
	    return FAIL;
	case COMMIT_TO_SUSPEND:
	case SUSPENDED:
	    return SUSPEND;
	case COMMIT_TO_UNSUSPEND:
	case RUNNING:
	    return UNSUSPEND;
	case COMMIT_TO_REMOVE:
	    return REMOVE;
	default:
	    return LAST_EXIT_INTENTION + 1;
	}
    }

    // Negotiate entry with all team members.
    // Blocks on achievement of COMMIT_TO_ENTER or COMMIT_TO_REFUSE_ENTRY.
    // Returns the commit set if COMMIT_TO_ENTER is retrieved, otherwise returns null.
    public Hashtable negotiateEntry(Object[] args, JointGoalStep initiatingGoal)
    {	 
	//Trace.trace(BehavingEntity.getBehavingEntity(), 
	//    "Negotiating entry for " + initiatingGoal + ", teamNeededForSuccess = " + initiatingGoal.getTeamNeededForSuccess());
	Trace.ablTrace("Negotiating entry for " + initiatingGoal + ", teamNeededForSuccess = " + initiatingGoal.getTeamNeededForSuccess());

	if (getState() != START) { 
	    throw new UnexpectedStateError(getState()); 
	}

	setState(INTENTION_TO_ENTER);
	Long uniqueID = new Long(uniqueEntryNegotiationID);
	escroedGoalStep = initiatingGoal; // set escroed goal to initiating goal for nicer debug printouts
	initiateNegotiationAction(initiatingGoal, ENTER);

	runningCommitSet.put(BehavingEntity.getBehavingEntity(), initiatingGoal);

	broadcastIntentionToEnter(uniqueID, args, initiatingGoal.getTeamNeededForSuccess(), initiatingGoal);

	// wait for COMMIT_TO_ENTER or COMMIT_TO_REFUSE_ENTRY to be achieved
	while( (getState() != COMMIT_TO_ENTER) &&
	       (getState() != COMMIT_TO_REFUSE_ENTRY) ) {
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	}

	// Trace.trace(BehavingEntity.getBehavingEntity(), "Entry negotiated");
	Trace.ablTrace("Entry negotiated");

	if (getState() == COMMIT_TO_ENTER)
	    completeNegotiationAction(initiatingGoal, ENTER);
	else
	    completeNegotiationAction(initiatingGoal, REFUSE_ENTRY);

	// when we get here, we've either commited to enter or commited to refuse entry
	if (getState() == COMMIT_TO_ENTER) {
	    setState(RUNNING);
	    return getCommitSet();
	}
	else if (getState() == COMMIT_TO_REFUSE_ENTRY) {
	    return null;
	}
	else
	    throw new UnexpectedStateError(getState());
    }

    // Process intention to enter.
    // From the start state, blocks on achievement of COMMIT_TO_ENTER or COMMIT_TO_REFUSE_ENTRY.
    public void processIntentionToEnter(Long uniqueEntryNegotiationID, BehavingEntity sender, Object[] args, boolean teamNeededForSuccess, 
					Set teamMembers, JointGoalStep sendingGoal, JointGoalStep escroedGoalStep)
    {
	BehavingEntity myBehavingEntity = BehavingEntity.getBehavingEntity();

	Trace.ablTrace("Process intention to enter in " + myBehavingEntity + " for goal " + sendingGoal);

	processIntentionAction(sender, sendingGoal, ENTER);
	
	if (getState() == START) {
	    // Beginning of the negotiation.
	    // Could only be in this state if we are not the initator of the negotation. 

	    Behavior newBeh = BehavingEntity.getBehavingEntity().chooseJointBehavior(args, new HashSet(), escroedGoalStep, teamMembers);
	    if (newBeh != null) {
		// found a joint behavior

		setState(INTENTION_TO_ENTER);
		initiateIntentionAction(sender, escroedGoalStep, ENTER);

		runningCommitSet.put(sender, sendingGoal);
		runningCommitSet.put(myBehavingEntity, escroedGoalStep);
		// broadcast intention-to-enter
		broadcastIntentionToEnter(uniqueEntryNegotiationID, args, escroedGoalStep.getTeamNeededForSuccess(), escroedGoalStep);

		// Check if we are ready to commit to enter.
		// This can only happen from the start state if there are only two entities in the team
		if (runningCommitSet.keySet().equals(teamMembers)) {
		    // all team members has commited - terminate entry negotiation and add escroedGoalStep
		    // and selected behavior to tree

		    setState(COMMIT_TO_ENTER);
		    commitToIntentionAction(escroedGoalStep, ENTER);
		    myBehavingEntity.terminateEntryNegotiation(uniqueEntryNegotiationID);

		    // Adds the escroed goal to the root collection behavior and performs appropriate ABT update.
		    escroedGoalStep.resetNegotiator(runningCommitSet, RUNNING);
		    myBehavingEntity.getRootCollectionBehavior().addChild(escroedGoalStep);
		    escroedGoalStep.addChild((Behavior)newBeh);
		    escroedGoalStep.executeBookkeeping();
		}
		else {
		    // Not ready to enter yet - block on achievement of COMMIT_TO_ENTER or COMMIT_TO_REFUSE_ENTRY
		    // achieved).
		    while( (getState() != COMMIT_TO_ENTER) &&
			   (getState() != COMMIT_TO_REFUSE_ENTRY) )
			((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

		    commitToIntentionAction(escroedGoalStep, convertCommitStateToIntention(getState()));
		    myBehavingEntity.terminateEntryNegotiation(uniqueEntryNegotiationID);
		    if (getState() == COMMIT_TO_ENTER) { 

			// Adds the escroed goal to the root collection behavior and performs appropriate ABT update.
			escroedGoalStep.resetNegotiator(runningCommitSet, RUNNING);
			myBehavingEntity.getRootCollectionBehavior().addChild(escroedGoalStep); 
			escroedGoalStep.addChild((Behavior)newBeh);
			escroedGoalStep.executeBookkeeping();
		    }
		    else if (getState() == COMMIT_TO_REFUSE_ENTRY) {
			escroedGoalStep = null;
		    }
		    else 
			throw new UnexpectedStateError(getState());
		}
	    }
	    else {
		// didn't find a joint behavior - broadcast intention to refuse entry
		setState(INTENTION_TO_REFUSE_ENTRY);
		initiateIntentionAction(null, escroedGoalStep, REFUSE_ENTRY);
		runningCommitSet.put(myBehavingEntity, new Integer(0));
		// escroedGoalStep = null;
		
		broadcastIntentionToRefuseEntry(uniqueEntryNegotiationID);
		while(getState() != COMMIT_TO_REFUSE_ENTRY)
		    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

		commitToIntentionAction(escroedGoalStep, REFUSE_ENTRY);
		myBehavingEntity.terminateEntryNegotiation(uniqueEntryNegotiationID);
		escroedGoalStep = null;
		// when we get here, we've committed
	    }
	}
	else if (getState() == INTENTION_TO_ENTER) {
	    // Waiting to complete a negotiation (either in negotiateEntry or the START case of processIntentionToEnter).

	    if (runningCommitSet.get(sender) == null)
		runningCommitSet.put(sender, sendingGoal);

	    // If all team members have commited, COMMIT_TO_ENTER.
 	    // A waiting thread (either in negotiateEntry or the START case of processIntentionToEnter) will eventually
	    // pick this up.
	    if (runningCommitSet.keySet().equals(teamMembers)) 
		setState(COMMIT_TO_ENTER);
	}
	else if (getState() == INTENTION_TO_REFUSE_ENTRY) {

	    // remove sender from the commit set (we intend to refuse entry, and someone intends to enter)
	    runningCommitSet.remove(sender);
	    
	    // fixme: narrowcast an intention-to-refuse-entry to the sender - for now, broadcast to all team members
	    broadcastIntentionToRefuseEntry(uniqueEntryNegotiationID);
	}
	else
	    throw new UnexpectedStateError(getState());
    }

    // Process intention to refuse entry.
    // From the start state, blocks on achievement of COMMIT_TO_REFUSE_ENTRY
    public void processIntentionToRefuseEntry(Long uniqueEntryNegotiationID, BehavingEntity sender)
    {
	processIntentionAction(sender, escroedGoalStep, REFUSE_ENTRY);

	BehavingEntity receiver = BehavingEntity.getBehavingEntity();

	// fixme: this case won't work right now because teamMembers will not have been set yet. 
	// What really needs to happen in this case is that a thread is spawned which waits for a processIntentionToEnter 
	// to be processed. Then the steps in the START case can happen.
	if (getState() == START) {
	    // Got an intention-to-refuse-entry before this negotiator even got a chance to participate.
	    // Jump directly to INTENTION_TO_REFUSE_ENTRY, add the sender and this negotiator to commit set, and broadcast

	    initiateIntentionAction(sender, null, REFUSE_ENTRY);

	    setState(INTENTION_TO_REFUSE_ENTRY);
	    runningCommitSet.put(receiver, new Integer(0));
	    runningCommitSet.put(sender, new Integer(0)); // Integer(0) is a dummy goal (not allowed to put null in a hash table)
	    broadcastIntentionToRefuseEntry(uniqueEntryNegotiationID);
	    
	    // Check if we are ready to commit to refuse enter.
	    // This can only happen from the start state if there are only two members in the team.
	    if (runningCommitSet.keySet().equals(teamMembers)) {
		// all team members have commited - move to COMMIT_TO_REFUSE_ENTRY 
		setState(COMMIT_TO_REFUSE_ENTRY);
		
		commitToIntentionAction(escroedGoalStep, REFUSE_ENTRY);
		BehavingEntity.getBehavingEntity().terminateEntryNegotiation(uniqueEntryNegotiationID);
	    }
	    else {
		// Not ready to exit yet - block on achievement of COMMIT_TO_REFUSE_ENTRY
		while( (getState() != COMMIT_TO_REFUSE_ENTRY) )
		    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
		commitToIntentionAction(escroedGoalStep, REFUSE_ENTRY);
		BehavingEntity.getBehavingEntity().terminateEntryNegotiation(uniqueEntryNegotiationID);
	    }
	}
	else if (getState() == INTENTION_TO_ENTER) {
	    // start a new commit set - we were intending to enter, but someone is intending to refuse entry
	    // broadcast an intention to refuse entry

	    setState(INTENTION_TO_REFUSE_ENTRY);
	    
	    initiateIntentionAction(null, escroedGoalStep, REFUSE_ENTRY);
	    runningCommitSet = new Hashtable();
	    runningCommitSet.put(receiver, new Integer(0));
	    runningCommitSet.put(sender, new Integer(0));
	    broadcastIntentionToRefuseEntry(uniqueEntryNegotiationID);

	    if (runningCommitSet.keySet().equals(teamMembers)) {
		// all team members have commited - move to COMMIT_TO_REFUSE_ENTRY 
		setState(COMMIT_TO_REFUSE_ENTRY);
		
		commitToIntentionAction(null, REFUSE_ENTRY);
		BehavingEntity.getBehavingEntity().terminateEntryNegotiation(uniqueEntryNegotiationID);
	    }
	    else {
		// Not ready to exit yet - block on achievement of COMMIT_TO_REFUSE_ENTRY
		while( (getState() != COMMIT_TO_REFUSE_ENTRY) )
		    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
		commitToIntentionAction(escroedGoalStep, REFUSE_ENTRY);
		BehavingEntity.getBehavingEntity().terminateEntryNegotiation(uniqueEntryNegotiationID);
	    }

	}
	else if (getState() == INTENTION_TO_REFUSE_ENTRY) {
	    // Waiting to complete a negotiation (either in negotiateEntry or the START case of 
	    // processIntentionToRefuseEntry).
	    // add the sender to the commit set if they haven't already been added

	    if (runningCommitSet.get(sender) == null)
		runningCommitSet.put(sender, new Integer(0)); // Integer(0) is a "dummy behavior"

	    // if all team members have commited, COMMIT_TO_REFUSE_ENTRY and notify waiting thread
	    if (runningCommitSet.keySet().equals(teamMembers)) {
		// all team members have commited - move to COMMIT_TO_REFUSE_ENTRY
		setState(COMMIT_TO_REFUSE_ENTRY);
	    }	
	}
	else 
	    throw new UnexpectedStateError(getState());	    
    }

    private boolean initiateSuccessNegotiation()
    {
	final BehavingEntity thisEntity = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(thisEntity);

	initiateNegotiationAction(negotiatingGoal, SUCCEED);

	setState(INTENTION_TO_SUCCEED);

	// Start a new negotiatingCommitSet and add myself to it.
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(thisEntity);

	broadcastIntentionToExit(SUCCEED);

	// wait for COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, or COMMIT_TO_REMOVE
	while ( (getState() != COMMIT_TO_SUCCEED) &&
		(getState() != COMMIT_TO_FAIL) &&
		(getState() != COMMIT_TO_REMOVE) )
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));

	if (getState() == COMMIT_TO_SUCCEED)
	    return true;
	else
	    return false;
    }

    // Negotiate success with all team members.
    // Blocks on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE.
    // Returns true if COMMIT_TO_SUCCEED achieved, false otherwise. 
    public boolean negotiateSuccess() 
    { 
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(BehavingEntity.getBehavingEntity());

	// Trace.trace(BehavingEntity.getBehavingEntity(), "Initiating success in static " + getState());
	Trace.ablTrace("Initiating success in state " + getState());

	// Because reflection might be used to initiate success at any time, this could happen in the middle of a 
	// prior negotiation, so have to work through all the cases.
	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING) {
		// If we arrive here, the joint goal's behavior has just been added to the ABT.
		// Since the success negotiation was initiated before entry, the subtree freeze happened before 
		// the behavior and behavior's steps were added to the ABT. So we need to immediately remove them before 
		// continuing the negotiation, otherwise they will continue to execute, creating errors. 
		
		negotiatingGoal.removeChild(true);
		return initiateSuccessNegotiation();
	    }
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return false;
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case COMMIT_TO_SUCCEED:
	case COMMIT_TO_FAIL:
	case COMMIT_TO_REMOVE:
	    // Shouldn't be possible for a meta-behavior to initiate success in a commited state. 
	    // The switch from intention to commitment happens atomically with respect to the decision cycle.
	    // This could only happen if a meta-behavior is holding onto an old reflection WME.

	    // throw new UnexpectedStateError(getState(), negotiatingGoal);

	    // Instead of throwing an error, just return false in this case. 
	    return false;
	    // break;

	case COMMIT_TO_REFUSE_ENTRY:
	    // Have already commited to refuse entry.
	case INTENTION_TO_SUCCEED:
	    // Have already begun negotiation to succeed.
	    // Ignore the new success negotiation - return false so that the calling thread doesn't do anything. 
	case INTENTION_TO_FAIL:
	    // Already intending to fail.
	    // Ignore the new success negotiation - return false so that the calling thread doesn't do anything. 
	case INTENTION_TO_REMOVE:
	    // Initiate success negotiation when I'm intending to remove.
	    // Originally intended to initiate success negotiation (succed trumps remove).
	    // Instead ignore (remove trumps success).
	    return false;
	    // break;

	case RUNNING:
	case INTENTION_TO_SUSPEND:
	    // Initiate success negotiation when I'm already intending to suspend.
	    // Initiate success negotiation (succeed trumps suspend).
	case SUSPENDED:
	    // Initiate success negotiation when I'm suspended. 
	    // Initiate success negotiation (succeed trumps suspend).
	case INTENTION_TO_UNSUSPEND:
	    // Initiate success negotiation when I'm intending to unsuspend. 
	    // Initiate success negotiation (succeed trumps unsuspend).
	case COMMIT_TO_SUSPEND:
	case COMMIT_TO_UNSUSPEND:
	    // Initiate success negotiation after having commited to suspend or unsuspend (but before switching
	    // to SUSPENDED or RUNNING). Initiate success negotiation (succeed trumps remove).
	    return initiateSuccessNegotiation();
	    // break;

	case WAIT_FOR_SUCCESS:
	    initiateNegotiationAction(negotiatingGoal, SUCCEED);

	    setState(INTENTION_TO_SUCCEED);

	    // No need to broadcast intention to exit because another thread will do it.

	    // wait for COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE.
	    while ( (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	    completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));

	    if (getState() == COMMIT_TO_SUCCEED)
		return true;
	    else
		return false;
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // Initiate an intention to succeed.
    // Called by processIntentionToSucceed().
    // Blocks on COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, or COMMIT_TO_REMOVE.
    private void initiateIntentionToSucceed(final BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	Trace.ablTrace("Initiating intention to succeed on " + receiver);
	Trace.ablTrace("Goal to succeed " + negotiatingGoal + ", teamNeededForSuccess = " + negotiatingGoal.getTeamNeededForSuccess());

 	// Not the initiator of the success negotiation
	setState(INTENTION_TO_SUCCEED);
	initiateIntentionAction(sender, negotiatingGoal, SUCCEED);

	// add the sender and receiver to the commit set
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(sender);
	negotiatingCommitSet.add(receiver);

	// fixme: if a teammember sends another intention to succeed after a commit (looks like it can happen when
	// a teammember initiates success, this entity waits for success, then broadcasts a negotiation, causing the
	// first teammember to resend a success intention), then this state is reentered, resulting in an endless 
	// wait for success.

	if (negotiatingGoal.getTeamNeededForSuccess()) {
	    // If team_needed_for_success, wait for this goal to succeed before freezing tree and broadcasting 
	    // intention to exit. 

	    // Trace.trace(BehavingEntity.getBehavingEntity(), "waiting for success in " + negotiatingGoal);
	    Trace.ablTrace("waiting for success in " + negotiatingGoal);

	    setState(WAIT_FOR_SUCCESS);
	    while ( (getState() != INTENTION_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_SUCCEED) && 
		    (getState() != COMMIT_TO_FAIL) && 
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	    Trace.ablTrace("goal succeeded: " + negotiatingGoal);
	}

	if (getState() == INTENTION_TO_SUCCEED) {
	    // If getState() == COMMIT_TO_FAIL || getState() == COMMIT_TO_REMOVE, skip all this stuff
	
	    JointGoalStep.blockOnSubtreeNegotiation(negotiatingGoal.freezeSubtreeAndNegotiateRemovalEntry());
	
	    broadcastIntentionToExit(SUCCEED);
	    
	    // Check if we are ready to commit to succeed.
	    // This can only happen on initiation if there are only two entities in the team.
	    if (negotiatingCommitSet.equals(teamMembers)) {
		// all team members have commited - move to COMMIT_TO_SUCCEED
		setState(COMMIT_TO_SUCCEED);
		commitToIntentionAction(negotiatingGoal, SUCCEED);
		negotiatingGoal.succeedStep();
	    }
	    else {
		// Not ready to enter yet - block on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, or COMMIT_TO_REMOVE
		while( (getState() != COMMIT_TO_SUCCEED) &&
		       (getState() != COMMIT_TO_FAIL) &&
		       (getState() != COMMIT_TO_REMOVE) )
		    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
		commitToIntentionAction(negotiatingGoal, convertCommitStateToIntention(getState()));

		if (getState() == COMMIT_TO_SUCCEED) 
		    negotiatingGoal.succeedStep();
		// if getState() == COMMIT_TO_FAIL || COMMIT_TO_REMOVE another thread is responsible for processing
	    }
	}
    }

    // Process intention to succeed.
    public void processIntentionToSucceed(BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);
	
	processIntentionAction(sender, negotiatingGoal, SUCCEED);

	switch (getState()) {
	case START:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_REFUSE_ENTRY:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case INTENTION_TO_ENTER:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING (COMMIT_TO_REFUSE_ENTRY can't happen because everyone in team has to 
	    // have found a behavior in order for someone to have already entered and succeeded).
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    switch (getState()) {
	    case RUNNING:
		initiateIntentionToSucceed(sender);
		break;
	    case COMMIT_TO_SUCCEED:
	    case COMMIT_TO_FAIL:
	    case COMMIT_TO_REMOVE:
		// A commit sneaks in before this success intention has a chance to be processed. I'm seeing a case
		// COMMIT_TO_REMOVE sneaks in because a self initiated intention to remove changes the state from RUNNING
		// to INTENTION_TO_REMOVE before this thread has a chance to run. 
		// In this case, ignore the success intention. 
		break;
	    case COMMIT_TO_REFUSE_ENTRY:
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    }
	    break;

	case RUNNING:
	case INTENTION_TO_SUSPEND:
	    // Receive an intention to succeed when I'm intending to suspend.
	    // Initiate an intention to succeed (succeed trumps suspend).
	case COMMIT_TO_SUSPEND:
	    // A message sneaks in between COMMIT_TO_SUSPEND and SUSPENDED.
	    // Initiate an intention to succeed (succeed trumps suspend).
	case SUSPENDED:
	    // Receive an intention to succeed when I'm suspended. 
	    // Initiate an intention to succeed.
	case INTENTION_TO_UNSUSPEND:
	    // Receive an intention to succeed when I'm intending to unsuspend.
	    // Initiate an intention to succeed (succeed trumps unsuspend).
	case COMMIT_TO_UNSUSPEND:
	    // A message sneaks in between COMMIT_TO_UNSUSPEND and RUNNING.
	    initiateIntentionToSucceed(sender);
	    break;

	case INTENTION_TO_REMOVE:
	    // Receive an intention to succeed when I'm intending to remove.
	    // Originally was going to initiate an intention to succeed (succeed trumps remove).
	    // Instead, just return (ignore succeed intention).
	    return;
	    // break;

	case INTENTION_TO_SUCCEED:
	case WAIT_FOR_SUCCESS: // We're waiting on this agent's goal to freeze before committing - however, we still want to add the broadcasting agent to the commit set. Didn't see this in Facade because we only had teams of two.
	    // Have already begun negotiation to succeed
	    // add the sender to the commit set
	    negotiatingCommitSet.add(sender); // OK to add without member check because of the semantics of sets
	    if (negotiatingCommitSet.equals(teamMembers))
		setState(COMMIT_TO_SUCCEED);
	    break;

	case COMMIT_TO_SUCCEED:
	    // I've already committed to succeed - recieving an extraneous message
	    // In a network environment, this could happen through message delays and rebroadcast.
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_FAIL:
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_REMOVE:
	    // fixme: for generality, do something smarter in this case
	    throw new MessageReceivedAfterCommitError(SUCCEED, getState(), negotiatingGoal);
	    // break;

	case INTENTION_TO_FAIL:
	    // Receive an intention to succeed when I'm already intending to fail.
	    // Remove the sender from the negotiation set.
	    negotiatingCommitSet.remove(sender);
	    break;
	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    private void initiateFailureNegotiation()
    {
	final BehavingEntity thisEntity = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(thisEntity);

	initiateNegotiationAction(negotiatingGoal, FAIL);

	setState(INTENTION_TO_FAIL);
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(BehavingEntity.getBehavingEntity());

	broadcastIntentionToExit(FAIL);
	
	// wait for COMMIT_TO_FAIL
	while (getState() != COMMIT_TO_FAIL)
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	// when we get here we've commited to fail - just return, allowing the thread to continue
    }

    // negotiate failure exit with all team members
    // returns true if the fail negotiation succeeds. The only way it can't succeed is if an attempt is made
    // to initiate failure negotiation when failure negotiation has already been initiated.
    public boolean negotiateFailure() 
    { 
	// Because reflection might be used to initiate failure at any time, this could happen in the middle of a 
	// prior negotiation, so have to work through all the cases.

	// Trace.trace(BehavingEntity.getBehavingEntity(), "Negotiating failure");
	Trace.ablTrace("Negotiating failure");

	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING) {
		// If we arrive here, the joint goal's behavior has just been added to the ABT.
		// Since the fail negotiation was initiated before entry, the subtree freeze happened before 
		// the behavior and behavior's steps were added to the ABT. So we need to immediately remove them before 
		// continuing the negotiation, otherwise they will continue to execute, creating errors. 
		
		negotiatingGoal.removeChild(true);
		initiateFailureNegotiation();
		return true;
	    }
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return false;
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case COMMIT_TO_SUCCEED:
	case COMMIT_TO_FAIL:
	case COMMIT_TO_REMOVE:
	    // Shouldn't be possible for a meta-behavior to initiate failure in a commited state. 
	    // The switch from intention to commitment happens atomically with respect to the decision cycle.
	    // This could only happen if a meta-behavior is holding onto an old reflection WME.
	    
	    // throw new UnexpectedStateError(getState(), negotiatingGoal);

	    // Instead of throwing an error, return false (ignore the negotiation).
	    return false;
	    // break;

	case COMMIT_TO_REFUSE_ENTRY:
	case INTENTION_TO_FAIL:
	    // Already intending to fail.
	    // Ignore the new failure negotiation - return false so that the calling thread doesn't do anything. 
	case INTENTION_TO_REMOVE:
	    // Initiate failure negotiation when I'm intending to remove. 
	    // Originally intended to initiate failure negotiation (failure trumps remove).
	    // Instead ignore the failure negotiation (remove trumps failure).
	    return false;
	    // break;

	case RUNNING:
	case INTENTION_TO_SUCCEED:
	    // Have already begun negotiation to succeed.
	    // Initiate failure negotiation (failure trumps success).
	case INTENTION_TO_SUSPEND:
	    // Initiate failure negotiation when I'm already intending to suspend.
	    // Initiate failure negotiation (failure trumps suspend).
	case SUSPENDED:
	    // Initiate failure negotiation when I'm suspended. 
	    // Initiate failure negotiation (failure trumps suspend).
	case INTENTION_TO_UNSUSPEND:
	    // Initiate failure negotiation when I'm intending to unsuspend. 
	    // Initiate failure negotiation (failure trumps unsuspend).
	case COMMIT_TO_SUSPEND:
	case COMMIT_TO_UNSUSPEND:
	    // Initiate failure negotiation (failure trumps suspend or unsuspend).
	case WAIT_FOR_SUCCESS:
	    // Failure trumps success.
	    initiateFailureNegotiation();
	    return true;
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // Initiate an intention to fail.
    // Called by processIntentionToFail().
    // Blocks on COMMIT_TO_FAIL.
    private void initiateIntentionToFail(final BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	setState(INTENTION_TO_FAIL);
	initiateIntentionAction(sender, negotiatingGoal, FAIL);
	    
	// add the sender and receiver to the commit set
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(sender);
	negotiatingCommitSet.add(receiver);
	    
	JointGoalStep.blockOnSubtreeNegotiation(negotiatingGoal.freezeSubtreeAndNegotiateRemovalEntry());

	broadcastIntentionToExit(FAIL);
	    
	// Check if we are ready to commit to fail.
	// This can only happen on initiaion if there are only two entities in the team.
	if (negotiatingCommitSet.equals(teamMembers)) {
	    // all team members have commited - move to COMMIT_TO_FAIL
	    setState(COMMIT_TO_FAIL);
	    commitToIntentionAction(negotiatingGoal,FAIL);
	    ((JointGoalStep)runningCommitSet.get(receiver)).failStep();
	}
	else {
	    // Not ready to enter yet - block on achievement of COMMIT_TO_FAIL
	    while(getState() != COMMIT_TO_FAIL)
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    
	    commitToIntentionAction(negotiatingGoal, FAIL);
	    ((JointGoalStep)runningCommitSet.get(receiver)).failStep();
	}
    }

    // Process intention to fail.
    public void processIntentionToFail(BehavingEntity sender)
    {
 	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	processIntentionAction(sender, negotiatingGoal, FAIL);

	switch (getState()) {
	case START:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_REFUSE_ENTRY:
	    // If we haven't entered the goal, it is an error to negotiate failure
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case INTENTION_TO_ENTER:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING (COMMIT_TO_REFUSE_ENTRY can't happen because everyone in team has to 
	    // have found a behavior in order for someone to have already entered and succeeded).
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
		initiateIntentionToFail(sender);
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    break;

	case COMMIT_TO_SUCCEED:
	    // I've already committed to succeed - recieving an extraneous message
	    // In a network environment, this could happen through message delays and rebroadcast.
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_FAIL:
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_REMOVE:
	    // fixme: for generality, do something smarter in this case
	    throw new MessageReceivedAfterCommitError(FAIL, getState(), negotiatingGoal);
	    // break;

	case RUNNING:
	case INTENTION_TO_SUCCEED:
	    // Have already begun negotiation to succeed.
	    // Failure trumps success, so negotiate failure.
	case INTENTION_TO_SUSPEND:
	    // Receive an intention to fail when I'm intending to suspend.
	    // Initiate an intention to fail (fail trumps suspend).
	case COMMIT_TO_SUSPEND:
	    // A message sneaks in between COMMIT_TO_SUSPEND and SUSPENDED.
	    // Initiate an intention to fail (fail trumps suspend).
	case SUSPENDED:
	    // Receive an intention to succeed when I'm suspended. 
	    // Initiate an intention to succeed.
	case INTENTION_TO_UNSUSPEND:
	    // Receive an intention to fail when I'm intending to unsuspend.
	    // Initiate an intention to fail (fail trumps unsuspend).
	case COMMIT_TO_UNSUSPEND:
	    // A message sneaks in between COMMIT_TO_UNSUSPEND and RUNNING.
	case WAIT_FOR_SUCCESS:
	    // Failure trumps suspend.
	    initiateIntentionToFail(sender);
	    break;

	case INTENTION_TO_REMOVE:
	    // Receive an intention to fail when I'm intending to remove.
	    // Originally intended to initiate an intention to fail (fail trumps remove).
	    // Instead ignore the failure negotiation (remove trumps failure).
	    break;

	case INTENTION_TO_FAIL:
	    // Already negotiating failure.
	    // Add the sender to the commit set
	    negotiatingCommitSet.add(sender); // OK to add without member check because of the semantics of sets
	    if (negotiatingCommitSet.equals(teamMembers))
		setState(COMMIT_TO_FAIL);
	    break;
	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // Initiate suspend negotiation.
    // Blocks on COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE, COMMIT_TO_SUSPEND. 
    private boolean initiateSuspendNegotiation()
    {
	final BehavingEntity thisEntity = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(thisEntity);

	initiateNegotiationAction(negotiatingGoal, SUSPEND);

	setState(INTENTION_TO_SUSPEND);
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(thisEntity);

	broadcastIntentionToExit(SUSPEND);
	
	// wait for COMMIT_TO_REMOVE, COMMIT_TO_FAIL, COMMIT_TO_SUCCEED, or COMMIT_TO_SUSPEND
	while ( (getState() != COMMIT_TO_REMOVE) &&
		(getState() != COMMIT_TO_FAIL) &&
		(getState() != COMMIT_TO_SUCCEED) &&
		(getState() != COMMIT_TO_SUSPEND) )
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
       
	completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	if (getState() == COMMIT_TO_SUSPEND) {
	    return true;
	}
	else
	    return false;
    }

    // Negotiate suspend with all team members.
    // Returns true if COMMIT_TO_SUSPEND is achieved, false otherwise.
    // Blocks on COMMIT_TO_SUSPEND, COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE
    public boolean negotiateSuspend() 
    { 
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(BehavingEntity.getBehavingEntity());

	// Trace.trace(BehavingEntity.getBehavingEntity(), "Initiating suspend in state " + getState());
	Trace.ablTrace("Initiating suspend in state " + getState());

	// Because reflection might be used to initiate suspend at any time, this could happen in the middle of a 
	// prior negotiation, so have to work through all the cases.
	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
		return initiateSuspendNegotiation();
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return false;
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;
	    
	case COMMIT_TO_SUCCEED:
	case COMMIT_TO_FAIL:
	case COMMIT_TO_REMOVE:
	    // Shouldn't be possible for a meta-behavior to initiate suspend in a commited state. 
	    // The switch from intention to commitment happens atomically with respect to the decision cycle.
	    // This could only happen if a meta-behavior is holding onto an old reflection WME.
	    
	    // throw new UnexpectedStateError(getState(), negotiatingGoal);

	    // Instead of throwing an error, return false (ignore the negotiation).
	    return false;
	    // break;

	case COMMIT_TO_REFUSE_ENTRY:
	    // Commited to refuse entry.
	case INTENTION_TO_FAIL:
	    // Already intending to fail.
	case INTENTION_TO_SUCCEED:
	    // Already intending to succeed.
	case INTENTION_TO_REMOVE:
	    // Already intending to remove.
	case SUSPENDED:
	    // Already suspended - don't need to negotiate suspend again.
	case INTENTION_TO_SUSPEND:
	    // Already intending to suspend - don't need to negotiate suspend again.
	    // Ignore the new suspend negotiation - return false so that the calling thread doesn't do anything. 
	    return false;
	    // break;

	case RUNNING:
	case WAIT_FOR_SUCCESS:
	    // Local goal still running (team member(s) have succeeded).
	    return initiateSuspendNegotiation();
	    // break;

	case COMMIT_TO_SUSPEND:
	    // Initiate suspend negotiation when I'm commited to suspend.
	    // Wait for SUSPENDED before initiating suspend.
	case COMMIT_TO_UNSUSPEND:
	    // Initiate suspend negotiation when I'm commited to unsuspend.
	    // Wait for RUNNING or SUSPENDED before initiating suspend.
	case INTENTION_TO_UNSUSPEND:
	    // Initiate suspend negotiation when I'm intending to unsuspend. 
	    // Complete the unsuspend negotiation first.
	    while( (getState() != COMMIT_TO_SUCCEED) &&
		   (getState() != COMMIT_TO_FAIL) &&
		   (getState() != COMMIT_TO_REMOVE) &&
		   (getState() != RUNNING) &&
		   (getState() != SUSPENDED) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    
	    if ((getState() == RUNNING) || (getState() == SUSPENDED))
		return initiateSuspendNegotiation();
	    else
		return false;
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // Initiate an intention to suspend.
    // Called by processIntentionToSuspend().
    // Blocks on COMMIT_TO_SUSPEND, COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, or COMMIT_TO_REMOVE.
    private void initiateIntentionToSuspend(final BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	// Not the initiator of the suspend negotiation
	initiateIntentionAction(sender, negotiatingGoal, SUSPEND);
	setState(INTENTION_TO_SUSPEND);
	    
	// add the sender and receiver to the commit set
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(sender);
	negotiatingCommitSet.add(receiver);
	    
	// freeze the tree rooted at this goal
	JointGoalStep.blockOnSubtreeNegotiation(negotiatingGoal.negotiateSuspensionOfSubtreeEntry());
	broadcastIntentionToExit(SUSPEND);
	    
	// Check if we are ready to commit to suspend.
	// This can only happen on initiation if there are only two entities in the team
	if (negotiatingCommitSet.equals(teamMembers)) {
	    // all team members have commited - move directly to SUSPENDED.
	    commitToIntentionAction(negotiatingGoal, SUSPEND);
	    setState(SUSPENDED);
	    negotiatingGoal.jointSuspend();
	}
	else {
	    // Not ready to enter yet - block on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE
	    // or SUSPENDED
	    while( (getState() != COMMIT_TO_SUCCEED) &&
		   (getState() != COMMIT_TO_FAIL) &&
		   (getState() != COMMIT_TO_REMOVE) &&
		   (getState() != COMMIT_TO_SUSPEND) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
	    commitToIntentionAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	    if (getState() == COMMIT_TO_SUSPEND) {
		negotiatingGoal.jointSuspend();
		setState(SUSPENDED);
	    }
	    // if currentState is anything other than SUSPENDED another thread is responsible for 
	    // processing.
	}	
    }
    
    // Process intention to suspend.
    public void processIntentionToSuspend(final BehavingEntity sender)
    {
 	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	processIntentionAction(sender, negotiatingGoal, SUSPEND);

	switch (getState()) {
	case START:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_REFUSE_ENTRY:
	    // If we haven't entered the goal, it is an error to negotiate failure
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case INTENTION_TO_ENTER:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING (COMMIT_TO_REFUSE_ENTRY can't happen because everyone in team has to 
	    // have found a behavior in order for someone to have already entered and succeeded).
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
	        initiateIntentionToSuspend(sender);
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    break;

	case COMMIT_TO_SUCCEED:
	    // I've already committed to succeed - recieving an extraneous message
	    // In a network environment, this could happen through message delays and rebroadcast.
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_FAIL:
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_REMOVE:
	    // fixme: for generality, do something smarter in this case
	    throw new MessageReceivedAfterCommitError(FAIL, getState(), negotiatingGoal);
	    // break;

	case SUSPENDED:
	    // Receive an intention to suspend when I'm suspended.
	    // Process the new suspend.
	case RUNNING:
	case WAIT_FOR_SUCCESS:
	    // Local goal still running (team member(s) have succeeded).
	    initiateIntentionToSuspend(sender);
	    break;

	case INTENTION_TO_UNSUSPEND:
	    // Receive an intention to suspend when I'm intending to unsuspend.
	    // Start a new intention to suspend then continue intention to unsuspend.
	    Set oldNegotiatingCommitSet = new HashSet(negotiatingCommitSet);
	    initiateIntentionToSuspend(sender);

	    // wait for RUNNING or SUSPENDED to be achieved.
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) &&
		    (getState() != SUSPENDED) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	    if ((getState() == RUNNING) || (getState() == SUSPENDED)) {
		// Continue back where we left off with with the INTENTION_TO_UNSUSPEND
		setState(INTENTION_TO_UNSUSPEND);
		negotiatingCommitSet = oldNegotiatingCommitSet;
	    }
	    break;

	case COMMIT_TO_SUSPEND:
	    // An intention to suspend sneaks in between COMMIT_TO_SUSPEND and SUSPEND.
	    // Wait for achievement of SUSPENDED before initiating suspend.
	case COMMIT_TO_UNSUSPEND:
	    // A message sneaks in between COMMIT_TO_UNSUSPEND and RUNNING.
	    // Wait for the RUNNING or SUSPENDED to be achived before initiating suspend.

	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) &&
		    (getState() != SUSPENDED) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if ((getState() == RUNNING) || (getState() == SUSPENDED)) {
		initiateIntentionToSuspend(sender);
	    }
	    break;

	case INTENTION_TO_SUCCEED:
	    // Already negotiating success.
	case INTENTION_TO_REMOVE:
	    // Already negotiating removal.
	case INTENTION_TO_FAIL:
	    // Already negotiating failure.
	    // Remove sender from commit set.
	    negotiatingCommitSet.remove(sender);
	    break;

	case INTENTION_TO_SUSPEND:
	    // Already negotiating suspend. 
	    // Add the sender to the commit set.
	    negotiatingCommitSet.add(sender); // OK to add without member check because of the semantics of sets
	    if (negotiatingCommitSet.equals(teamMembers))
		setState(COMMIT_TO_SUSPEND);
	    break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // called by negotiateUnsuspend().
    // Blocks on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE, SUSPENDED or RUNNING.
    private boolean initiateUnsuspendNegotiation()
    {
	final BehavingEntity thisEntity = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(thisEntity);

	initiateNegotiationAction(negotiatingGoal, UNSUSPEND);

	setState(INTENTION_TO_UNSUSPEND);
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(thisEntity);

	broadcastIntentionToExit(UNSUSPEND);
	
	// wait for COMMIT_TO_REMOVE, COMMIT_TO_FAIL, COMMIT_TO_SUCCEED, or COMMIT_TO_UNSUSPEND
	while ( (getState() != COMMIT_TO_REMOVE) &&
		(getState() != COMMIT_TO_FAIL) &&
		(getState() != COMMIT_TO_SUCCEED) &&
		(getState() != COMMIT_TO_UNSUSPEND) )
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
       
	completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	if (getState() == COMMIT_TO_UNSUSPEND) {
	    return true;
	}
	else
	    return false;
    }

    // Negotiate unsuspend re-entry with all team members
    // Returns true if COMMIT_TO_UNSUSPEND is achieved, false otherwise.
    // Blocks on COMMIT_TO_UNSUSPEND, COMMIT_TO_SUSPEND, COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE
    // fixme: may not work in state WAIT_FOR_SUCCESS
    public boolean negotiateUnsuspend() 
    { 
	// Because reflection might be used to initiate unsuspend at any time, this could happen in the middle of a 
	// prior negotiation, so have to work through all the cases.

	// Trace.trace(BehavingEntity.getBehavingEntity(), "Initiating unsuspend in state " + getState());
	Trace.ablTrace("Initiating unsuspend in state " + getState());

	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
		return initiateUnsuspendNegotiation();
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return false;
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case COMMIT_TO_SUCCEED:
	case COMMIT_TO_FAIL:
	case COMMIT_TO_REMOVE:
	    // Shouldn't be possible for a meta-behavior to initiate unsuspend in a commited state. 
	    // The switch from intention to commitment happens atomically with respect to the decision cycle.
	    // This could only happen if a meta-behavior is holding onto an old reflection WME.
	    
	    // throw new UnexpectedStateError(getState(), negotiatingGoal);

	    // Instead of throwing an error, return false (ignore the negotiation).
	    return false;
	    // break;

	case COMMIT_TO_SUSPEND:
	case COMMIT_TO_UNSUSPEND:
	    // Finish commit processing before initiating the new suspend.
	    while ( (getState() != COMMIT_TO_REMOVE) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != SUSPENDED) &&
		    (getState() != RUNNING) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	    if ((getState() == SUSPENDED) || (getState() == RUNNING))
		return initiateUnsuspendNegotiation();
	    else
		return false;
	    // break;

	case INTENTION_TO_SUSPEND:
	    // Initiate unsuspend negotiation when I'm already intending to suspend. 
	    // Finish the current suspend negotiation first. 
	case INTENTION_TO_UNSUSPEND:
	    // Initiate unsuspend negotiation when I'm already intending to unsuspend. 
	    // Finish current unsuspend processing first.
	    while ( (getState() != COMMIT_TO_REMOVE) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != SUSPENDED) &&
		    (getState() != RUNNING) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();

	    if (getState() == SUSPENDED)
		return initiateUnsuspendNegotiation();
	    else
		return false;
	    // break;

	case COMMIT_TO_REFUSE_ENTRY:
	    // Already commited to refuse entry.
	case INTENTION_TO_FAIL:
	    // Already intending to fail.
	case INTENTION_TO_SUCCEED:
	    // Already intending to succeed.
	case RUNNING:
	    // Already running - ignore the unsuspend.
	case INTENTION_TO_REMOVE:
	    // Already intending to remove.
	    // Ignore the new unsuspend negotiation - return false so that the calling thread doesn't do anything. 
	    return false;
	    // break;

	case SUSPENDED:
	    // Suspended - initiate unsuspend negotiation
	    return initiateUnsuspendNegotiation();
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    // fixme: may not work in state WAIT_FOR_SUCCESS
    private void initiateIntentionToUnsuspend(final BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	// Not the initiator of the unsuspend negotiation
	initiateIntentionAction(sender, negotiatingGoal, UNSUSPEND);
	setState(INTENTION_TO_UNSUSPEND);
	    
	// add the sender and receiver to the commit set
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(sender);
	negotiatingCommitSet.add(receiver);
	    
	broadcastIntentionToExit(UNSUSPEND);
	    
	// Check if we are ready to commit to suspend.
	// This can only happen on initiation if there are only two entities in the team
	if (negotiatingCommitSet.equals(teamMembers)) {
	    // all team members have commited 
	    commitToIntentionAction(negotiatingGoal, UNSUSPEND);
	    negotiatingGoal.jointUnsuspendEntry();
	    if (negotiatingGoal.isSuspended())
		setState(SUSPENDED);
	    else
		setState(RUNNING);
	}
	else {
	    // Not ready to enter yet - block on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, COMMIT_TO_REMOVE
	    // or COMMIT_TO_UNSUSPEND
	    while( (getState() != COMMIT_TO_SUCCEED) &&
		   (getState() != COMMIT_TO_FAIL) &&
		   (getState() != COMMIT_TO_REMOVE) &&
		   (getState() != COMMIT_TO_UNSUSPEND) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
	    commitToIntentionAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	    if (getState() == COMMIT_TO_UNSUSPEND) {
		negotiatingGoal.jointUnsuspendEntry();
		if (negotiatingGoal.isSuspended())
		    setState(SUSPENDED);
		else
		    setState(RUNNING);
	    }
	    // if currentState is anything other than COMMIT_TO_SUSPEND another thread is responsible for 
	    // processing.
	}
    }

    // Process intention to unsuspend.
    // fixme: may not work in state WAIT_FOR_SUCCESS
    public void processIntentionToUnsuspend(final BehavingEntity sender)
    {
 	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	processIntentionAction(sender, negotiatingGoal, UNSUSPEND);

	switch (getState()) {
	case START:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_REFUSE_ENTRY:
	    // If we haven't entered the goal, it is an error to negotiate failure
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case INTENTION_TO_ENTER:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING (COMMIT_TO_REFUSE_ENTRY can't happen because everyone in team has to 
	    // have found a behavior in order for someone to have already entered and succeeded).
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
		initiateIntentionToUnsuspend(sender);
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    break;

	case COMMIT_TO_SUCCEED:
	    // I've already committed to succeed - recieving an extraneous message
	    // In a network environment, this could happen through message delays and rebroadcast.
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_FAIL:
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_REMOVE:
	    // fixme: for generality, do something smarter in this case
	    throw new MessageReceivedAfterCommitError(FAIL, getState(), negotiatingGoal);
	    // break;

	case SUSPENDED:
	    // Receive an intention to unsuspend when I'm suspended.
	    initiateIntentionToUnsuspend(sender);
	    break;

	case RUNNING:
	    // Receive an intention to unsuspend when I'm already running.
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;
	
	case COMMIT_TO_SUSPEND:
	    // An intention to unsuspend sneaks in between COMMIT_TO_SUSPEND and SUSPEND.
	    // Wait for suspend processing to complete before initiating unsuspend.
    	case INTENTION_TO_SUSPEND:
	    // Receive an intention to unsuspend when I'm intending to suspend.
	    // Wait for suspend processing to complete before initiating unsuspend.
	case COMMIT_TO_UNSUSPEND:
	    // A message sneaks in between COMMIT_TO_UNSUSPEND and RUNNING.
	    // Wait for unsuspend processing to complete before initiating unsuspend.

	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) &&
		    (getState() != SUSPENDED) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if ((getState() == RUNNING) || (getState() == SUSPENDED)) {
		initiateIntentionToUnsuspend(sender);
	    }
	    break;

	case INTENTION_TO_SUCCEED:
	    // Already negotiating success.
	case INTENTION_TO_REMOVE:
	    // Already negotiating removal.
	case INTENTION_TO_FAIL:
	    // Already negotiating failure.
	    // Ignore intention to unsuspend.
	    break;

	case INTENTION_TO_UNSUSPEND:
	    // Already negotiating unsuspend.
	    negotiatingCommitSet.add(sender); // OK to add without member check because of the semantics of sets
	    if (negotiatingCommitSet.equals(teamMembers))
		setState(COMMIT_TO_UNSUSPEND);
	    break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    private boolean initiateRemovalNegotiation()
    {
	final BehavingEntity thisEntity = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(thisEntity);

	initiateNegotiationAction(negotiatingGoal, REMOVE);

	setState(INTENTION_TO_REMOVE);
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(thisEntity);

	broadcastIntentionToExit(REMOVE);
	
	// wait for COMMIT_TO_REMOVE, COMMIT_TO_FAIL, or COMMIT_TO_SUCCEED
	while ( (getState() != COMMIT_TO_REMOVE) &&
		(getState() != COMMIT_TO_FAIL) &&
		(getState() != COMMIT_TO_SUCCEED) )
	    ((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
       
	completeNegotiationAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	if (getState() == COMMIT_TO_REMOVE)
	    return true;
	else
	    return false;
    }

    // Negotiate removal of a joint behavior (something higher in the tree caused it to succeed or fail).
    // Blocks on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL, or COMMIT_TO_REMOVE.
    // Returns true if COMMIT_TO_REMOVE is achieved, false otherwise.
    public boolean negotiateRemoval() 
    { 
	// Trace.trace(BehavingEntity.getBehavingEntity(), "Negotiating removal");
	Trace.ablTrace("Negotiating removal");

	// Because reflection might be used to initiate removal At any time, this could happen in the middle of a 
	// prior negotiation, so have to work through all the cases.
	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING) {
		// If we arrive here, the joint goal's behavior has just been added to the ABT.
		// Since the remove negotiation was initiated before entry, the subtree freeze happened before 
		// the behavior and behavior's steps were added to the ABT. So we need to immediately remove them before 
		// continuing the negotiation, otherwise they will continue to execute, creating errors. 
		
		// System.out.println("!!!!!!!!!!!!!!! Removing immediately after entry: " + negotiatingGoal.getSignature());
		negotiatingGoal.removeChild(true);
		return initiateRemovalNegotiation();
	    }
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return false;
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    // break;

	case COMMIT_TO_SUCCEED:
	case COMMIT_TO_FAIL:
	case COMMIT_TO_REMOVE:
	    // Instead of throwing an error, just return false. 
	    // throw new UnexpectedStateError(getState(), negotiatingGoal);
	    return false;
	    // break;

	case COMMIT_TO_REFUSE_ENTRY:
	    // Already commited to refuse entry.
	case INTENTION_TO_FAIL:
	    // Already intending to fail.
	    // Fail trumps remove, so ignore the new removal negotiation.
	case INTENTION_TO_REMOVE:
	    // Already intending to remove.
	    // Ignore the new removal negotiation - return false so that the calling thread doesn't do anything. 
	    return false; 
	    // break;

	case RUNNING:
	case COMMIT_TO_SUSPEND:
	case COMMIT_TO_UNSUSPEND:
	case INTENTION_TO_SUSPEND:
	    // Initiate removal negotiation when I'm already intending to suspend.
	    // Initiate removal negotiation (removal trumps suspend).
	case SUSPENDED:
	    // Initiate removal negotiation when I'm suspended. 
	    // Initiate removal negotiation (removal trumps suspend).
	case INTENTION_TO_UNSUSPEND:
	    // Initiate removal negotiation when I'm intending to unsuspend. 
	    // Initiate removal negotiation (removal trumps unsuspend).
	case INTENTION_TO_SUCCEED:
	    // Have already begun negotiation to succeed.
	    // Switch to intention to remove.
	case WAIT_FOR_SUCCESS:
	    // Removal trumps success.
	    return initiateRemovalNegotiation();
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    private void removeGoal(JointGoalStep g)
    {
	g.removeGoal();
	g.cleanupParentIfRoot();
    }

    // Initiate an intention to remove.
    // Called by processIntentionToRemove().
    // Blocks on COMMIT_TO_SUCCEED or COMMIT_TO_FAIL or COMMIT_TO_REMOVE
    private void initiateIntentionToRemove(final BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	// Not the initiator of the remove negotiation
	initiateIntentionAction(sender, negotiatingGoal, REMOVE);
	setState(INTENTION_TO_REMOVE);
	    
	// add the sender and receiver to the commit set
	negotiatingCommitSet.clear();
	negotiatingCommitSet.add(sender);
	negotiatingCommitSet.add(receiver);
	    
	// freeze the tree rooted at this goal
	JointGoalStep.blockOnSubtreeNegotiation(negotiatingGoal.freezeSubtreeAndNegotiateRemovalEntry());
	broadcastIntentionToExit(REMOVE);
	    
	// Check if we are ready to commit to remove.
	// This can only happen on initiation if there are only two entities in the team
	if (negotiatingCommitSet.equals(teamMembers)) {
	    // all team members have commited - move to COMMIT_TO_REMOVE
	    commitToIntentionAction(negotiatingGoal, REMOVE);
	    setState(COMMIT_TO_REMOVE);
	    removeGoal((JointGoalStep)runningCommitSet.get(receiver));
	}
	else {
	    // Not ready to enter yet - block on achievement of COMMIT_TO_SUCCEED, COMMIT_TO_FAIL or COMMIT_TO_REMOVE
	    while( (getState() != COMMIT_TO_SUCCEED) &&
		   (getState() != COMMIT_TO_FAIL) &&
		   (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
		
	    commitToIntentionAction(negotiatingGoal, convertCommitStateToIntention(getState()));
	    if (getState() == COMMIT_TO_REMOVE) {
		removeGoal((JointGoalStep)runningCommitSet.get(receiver));
	    }
	    // if getState() == COMMIT_TO_FAIL, another thread is responsible for failing
	}
    }

    public void processIntentionToRemove(BehavingEntity sender)
    {
	final BehavingEntity receiver = BehavingEntity.getBehavingEntity();
	final JointGoalStep negotiatingGoal = (JointGoalStep)runningCommitSet.get(receiver);

	processIntentionAction(sender, negotiatingGoal, REMOVE);

	switch (getState()) {
	case START:
	case INTENTION_TO_ENTER:
	case INTENTION_TO_REFUSE_ENTRY:
	case COMMIT_TO_ENTER:
	case COMMIT_TO_REFUSE_ENTRY:
	    // Not done negotiating entry. Wait until we've achieved RUNNING or COMMIT_TO_REFUSE_ENTRY
	    while ( (getState() != RUNNING) &&
		    (getState() != COMMIT_TO_REFUSE_ENTRY) &&
		    (getState() != COMMIT_TO_SUCCEED) &&
		    (getState() != COMMIT_TO_FAIL) &&
		    (getState() != COMMIT_TO_REMOVE) )
		((JointGoalNegotiationThread)Thread.currentThread()).waitForDecisionCycle();
	    if (getState() == RUNNING)
		initiateIntentionToRemove(sender);
	    else if (getState() == COMMIT_TO_REFUSE_ENTRY)
		return; // skip removal negotiation
	    else
		throw new UnexpectedStateError(getState(), negotiatingGoal);
	    break;

	case RUNNING:
	case INTENTION_TO_SUSPEND:
	    // Receive an intention to remove when I'm intending to suspend.
	    // Initiate an intention to remove (remove trumps suspend).
	case COMMIT_TO_SUSPEND:
	    // A message sneaks in between COMMIT_TO_SUSPEND and SUSPENDED.
	    // Initiate an intention to remove (remove trumps suspend).
	case SUSPENDED:
	    // Receive an intention to remove when I'm suspended. 
	    // Initiate an intention to remove.
	case INTENTION_TO_UNSUSPEND:
	    // Receive an intention to remove when I'm intending to unsuspend.
	    // Initiate an intention to remove (remove trumps unsuspend).
	case COMMIT_TO_UNSUSPEND:
	    // A message sneaks in between COMMIT_TO_UNSUSPEND and RUNNING.
  	case INTENTION_TO_SUCCEED:
	    // Receive an intention to remove whem I'm already intending to succeed
	    // Initiate an intention to remove (remove trumps succeed).
	case WAIT_FOR_SUCCESS:
	    // Removal trumps success.
	    initiateIntentionToRemove(sender);
	    break;

	case INTENTION_TO_REMOVE:
	    // Have already begun negotiation to remove

	    negotiatingCommitSet.add(sender); // OK to add without member check because of the semantics of sets
	    if (negotiatingCommitSet.equals(teamMembers))
		setState(COMMIT_TO_REMOVE);
	    break;

	case INTENTION_TO_FAIL:
	    // Receive an intention to remove when I'm already intending to fail.	    
	    // Remove sender from commit set.
	    negotiatingCommitSet.remove(sender);
	    break;

	case COMMIT_TO_SUCCEED:
	    // I've already committed to succeed - recieving an extraneous message
	    // In a network environment, this could happen through message delays and rebroadcast.
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_FAIL:
	    // fixme: for generality, do something smarter in this case
	case COMMIT_TO_REMOVE:
	    // fixme: for generality, do something smarter in this case
	    throw new MessageReceivedAfterCommitError(SUCCEED, getState(), negotiatingGoal);
	    // break;

	default:
	    throw new UnexpectedStateError(getState(), negotiatingGoal);
	}
    }

    private void broadcastIntentionToEnter(final Long uniqueEntryNegotiationID, 
					   final Object[] args, 
					   final boolean teamNeededForSuccess, 
					   final JointGoalStep g) 
    {
	final BehavingEntity sender = BehavingEntity.getBehavingEntity();

	// remove this behaving entity from the set of entities to broadcast to
	Set broadcastSet = new HashSet(teamMembers);
	broadcastSet.remove(sender);

	// Iterate through the broadcast set, sending an intention to enter to each entity in the set.
	Iterator iter = broadcastSet.iterator();
	while(iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    destination.queueIntentionToEnter(uniqueEntryNegotiationID, sender, args, teamNeededForSuccess, teamMembers, g);
	}

	// fixme: eventually get this thread code working for supporting networked negotiation
	// Even though queueIntentionToEnter is non-blocking, each intention is sent in it's own thread to mitigate
	// communication delays.
	/* while (iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    new Thread() {
		    public void run()
		    {
			destination.queueIntentionToEnter(uniqueEntryNegotiationID, sender, args, teamNeededForSuccess, teamMembers, g);
		    } 
		    }.start();
	    } 
	*/
    }

    private void broadcastIntentionToRefuseEntry(final Long uniqueEntryNegotiationID)
    {
	final BehavingEntity sender = (BehavingEntity)BehavingEntity.getBehavingEntity();

	// remove this behaving entity from the set of entities to broadcast to
	Set broadcastSet = new HashSet(teamMembers);
	broadcastSet.remove(sender);

	// iterate through the broadcast set, sending an intention to enter to each entity in the set, in it's own thread
	Iterator iter = broadcastSet.iterator();
	while (iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    destination.queueIntentionToRefuseEntry(uniqueEntryNegotiationID, sender, teamMembers);
	}

	// fixme: eventually get this thread code working for supporting networked negotiation
	// Even though queueIntentionToRefuseEntry is non-blocking, each intention is sent in it's own thread to mitigate
	// communication delays.
	/*
	while (iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    new Thread() { 
		    public void run() 
		    {
			destination.queueIntentionToRefuseEntry(uniqueEntryNegotiationID, sender, teamMembers);
		    } 
		}.start();
	}
	*/
    }

    private void broadcastIntentionToExit(final int intention)
    {
	final BehavingEntity sender = (BehavingEntity)BehavingEntity.getBehavingEntity();

	// remove this behaving entity from the set of entities to broadcast to
	Set broadcastSet = new HashSet(teamMembers);
	broadcastSet.remove(sender);

	// iterate through the broadcast set, sending an intention to enter to each entity in the set, in it's own thread
	Iterator iter = broadcastSet.iterator();
	while (iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    final JointGoalStep goal = (JointGoalStep)runningCommitSet.get(destination);
	    destination.queueIntentionToExit(sender, goal, intention); 
	}

	// fixme: eventually get this thread code working for supporting networked negotiation
	// Even though queueIntentionToExit is non-blocking, each intention is sent in it's own thread to mitigate
	// communication delays.	
	/*
	while (iter.hasNext()) {
	    final BehavingEntity destination = (BehavingEntity)iter.next();
	    final JointGoalStep goal = (JointGoalStep)runningCommitSet.get(destination);
	    new Thread() {
		    public void run() { destination.queueIntentionToExit(sender, goal, intention); }
		}.start();
	}
	*/
    }
}

