// Abstract superclass for all joint behavior negotiation threads

// package access - shouldn't be used inside user code

package abl.runtime;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

abstract class JointGoalNegotiationThread extends Thread {

	private boolean isWaiting = true; // true when the
										// JointGoalNegotiationThread is waiting
										// to run, false otherwise

	private boolean isStarted = false; // true when the thread has been started

	private final JointGoalNegotiator negotiator; // The negotiator associated
													// with this negotiation
													// thread.

	final String descriptorString; // Package access because it's looked at by
									// JointGoalNegotiatorDebug

	private final String stackTrace;

	private final int originalPriority;

	private final JointGoalStep negotiatingStep; // The step associated with
													// this thread. Either this
													// or negotiator is non-null

	// version of constructor that takes a negotiator
	JointGoalNegotiationThread(JointGoalNegotiator negotiator, String desc) {
		descriptorString = desc;

		this.negotiator = negotiator;
		negotiatingStep = null;
		ByteArrayOutputStream charBufferStream = new ByteArrayOutputStream();
		new Throwable().printStackTrace(new PrintStream(charBufferStream));
		stackTrace = charBufferStream.toString();
		originalPriority = Thread.currentThread().getPriority();
	}

	// Version of constructor that takes a step
	JointGoalNegotiationThread(JointGoalStep s, String desc) {
		descriptorString = desc;

		negotiatingStep = s;
		negotiator = null;
		ByteArrayOutputStream charBufferStream = new ByteArrayOutputStream();
		new Throwable().printStackTrace(new PrintStream(charBufferStream));
		stackTrace = charBufferStream.toString();
		originalPriority = Thread.currentThread().getPriority();
	}

	public synchronized boolean getIsWaiting() {
		return isWaiting;
	}

	private synchronized void setIsWaiting(boolean b) {
		isWaiting = b;
	}

	public synchronized boolean getIsStarted() {
		return isStarted;
	}

	private synchronized void setIsStarted(boolean b) {
		isStarted = b;
	}

	public void start() {
		// Negotiation threads should only be started by the decision cycle.
		assert (Thread.currentThread() == BehavingEntity.getBehavingEntity()
				.getDecisionCycleThread());
		setIsStarted(true);
		setIsWaiting(false);
		// Set max priority so that decision cycle thread hits wait before
		// negotiation thread
		Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
		super.start(); // do the normal thread start stuff
		waitForNegotiation();
	}

	synchronized void waitForDecisionCycle() {
		assert (!isWaiting);

		isWaiting = true;
		// Set max priority so that negotiation thread hits wait before decision
		// cycle thread
		Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
		notify(); // wake up the decision cycle
		try {
			wait();
			yield();
			Thread.currentThread().setPriority(originalPriority); // set back
																	// to
																	// original
																	// priority
		} // wait for the decision cycle
		catch (InterruptedException e) {
			System.out.println(this);
			throw new AblRuntimeError(e);
		}
	}

	// called by decision cycle
	synchronized void continueNegotiation() {
		// must not be called by this negotiation thread
		assert (this != Thread.currentThread());

		if (isWaiting) {
			// negotiation thread is waiting
			// Set max priority so that negotiation thread hits wait before
			// decision cycle thread
			Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
			notify(); // wake up the negotiation thread
			isWaiting = false;
			try {
				wait();
				yield();
				Thread.currentThread().setPriority(originalPriority); // set
																		// back
																		// to
																		// original
																		// priority
			} // decision cycle waits for negotiation to complete or
				// waitForDecisionCycle to be called
			catch (InterruptedException e) {
				System.out.println(this);
				throw new AblRuntimeError(e);
			}
		}
	}

	private synchronized void waitForNegotiation() {
		// must not be called by this negotiation thread
		assert (this != Thread.currentThread());
		try {
			wait();
			yield();
			Thread.currentThread().setPriority(originalPriority); // set back
																	// to
																	// original
																	// priority
		} // decision cycle waits for negotiation to complete or
			// waitForDecisionCycle to be called
		catch (InterruptedException e) {
			System.out.println(this);
			throw new AblRuntimeError(e);
		}
	}

	public String toString() {
		ByteArrayOutputStream bStream = new ByteArrayOutputStream();
		PrintStream pStream = new PrintStream(bStream);
		pStream.println("JointGoalNegotiationThread: " + descriptorString);
		pStream.println("Thread is alive: " + this.isAlive());
		pStream.println("Current execution state:");
		if (negotiator != null)
			negotiator.printNegotiationHistory(pStream);
		else if (negotiatingStep != null && negotiatingStep.negotiator != null)
			negotiatingStep.negotiator.printNegotiationHistory(pStream);
		else
			pStream.println("No negotiation history");
		pStream
				.println("Stacktrace indicates negotiation initiator - NOT AN ERROR");
		pStream.println(stackTrace);
		return bStream.toString();
	}

	String shortToString() {
		return "JointGoalNegotiationThread: " + descriptorString;
	}
}
