// Abstract superclass for goal steps. All concrete goal steps appearing in a ABL program are subclasses of GoalStep. 

package abl.runtime;

import java.util.*;
import java.beans.*;
import java.lang.reflect.Method;

public class GoalStep extends ExecutableStep {

	protected Behavior child = null; // Child behavior reference

	// Enum constants for goal execution type
	public static final short SUBGOAL = 0;

	public static final short SPAWNGOAL_DEFAULT = 1;

	public static final short SPAWNGOAL_AT_NODE = 2;

	// Goal signature.
	private String signature;

	// Goal execution type flag indicating whether goal step is a subgoal or
	// spawngoal - defaults to subgoal.
	// The flag is set in the constructor on the concrete class.
	private short goalExecutionType = SUBGOAL;

	/**
	 * Hash set of behaviors which have already been tried for this goal (as a
	 * set, it is a collection that contains no duplicate elements). The
	 * behaviors are stored by a reference to the behavior ID. A reference to
	 * failedBehaviors is passed to the behavior arbiter on behaving entity so
	 * that they can be removed from the conflict set.
	 */
	protected HashSet failedBehaviors = new HashSet();

	private PropertyChangeSupport changes = new PropertyChangeSupport(this);

	private boolean succeeding = false;

	private NegotiateSubtreeRemovalThread currentNegotiationThread = null;

	private Object[] args; // The bound args for this goal step

	private MultiStepBehaviorWME rerootParent = null; // Reroot parent for
														// explicitly rooted
														// spawngoals

	// for spawngoal steps, set to true after the goal has been rerooted. Allows
	// execute to tell the difference between
	// the first execution of a spawngoal (which should just move the goal) and
	// subsequent execution.
	private boolean rerooted = false;

	// Constructor for GoalStep calls the super constructor.
	public GoalStep(int stepID, Behavior arg_parent, boolean arg_persistent,
			boolean arg_persistentWhenSucceeds,
			boolean arg_persistentWhenFails, boolean arg_ignoreFailure,
			boolean arg_effectOnly, boolean arg_teamEffectOnly,
			short arg_priority, short arg_priorityModifier, boolean arg_post,
			String arg_postMemory, Method arg_execute, Method arg_successTest,
			Method arg_successTestSensorFactory,
			AblNamedPropertySupport arg_propertyTable, String arg_signature,
			String[] arg_stepsIConflictWith) {
		super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds,
				arg_persistentWhenFails, arg_ignoreFailure, arg_effectOnly,
				arg_teamEffectOnly, arg_priority, arg_priorityModifier,
				arg_post, arg_postMemory, arg_execute, arg_successTest,
				arg_successTestSensorFactory, arg_propertyTable, GOAL,
				arg_stepsIConflictWith);
		signature = arg_signature;
		name = signature.substring(0, signature.indexOf("("));
	}

	// Version of constructor for explicit set of goalExecutionType
	public GoalStep(int stepID, Behavior arg_parent, boolean arg_persistent,
			boolean arg_persistentWhenSucceeds,
			boolean arg_persistentWhenFails, boolean arg_ignoreFailure,
			boolean arg_effectOnly, boolean arg_teamEffectOnly,
			short arg_priority, short arg_priorityModifier, boolean arg_post,
			String arg_postMemory, Method arg_execute, Method arg_successTest,
			Method arg_successTestSensorFactory,
			AblNamedPropertySupport arg_propertyTable, String arg_signature,
			String[] arg_stepsIConflictWith, short arg_goalExecutionType) {
		super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds,
				arg_persistentWhenFails, arg_ignoreFailure, arg_effectOnly,
				arg_teamEffectOnly, arg_priority, arg_priorityModifier,
				arg_post, arg_postMemory, arg_execute, arg_successTest,
				arg_successTestSensorFactory, arg_propertyTable, GOAL,
				arg_stepsIConflictWith);
		signature = arg_signature;
		name = signature.substring(0, signature.indexOf("("));
		goalExecutionType = arg_goalExecutionType;
	}

	// Public accessor for signature.
	String getSignature() {
		return signature;
	}

	// package assessor for goalExecutionType.
	short getGoalExecutionType() {
		return goalExecutionType;
	}

	// package accessor for rerooted
	boolean getRerooted() {
		return rerooted;
	}

	// returns false for all concrete GoalSteps
	boolean isJointGoal() {
		return false;
	}

	// Interface for adding a failed behavior to failedBehaviors.
	protected void addFailedBehavior(Behavior beh) {
		failedBehaviors.add(new Integer(beh.getID()));
	}

	private static String formatAction(int action) {
		switch (action) {
		case NegotiateSubtreeRemovalThread.SUCCEED:
			return "SUCCEED";
		case NegotiateSubtreeRemovalThread.FAIL:
			return "FAIL";
		case NegotiateSubtreeRemovalThread.RESET:
			return "RESET";
		default:
			return "INVALID_ACTION";
		}
	}

	private class NegotiateSubtreeRemovalThread extends
			JointGoalNegotiationThread {
		static final int INVALID_ACTION = 0;

		static final int SUCCEED = 1;

		static final int FAIL = 2;

		static final int RESET = 3;

		private List jointGoals = new Vector(0);

		private int actionOnCommit = INVALID_ACTION;

		NegotiateSubtreeRemovalThread(List jointGoals, int actionOnCommit) {
			super((JointGoalStep) null, GoalStep.this
					+ " NegotiateSubtreeRemovalThread("
					+ formatAction(actionOnCommit) + ")");
			assert ((actionOnCommit > INVALID_ACTION) && (actionOnCommit <= RESET));

			this.jointGoals = jointGoals;
			this.actionOnCommit = actionOnCommit;
		}

		int getActionOnCommit() {
			return actionOnCommit;
		}

		public void run() {
			Iterator iter = jointGoals.iterator();
			while (iter.hasNext()) {
				JointGoalStep js = (JointGoalStep) iter.next();
				if ((js.negotiator != null)
						&& (js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_SUCCEED)
						&& (js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_FAIL)
						&& (js.negotiator.getState() != JointGoalNegotiator.COMMIT_TO_REMOVE)) {

					((JointGoalNegotiationThread) Thread.currentThread())
							.waitForDecisionCycle();
				}

				// fixme: The only situation in which the negotiator is nulled
				// now is failure to find a behavior. May be able
				// to revert to the old code and assert that the negotiator
				// should not be null.
			}
			// When we get here everyone has commited.
			switch (actionOnCommit) {
			case SUCCEED:
				succeedStepSuper();
				break;
			case FAIL:
				failStepSuper();
				break;
			case RESET:
				resetStepBody();
				break;
			default:
				throw new AblRuntimeError("Unexpected action type "
						+ actionOnCommit);
			}
		}
	}

	// fixme: defined because GoalStep.super.<method> doesn't seem to work
	// inside an inner class.
	private void succeedStepSuper() {
		super.succeedStep();
	}

	private void failStepSuper() {
		super.failStep();
	}

	void succeedStep() {
		// Trace.ablTrace("(GoalStep)" + this.getSignature() + "succeedStep()");

		// Mark this goal as succeeding
		succeeding = true;

		if (!isJointGoal()) {
			// Not a joint goal - freeze the subtree and block on completion of
			// subtree negotiation.

			final List jointGoals = freezeSubtreeAndNegotiateRemoval();
			if (!jointGoals.isEmpty()) {
				// There are some joint goals in the subtree - wait for them to
				// commit.

				if (currentNegotiationThread == null
						|| currentNegotiationThread.getActionOnCommit() != NegotiateSubtreeRemovalThread.SUCCEED) {
					// Only register the succeed thread if we aren't currently
					// negotiating subtree removal or
					// we're not already negotiating succeed
					final NegotiateSubtreeRemovalThread negotiateSuccessThread = new NegotiateSubtreeRemovalThread(
							jointGoals, NegotiateSubtreeRemovalThread.SUCCEED);
					if (currentNegotiationThread != null)
						BehavingEntity.getBehavingEntity()
								.unregisterNegotiationThread(
										currentNegotiationThread);
					BehavingEntity.getBehavingEntity()
							.registerNegotiationThread(negotiateSuccessThread);
					currentNegotiationThread = negotiateSuccessThread;
				}
			} else
				// There are no joint goals in the subtree - can succeed
				// immediately.
				super.succeedStep();
		} else {
			// This is a joint goal - freeze and subtree block has already been
			// handled by JointGoalStep.succeedStep()
			if (child != null)
				// If the child has team_effect_only steps or has been succeeded
				// by a team member (one_needed_for_success),
				// need to remove the children of the child behavior.
				child.removeChildren();

			super.succeedStep();
		}
	}

	void failStep() {
		// Trace.ablTrace("(GoalStep)" + this.getSignature() + "failStep()");

		if (!isJointGoal()) {
			// Not a joint goal - freeze the subtree and block on completion of
			// subtree negotiation.

			final List jointGoals = freezeSubtreeAndNegotiateRemoval();
			if (!jointGoals.isEmpty()) {
				// There are some joint goals in the subtree - wait for them to
				// commit.

				if (currentNegotiationThread == null
						|| currentNegotiationThread.getActionOnCommit() != NegotiateSubtreeRemovalThread.FAIL) {
					// Only register the fail thread if we aren't currently
					// negotiating subtree removal or
					// we're not already negotiating FAIL
					final NegotiateSubtreeRemovalThread negotiateFailureThread = new NegotiateSubtreeRemovalThread(
							jointGoals, NegotiateSubtreeRemovalThread.FAIL);
					if (currentNegotiationThread != null)
						BehavingEntity.getBehavingEntity()
								.unregisterNegotiationThread(
										currentNegotiationThread);
					BehavingEntity.getBehavingEntity()
							.registerNegotiationThread(negotiateFailureThread);
					currentNegotiationThread = negotiateFailureThread;
				}
			} else {
				// There are no joint goals in the subtree - can fail
				// immediately.
				super.failStep();
			}
		} else
			// This is a joint goal - freeze and subtree block has already been
			// handled by JointGoalStep.succeedStep()
			super.failStep();
	}

	void resetStep() {
		if (!isJointGoal()) {
			// Not a joint goal - freeze the subtree and block on completion of
			// subtree negotiation.

			final List jointGoals = freezeSubtreeAndNegotiateRemoval();
			if (!jointGoals.isEmpty()) {
				// There are some joint goals in the subtree - wait for them to
				// commit.

				if (currentNegotiationThread == null
						|| currentNegotiationThread.getActionOnCommit() != NegotiateSubtreeRemovalThread.RESET) {
					// Only register the reset thread if we aren't currently
					// negotiating subtree removal or
					// we're not already negotiating RESET.
					final NegotiateSubtreeRemovalThread negotiateResetThread = new NegotiateSubtreeRemovalThread(
							jointGoals, NegotiateSubtreeRemovalThread.RESET);
					if (currentNegotiationThread != null)
						BehavingEntity.getBehavingEntity()
								.unregisterNegotiationThread(
										currentNegotiationThread);
					BehavingEntity.getBehavingEntity()
							.registerNegotiationThread(negotiateResetThread);
					currentNegotiationThread = negotiateResetThread;
				}

			} else
				// There are no joint goals in the subtree - can succeed
				// immediately.
				resetStepBody();
		} else
			// This is a joint goal - freeze and subtree block has already been
			// handled by JointGoalStep.succeedStep()
			resetStepBody();
	}

	// Resets this goal step. Called when you want to leave the goal step in the
	// ABT but make it "fresh".
	private void resetStepBody() {
		removeChild(true);
		failedBehaviors.clear();
		BehavingEntity.getBehavingEntity().resetStep(this);

	}

	/*
	 * Adds a child (a behavior) to the node. Throws a runtime error if there is
	 * already a child (child is not null), or if the ABTNode to add is null.
	 */
	void addChild(Behavior node) {
		if (node == null)
			throw new AblRuntimeError("Null ABTNode passed to addChild().");
		if (child != null)
			throw new AblRuntimeError(
					"addChild() called on a GoalStep with non-null child.");
		child = node;
		BehavingEntity.getBehavingEntity().addBehavior(child); // Update
																// BehavingEntity
																// state.
		changes.firePropertyChange("child", null, node); // Fire property
															// change for any
															// listeners.
		child.addChildren(); // Add step(s) as children to the child
								// behavior.
	}

	// Called by a parent behavior which is removing its steps.
	// Recursively remove the tree rooted at this goal.
	void removeGoal() {
		removeChild(true);
	}

	// Removes a child (a behavior) from a node.
	void removeChild(boolean isRecursive) {
		if (child != null) {
			if (isRecursive) {
				// Don't need to freeze, because removeChild is only called by
				// succeedBehavior or failBehavior, both of
				// which initiate freezeSubtreeAndNegotiateRemoval().
				child.removeBehavior(); // recursively remove child
			}

			changes.firePropertyChange("child", child, null); // Fire property
																// change for
																// any listeners
			child = null;
		}
	}

	// Returns true if the goal step is currently executing. A goal
	// step is executing if it has been expanded and it is not
	// suspended.
	boolean isExecuting() {
		if (isExpanded() && !isSuspended())
			return true;
		else
			return false;
	}

	// Returns true if the goal step has been expanded.
	boolean isExpanded() {
		if (child != null)
			return true;
		else
			return false;
	}

	// Called in chooseBehavior() when a behavior is found.
	final protected void executeBookkeeping() {
		final int parentBehaviorType = parent.getBehaviorType();
		if ((parentBehaviorType == Behavior.PARALLEL)
				|| (parentBehaviorType == Behavior.COLLECTION))
			((MultiStepBehavior) parent).pursueGoal(this);

		super.executeBookkeeping();
	}

	// reroots the goal at newParent
	private void rerootGoal(MultiStepBehaviorWME newParent) {
		// success of the step will cause removal of the corresponding
		// reflection WME
		succeedStep(); // Step succeeds immediately

		// this change is transparent to the reflection WMEs
		if (newParent == null)
			// spawning at root
			parent = BehavingEntity.getBehavingEntity()
					.getRootCollectionBehavior();
		else {
			// spawning at parent
			final Behavior tempParent = newParent.getBehavior();
			if (tempParent == null) {
				// an attempt was made to spawn a goal at an invalid behavior.
				// Print out warning info and fail.
				System.err
						.println("WARNING: unexected NULL behavior when attempting to reroot goal "
								+ this
								+ " at behavior "
								+ newParent
								+ ". Failing the step");
				failStep();
				return;
			} else
				parent = tempParent;
		}
		((MultiStepBehavior) parent).addChild(this); // this will cause a new
														// reflection wme to be
														// created
	}

	/*
	 * Chooses a behavior for a goal step given the signature, arguments and
	 * current set of previously failed behaviors. Returns the behavior if one
	 * is found, null otherwise.
	 */

	void chooseBehavior(final Object[] args) {
		/*
		 * Find an applicable behavior given the signature and arguments. If no
		 * applicable behavior found, returns null.
		 */
		final Behavior beh = BehavingEntity.getBehavingEntity()
				.chooseIndividualBehavior(args, failedBehaviors, this);

		if (beh != null) {
			/*
			 * If a behavior was found, add it as a child of this goal.
			 * addChild() also adds the children (steps) of the behavior as
			 * naked behaviors (with no steps) are never found in the ABT.
			 */

			assert (!beh.isJointBehavior());

			addChild(beh);
			executeBookkeeping();
		} else {
			// No behavior found. Fail the step.
			failStep();
		}
	}

	/*
	 * Returns true if this step is in the line of expansion of the most
	 * recently pursued goal, false otherwise. Overrides currentLineOfExpansion
	 * defined on Step.
	 */
	boolean currentLineOfExpansion(GoalStep currentGoal) {
		/*
		 * Asks the parent if this goal is the last pursued goal. Ignores the
		 * passed in currentgoal.
		 */
		return parent.currentLineOfExpansion(this);
	}

	/** Suspend every executable step of the subtree rooted at this GoalStep. */
	void suspend(ExecutableStep step) {
		suspend(step, true);
	}

	void suspend(ExecutableStep step, boolean isRecursive) {
		if (!isSuspended())
			// If the step is not already currently suspended, perform
			// bookkeeping on behaving entity
			BehavingEntity.getBehavingEntity().suspendStep(this);
		super.suspend(step);
		if (isRecursive)
			if (child != null) {
				child.suspend(step);
			}
	}

	void suspendSkipJointGoals(ExecutableStep step) {
		if (!isSuspended())
			// If the step is not already currently suspended, perform
			// bookkeeping on behaving entity
			BehavingEntity.getBehavingEntity().suspendStep(this);
		super.suspend(step);
		if (child != null) {
			child.suspendSkipJointGoals(step);
		}
	}

	// Meta control for suspend
	void metaSuspend() {
		metaSuspend(true);
	}

	void metaSuspend(boolean isRecursive) {
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().suspendStep(this);
		super.metaSuspend();
		if (isRecursive)
			if (child != null)
				child.metaSuspend();
	}

	void metaSuspendSkipJointGoals() {
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().suspendStep(this);
		super.metaSuspend();
		if (child != null)
			child.metaSuspendSkipJointGoals();
	}

	// Joint suspend. negotiateSuspensionOfSubtree() is responsible for
	// recursing the subtree so jointSuspend()
	// doesn't recurse.
	void jointSuspend() {
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().suspendStep(this);
		super.jointSuspend();
	}

	// Unsuspend the goal step. Unsuspending the children is handled by the
	// unsuspender.
	void unsuspend(ExecutableStep step) {
		super.unsuspend(step);
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().unsuspendStep(this);
	}

	// Meta control for unsuspend
	void metaUnsuspend() {
		super.metaUnsuspend();
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().unsuspendStep(this);
		if (child != null)
			child.metaUnsuspend();
	}

	void jointUnsuspend() {
		super.jointUnsuspend();
		if (!isSuspended())
			BehavingEntity.getBehavingEntity().unsuspendStep(this);
		if (child != null)
			child.jointUnsuspend();
	}

	void addChildChangeListener(PropertyChangeListener l) {
		changes.addPropertyChangeListener("child", l);
	}

	void removeChildChangeListener(PropertyChangeListener l) {
		changes.removePropertyChangeListener("child", l);
	}

	// Negotiates the removal of the joint goals in the subtree and returns a
	// list of these goals.
	List<Step> freezeSubtreeAndNegotiateRemoval() {
		BehavingEntity.getBehavingEntity().removeSuccessTest(this);
		if (child != null)
			return child.freezeSubtreeAndNegotiateRemoval();
		else
			return new Vector<Step>(0); // return an empty list
	}

	// Negotiates the suspension of the joint goals in the subtree and returns a
	// list of these goals.
	// Called during suspension initiated by team member.
	List negotiateSuspensionOfSubtree() {
		jointSuspend();
		if (child != null)
			return child.negotiateSuspensionOfSubtree();
		else
			return new Vector(0); // return an empty list
	}

	/** Negotiates the suspension of the joint goals in the subtree and returns a
	* list of these goals.
	* Called during suspension initiated by this agent. */
	List negotiateSuspensionOfSubtree(ExecutableStep step) {
		if (step != null)
			suspend(step, false);
		else
			metaSuspend(false);
		if (child != null)
			return child.negotiateSuspensionOfSubtree(step);
		else
			return new Vector(0); // return an empty list
	}

	void processStepRemoval() {
		if (succeeding) {
			if (goalExecutionType == SUBGOAL || rerooted == true)
				super.processStepRemoval();
			else {
				// processing step removal for a spawngoal reroot - next removal
				// should unsuspend suspended steps
				rerooted = true;
				succeeding = false;
			}
		} else
			super.processStepRemoval();
	}

	private final void bindArgs() {
		if (execute != null) {
			// The goal step has args, so a bind args case has been defined in
			// GoalStepExecute.
			final Object[] exeArgs = { new Integer(stepID),
					parent.getBehaviorVariableFrame(),
					BehavingEntity.getBehavingEntity() };
			try {
				final Object[] tempArgs = (Object[]) execute.invoke(null,
						exeArgs);
				if (goalExecutionType != SPAWNGOAL_AT_NODE) {
					args = tempArgs;
				} else {
					// Args shouldn't include the first element of the array
					final int len = tempArgs.length - 1;
					args = new Object[len];
					for (int i = 0; i < len; i++)
						args[i] = tempArgs[i + 1];

					rerootParent = (MultiStepBehaviorWME) tempArgs[0];
				}
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking execute", e);
			}
		} else
			args = null; // No args
	}

	void execute() {
		checkForConflictsOnExecution();
		if (!isSuspended()) {
			if (goalExecutionType != SUBGOAL) {
				if (!rerooted) {
					bindArgs();

					// If this goal is not an explicitly rooted spawngoal,
					// rerootParent == null, otherwise rerootParent
					// is set by bindArgs().
					rerootGoal(rerootParent);
				} else {
					// Already rerooted - just execute with the bound arguments
					chooseBehavior(args);
				}
			} else {
				// Not a spawngoal, bind the args and immediately choose a
				// behavior.
				bindArgs();
				chooseBehavior(args);
			}
		}
	}
}
