// Interface for steps which execute over time (goal and primitive steps). 

package abl.runtime;

import java.lang.reflect.Method;

public abstract class ExecutableStep extends Step
{
    // Array of step names (act or goal names) that conflict with this step. 
    protected String[] stepsIConflictWith = null;

    // Name of the executable step. Set by the constructor of the
    // concrete goal or primitive step class.
    protected String name;

    public ExecutableStep(int stepID, Behavior arg_parent, boolean arg_persistent, boolean arg_persistentWhenSucceeds, 
			  boolean arg_persistentWhenFails, boolean arg_ignoreFailure, boolean arg_effectOnly, 
			  boolean arg_teamEffectOnly, short arg_priority, short arg_priorityModifier, boolean arg_post, 
			  String arg_postMemory, Method arg_execute, Method arg_successTest, 
			  Method arg_successTestSensorFactory, AblNamedPropertySupport arg_propertyTable, 
			  short arg_stepType, String[] arg_stepsIConflictWith)
    {
	super(stepID, arg_parent, arg_persistent, arg_persistentWhenSucceeds, arg_persistentWhenFails, arg_ignoreFailure, 
	      arg_effectOnly, arg_teamEffectOnly, arg_priority, arg_priorityModifier, arg_post, arg_postMemory, 
	      arg_execute, arg_successTest, arg_successTestSensorFactory, arg_propertyTable, arg_stepType);
	stepsIConflictWith = arg_stepsIConflictWith;
    }

    protected void checkForConflictsOnExecution()
    {
	if (stepsIConflictWith != null) {
	    BehavingEntity.getBehavingEntity().findConflictsWithCurrentlyExecutingSteps(this, BehavingEntity.suspendOnExecute);
	}
    }

    protected void initConflictSteps(String[] steps)
    {
	stepsIConflictWith = steps;
    }

    // Returns true if the step is currently executing, false otherwise. 
    abstract boolean isExecuting();

    // get accessor for stepsIConflictWith
    String[] getConflicts()
    {
	if (stepsIConflictWith != null)
	    return (String[])stepsIConflictWith.clone();
	else
	    return new String[0];
    }

    private void unsuspendStepsIHaveSuspended()
    {
	// Use arrays to avoid concurrent modification errors. 
	Step[] stepsIHaveSuspendedArray = 
	    (Step[])stepsIHaveSuspended.toArray(new Step[stepsIHaveSuspended.size()]);
	int i;
	
	for(i = 0; i < stepsIHaveSuspendedArray.length; i++) {
	    stepsIHaveSuspendedArray[i].unsuspend(this);
	}
    }

    // Unsuspended any steps suspended on this step. Remove this step from the suspender list of any suspending steps. 
    void processStepRemoval()
    {
	// Use arrays to avoid concurrent modification errors. 
	unsuspendStepsIHaveSuspended();

	updateStepsSuspendingMe();
    }
    
    // Returns the step name. 
    String getName()
    {
	return name;
    }

    // Adds a step to the list of suspending steps. 
    void suspend(ExecutableStep step)
    {
	if (isExecuting()) {
	    // If this step is executing when it's suspended, unsuspended any steps suspended on this step.
	    unsuspendStepsIHaveSuspended();
	}

	super.suspend(step);
    }

    // Meta controls for suspend
    void metaSuspend()
    {
	if (isExecuting()) {
	    // If this step is executing when it's suspended, unsuspended any steps suspended on this step.
	    unsuspendStepsIHaveSuspended();
	}

	super.metaSuspend();
    }	    

    // Joint suspension (suspension initiated by team member)
    void jointSuspend()
    {
	if (isExecuting())
	    // If this step is executing when it's suspended, unsuspended any steps suspended on this step.
	    unsuspendStepsIHaveSuspended();

	super.jointSuspend();
    }	    
    
    // Adds a step to the list of steps I'm suspending.
    void suspenderFor(Step step)
    {
	stepsIHaveSuspended.add(step);
    }

    // Removes a step from the list of steps I'm suspending.
    void unsuspenderFor(Step step)
    {
	stepsIHaveSuspended.remove(step);
    }
}
