/* GUI debugger for ABL. */
package abl.runtime;

import javax.swing.*;
import javax.swing.tree.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import wm.*;
// import debug.Assert;

public class Debugger {

    // The number of decision cycles to count in order to compute the number of decision cycles per second
    private static final String DECISION_CYCLE_TEXT = "Decisions per second";
    private static final String CONDITION_TIME_TEXT = "Continuous condition time/100 cycles";

    private static final int minimumViewWidth = 200;
    private static final int preferredViewWidth = 500;
    private static final int minimumViewHeight = 100;
    private static final int preferredViewHeight = 200;

    private static final int minimumTopPaneWidth = 2 * minimumViewWidth;
    private static final int preferredTopPaneWidth = 2 * preferredViewWidth;
    private static final int minimumTopPaneHeight = 2 * minimumViewHeight;
    private static final int preferredTopPaneHeight = 2 * preferredViewHeight;

    private static final int minimumMainSplitPaneWidth = minimumViewWidth;
    private static final int preferredMainSplitPaneWidth = preferredViewWidth;
    private static final int minimumMainSplitPaneHeight = minimumTopPaneHeight + minimumViewHeight;
    private static final int preferredMainSplitPaneHeight = preferredTopPaneHeight + preferredViewHeight;

    private static final int minimumButtonPaneWidth = minimumMainSplitPaneWidth;
    private static final int preferredButtonPaneWidth = preferredMainSplitPaneWidth;
    private static final int buttonPaneHeight = 90;

    private BehavingEntity entity; // the entity being debugged 

    private WorkingMemoryDebugger wmPanel; // Debug pane for working memory.
    private JTree ABTTree; // Tree display for the ABT.
    protected JTextArea traceArea;
    
    // fixme: declarations for styled documents
    // protected JTextPane traceArea; 
    // protected Document traceDocument;

    protected JScrollBar traceViewScroller;
    protected JScrollPane traceView;
    private boolean bBreakDecisionCycle = true; 
    private JLabel decisionsPerSecondLabel; 
    private JLabel accumulatedConditionTimeLabel;
    private JCheckBox ABTUpdateTreeButton; 

    private int decisionCycleCounterForDecisionsPerMillisecond = 0;
    private int decisionCycleCounterForConditionTime = 0;
    private long accumulatedContinuousConditionTime = 0;
    private long startTime = System.currentTimeMillis();

    protected JCheckBox traceToScreenChkBox;
    protected JCheckBox traceToBufferChkBox;

    protected final TraceListener traceListener = new TraceListener();
    protected java.util.List traceBuffer = new LinkedList();
    protected Set tracedBehaviors = new HashSet();

    private JFrame debugFrame;
    private JButton breakBtn;
    private JButton stepBtn;
    private JButton continueBtn;

    private final AblEventSupport listenerSupport = new AblEventSupport();
    private final static HashSet debuggedBehaviors = new HashSet();

    // fixme: added to support auto-scrolling during trace addition (an optimization). Fix or remove
    /* protected class TextAreaRowHeight extends JTextArea
    {
	int getRowHeightAccessible() { return super.getRowHeight(); }
	} */

    protected class ABTCellRenderer extends DefaultTreeCellRenderer 
    {
	
	public Component getTreeCellRendererComponent(JTree tree,
						      Object value,
						      boolean sel,
						      boolean expanded,
						      boolean leaf,
						      int row,
						      boolean hasFocus) 
	{
	    
	    super.getTreeCellRendererComponent(tree, value, sel,
					       expanded, leaf, row,
					       hasFocus);
	    // If the current node (value) is a suspended step, change the color used to render the node to red.
	    // If the current node (value) is a negotiating step, change the color used to render the node to blue.
	    try {
		DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
		Step step = (Step)node.getUserObject();
		if (step.isSuspended())
		    setForeground(Color.red);
		else if ((step.getStepType() == Step.GOAL) &&
			 (((GoalStep)step).isJointGoal()) &&
			 (((JointGoalStep)step).getIsNegotiating()))
		    setForeground(Color.blue);
		else
		    setForeground(Color.black);
		return this;
	    } 
	    catch (ClassCastException e) { 
		// Node doesn't contain a step
		setForeground(Color.black);
		return this; 
	    }
	    catch (NullPointerException e) {
		// Node is empty
		setForeground(Color.black);
		return this;
	    }
	}
    }

    class TraceListener implements AblListener
    {
	final static String lineSep = "\n";
	final Hashtable timeTable = new Hashtable();

	private String getNegotiationActionString(JointGoalNegotiatorDebug.JointGoalNegotiationInfo i)
	{
	    switch (i.intention) {
	    case JointGoalNegotiator.ENTER:
		return "entry";
	    case JointGoalNegotiator.REFUSE_ENTRY:
		return "refuse entry";
	    case JointGoalNegotiator.SUCCEED:
		return "succeed";
	    case JointGoalNegotiator.FAIL:
		return "fail";
	    case JointGoalNegotiator.REMOVE:
		return "remove";
	    case JointGoalNegotiator.SUSPEND:
		return "suspend";
	    case JointGoalNegotiator.UNSUSPEND:
		return "unsuspend";
	    default:
		return "unrecognized action";
	    }
	    
	}

	// Given an AblEvent, sets the elapsedTime if the AblEvent is a completion event
	public AblEvent computeElapsedTime(AblEvent e) {
	    Object source = e.getSource();
	    Long time = (Long)timeTable.get(source);
	    int eventType = e.getType();
	    switch (e.getType()) {
	    case AblEvent.ACT_EXECUTION:
	    case AblEvent.SUBGOAL_EXECUTION:
	    case AblEvent.MENTAL_STEP_EXECUTION:
		timeTable.put(source, new Long(System.currentTimeMillis()));
		break;
	    case AblEvent.BEHAVIOR_COMPLETION:
	    case AblEvent.ACT_COMPLETION:
	    case AblEvent.SUBGOAL_COMPLETION:
	    case AblEvent.MENTAL_STEP_COMPLETION:
		if (time != null) {
		    e.setElapsedTime(System.currentTimeMillis() - time.longValue());
		    timeTable.remove(source);
		}
		break;
	    case AblEvent.BEHAVIOR_EXECUTION:
		timeTable.put(source, new Long(System.currentTimeMillis()));
		Object o = e.getObject();
		if (o != null) 
		    time = (Long)timeTable.get(o);
		if (time != null) 
		    e.setElapsedTime(System.currentTimeMillis() - time.longValue());
		break;
	    }
	    return e;
	}

	public String formatBehaviorTrace(AblEvent e)
	{

	    Object o = e.getObject();
	    Object source = e.getSource();
	    String actionMessage;
	    long time = e.getElapsedTime();
	
	    switch (e.getType()) {
	    case AblEvent.PRECONDITION_EXECUTION:
		actionMessage = "precondition tested: " + o;
		break;
	    case AblEvent.BEHAVIOR_EXECUTION:
		actionMessage = "behavior chosen";
		break;
	    case AblEvent.BEHAVIOR_COMPLETION:
		if (((Boolean)o).booleanValue())
		    actionMessage = "behavior succeeded";
		else
		    actionMessage = "behavior failed";
		break;
	    case AblEvent.CONTEXT_CONDITION_FAILURE:
		actionMessage = "context condition failed";
		break;
	    case AblEvent.ACT_EXECUTION:
		actionMessage = "execution";
		break;
	    case AblEvent.ACT_COMPLETION:
		if (((Boolean)o).booleanValue())
		    actionMessage = "succeeded";
		else 
		    actionMessage = "act failed";
		break;
	    case AblEvent.SUBGOAL_EXECUTION:
		actionMessage = "execution";
		break;
	    case AblEvent.SUBGOAL_COMPLETION:
		if (((Boolean)o).booleanValue())
		    actionMessage = "succeeded";
		else 
		    actionMessage = "failed";
		break;
	    case AblEvent.MENTAL_STEP_EXECUTION:
		actionMessage = "execution";
		break;
	    case AblEvent.MENTAL_STEP_COMPLETION:
		actionMessage = "succeeded";
		break;
	    case AblEvent.WAIT_STEP_COMPLETION:
		actionMessage = "succeeded";
		break;
	    case AblEvent.FAILSTEP_EXECUTION:
		actionMessage = "execution";
		break;
	    case AblEvent.FAILSTEP_COMPLETION:
		actionMessage = "failed";
		break;		
	    case AblEvent.SUCCEEDSTEP_EXECUTION:
		actionMessage = "execution";
		break;
	    case AblEvent.SUCCEEDSTEP_COMPLETION:
		actionMessage = "succeeded";
		break;		
	    case AblEvent.SUCCESS_TEST_SUCCESS:
		actionMessage = "success test succeeded";
		break;
	    case AblEvent.SPAWNGOAL_AT_ROOT:
		actionMessage = "re-rooting subgoal at ABT root";
		break;
	    case AblEvent.INITIATE_NEGOTIATION:
		actionMessage = "initiating " + getNegotiationActionString((JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o) + " negotiation";
		break;
	    case AblEvent.COMPLETE_NEGOTIATION:
		actionMessage = "completing " + getNegotiationActionString((JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o) + " negotiation";
		break;
	    case AblEvent.INITIATE_INTENTION:
		actionMessage = "initiating " + getNegotiationActionString((JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o) + " intention";
		break;
	    case AblEvent.COMMIT_TO_INTENTION:
		actionMessage = "committing to " + getNegotiationActionString((JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o) + " intention";
		break;
	    case AblEvent.PROCESS_INTENTION:
		actionMessage = "process " + getNegotiationActionString((JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o) + " intention";
		break;
	    default:
		actionMessage = "unrecognized action " + e.getType();
	    }
	    int nestLevel = e.getNestLevel();
	    StringBuffer blankBuffer = new StringBuffer(2 * nestLevel);
	    for(int i = 0; i < nestLevel; i++)
		blankBuffer.append("  ");
	    
	    String sourceMessage = "";
	    switch (e.getSourceType()) {
	    case AblEvent.BEHAVIOR:
		if ((e.getType() == AblEvent.PRECONDITION_EXECUTION) || 
		    (e.getType() == AblEvent.BEHAVIOR_EXECUTION))
		    sourceMessage = ((__BehaviorDesc)source).uniqueName;
		else
		    sourceMessage = ((__BehaviorDesc)source).signature; 
		break;
	    case AblEvent.NEGOTIATOR:
		JointGoalNegotiatorDebug.JointGoalNegotiationInfo i = (JointGoalNegotiatorDebug.JointGoalNegotiationInfo)o;
		if (i.g != null)
		    sourceMessage = i.g.getSignature();
		if (i.behavingEntity != null) 
		    sourceMessage = sourceMessage + " from " + i.behavingEntity;
		break;
	    case AblEvent.STEP:
		sourceMessage = ((Step)source).toString();
		break;
	    default:
		sourceMessage = "unrecognized source " + e.getSourceType();
	    }
	    if (time != -1)
		actionMessage = actionMessage + " elapsed time = " + time;

	    if (sourceMessage.equals(""))
		return blankBuffer + actionMessage + lineSep;
	    else
		return blankBuffer + sourceMessage + ": " + actionMessage + lineSep;
	}	    
	
	public void eventHappened(final AblEvent e)
	{
	    if (traceToScreenChkBox.isSelected()) {
		SwingUtilities.invokeLater(new Runnable() 
		    {
			public void run() { 
			    traceArea.append(formatBehaviorTrace(computeElapsedTime(e))); 
			}
		    });
		
		// fixme: string insert for styled documents
		/* try {
		    traceDocument.insertString(traceDocument.getLength(), formatBehaviorTrace(e), null);
		} catch (BadLocationException ex) { throw new RuntimeError("Trace error", ex); }
		*/ 
		
		// fixme: figure out how to page to a new view
		/* int extent = traceView.getViewport().getExtentSize().height;
		int scrollMax = traceViewScroller.getMaximum();
		System.out.println("exent: " + extent);
		System.out.println("scrollmax: " + scrollMax);
		if (scrollMax > traceViewScroller.getValue() + extent) {
		    System.out.println("Setting new rows");
		    Point p = traceView.getViewport().getViewPosition();
		    p.setLocation(p.getX(), p.getY() + extent);
		    traceView.getViewport().setViewPosition(p);
		    // traceArea.setRows(traceArea.getLineCount() + extent/traceArea.getRowHeightAccessible());
		    // traceViewScroller.setValue(traceViewScroller.getMaximum() - extent);
		    } */
	    }
	    
	    if (traceToBufferChkBox.isSelected()) 
		traceBuffer.add(computeElapsedTime(e));
	}
    }

    class StringIgnoreCaseComparator implements Comparator {
	public int compare(Object o1, Object o2)
	{
	    return ((String)o1).compareToIgnoreCase((String)o2);
	}
    }

    class SelectTraceDialog extends JDialog
    {
	private final JPanel contentPane;
	private final JPanel checkBoxes;
	private final JScrollPane checkBoxesView;
	private final JPanel buttonContainer;
	private final JButton okBtn;
	private final JButton cancelBtn;
	private final JButton selectAllBtn;
	private final JButton clearAllBtn;
	final SelectTraceDialog me; // used by enclosed classes

	class ScrollableJPanel extends JPanel implements Scrollable
	{
	    public Dimension getPreferredScrollableViewportSize() { return this.getPreferredSize(); }
	    public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) 
	    { 
		return new JCheckBox("Sample").getPreferredSize().height;
	    }
	    public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction)
	    {
		JCheckBox sample = new JCheckBox("Sample");
		if (orientation == SwingConstants.VERTICAL)
		    return visibleRect.height - sample.getPreferredSize().height;
		else
		    return visibleRect.width - sample.getPreferredSize().height;
	    }
	    public boolean getScrollableTracksViewportWidth() { return false; }
	    public boolean getScrollableTracksViewportHeight() { return false; }
	}

	SelectTraceDialog()
	{
	    me = this;

	    setTitle("Select behaviors to trace");

	    // Set the content pane
	    contentPane = new JPanel();
	    setContentPane(contentPane);
	    contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));

	    checkBoxes = new ScrollableJPanel();
	    checkBoxes.setLayout(new BoxLayout(checkBoxes, BoxLayout.Y_AXIS));
	    
	    checkBoxesView = new JScrollPane(checkBoxes);
	    checkBoxesView.getViewport().setScrollMode(JViewport.BACKINGSTORE_SCROLL_MODE);
	    
	    buttonContainer = new JPanel();
	    buttonContainer.setLayout(new BoxLayout(buttonContainer, BoxLayout.X_AXIS));
	    buttonContainer.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

	    // create the buttons
	    selectAllBtn = new JButton("Select All");
	    clearAllBtn = new JButton("Clear All");
	    okBtn = new JButton("OK");
	    cancelBtn = new JButton("Cancel");

	    // add the OK listener
	    okBtn.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			Component[] comp = checkBoxes.getComponents();
			for(int i = 0; i < comp.length; i++) {
			    JCheckBox cbox = (JCheckBox)comp[i];
			    if (cbox.isSelected()) {
				// check box is selected - add listener if it wasn't previously in the selected set
				if (!tracedBehaviors.contains(cbox.getText())) {
				    entity.traceBehaviorSignature(cbox.getText());
				    tracedBehaviors.add(cbox.getText()); // add behavior sig to hash set
				}
			    }
			    else {
				// check box is not selected - remove listener it is was in the selected set
				if (tracedBehaviors.contains(cbox.getText())) {
				    entity.untraceBehaviorSignature(cbox.getText());
				    tracedBehaviors.remove(cbox.getText()); // remove behavior sig from hash set
				}
			    }
			}
			me.dispose();
		    } });

	    // add the cancel listener
	    cancelBtn.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			me.dispose();
		    }
		});

	    // add the select all listener
	    selectAllBtn.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			Component[] comp = checkBoxes.getComponents();
			for(int i = 0; i < comp.length; i++) 
			    ((JCheckBox)comp[i]).setSelected(true); 
		    }
		});

	    // add the clear all listener
	    clearAllBtn.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			Component[] comp = checkBoxes.getComponents();
			for(int i = 0; i < comp.length; i++) 
			    ((JCheckBox)comp[i]).setSelected(false); 
		    }
		});
			
	    // add the containers
	    getContentPane().add(checkBoxesView);
	    getContentPane().add(buttonContainer);
	    buttonContainer.add(selectAllBtn);
	    buttonContainer.add(clearAllBtn);
	    buttonContainer.add(Box.createHorizontalStrut(30));
	    buttonContainer.add(okBtn);
	    buttonContainer.add(cancelBtn);
	    
	    // set the maximum height of the button container
	    Dimension buttonContainerSize = buttonContainer.getMaximumSize();
	    buttonContainer.setMaximumSize(new Dimension(buttonContainerSize.width, 30));

	    // set the minimum size of the button container
	    buttonContainer.setMinimumSize(new Dimension(200, 30));

	    // add the check boxes
	    
	    Set behSet = entity.getRegisteredBehaviors();
	    String[] behs = (String[])behSet.toArray(new String[behSet.size()]);
	    
	    // sort the behavior names for ease of selection
	    Arrays.sort(behs, new StringIgnoreCaseComparator());

	    for(int i = 0; i < behs.length; i++) {
		boolean select = tracedBehaviors.contains(behs[i]); // get the selected state of the behavior named behs[i]
		JCheckBox behaviorCheckBox = new JCheckBox(behs[i], select);
		checkBoxes.add(behaviorCheckBox);
	    }

	    // If there are more checkboxes than can fit on the screen, set the height of the check box scroll pane to the maximum height that 
	    // will fit on the screen.
	    JCheckBox tempBox = new JCheckBox();
	    int currentPreferredHeight = tempBox.getPreferredSize().height * behs.length;
	    Dimension screenSize = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration().getBounds().getSize();
	    if (currentPreferredHeight > (screenSize.height - 60)) 
		checkBoxesView.setPreferredSize(new Dimension(checkBoxes.getPreferredSize().width, screenSize.height - 100)); 
	   
	    setModal(true);
	    pack();
	}
    }

    // fixme: added to support styled text with no wrapping. Fix or delete.
    /* 
    // view for the trace pane which doesn't support breaking
    class TraceView extends LabelView 
    {
	public TraceView(Element elem) { super(elem); }

	public int getBreakWeight(int axis, float pos, float len)
	{
	    if (axis == Y_AXIS)
		return super.getBreakWeight(axis, pos, len);
	    else {
		System.out.println("getBreakWeight");
		Document doc = getDocument(); // get the doc associated with this view
		try {
		    if (doc.getText((int)pos, 1).equals("\n"))
			return ForcedBreakWeight;
		    else
			return BadBreakWeight;
		} catch (BadLocationException e) { throw new RuntimeError("Error in trace view", e); }
	    }
	}

    }

    class TraceViewFactory implements ViewFactory
    {
	public View create(Element elem) { return new TraceView(elem); }
    }
    
    class MyBoxView extends BoxView
    {
	public MyBoxView(Element elem) { super(elem, Y_AXIS); }
	public ViewFactory getViewFactory() 
	{ 
	    return new TraceViewFactory(this);
	}
    }
		
	    
    // editor kit implementing ViewFactory which returns NoBreakView
    class TraceEditorKit extends StyledEditorKit implements ViewFactory
    {
	private MyBoxView view = null;
	public ViewFactory getViewFactory() { return this; }

	public View create(Element elem) { return new MyBoxView(elem); }
    } */


    public Debugger(BehavingEntity entity) {
	try {
	    UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
	} catch (Exception e) {throw new AblRuntimeError("Error setting the look and feel in the debugger");}

	wmPanel = entity.getWMDebugger();

	/* When the tree view for the ABT is first created, it is
           empty. On each iteration through the decision cycle, the
           ABT is traversed to create a TreeModel. This TreeModel is
           then set as the current model of the tree view. */
	ABTTree = new JTree((Object[])null);
	ABTTree.setRootVisible(true);
	ABTTree.setCellRenderer(new ABTCellRenderer());
	JPanel ABTTreeViewPanel = new JPanel();
	ABTUpdateTreeButton = new JCheckBox("Continuously update ABT");
	JScrollPane ABTTreeView = new JScrollPane(ABTTree);
	ABTTreeView.getViewport().setScrollMode(JViewport.BACKINGSTORE_SCROLL_MODE);
	ABTTreeViewPanel.setLayout(new BoxLayout(ABTTreeViewPanel, BoxLayout.Y_AXIS));
	ABTTreeViewPanel.add(ABTUpdateTreeButton);
	ABTTreeViewPanel.add(ABTTreeView);
	ABTTreeViewPanel.setBorder(BorderFactory.createTitledBorder("Active Behavior Tree"));

	// Create the buttons
	breakBtn = new JButton("Break");
	stepBtn = new JButton("Step");
	continueBtn = new JButton("Continue");

	// Add the listeners to the buttons
	breakBtn.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		bBreakDecisionCycle = true;
		debugFrame.getRootPane().setDefaultButton(continueBtn);
	    } });

	stepBtn.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    releaseDecisionThread();
		} });

	continueBtn.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    wmPanel.clearWMView();
		    clearABTTreeModel();
		    bBreakDecisionCycle = false;
		    releaseDecisionThread();
		    debugFrame.getRootPane().setDefaultButton(breakBtn);
		} });

	JSplitPane topPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, wmPanel, ABTTreeViewPanel);
	topPane.setOneTouchExpandable(true);
	wmPanel.setMinimumSize(new Dimension(minimumViewWidth, minimumViewHeight));
	ABTTreeViewPanel.setMinimumSize(new Dimension(minimumViewWidth, minimumViewHeight));
	wmPanel.setPreferredSize(new Dimension(preferredViewWidth, preferredViewHeight));
	ABTTreeViewPanel.setPreferredSize(new Dimension(preferredViewWidth, preferredViewHeight));
	
	// Define the trace panel

	// fixme: use a text pane for styled text. Need to figure out how not to wrap.
	// traceArea = new JTextPane();
	// traceArea.setEditorKit(new TraceEditorKit());
	// traceDocument = traceArea.getDocument();

	traceArea = new JTextArea();
	traceArea.setEditable(false);
	traceView = new JScrollPane(traceArea); 
	traceViewScroller = traceView.getVerticalScrollBar();
	traceView.getViewport().setScrollMode(JViewport.BACKINGSTORE_SCROLL_MODE);
	
	// Lame support for searching trace. When you click on the trace, it selects all the text in the TextArea and 
	// copies it to the clipboard. You can then paste into your favorite editor and do searches.
	traceArea.addMouseListener(new MouseAdapter() {
		public void mouseClicked(MouseEvent e) {
		    traceArea.selectAll();
		    traceArea.copy();
		}
	    } );

	JButton selectTraceBtn = new JButton("Select Trace");
	traceToScreenChkBox = new JCheckBox("Trace to screen");
	traceToBufferChkBox = new JCheckBox("Trace to buffer");
	JButton clearScreenBtn = new JButton("Clear Screen");
	JButton clearBufferBtn = new JButton("Clear Buffer");
	JButton replayBufferBtn = new JButton("Replay Buffer");

	selectTraceBtn.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		new SelectTraceDialog().setVisible(true);
	    } });

	clearScreenBtn.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    traceArea.setText(""); 
		}
	    });

	clearBufferBtn.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    traceBuffer.clear(); 
		}
	    });

	replayBufferBtn.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    traceArea.setText(""); 
		    Iterator i = traceBuffer.iterator();
		    while(i.hasNext()) {
			AblEvent event = (AblEvent)i.next();
			if (event.getSourceType() == AblEvent.BEHAVIOR) {
			    __BehaviorDesc desc = (__BehaviorDesc)event.getSource();
			    if (tracedBehaviors.contains(desc.signature)) {
				traceArea.append(traceListener.formatBehaviorTrace(event));
			    
				// fixme: string insert for styled documents
				/* try {
				   traceDocument.insertString(traceDocument.getLength(), 
				   traceListener.formatBehaviorTrace(event),
				   null);
				   } catch (BadLocationException ex) { throw new RuntimeError("Trace error", ex); }
				*/ 
			    }
			}
			else if (event.getSourceType() == AblEvent.NEGOTIATOR) {
			    JointGoalNegotiatorDebug.JointGoalNegotiationInfo info = (JointGoalNegotiatorDebug.JointGoalNegotiationInfo)event.getObject();
			    if (info.g != null)
				if (tracedBehaviors.contains(info.g.getParent().getSignature()))
				    traceArea.append(traceListener.formatBehaviorTrace(event));
			    else
				traceArea.append(traceListener.formatBehaviorTrace(event));
			}
			else {
			    // a step source
			    Step s = (Step)event.getSource();
			    if (tracedBehaviors.contains(s.getParent().getSignature())) {
				traceArea.append(traceListener.formatBehaviorTrace(event));

				// fixme: string insert for styled documents
				/* try {
				    traceDocument.insertString(traceDocument.getLength(),
				    traceListener.formatBehaviorTrace(event),
				    null);
				    } catch (BadLocationException ex) { throw new RuntimeError("Trace error", ex); } 
				*/
			    }
			}
		    }
		}
	    });

	JPanel traceViewPanel = new JPanel();
	traceViewPanel.setLayout(new BoxLayout(traceViewPanel, BoxLayout.Y_AXIS));

	// setup the button panels
	JPanel buttonPanelMain = new JPanel();
	buttonPanelMain.setLayout(new BoxLayout(buttonPanelMain, BoxLayout.Y_AXIS));
	buttonPanelMain.setMaximumSize(new Dimension(400, 120));
	buttonPanelMain.setAlignmentX(Component.CENTER_ALIGNMENT);

	JPanel buttonPanel1 = new JPanel();
	buttonPanel1.setLayout(new BoxLayout(buttonPanel1, BoxLayout.X_AXIS));
	buttonPanel1.setAlignmentX(Component.LEFT_ALIGNMENT);

	JPanel buttonPanel2 = new JPanel();
	buttonPanel2.setLayout(new BoxLayout(buttonPanel2, BoxLayout.X_AXIS));
	buttonPanel2.setAlignmentX(Component.LEFT_ALIGNMENT);

	traceViewPanel.add(buttonPanelMain);
	buttonPanelMain.add(buttonPanel1);
	buttonPanelMain.add(buttonPanel2);
	
	// add buttons to button panels
	buttonPanel1.add(selectTraceBtn);
	buttonPanel1.add(Box.createHorizontalStrut(10));
	buttonPanel1.add(traceToScreenChkBox);
	buttonPanel1.add(Box.createHorizontalStrut(10));
	buttonPanel1.add(traceToBufferChkBox);
	buttonPanel2.add(clearScreenBtn);
	buttonPanel2.add(Box.createHorizontalStrut(10));
	buttonPanel2.add(clearBufferBtn);
	buttonPanel2.add(Box.createHorizontalStrut(10));
	buttonPanel2.add(replayBufferBtn);

	// add the trace view
	traceViewPanel.add(Box.createVerticalStrut(10));
	traceViewPanel.add(traceView);
	traceViewPanel.setBorder(BorderFactory.createTitledBorder("Behavior Trace"));
	

	JSplitPane mainSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, topPane, traceViewPanel);
	mainSplitPane.setOneTouchExpandable(true);
	topPane.setMinimumSize(new Dimension(minimumTopPaneWidth, minimumTopPaneHeight));
	topPane.setPreferredSize(new Dimension(preferredTopPaneWidth, preferredTopPaneHeight));
	mainSplitPane.setMinimumSize(new Dimension(minimumMainSplitPaneWidth, minimumMainSplitPaneHeight));
	mainSplitPane.setPreferredSize(new Dimension(preferredMainSplitPaneWidth, preferredMainSplitPaneHeight));
	mainSplitPane.setAlignmentX(1);

	// Decisions per second display. 
	decisionsPerSecondLabel = new JLabel(DECISION_CYCLE_TEXT);
	accumulatedConditionTimeLabel = new JLabel(CONDITION_TIME_TEXT);

	JPanel buttonPane = new JPanel();
	Box labels = new Box(BoxLayout.Y_AXIS);
	Box buttons = new Box(BoxLayout.X_AXIS);
	buttonPane.add(labels);
	buttonPane.add(Box.createRigidArea(new Dimension(50, 0)));
	buttonPane.add(buttons);
	labels.add(decisionsPerSecondLabel);
	labels.add(accumulatedConditionTimeLabel);
	buttons.add(breakBtn);
	buttons.add(Box.createRigidArea(new Dimension(10, 0)));
	buttons.add(stepBtn);
	buttons.add(Box.createRigidArea(new Dimension(10, 0)));
	buttons.add(continueBtn);
	buttonPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
	buttonPane.setAlignmentX(1);
	buttonPane.setMinimumSize(new Dimension(minimumButtonPaneWidth, buttonPaneHeight));
	buttonPane.setPreferredSize(new Dimension(preferredButtonPaneWidth, buttonPaneHeight));

	this.entity = entity;
	String longEntityName = entity.getClass().getName();
	String shortEntityName = longEntityName.substring(longEntityName.lastIndexOf('.') + 1);
	debugFrame = new JFrame(shortEntityName + " Debugger");
	debugFrame.getContentPane().setLayout(new BoxLayout(debugFrame.getContentPane(), BoxLayout.Y_AXIS));
	debugFrame.getContentPane().add(mainSplitPane);
	debugFrame.getContentPane().add(buttonPane);
	debugFrame.getRootPane().setDefaultButton(continueBtn);
	debugFrame.pack();
	debugFrame.setVisible(true); 

	// Add a listener to the debug listeners
	
	listenerSupport.addAblListener(traceListener); 

	// fixme: remove? Figure out what's going on now with tracing joint intentions. I don't actually see any code
	// that calls this listener. It looks like entry negotiations are traced by InitiatedJointGoalStepDebug
	// Add listener to BehavingEntity
	// fixme: eventually add a GUI which lets you turn entry negotiation tracing on and off
	// entity.addAblListener(traceListener);

	// Add listener to InitiatedJointGoalStepDebug. Need to add listener support to InitiatedJointGoalStepDebug.
	// Now this is hardwired into the InitiatedJointGoalStepDebug class
	// fixme: eventually add a GUI which lets you turn negotiation tracing on and off
	
    } 

    public boolean getBreakDecisionCycle() {
	return bBreakDecisionCycle;
    }

    protected void setTreeModel(JTree tree, DefaultTreeModel treeModel) {
	tree.setModel(treeModel);
	LinkedList paths = getAllLeafPaths((TreeNode)treeModel.getRoot(), treeModel);
	ListIterator pathIter = paths.listIterator();
	while(pathIter.hasNext())
	    tree.makeVisible((TreePath)pathIter.next());
    }

    public void setABTTreeModel(DefaultTreeModel treeModel) {
	setTreeModel(ABTTree, treeModel);
    }

    protected void clearABTTreeModel() { ABTTree.setModel(new DefaultTreeModel(new DefaultMutableTreeNode("ABT not available"))); }

    public boolean updateABT() { return ABTUpdateTreeButton.isSelected(); }
    
    protected LinkedList getAllLeafPaths(TreeNode node, DefaultTreeModel model) {
	LinkedList pathList = new LinkedList();
	int childCount = node.getChildCount();
	if (childCount == 0) {
	    // Leaf node; return a LinkedList containing the single path to this child.
	    pathList.add(new TreePath(model.getPathToRoot(node)));
	    return pathList;
	}
	else {
	    // Not a leaf; return a LinkedList containing the appended
	    // paths of all leafs nodes in the child subtrees
	    for (int i = 0; i < childCount; i++)
		pathList.addAll(getAllLeafPaths(node.getChildAt(i), model));
	    return pathList;
	}
    }
	    
    // if the user has selected break, updates the GUI and traps the
    // calling thread until the user selects continue.
    // Parameter: The number of milliseconds spent running continuous conditions on the last decision cycle.
    public synchronized void debug(long continuousConditionTime) {
	assert (continuousConditionTime >= 0);

	if (bBreakDecisionCycle) {
	    System.out.println(BehavingEntity.getBehavingEntity().getBehavingEntityShortName() + ": hit breakpoint");
	    decisionCycleCounterForDecisionsPerMillisecond = 0;
	    wmPanel.update();
	    setABTTreeModel(entity.getABTTreeModel());
	    BehavingEntity.getBehavingEntity().printNegotiationThreads();
	    BehavingEntity.getBehavingEntity().printLeafSteps();
	    BehavingEntity.getBehavingEntity().printAtomicSteps();
	    try {
		wait();
	    } catch (Exception e) {throw new AblRuntimeError("Unexpected interruption in Debugger.debug()");} 
	}
	else {
	    // Debugger update for continuously monitored WMs is
	    // handled by WorkingMemory

	    if (updateABT())
		setABTTreeModel(entity.getABTTreeModel());
	    long currentTime = System.currentTimeMillis();
	    if ((currentTime - startTime) >= 1000) {
		// A second has gone by.
		float decisionsPerMillisecond = (float) decisionCycleCounterForDecisionsPerMillisecond 
		    / (float) (currentTime - startTime);
		decisionsPerSecondLabel.setText(DECISION_CYCLE_TEXT + " " + Math.round(decisionsPerMillisecond * 1000));
		decisionCycleCounterForDecisionsPerMillisecond = 0;
		startTime = System.currentTimeMillis();
	    }
	    else
		decisionCycleCounterForDecisionsPerMillisecond++;
	    
	    accumulatedContinuousConditionTime += continuousConditionTime;
	    if (decisionCycleCounterForConditionTime == 100) {
		accumulatedConditionTimeLabel.setText(CONDITION_TIME_TEXT + " " + accumulatedContinuousConditionTime);
		decisionCycleCounterForConditionTime = 0;
		accumulatedContinuousConditionTime = 0;
	    }
	    else
		decisionCycleCounterForConditionTime++;

	}
    }

    public synchronized void breakSenseCycle() {}
    
    public synchronized void setTraceToBuffer(boolean b) {
	traceToBufferChkBox.setSelected(b);
    }

    public synchronized void setTraceToScreen(boolean b) {
	traceToScreenChkBox.setSelected(b);
    }

    public synchronized void releaseDecisionThread() {
	notify();
    }

    // Sets the debugger to break on the next decision cycle.
    public void breakNextDecisionCycle()
    {
	bBreakDecisionCycle = true;
    }


    // Interface for tracing 
    public void addABLListener(AblListener behaviorListener) { listenerSupport.addAblListener(behaviorListener); }
    public void removeABLListener(AblListener behaviorListener) { listenerSupport.removeAblListener(behaviorListener); }

    void traceAblExecutionEvent(int type, Step source, Object obj, int nestLevel, int behaviorID) 
    {
	if (debuggedBehaviors.contains(new Integer(behaviorID)))
	    listenerSupport.fireAblEvent(type, source, obj, nestLevel);
    }

    void traceAblExecutionEvent(int type, __BehaviorDesc source, Object obj, int nestLevel, int behaviorID) 
    {
	if (debuggedBehaviors.contains(new Integer(behaviorID)))
	    listenerSupport.fireAblEvent(type, source, obj, nestLevel);
    }

    void traceBehavior(int behaviorID) { debuggedBehaviors.add(new Integer(behaviorID)); }
    void untraceBehavior(int behaviorID) { debuggedBehaviors.remove(new Integer(behaviorID)); }

};
