/* Abstract superclass for behaviors (sequential, parallel and collection). */

package abl.runtime;

import java.util.*;
import java.lang.reflect.*;
import java.beans.*;
// import debug.*;

public abstract class Behavior implements ABTNode {

	// Constants for behaviorType
	public final static short SEQUENTIAL = 0;
	public final static short PARALLEL = 1;
	public final static short COLLECTION = 2;
	public final static short ADAPTIVE = 3;

	// named property support
	final AblNamedPropertySupport propertyTable = new AblNamedPropertySupport();

	// Parent goal of this behavior.
	protected GoalStep parent;

	// Priority inherited from goal (default 0). The steps of a
	// behavior inherit the behavior's priority (as modified by
	// priority modifiers).
	private short priority = 0;

	// True if the steps of this behavior are atomic - default false.
	protected boolean isAtomic = false;

	protected PropertyChangeSupport changes = new PropertyChangeSupport(this);

	// Set to true when a context condition associated with the behavior is deactivated.
	// Allows BehavingEntity.removeBehavior() to be idempotent.
	protected boolean behaviorRemoved = false;

	boolean getBehaviorRemoved() {
		return behaviorRemoved;
	}
	void setBehaviorRemoved() {
		behaviorRemoved = true;
	}

	// The behavior signature - set in the constructor
	protected final String signature;

	// An array of objects storing the variables defined in the behavior scope.
	protected final Object[] behaviorVariableFrame;

	/** 
	 * An array of integer step IDs. Used by MultiStepBehavior.getAllSteps() and SequentialBehavior.getNextStep() to
	   pass to the step factories.
	 */ 
	protected /*final*/ __StepDesc[] stepDescs;

	// Behavior specificity - defaults to 0 (set by the compiler)
	protected short specificity;

	// HashSet of BehavingEntity references
	protected Set<BehavingEntity> teamMembers = null;

	// Unique ID for this behavior. Used to find the right case in behavior methods.
	final protected int behaviorID;

	final private Method contextCondition;
	final private Method contextConditionSensorFactory;

	final private Method successCondition;
	final private Method successConditionSensorFactory;

	Object[] getBehaviorVariableFrame() {
		return behaviorVariableFrame;
	}

	public Behavior(
			GoalStep arg_parent, 
			Method arg_contextCondition, 
			Method arg_contextConditionSensorFactory, 
			Method arg_successCondition,
			Method arg_successConditionSensorFactory,
			boolean arg_isAtomic, 
			String arg_signature, 
			short arg_specificity, 
			int arg_behaviorID, 
			Object[] arg_behaviorVariableFrame, 
			__StepDesc[] arg_stepDescs) {
		parent = arg_parent;
		if (arg_parent != null && isAtomic == false)
			// Not the root behavior
			isAtomic = parent.getIsAtomic(); // inherit isAtomic from parent
		else
			isAtomic = arg_isAtomic; // explicitly set isAtomic if isAtomic argument is true

		contextCondition = arg_contextCondition;
		contextConditionSensorFactory = arg_contextConditionSensorFactory;
		successCondition = arg_successCondition;
		successConditionSensorFactory = arg_successConditionSensorFactory;
		signature = arg_signature;
		specificity = arg_specificity;
		behaviorID = arg_behaviorID;
		behaviorVariableFrame = arg_behaviorVariableFrame;
		stepDescs = arg_stepDescs;

		// Inherit the priority from the parent
		if (parent != null)
			priority = parent.getPriority();
		else
			priority = 0;
	}

	// Version of the constructor for joint behaviors.
	public Behavior(
			GoalStep arg_parent, 
			Method arg_contextCondition, 
			Method arg_contextConditionSensorFactory, 
			Method arg_successCondition,
			Method arg_successConditionSensorFactory,
			boolean arg_isAtomic, 
			String arg_signature, 
			short arg_specificity, 
			int arg_behaviorID, 
			Object[] arg_behaviorVariableFrame, 
			__StepDesc[] arg_stepDescs, 
			BehavingEntity[] arg_teamMembers) {
		this(arg_parent, arg_contextCondition, arg_contextConditionSensorFactory, arg_successCondition, arg_successConditionSensorFactory, arg_isAtomic, arg_signature, arg_specificity, arg_behaviorID, arg_behaviorVariableFrame, arg_stepDescs);
		assert(arg_teamMembers != null && arg_teamMembers.length > 0);
		teamMembers = new HashSet<BehavingEntity>();
		for (int i = 0; i < arg_teamMembers.length; i++)
			teamMembers.add(arg_teamMembers[i]);
	}

	int getID() {
		return behaviorID;
	}

	short getPriority() {
		return priority;
	}

	/** Note: Ben Weber, changed from default to public access */
	public final GoalStep getParent() {
		return parent;
	}

	public String toString() {
		try {
			String behaviorTypeString;
			switch (getBehaviorType()) {
				case SEQUENTIAL :
					behaviorTypeString = "sequential";
					break;
				case PARALLEL :
					behaviorTypeString = "parallel";
					break;
				case COLLECTION :
					behaviorTypeString = "collection";
					break;
				default :
					behaviorTypeString = "UNEXPECTED BEHAVIOR TYPE";
			}

			return behaviorTypeString + " " + getSignature() + " priority: " + priority;
		} catch (Exception e) {
			throw new AblRuntimeError("Reflection error in SequentialBehavior.toString()");
		}
	}

	// Public accessor for behavior type.
	abstract short getBehaviorType();

	// Returns true if the behavior has a context condition
	boolean getHasContextCondition() {
		if (contextCondition != null)
			return true;
		else
			return false;
	}

    // Returns true if the behavior has a success condition, otherwise returns false
    boolean getHasSuccessCondition() {
	if (successCondition != null) 
	    return true;
	else
	    return false; 
    }

    short getSpecificity() {
	return specificity;
    }

	// fixme: currently unused - may eventually want to add changing specificity to the reflection interface
	void setSpecificity(short arg_specificity) {
		specificity = arg_specificity;
	}

	String getSignature() {
		return signature;
	}

	// A behavior is joint if teamMembers has been defined (is not null).
	boolean isJointBehavior() {
		if (teamMembers != null)
			return true;
		else
			return false;
	}

	// Package accessor for teamMembers.
	// If called on a non-joint behavior, asserts.
	final Set getTeamMembers() {
		assert(isJointBehavior());
		return teamMembers;
	}

	final int getBehaviorID() {
		return behaviorID;
	}
	// Adds behavior steps as children to this behavior.
	abstract void addChildren();

	// Recursively removes all child steps. Removes leaf steps from behaving entity.
	abstract void removeChildren();

	// Returns true if the behavior has any team_effect_only steps.
	abstract protected boolean hasTeamEffectOnlySteps();

	// Recursively removes a single child. Updates BehavingEntity state.
	protected void removeChild(final Step child, boolean isCallerChild) {
		final int stepType = child.getStepType();

		switch (stepType) {
			case Step.PRIMITIVE :
				if (((PrimitiveStep) child).isExecuting()) {
					// Primitive act is executing. Abort it.
					 ((PrimitiveStep) child).abort();
				}
				((PrimitiveStep) child).processStepRemoval(); // Unsuspend any steps suspended on this step.
				BehavingEntity.getBehavingEntity().removeStep(child);
				changes.firePropertyChange("child", child, null); // Fire property change for any listeners
				break;
			case Step.GOAL :
				if (((GoalStep) child).isExpanded())
					// Goal step has been chosen for execution (it has a child behavior). Recursively remove the child.
					 ((GoalStep) child).removeGoal();
				((GoalStep) child).processStepRemoval(); // Unsuspend any steps suspended on this step.
				BehavingEntity.getBehavingEntity().removeStep(child);
				changes.firePropertyChange("child", child, null); // Fire property change for any listeners
				break;
			case Step.MENTAL :
				// Mark MentalStep state so that it knows it's been deleted.
				// Need this so that mental steps which end up removing
				// themselves through reflection don't cause a problem
				// when they succeed.
				 ((MentalStep) child).delete();
				BehavingEntity.getBehavingEntity().removeStep(child);
				changes.firePropertyChange("child", child, null); // Fire property change for any listeners
				break;
			case Step.WAIT :
			case Step.SUCCEED :
			case Step.FAIL :
			case Step.EXECUTE_POLICY :
				BehavingEntity.getBehavingEntity().removeStep(child);
				changes.firePropertyChange("child", child, null); // Fire property change for any listeners
				break;
			default :
				throw new AblRuntimeError("Unexpected step type " + stepType);
		}
	}

	private void failBehaviorBody() {
		BehavingEntity.getBehavingEntity().removeBehavior(this); // Update BehavingEntity state.
		removeChildren(); // behavior responsible for removing it's own children

		if (parent != null) {
			// Not the root collection behavior

			parent.addFailedBehavior(this);

			parent.removeChild(false); // remove this behavior from the goal

			// Need to add the parent back to the set of leaf steps on failure of the behavior.
			BehavingEntity.getBehavingEntity().resetStep(parent);
		}
	}

	/* When a behavior fails, it is added to the set of behaviors
	   which have been attempted for the parent goal, and deleted from
	   the tree */
	void failBehavior() {
		final List jointGoals = freezeSubtreeAndNegotiateRemoval();
		if (!jointGoals.isEmpty()) {
			// There are joint goals in the subtree - spawn a thread which blocks on completion of subtree negotiation
			JointGoalNegotiationThread negotiateFailThread = new JointGoalNegotiationThread((JointGoalStep) null, Behavior.this +" negotiateFailThread") {
				public void run() {
					JointGoalStep.blockOnSubtreeNegotiation(jointGoals);

					// When we get here every goal in subtree has commited.
					failBehaviorBody();
				}
			};
			BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateFailThread);
		} else
			// No joint goals in subtree
			failBehaviorBody();
	}

	// When a behavior is removed from the tree, recursively remove children. Entry point for removal.
	void removeBehavior() {
		BehavingEntity.getBehavingEntity().removeBehavior(this); // Update BehavingEntity state.
		removeChildren(); // behavior responsible for removing it's own children
	}

	private void succeedBehaviorBody() {
		BehavingEntity.getBehavingEntity().removeBehavior(this);
		removeChildren(); // remove children
		if (parent != null) {
			// Parent is not null (this is not the root behavior)
			parent.removeChild(false);
			parent.succeedStep();
		}
	}

	/* When a behavior succeeds, it calls succeed on its parent
	   goal. Succeed on the parent goal is responsible for updating
	   ABT state and recursively removing tree rooted at the parent
	   goal. */
	void succeedBehavior() {
		final List jointGoals = freezeSubtreeAndNegotiateRemoval();

		if (!jointGoals.isEmpty()) {
			// There are joint goals in the subtree - spawn a thread which blocks on completion of subtree negotiation
			JointGoalNegotiationThread negotiateSucceedThread = new JointGoalNegotiationThread((JointGoalStep) null, Behavior.this +" negotiateSucceedThread") {
				public void run() {
					JointGoalStep.blockOnSubtreeNegotiation(jointGoals);

					// When we get here every goal in subtree has commited.
					succeedBehaviorBody();
				}
			};
			BehavingEntity.getBehavingEntity().registerNegotiationThread(negotiateSucceedThread);
		} else if (isJointBehavior() && hasTeamEffectOnlySteps()) {
			parent.succeedStep(); // Succeed the parent joint goal - initiates success negotiation while leaving team_effect_only steps in the tree
		} else {
			// No joint goals or team_effect_only steps in subtree
			succeedBehaviorBody();
		}
	}

	// returns true if no context condition defined for this behavior, otherwise invokes the context condition.
	boolean contextCondition() {
		if (contextCondition == null)
			// no context condition - always true.
			return true;
		else {
			Object[] args = { new Integer(behaviorID), behaviorVariableFrame, BehavingEntity.getBehavingEntity()};
			try {
				Boolean b = (Boolean) contextCondition.invoke(null, args);
				return b.booleanValue();
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking context condition", e);
			}
		}
	}

	// For those Behaviors with context conditions referencing sensed wmes, returns a list of
	// the SensorActivations needed by the context condition.
	final SensorActivation[] getContextConditionSensorActivations() {
		if (contextConditionSensorFactory == null)
			return null; // no sensor factory
		else {
			Object[] args = { new Integer(behaviorID)};
			try {
				return (SensorActivation[]) contextConditionSensorFactory.invoke(null, args);
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking context condition sensor factory", e);
			}
		}
	}

    // returns false if no success condition defined for this behavior, otherwise invokes the success condition.
	boolean successCondition() {
		if (successCondition == null)
			// no success condition - always false.
			return false;
		else {
			final Object[] args = { new Integer(behaviorID), behaviorVariableFrame, BehavingEntity.getBehavingEntity()};
			try {
				final Boolean b = (Boolean) successCondition.invoke(null, args);
				return b.booleanValue();
			} catch (Exception e) {
				throw new AblRuntimeError("Error invoking success condition", e);
			}
		}
	}

    // For those Behaviors with success conditions referencing sensed wmes, returns a list of
    // the SensorActivations needed by the success condition.
    final SensorActivation[] getSuccessConditionSensorActivations() {
	if (successConditionSensorFactory == null)
	    return null; // no sensor factory
	else {
	    final Object[] args = { new Integer(behaviorID)};
	    try {
		return (SensorActivation[]) successConditionSensorFactory.invoke(null, args);
	    } catch (Exception e) {
		throw new AblRuntimeError("Error invoking success condition sensor factory", e);
	    }
	}
    }

	/* Returns true if the current goal is the last pursued goal of
	   this behavior, false otherwise. Defined on MultiStepBehavior
	   and SequentialBehavior. */
	abstract boolean currentLineOfExpansion(GoalStep currentGoal);

	// Suspends the steps of the behavior.
	abstract void suspend(ExecutableStep step);

	// Suspends the steps of the behavior, skipping joint goals
	abstract void suspendSkipJointGoals(ExecutableStep step);

	// Negotiation joint unsuspend of the steps of the behavior.
	abstract void jointUnsuspend();

	// Unsuspends the steps of the behavior.
	abstract void unsuspend(ExecutableStep step);

	// Meta control for suspending the steps of a behavior.
	abstract void metaSuspend();

	// Meta control for suspending the steps of this behavior (skipping joint goals).
	abstract void metaSuspendSkipJointGoals();

	// Meta control for unsuspending the steps of a behavior.
	abstract void metaUnsuspend();

	// reference to the reflection wme wrapping this behavior
	private BehaviorWME behaviorWME = null;

	public void addChildChangeListener(PropertyChangeListener l) {
		changes.addPropertyChangeListener("child", l);
	}
	public void removeChildChangeListener(PropertyChangeListener l) {
		changes.removePropertyChangeListener("child", l);
	}

	// only classes in abl.runtime can set the reflection wme
	void setReflectionWME(BehaviorWME behaviorWME) {
		this.behaviorWME = behaviorWME;
	}

	// concrete subclasses of Step can get the reflection wme
	protected BehaviorWME getReflectionWME() {
		return behaviorWME;
	}

	abstract List<Step> freezeSubtreeAndNegotiateRemoval();

	protected abstract List freezeNonTeamEffectOnlySubtreeAndNegotiateRemoval();

	abstract List negotiateSuspensionOfSubtree();

	abstract List negotiateSuspensionOfSubtree(ExecutableStep step);

	// fixme: Move this logic into freezeStep() defined on Step (and subclasses).
	protected List<Step> freezeChild(Step s) {
		// Trace.ablTrace("(Behavior)" + this.getSignature() + ".freezeChild(" + s + ")");

		final int stepType = s.getStepType();

		switch (stepType) {
			case Step.PRIMITIVE :
				if (((PrimitiveStep) s).isExecuting()) {
					// Primitive act is executing. Abort it.
					 ((PrimitiveStep) s).abort();
				}

				BehavingEntity.getBehavingEntity().removeStep(s);
				s.updateStepsSuspendingMe();
				return new Vector<Step>(0);
			case Step.GOAL :
				BehavingEntity.getBehavingEntity().removeStep(s);
				s.updateStepsSuspendingMe();
				return ((GoalStep) s).freezeSubtreeAndNegotiateRemoval();
			case Step.MENTAL :
				// mark MentalStep state so that it knows it's been
				// deleted. Need this so that mental steps which end up
				// initiating their own removal through reflection don't cause a
				// problem when they succeed.
				 ((MentalStep) s).delete();

				BehavingEntity.getBehavingEntity().removeStep(s);
				s.updateStepsSuspendingMe();
				return new Vector<Step>(0);
			case Step.WAIT :
			case Step.FAIL :
			case Step.SUCCEED :
			case Step.EXECUTE_POLICY :
				BehavingEntity.getBehavingEntity().removeStep(s);
				s.updateStepsSuspendingMe();
				return new Vector<Step>(0);
			default :
				throw new AblRuntimeError("Unexpected step type " + stepType);
		}
	}

	protected List negotiateSuspensionOfChild(Step stepToSuspend, ExecutableStep stepToSuspendOn) {
		final int stepType = stepToSuspend.getStepType();

		switch (stepType) {
			case Step.PRIMITIVE :
			case Step.MENTAL :
			case Step.WAIT :
			case Step.SUCCEED :
			case Step.FAIL :
				if (stepToSuspendOn != null)
					stepToSuspend.suspend(stepToSuspendOn);
				else
					stepToSuspend.metaSuspend();
				return new Vector(0);
			case Step.GOAL :
				return ((GoalStep) stepToSuspend).negotiateSuspensionOfSubtree(stepToSuspendOn);
			default :
				throw new AblRuntimeError("Unexpected step type " + stepType);
		}
	}

	protected List negotiateSuspensionOfChild(Step stepToSuspend) {
		final int stepType = stepToSuspend.getStepType();

		switch (stepType) {
			case Step.PRIMITIVE :
			case Step.MENTAL :
			case Step.WAIT :
			case Step.SUCCEED :
			case Step.FAIL :
				// fimxe: shouldn't really suspend non-joint steps until joint steps have suspended
				// Eventually use same idiom (suspendSkipJointGoals()) as for suspending on a step
				stepToSuspend.jointSuspend();
				return new Vector(0);
			case Step.GOAL :
				return ((GoalStep) stepToSuspend).negotiateSuspensionOfSubtree();
			default :
				throw new AblRuntimeError("Unexpected step type " + stepType);
		}
	}

	// returns true if this is the root behavior, false otherwise.
	boolean isRootBehavior() {
		if (parent == null)
			return true;
		else
			return false;
	}
}
