
package abl.runtime;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Random;
import java.util.Set;
import java.util.Vector;

import javax.swing.tree.DefaultTreeModel;

import wm.WME;
import wm.WorkingMemory;
import wm.WorkingMemoryDebugger;
import abl.compiler.AblDebuggerConstants;

// fixme: remove. For debugging purposes only.
// import facade.nativeinterface.NativeAnimationInterface;

/** All behaving entities are subclasses of BehavingEntity. */
public abstract class BehavingEntity implements AblDebuggerConstants {
	protected class BehaviorLibrary {
		Hashtable behaviorNames; // Keys are signatures, values are lists of

		// concrete behavior class names.

		/**
         * Hash table of all behaviors. Used by the behavior arbitration
         * mechanism to choose a behavior for a goal. Keys are behavior
         * signitures; values are concrete behavior classes. The hash table is
         * initialized and set to the appropriate behaviors in the constructor
         * of the concrete BehavingEntity.
         */
		Hashtable behaviors;

		public BehaviorLibrary(int initialSize) {
			behaviors = new Hashtable(initialSize);
		}

        /*
         * Modified by Gillian Smith, 07-27-10 to support inheritance. __BehaviorDesc have multiple
         * signatures associated with them. This method iterates over all the signatures for a behavior
         * and adds the behavior to that entry in the behaviors hashtable.
         */
		public synchronized void registerBehavior(__BehaviorDesc behDesc) {
            //System.err.println("Registering " + behDesc.signature);
            for (String sig : behDesc.signatures) {
                //System.err.println("\t" + sig);
                List behaviorDescList = (List) behaviors.get(sig);

                if (behaviorDescList == null) {
                    // No behavior with this signature has yet been
                    // registered. Create the list, add behavior
                    // name to it, and hash the list under the signature.
                    behaviorDescList = new Vector();
                    behaviorDescList.add(behDesc);
                    if (debugLevel == GUI_DEBUGGER && sig.equals(behDesc.signature)) {
                        behDesc.uniqueName = behDesc.signature.substring(0,
                                behDesc.signature.indexOf('('))
                                + "_1";
                    }

                    behaviors.put(sig, behaviorDescList);
                } else {
                    // Behavior with this signature has previously been registered.
                    // Add the behavior class name to the list.
                    behaviorDescList.add(behDesc);
                    if (debugLevel == GUI_DEBUGGER && sig.equals(behDesc.signature))
                        behDesc.uniqueName = behDesc.signature.substring(0,
                                behDesc.signature.indexOf('('))
                                + "_" + behaviorDescList.size();
                }
            }
            //System.err.flush();
		}

		/** Given a signature, returns a List of __BehaviorDesc */
		public synchronized List lookupBehavior(String signature) {
			List behaviorDescList = (List) behaviors.get(signature);
			if (behaviorDescList == null)
				// no behaviors with this signature have been added yet.
				return new Vector(0);
			else
				return new ArrayList(behaviorDescList);
		}

		// Returns a set of registered behavior signatures
		private synchronized Set getRegisteredBehaviors() {
			return new HashSet(behaviors.keySet());
		}
	}

	/** Constants for use in findConflictsWithCurrentlyExecutingSteps() */
	// fixme: suspendOnInstantiate appears to be unused
	public final static int suspendOnInstantiate = 0;

	public final static int suspendOnExecute = 1;

	// Holds WMEs
	protected final WorkingMemory workingMemory;

	// Top node of ABT.
	protected CollectionBehavior ABT;

	// Leaf steps that can currently be choosen for execution.
	// protected HashSet leafSteps = new HashSet();
	// fixme: why am I synchronizing the leafSteps collection?
	// recently sychronized all ABT modification and access routines on
	// BehavingEntity (e.g. behave(), resetStep(), etc.)
	// so there is probably no reason to synchronize the set.
	protected Set leafSteps = Collections.synchronizedSet(new HashSet());

	// Atomic steps than can currently be choosen for
	// execution. Any leaf step of the ABT is either in the set
	// leafSteps or the set atomicSteps.
	protected HashSet atomicSteps = new HashSet();

	// Primitive steps that are currently executing.
	protected HashSet executingPrimitiveSteps = new HashSet();

	// Executing steps with success tests (goals and primitive acts).
	protected HashSet successTestStepsNoSensing = new HashSet();

	// Behaviors in ABT with context condtions.
	protected HashSet contextConditionBehaviorsNoSensing = new HashSet();

	// Behaviors in ABT with success conditions that don't require sensing.
	final protected HashSet successConditionBehaviorsNoSensing = new HashSet();

	// Executing steps with success tests (goals and primitive acts).
	protected HashSet successTestStepsSensing = new HashSet();

	// Behaviors in ABT with context condtions.
	protected HashSet contextConditionBehaviorsSensing = new HashSet();

	// Behaviors in ABT with success conditions that do require sensing
	final protected HashSet successConditionBehaviorsSensing = new HashSet();

	// Steps in ABT that are currently executing (goals and primitive acts).
	protected Hashtable executingSteps = new Hashtable();

	// The behavior libraries are initialized in the constructor of the concrete
	// BehavingEntity.
	protected BehaviorLibrary individualBehaviorLibrary;

	protected BehaviorLibrary jointBehaviorLibrary;

	/*
	 * Reference to this behaving entity. Made an InheritableThreadLocal because
	 * each behaving entity runs in its own thread; two different behaving
	 * entities will each have the correct self reference. Must be inheritable
	 * because a single behaving entity has multiple child threads.
	 */
	protected final static InheritableThreadLocal entity = new InheritableThreadLocal();

	// Behavior specificity comparator used in chooseBehavior().
	private static BehaviorSpecificityComparator behComparator = new BehaviorSpecificityComparator();

	// Step priority comparator used in chooseStep().
	private static StepPriorityComparator stepPriorityComparator = new StepPriorityComparator();

	// Step expansion comparator (current line of expansion) used in
	// chooseStep().
	private static StepExpansionComparator stepExpansionComparator = new StepExpansionComparator();

	// Random number generator for use in step and behavior selection.
	Random randomGen = new Random();

	// Instance of Object[] used for retrieving the Class instance of Object[].
	static Object[] tempObjArray = new Object[1];

	// Emotional state hash table (add later).

	// Set to true to turn on the current line of expansion heuristic. Defaults
	// to true. Can be turned off by an ABL command
	// line switch.
	protected boolean bCurrentLineOfExpansion = true;

	// Set to true to turn on step conflict maintainance.
	protected boolean bStepConflicts = true;

	// Set to true to debug the behaving entity.
	protected byte debugLevel = NO_DEBUG;

	// If code is compiled with the debug flag, the concrete BehavingEntity
	// class creates a Debugger.
	protected Debugger debuggerGUI = null;

	protected boolean bDecisionCycleSMCall = false; // true if the behaving

	// entity has a decision
	// cycle callback

	protected boolean asynchronousSenseCycle = true;

	private static Hashtable entityTable = new Hashtable();

	// Hashtable of current entry negotiators - unique id (Integer) is key,
	// negotiator is value
	private Hashtable currentEntryNegotiators = new Hashtable();

	// List of active negotiation threads
	private List activeNegotiationThreads = Collections
			.synchronizedList(new LinkedList());;

	// Reference to the thread running the decision cycle. Used to check that
	// methods are being called by
	// appropriate threads.
	private Thread decisionCycleThread = null;

	private boolean negotiatorCommittedDuringNegotiation = false;

	private long continuousConditionTime = 0;

	private ShutdownHook shutdownHook;

	private CollectionBehaviorWME rootCollectionBehaviorWME = null;

	/**
	 * Note: added by Ben Weber
	 * 
	 * This is set to false when stopBehaving is called.
	 */
	private boolean behaving = true;

	private boolean paused = false;

	/**
	 * Note: Ben Weber
	 * 
	 * List of behaving listeners
	 */
	private ArrayList<BehavingListener> behavingListeners = new ArrayList<BehavingListener>();
	
	private class SensedConditionMonitor {
		private boolean ready = false;

		synchronized boolean sensedConditionsReadyToRun() {
			boolean readyTemp = ready;
			ready = false;
			return readyTemp;
		}

		synchronized void setSensedConditionsReadyToRun(boolean b) {
			ready = b;
		}
	}

	SensedConditionMonitor senseMonitor = new SensedConditionMonitor();

	// Manages changes to active sensors and the sensing of these active sensors
	/**
	 * @author Michael Mateas
	 *
	 */
	private class ActiveContinuousSensors {
		// Set of sensor activations associated with currently active
		// context conditions and success tests.
		private LinkedList activeSensors = new LinkedList();

		// Adds a sensor activation to the set of current sensor
		// activations. Called by addBehavior() for those behaviors
		// with context conditions which return a non-null value from
		// getContextConditionSensorActivations() and those behaviors
		// with success conditions which return a non-null value from
		// getSuccessCondtionSensorActivations(), and called by
		// addStep() for those steps with success tests which return a
		// non-null value from getSuccessTestSensorActivations().
		synchronized void activateSensor(SensorActivation s) {
			ListIterator iter = activeSensors.listIterator();
			SensorActivation currentSensor;
			while (iter.hasNext()) {
				currentSensor = (SensorActivation) iter.next();

				if (s.equals(currentSensor)) {
					/*
					 * This SensorActivation (combination of sensor and
					 * arguments) has previously been registered as an active
					 * sensor. Increment the reference count.
					 */

					currentSensor.referenceCount++;
					return;
				}
			}
			// If got this far, this SensorActivation (combination of
			// sensor and arguments) has not previously been
			// registered as an active sensor. Add it to the set of
			// active sensors.

			activeSensors.add(s);
			s.activeSensor.initializeContinuous(s.arguments);
			this.notify();
		}

		synchronized void activateSensors(SensorActivation[] activations) {
			if (activations != null)
				for (int i = 0; i < activations.length; i++)
					activateSensor(activations[i]);
		}

		
		/** Removes a sensor activation from the set of current sensor activations. Called by removeBehavior() for those behaviors
		* with context conditions which return a non-null value from
		* getContextConditionSensorActivations() or a non-null value from getSuccessConditionSensorActivations()
		* and called by removeStep() for those steps with success tests which
		* return a non-null value from getSuccessTestSensorActivations(). */		
		synchronized void deactivateSensor(SensorActivation s) {
			ListIterator iter = activeSensors.listIterator();
			SensorActivation currentSensor;
			while (iter.hasNext()) {
				currentSensor = (SensorActivation) iter.next();
				if (s.equals(currentSensor)) {
					// An equivalent SensorActivation was found. If
					// the reference count would go to zero, remove
					// the sensor, otherwise decrement the reference
					// count.
					if (currentSensor.referenceCount == 1) {
						iter.remove();
					} else {
						currentSensor.referenceCount--;
					}
					return;
				}
			}
			// If got this far, then deactiveSensor() was called with
			// a SensorActivaiton which is not currently in the set of
			// active sensors.
		}

		synchronized void deactivateSensors(SensorActivation[] activations) {
			if (activations != null)
				for (int i = 0; i < activations.length; i++)
					deactivateSensor(activations[i]);
		}

		synchronized void sense() {
			if (!activeSensors.isEmpty()) {
				runSensors(activeSensors, true);
				senseMonitor.setSensedConditionsReadyToRun(true);
			} else {
                            if (asynchronousSenseCycle) {
                                // we need to wait for sensors to appear in the asynchronous case,
                                // otherwise we would spin if we didn't wait
				try {
					this.wait();
				} // Wait for sensors to be added to activate sensors
				catch (InterruptedException e) {
					throw new RuntimeError("Sense cycle interrupted");
				}
                            }
			}
		}
	}

	ActiveContinuousSensors activeSensors = new ActiveContinuousSensors();

	// The name of the working memory is the short name of the concrete behaving
	// entity class.
	public BehavingEntity() {
		entity.set(this);
		String longName = this.getClass().getName();
		// fixme: behaving entity's shouldn't automatically have
		// publically-accessible memories
		workingMemory = new WorkingMemory(longName.substring(longName
				.lastIndexOf('.') + 1)
				+ "Memory");
	}

	public String getBehavingEntityShortName() {
		String longName = this.getClass().getName();
		return longName.substring(longName.lastIndexOf('.') + 1);
	}

	public String toString() {
		return getBehavingEntityShortName();
	}

	// Returns a reference to the root of the ABT.
	CollectionBehavior getRootCollectionBehavior() {
		return (CollectionBehavior) ABT;
	}

	// Returns a reference to the behaving entity containing a behavior.
	public static BehavingEntity getBehavingEntity() {
		return (BehavingEntity) entity.get();
	}

	// Registers a behaving entity with a name.
	// fixme: doesn't check for duplicate names
	protected static void registerEntity(String name, BehavingEntity entity) {
		synchronized (entityTable) {
			entityTable.put(name, entity);
		}
	}

	// Returns a reference to the named behaving entity. Currently assumes that
	// all entities live in the same
	// process on the same machine.
	// fixme: change this to support fully distributed behaving entities
	public static BehavingEntity getBehavingEntity(String name) {
		synchronized (entityTable) {
			BehavingEntity specifiedEntity = (BehavingEntity) entityTable
					.get(name);
			if (specifiedEntity == null)
				throw new AblRuntimeError(
						"No behaving entity found for entity name " + name);
			else
				return specifiedEntity;
		}
	}

	Thread getDecisionCycleThread() {
		return decisionCycleThread;
	}

	// for debugging - shutdown hook prints out information about where we were
	// in the decision cycle.
	class ShutdownHook extends Thread {

		public String shutdownMessage; // Set by behave()

		public void run() {
			System.err.println(shutdownMessage);
			printNegotiationThreads();
			System.err.flush(); // Flush the err PrintStream after writing
			// shutdown info.
		}

	}

	// Start the top level exectution loop for this behaving entity.
	public void startBehaving() {
		decisionCycleThread = Thread.currentThread();

		// thread class for the sense cycle
		class SenseCycleThread extends Thread {
			public void run() {
				while (behaving) {			// Note: Ben Weber, changed "true" to "behaving" so the loop can exit
					activeSensors.sense();
				}
			}
		}

		if (asynchronousSenseCycle) 
			new SenseCycleThread().start(); // start the parallel sense cycle
		
		shutdownHook = new ShutdownHook();

		Runtime.getRuntime().addShutdownHook(shutdownHook);

		while (behaving) {					// Note: Ben Weber, changed "true" to "behaving" so the loop can exit
			Thread.currentThread().setName("DecisionCycleThread");
			behave();
		}

		/**
		 * Note: Ben Weber
		 * 
		 * Wake of the sense thread so it knows to exit. If this is not called, 
		 * it is possible that the sensor thread will never exit.
		 */
		synchronized (this.activeSensors) {
			this.activeSensors.notifyAll(); 
		}
	}
	
	/**
	 * Note: Ben Weber
	 * 
	 * Tells this class to stop behavior. This causes the main loop to break out 
	 * and then notifies the asynchronous thread to break out as well.
	 */
	public void stopBehaving() {
		behaving = false;
	}

	/**
	 * Note: Ben Weber
	 * 
	 * Enable synchronous sensors.
	 */
	public void setUseSynchrousSensors() {
		asynchronousSenseCycle = false;
	}
	
	/**
	 * Note: Ben Weber
	 * 
	 * Adds a behaving listener to this behaving entity.
	 */
	public void addBehavingListener(BehavingListener listener) {
		behavingListeners.add(listener);
	}
	
	/**
	 * Note: Ben Weber
	 * 
	 * Pauses ABL
	 */
	public void pause() {
		paused = true;
	}
	
	/**
	 * Note: Ben Weber
	 * 
	 * Un-pauses ABL
	 */
	public void unpause() {
		paused = false;
	}
	
	/**
	 * Note: Ben Weber
	 * 
	 * Returns true if ABL is paused
	 */
	public boolean isPaused() {
		return paused;
	}

	
	/*
	 * Implements the top level ABL execution loop. Implements the loop defined
	 * in Bryan's thesis. 1. If needed adjust the active behavior tree for
	 * actions that have finished. 2. Else if needed adjust the ABT based on
	 * changes in the world (context conditions and success tests). 3. Else if
	 * needed adjust suspended goals. 4. Else pick a leaf step to execute and
	 * execute it.
	 * 
	 * Bryan sent an email explaining why each step of the loop is part of a
	 * branching if-then-else rather than just running sequentially (longer
	 * discussion in HapDesignQ&A.txt). But for now I'm just going to do the
	 * test sequentially. Eventually I will need to maintain some kind of flag
	 * that keeps track of whether the working memory has been modified. Another
	 * flag I'll need is whether a primitive act has succeeded or failed. A
	 * third flag I'll need is whether any change has been made to the ABT.
	 */

	public void behave() {

		/**
		 * Note: Ben Weber
		 * 
		 * returns if currently paused
		 */
		if (paused) {
			try {
				Thread.sleep(100);
			}
			catch (Exception e) {}
			
			return;
		}
		
		/**
		 * Note: Ben Weber
		 * 
		 * Notify listens that behave is being invoked.
		 */
		for (BehavingListener listener : behavingListeners) {
			listener.onBehave(executingSteps, leafSteps);
		}		

		// Loop through all threads waiting on a negotiation condition checking
		// if any can run
		shutdownHook.shutdownMessage = getBehavingEntityShortName()
				+ " was running negotiation threads";
		runNegotiationThreads();

		// fixme: remove?
		// Make sure that all negotiation threads are waiting.
		/*
		 * final int originalPriority = Thread.currentThread().getPriority();
		 * Thread.currentThread().setPriority(Thread.MAX_PRIORITY); final
		 * Iterator iter = new ArrayList(activeNegotiationThreads).iterator();
		 * while(iter.hasNext()) { final JointGoalNegotiationThread t =
		 * (JointGoalNegotiationThread)iter.next(); // Thread must either not be
		 * started, be waiting, or not be alive. // assert !t.getIsStarted() ||
		 * t.getIsWaiting() || !t.isAlive(): t.shortToString() + " is still
		 * running: isStarted = " + t.getIsStarted() + // " isWaiting = " +
		 * t.getIsWaiting() + " isAlive = " + t.isAlive(); }
		 * Thread.currentThread().setPriority(originalPriority);
		 */

		if (bDecisionCycleSMCall) {
			shutdownHook.shutdownMessage = getBehavingEntityShortName()
					+ " was in the decision cycle SM call";
			decisionCycleSMCall();
		}

		workingMemory.deleteMarkedTransientWMEs();

		// First check if any executing PrimitiveStep has completed.
		shutdownHook.shutdownMessage = getBehavingEntityShortName()
				+ " was checking for completed primitive actions";
		Iterator primStepIterator = executingPrimitiveSteps.iterator();
		PrimitiveStep primStep;
		while (primStepIterator.hasNext()) {
			primStep = (PrimitiveStep) primStepIterator.next();

			if (primStep.getCompletionStatus() == PrimitiveAction.SUCCESS) {
				primStep.succeedStep();
				return;
			} else if (primStep.getCompletionStatus() == PrimitiveAction.FAILURE) {
				primStep.failStep();
				return;
			}
		}

		// Now check if there are any atomic steps to
		// execute. If there are atomic steps, skip sensing
		// and choose one of the steps, otherwise process as usual.
		shutdownHook.shutdownMessage = getBehavingEntityShortName()
				+ " was checking continuously monitored conditions";
		if (atomicSteps.isEmpty()) {
			// No atomic steps

			/*
			 * Now check the sensors associated with any continuously monitored
			 * conditions (i.e. success tests and context conditions). This
			 * prepares working memory for testing the monitored conditions.
			 */
			if (!asynchronousSenseCycle) {
				activeSensors.sense();
				runSuccessTests(true); // run the sensing success tests
				runContextConditions(true); // run the sensing context
				// conditions
				runSuccessConditions(true); // run the sensing success
				// conditions
				workingMemory.markTransientWMEs();
			}

			long continuousConditionStartTime = System.currentTimeMillis();

			runSuccessTests(false); // run the no-sensing success tests
			runContextConditions(false); // run the no-sensing context
			// conditions
			runSuccessConditions(false); // run the no-sensing success
			// conditions

			// fixme: currently only measuring the amount of time spent testing
			// no-sensing continuous conditions.
			// fixme: current time measurements don't take account of the time
			// the thread is actually scheduled to
			// run - only measures elapsed system time.
			continuousConditionTime = System.currentTimeMillis()
					- continuousConditionStartTime;

			if (asynchronousSenseCycle
					&& senseMonitor.sensedConditionsReadyToRun()) {
				runSuccessTests(true); // run the sensing success tests
				runContextConditions(true); // run the sensing context
				// conditions
				runSuccessConditions(true); // run the sensing success
				// conditions
				workingMemory.markTransientWMEs();
			}

			/*
			 * Adjust which goals are marked as suspended because of conflicts.
			 * Because of successes and failures occuring above, some currently
			 * suspended goals may be unsuspended and some currently unsuspended
			 * goals may be suspended. When a goal is suspended, if it is not a
			 * leaf step, then all leaf steps below the goal must be marked as
			 * suspended (but not the goals along the way). When unsuspending a
			 * goal, if it is not a leaf step, all leaf steps below the goal are
			 * unsuspended, except for leaf steps of subtrees rooted at a goal
			 * marked as suspended. These adjustments are handled by methods
			 * defined on ExecutableStep.
			 */

		}

		// while running atomically a behaving entity with
		// asynchronous sensing continues to sense - this is all right
		// since the continuous conditions aren't being run.

		shutdownHook.shutdownMessage = getBehavingEntityShortName()
				+ " was selecting a step";
		// Else, pick a leaf step and execute it.
		Step stepToExecute = chooseStep();

		// If the debug flag has been set, update the debugger GUI
		if (debugLevel == GUI_DEBUGGER) {
			debuggerGUI.debug(continuousConditionTime);
		}

		if (stepToExecute != null) {
			// If there was a step to choose for execution, execute it.
			shutdownHook.shutdownMessage = getBehavingEntityShortName()
					+ " was executing step " + stepToExecute + " of behavior "
					+ stepToExecute.getParent();

			stepToExecute.execute();
		} else {
			// there were no steps to choose - wait 1/60 of a second
			try {
				shutdownHook.shutdownMessage = getBehavingEntityShortName()
						+ " was waiting because there was no step to execute";
				Thread.sleep(17);
			} catch (InterruptedException exception) {
				throw new RuntimeError("Unexpected interruption");
			}
		}
	}

	// Prints out the threads in activeNegotiationThreads
	void printNegotiationThreads() {

		synchronized (activeNegotiationThreads) { // synchronize to avoid
			// concurrent modification
			Iterator iter = activeNegotiationThreads.iterator();
			if (!iter.hasNext())
				System.err.println("No active negotiation threads in "
						+ getBehavingEntityShortName());
			while (iter.hasNext())
				System.err.println(iter.next());
		}
	}

	private final ThreadGroup timeCheckThreads = new ThreadGroup(
			"TimeCheckThreads");

	private void runNegotiationThreads() {
		negotiatorCommittedDuringNegotiation = false;

		// TimeCheckThread makes sure that no negotiation thread blocks the
		// decision cycle for more than a second.
		// The offending JointGoalNegotiationThread is interrupted and spits out
		// diagnostic information.
		class TimeCheckThread extends Thread {
			JointGoalNegotiationThread monitoredThread;

			TimeCheckThread(JointGoalNegotiationThread t) {
				super(timeCheckThreads, null, "tct"); // Add the new
				// TimeCheckThread to
				// the TimeCheckThreads
				// group.
				monitoredThread = t;
			}

			public void run() {
				try {
					sleep(1000);
					System.err
							.println("WARNING: a negotiation thread took longer than a second to execute. The thread was interrupted.");
					monitoredThread.interrupt();
				} catch (InterruptedException e) {
				} // Do nothing - thread will be interrupted after negotiation
				// thread returns
			}
		}

		JointGoalNegotiationThread[] threads = (JointGoalNegotiationThread[]) activeNegotiationThreads
				.toArray(new JointGoalNegotiationThread[activeNegotiationThreads
						.size()]);
		for (int i = 0; i < threads.length; i++) {
			if (!threads[i].getIsStarted()) {
				// t hasn't been started yet - start it up
				TimeCheckThread tct = new TimeCheckThread(threads[i]);
				tct.start();
				threads[i].start(); // block until thread completes or waits
				tct.interrupt();
			} else if (!threads[i].isAlive()) {
				// t is done running - wake waiters and remove it from the
				// activeNegotiationThreads list
				activeNegotiationThreads.remove(threads[i]);
			} else {
				TimeCheckThread tct = new TimeCheckThread(threads[i]);
				tct.start();
				threads[i].continueNegotiation(); // give thread the go-ahead
				// to continue - block until
				// thread completes or waits
				tct.interrupt();
			}
		}

		if (negotiatorCommittedDuringNegotiation) { // someone committed during
			// negotiation - rerun the
			// negotiators.
			runNegotiationThreads();
		}
	}

	// Caller commited during negotiation. Sets flag indicating negotiation
	// threads should be rerun.
	void jointGoalNegotiatorCommitted() {
		try {
			JointGoalNegotiationThread t = (JointGoalNegotiationThread) Thread
					.currentThread();
		} catch (ClassCastException e) {
			throw new AblRuntimeError("Non-negotiation thread "
					+ Thread.currentThread()
					+ " attempted jointGoalNegotiatorCommitted.");
		}

		negotiatorCommittedDuringNegotiation = true;
	}

	// Get this entity's working memory
	public WorkingMemory getWorkingMemory() {
		return workingMemory;
	}

	// Adds WME to working memory.
	public void addWME(WME w) {
		workingMemory.addWME(w);
	}

        public void addWMEs(WME... ws) {
            for (int i=0; i<ws.length; i++) 
                workingMemory.addWME(ws[i]);
        }

	// fixme: add back support for WME modification
	/*
	 * // Modifies a WME in working memory. public void modifyWME(WME
	 * wmeToModify, WME wmeWithNewValues) { try {
	 * workingMemory.modifyWME(wmeToModify, wmeWithNewValues); } catch
	 * (WorkingMemoryException e) {throw new AblRuntimeError(e);} }
	 */

	// Removes a WME from working memory. If the wme is not in working
	// memory does nothing (doesn't throw an error).
	public void deleteWME(WME wmeToDelete) {
		workingMemory.deleteWME(wmeToDelete);
	}

	// Deletes all WMEs with a given class name from working
	// memory. If the WME class is not found in working memory, does
	// nothing.
	public void deleteAllWMEClass(String wmeClassName) {
		workingMemory.deleteAllWMEClass(wmeClassName);
	}

	// Given a WME class name, returns the list of WMEs in working memory with
	// that class name.
	public LinkedList lookupWME(String wmeClassName) {
		return workingMemory.lookupWME(wmeClassName);
	}

	// Given the name of a signature bearing reflection WME (e.g. "GoalStepWME")
	// and a signature name, returns a list of all
	// WMEs sharing that signature.
	public List lookupReflectionWMEBySignature(String wmeClassName,
			String signature) {
		return workingMemory.lookupReflectionWMEBySignature(wmeClassName,
				signature);
	}

	// Given the name of a reflection WME and a user property, returns a list of
	// all reflection WMEs bearing that property.
	public List lookupReflectionWMEByUserProperty(String wmeClassName,
			String userPropertyName) {
		return workingMemory.lookupReflectionWMEByUserProperty(wmeClassName,
				userPropertyName);
	}

	// Get a WorkingMemoryDebugger for the behaving entity's working memory.
	public WorkingMemoryDebugger getWMDebugger() {
		return new WorkingMemoryDebugger(workingMemory);
	}

	// Called when a step already in the ABT is reset.
	void resetStep(Step s) // { abt.resetStep(); }
	{
		/*
		 * Removes the step from the set of executing primitive acts. If the
		 * step isn't an executing primitive act, does nothing.
		 */

		executingPrimitiveSteps.remove(s);

		// Remove from the table of executing steps.
		removeExecutingStep(s);

		// Adds the step to the set of steps that can be chosen for execution.
		if (s.getStepType() != Step.WAIT) {
			if (s.getIsAtomic())
				atomicSteps.add(s);
			else
				leafSteps.add(s);
		}
	}

	// Called when a primitive act is aborted.
	void abortStep(PrimitiveStep s) // { abt.abortStep(s); }
	{
		// Remove any success test associated with the aborting step
		removeSuccessTest(s);
		
		executingPrimitiveSteps.remove(s);

		// Remove from the table of executing steps.
		removeExecutingStep(s);
	}

	// Suspends an ExecutableStep
	void suspendStep(ExecutableStep s) {
		boolean inFringe;
		
		// Remove the success test. Don't want it to be tested while we're suspended. 
		removeSuccessTest(s);
		
		if (s.getStepType() == Step.GOAL)
			inFringe = !((GoalStep) s).isExpanded();
		else
			inFringe = !s.isExecuting();

		if (!inFringe) {
			// The step is not in leaf (or atomic) steps. Remove from appropriate executing sets.
			removeExecutingStep(s);
			if (s.getStepType() == Step.PRIMITIVE)
				executingPrimitiveSteps.remove(s);
		} else {
			// Step is in fringe, so it's a leaf (or atomic) step.
			if (s.getIsAtomic())
				atomicSteps.remove(s);
			else
				leafSteps.remove(s);
		}
	}
	
	
	/**
	 * Suspends WaitSteps. For WaitSteps, suspending only involves removing the success test.  
	 */
	void suspendStep(WaitStep s) {
		removeSuccessTest(s);
	}

	// Unsuspends an executable step.
	void unsuspendStep(ExecutableStep s) {
		if ((s.getStepType() == Step.GOAL) && ((GoalStep) s).isExpanded()) {
			// The suspended goal step has been expanded; add it back to the
			// executing set
			HashSet stepSet = (HashSet) executingSteps.get(s.getName());
			if (stepSet != null)
				stepSet.add(s);
			else {
				stepSet = new HashSet();
				stepSet.add(s);
				executingSteps.put(((ExecutableStep) s).getName(), stepSet);
			}
		} else {
			// The step is either a primitive step or an unexpanded goal step.
			// Add it back to leafSteps.
			if (s.getIsAtomic())
				atomicSteps.add(s);
			else
				leafSteps.add(s);
		}
		
		// Reactivate any success test associated with the step.
		addSuccessTest(s);
	}
	
	/**
	 * Unsuspends a WaitStep. For WaitSteps, unsuspending only involves reactivating the success test. 
	 */
	void unsuspendStep(WaitStep s) {
		addSuccessTest(s);
	}

	// Return a DefaultTreeModel containing a representation of the ABT. Used by
	// the debugger.
	DefaultTreeModel getABTTreeModel() {
		return new DefaultTreeModel(((DebuggableBehavior) ABT).getTree());
	}

	// Removes a success test.
	void removeSuccessTest(Step s) {
		if (s.getHasSuccessTest()
				&& (successTestStepsSensing.contains(s) || successTestStepsNoSensing.contains(s))) {
			// The step has a success test. Remove it from the set of steps with
			// success tests.
			SensorActivation[] activations = s.getSuccessTestSensorActivations();
			if (activations != null)
				successTestStepsSensing.remove(s);
			else
				successTestStepsNoSensing.remove(s);

			// If there are sensor activations associated with the
			// success test, remove them from the set of active
			// sensors.
			if (activations != null) {
				activeSensors.deactivateSensors(activations);
			}

		}
	}
	
	private void addSuccessTest(Step s) {
		if (s.getHasSuccessTest()) {
			// The step has a success test. Add it to the set of steps with
			// success tests.
			SensorActivation[] activations = s
					.getSuccessTestSensorActivations();
			if (activations != null) {
				successTestStepsSensing.add(s);
			}
			else
				successTestStepsNoSensing.add(s);

			/*
			 * If there are sensor activations associated with the success test,
			 * add them to the set of active sensors.
			 */
			if (activations != null) {
				activeSensors.activateSensors(activations);
			}
		}

	}
	
	/** Called when a step is removed from the ABT. Idempotent. */
	void removeStep(Step s) {
		removeSuccessTest(s);

		// Removes the step from the set of executing primitive
		// acts. If the step isn't an executing primitive act,
		// does nothing.
		executingPrimitiveSteps.remove(s);

		// Removes the step from the set of steps that can be
		// chosen for execution. If the step isn't in the set,
		// does nothing.
		if (s.getIsAtomic())
			atomicSteps.remove(s);
		else
			leafSteps.remove(s);

		// Remove the step from the table of executing steps.
		removeExecutingStep(s);
	}

	// Called when a new step is added to the ABT.
	void addStep(Step s) // {abt.addStep(s); }
	{
		if (s.getStepType() != Step.WAIT) {
			// If not a wait step, make it available for execution.
			if (s.getIsAtomic())
				atomicSteps.add(s);
			else
				leafSteps.add(s);
		}

		addSuccessTest(s);
	}

	// Removes a goal or primitive step from the table of executing steps.
	void removeExecutingStep(Step s) {
		if (s.getStepType() == Step.PRIMITIVE || s.getStepType() == Step.GOAL) {
			HashSet stepSet = (HashSet) executingSteps.get(((ExecutableStep) s)
					.getName());
			if (stepSet != null) {
				stepSet.remove(s);
			}
		}
	}

	/** Called when a step is executed. Updates BehavingEntity state for the step. */
	void executeStep(Step s) {

		// Remove step from set of executable steps.
		if (s.getIsAtomic())
			atomicSteps.remove(s);
		else {
			leafSteps.remove(s);
		}

		int stepType = s.getStepType();
		if (stepType == Step.PRIMITIVE)
			// s is a primitive act. Add to set of executing primitive acts.
			executingPrimitiveSteps.add(s);

		// Add the step to the table of executing steps (keyed by step name).
		if (stepType == Step.PRIMITIVE || stepType == Step.GOAL) {
			HashSet stepSet = (HashSet) executingSteps.get(((ExecutableStep) s)
					.getName());
			if (stepSet != null)
				stepSet.add(s);
			else {
				stepSet = new HashSet();
				stepSet.add(s);
				executingSteps.put(((ExecutableStep) s).getName(), stepSet);
			}
		}
	}

	// Called when a behavior is added to the ABT.
	void addBehavior(Behavior b) {
		if (b.getHasContextCondition()) {
			// If the behavior has a context condition, add it to the set of
			// behaviors with context conditions.
			final SensorActivation[] activations = b
					.getContextConditionSensorActivations();
			if (activations != null)
				contextConditionBehaviorsSensing.add(b);
			else
				contextConditionBehaviorsNoSensing.add(b);

			/*
			 * If there are sensor activations associated with the context
			 * condition, add them to the set of active sensors.
			 */
			if (activations != null) {
				activeSensors.activateSensors(activations);

				// fixme: remove
				/*
				 * for(int i = 0; i < activations.length; i++) if
				 * (activations[i].activeSensor.getClass().getName().equals("facade.sensor.GraceAnimationCueSensor"))
				 * System.out.println("Activating sensors for behavior " +
				 * b.getClass().getName());
				 */
			}
		}
		if (b.getHasSuccessCondition()) {
			// If the behavior has a success condition, add it to the set of
			// behaviors with success conditions
			final SensorActivation[] activations = b
					.getSuccessConditionSensorActivations();
			if (activations != null)
				successConditionBehaviorsSensing.add(b);
			else
				successConditionBehaviorsNoSensing.add(b);

			// If there are sensor activations associated with the success
			// condition, hadd them to the set of active
			// sensors.
			if (activations != null) {
				activeSensors.activateSensors(activations);
			}
		}

	}

	// Called when a behavior is removed from the ABT.
	// idempotent (because of behaviorRemoved state maintained on Behavior)
	void removeBehavior(Behavior b) {

		if ((b.getHasContextCondition() || b.getHasSuccessCondition())
				&& !b.getBehaviorRemoved()) {
			// Remove behavior from context condition and/or success condition
			// behaviors.

			final SensorActivation[] contextConditionActivations = b
					.getContextConditionSensorActivations();
			if (contextConditionActivations != null)
				contextConditionBehaviorsSensing.remove(b);
			else
				contextConditionBehaviorsNoSensing.remove(b);

			final SensorActivation[] successConditionActivations = b
					.getSuccessConditionSensorActivations();
			if (successConditionActivations != null)
				successConditionBehaviorsSensing.remove(b);
			else
				successConditionBehaviorsNoSensing.remove(b);

			b.setBehaviorRemoved();

			/*
			 * If there are sensor activations associated with the context
			 * and/or success conditions, remove them from the set of active
			 * sensors.
			 */
			if (contextConditionActivations != null) {
				activeSensors.deactivateSensors(contextConditionActivations);

				// fixme: remove
				/*
				 * for(int i = 0; i < activations.length; i++) if
				 * (activations[i].activeSensor.getClass().getName().equals("facade.sensor.GraceAnimationCueSensor")) {
				 * NativeAnimationInterface.dprintf("Deactivate
				 * AnimationCueSensor for behavior " + b.getClass().getName());
				 * System.out.println("Deactivating sensors for behavior " +
				 * b.getClass().getName()); }
				 */
			}
			if (successConditionActivations != null) {
				activeSensors.deactivateSensors(successConditionActivations);
			}
		}

	}

	/*
	 * Runs the context conditions and modifies the ABT. Returns true if a
	 * context condition unfires (becomes false).
	 */
	private boolean runContextConditions(boolean runSensedConditions) {

		Iterator contextConditionIterator;

		if (runSensedConditions)
			contextConditionIterator = contextConditionBehaviorsSensing
					.iterator();
		else
			contextConditionIterator = contextConditionBehaviorsNoSensing
					.iterator();
		Behavior contextConditionBehavior;
		while (contextConditionIterator.hasNext()) {
			contextConditionBehavior = (Behavior) contextConditionIterator
					.next();
			if (!contextConditionBehavior.contextCondition()) {
				contextConditionBehavior.failBehavior();
				runContextConditions(runSensedConditions);
				return true;
			}
		}
		return false;
	}

	// Runs the success conditions and modifies the ABT. Returns true if a
	// success conditions fires (becomes true).
	private boolean runSuccessConditions(boolean runSensedConditions) {
		Iterator successConditionIterator;

		if (runSensedConditions)
			successConditionIterator = successConditionBehaviorsSensing
					.iterator();
		else
			successConditionIterator = successConditionBehaviorsNoSensing
					.iterator();
		while (successConditionIterator.hasNext()) {
			final Behavior successConditionBehavior = (Behavior) successConditionIterator
					.next();
			if (successConditionBehavior.successCondition()) {
				successConditionBehavior.succeedBehavior();
				runSuccessConditions(runSensedConditions);
				return true;
			}
		}
		return false;
	}

	/*
	 * Runs the successs tests and modifies the ABT. Returns true if a
	 * succcessTest() fired, false otherwise.
	 */
	private boolean runSuccessTests(boolean runSensedConditions) {

		Iterator successStepIterator;
		if (runSensedConditions)
			successStepIterator = successTestStepsSensing.iterator();
		else
			successStepIterator = successTestStepsNoSensing.iterator();

		Step successStep;
		while (successStepIterator.hasNext()) {
			successStep = (Step) successStepIterator.next();
			if (successStep.successTest()) {
				/*
				 * If a single success test succeeds, modify the tree and exit.
				 * If multiple success tests are true, the ABT modifications
				 * from these success tests will be processed on successive
				 * calls to behave().
				 */
				successStep.succeedStep();
				runSuccessTests(runSensedConditions);
				return true;
			}
		}
		return false;
	}

	// Entry point for choosing an individual behavior.
	Behavior chooseIndividualBehavior(Object[] args, HashSet failedBehaviors,
			GoalStep g) {
		return chooseBehavior(args, failedBehaviors, g, false, null);
	}

	// Entry point for choosing a joint behavior.
	Behavior chooseJointBehavior(Object[] args, Set failedBehaviors, GoalStep g) {
		return chooseBehavior(args, failedBehaviors, g, true, null);
	}

	// Entry point for choosing a joint behavior with a specific set of team
	// members
	Behavior chooseJointBehavior(Object[] args, Set failedBehaviors,
			GoalStep g, Set teamMembers) {
		return chooseBehavior(args, failedBehaviors, g, true, teamMembers);
	}

	/*
	 * Given a behavior signature, args and a set of failed behaviors, returns a
	 * behavior whose precondition is satisfied by the args and which is not in
	 * the set of failedBehaviors. If no behavior is found, returns null.
	 * Implements the behavior arbitration algorithm defined in Bryan Loyall's
	 * thesis.
	 */
	private Behavior chooseBehavior(Object[] args, Set failedBehaviors,
			GoalStep g, boolean findJointBehaviors, Set teamMembers) {

		// Illegal to specify teammates if you aren't looking for joint
		// behaviors.
		assert ((!findJointBehaviors && teamMembers == null) || findJointBehaviors);

		List behList;
		final String signature = g.getSignature();

		// First find behaviors with matching signature.
		if (findJointBehaviors) {
			// looking for joint behaviors
			behList = jointBehaviorLibrary.lookupBehavior(signature);

			if (behList.isEmpty()) {
				// No behaviors were found matching the signature. Return
				// immediately with null.
				return null;
			} else if (teamMembers != null) {
				// filter behList so it only contains behaviors with the
				// specified teamMembers
				final Iterator iter = behList.iterator();
				while (iter.hasNext()) {
					final __BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
					final Set behTeamMembers = new HashSet(
							behDesc.teamMembers.length * 2);
					for (int i = 0; i < behDesc.teamMembers.length; i++)
						behTeamMembers
								.add(getBehavingEntity(behDesc.teamMembers[i]));
					if (!behTeamMembers.equals(teamMembers))
						iter.remove();
				}
			}
		} else {
			// looking for individual behaviors
			behList = (List) individualBehaviorLibrary
					.lookupBehavior(signature);
			if (behList == null) {
				// No behaviors were found matching the signature. Return
				// immediately with null.
				return null;
			}
		}

        //debug: print out the list of behaviors to choose from
/*        System.err.println("Choosing from behaviors for signature: " + signature);
        for (Object b : behList) {
            System.err.println("\t" + ((__BehaviorDesc) b).signature);
            System.err.flush();
        }*/

		// Now filter out all failed behaviors
		Iterator iter = behList.iterator();
		while (iter.hasNext()) {
			final __BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
			if (failedBehaviors.contains(new Integer(behDesc.behaviorID))) {
				iter.remove();
			}
		}

		// If there are no candidate behaviors left after removing the failed
		// behaviors, return null.
		if (behList.isEmpty())
			return null;

		// List of satisfied behavior classes.
		final List satisfiedBeh = new Vector(100);

		// Before running the preconditions, accumulate all sensor
		// activations referred to by any precondition of any behavior
		// with this signature, and call senseOneShot() on each of
		// these sensors.
		iter = behList.iterator();
		HashSet sensors = new HashSet();
		try {
			while (iter.hasNext()) {
				final __BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
				if (behDesc.preconditionSensorFactory != null) {
					// Behavior has a precondition sensor factory.

					final Object[] sensorFactoryArgs = { new Integer(
							behDesc.behaviorID) };
					final SensorActivation[] activations = (SensorActivation[]) behDesc.preconditionSensorFactory
							.invoke(null, sensorFactoryArgs);
					for (int activationCounter = 0; activationCounter < activations.length; activationCounter++)
						sensors.add(activations[activationCounter]);
				}
			}
		} catch (Exception e) {
			throw new AblRuntimeError("Reflection error", e);
		}

		// invoke the sensors - senseOneShot
		if (!sensors.isEmpty())
			runSensors(new Vector(sensors), false);

		// Now test preconditions, ignoring any behaviors in the set
		// failedBehaviors.
		iter = behList.iterator();

		try {
			while (iter.hasNext()) {
				final __BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
				if (behDesc.precondition != null) {
					// Behavior has a precondition
					Object[] argArray = { new Integer(behDesc.behaviorID),
							args, new Hashtable(), this };
					// Method invoked with null instance since it is static.
					final Boolean b = (Boolean) behDesc.precondition.invoke(
							null, argArray);
					if (debugLevel == GUI_DEBUGGER)
						debuggerGUI.traceAblExecutionEvent(
								AblEvent.PRECONDITION_EXECUTION, behDesc, b,
								((DebuggableStep) g).getNestLevel() + 1,
								behDesc.behaviorID);

					if (b.booleanValue()) {
						// Precondition satisifed; store the satisfied behavior
						// and the hash table of variables
						// bound by the precondition.
						satisfiedBeh.add(new SatisfiedBehavior(behDesc,
								(Hashtable) argArray[2]));
					}
				} else {
					// Add behaviors without preconditions to satisfiedBeh (null
					// precondition is always true)
					if (debugLevel == GUI_DEBUGGER)
						debuggerGUI
								.traceAblExecutionEvent(
										AblEvent.PRECONDITION_EXECUTION,
										behDesc,
										new Boolean(true),
										((DebuggableStep) g).getNestLevel() + 1,
										behDesc.behaviorID);

					satisfiedBeh.add(new SatisfiedBehavior(behDesc, null));
				}
			}

			if (!satisfiedBeh.isEmpty()) {
				// Behaviors with satisfied preconditions were found.

				// Now sort the satisfied behaviors by specificity.
				SatisfiedBehavior[] behArray = (SatisfiedBehavior[]) satisfiedBeh
						.toArray(new SatisfiedBehavior[satisfiedBeh.size()]);
				Arrays.sort(behArray, behComparator);

				// Now create a List consisting of all most-specific behaviors.
				List mostSpecificBeh = new ArrayList(satisfiedBeh.size());
				mostSpecificBeh.add(behArray[0]); // The first behavior is
				// always most-specific.
				for (int j = 1; j < behArray.length; j++)
					if (behComparator.compare(behArray[0], behArray[j]) == 0)
						mostSpecificBeh.add(behArray[j]);

				// Now randomly return an instantiated behavior from among the
				// most specific behaviors.
				__BehaviorDesc winningBeh;
				Hashtable preconditionBoundVariables;
				if (mostSpecificBeh.size() == 1) {
					winningBeh = ((SatisfiedBehavior) mostSpecificBeh.get(0)).behDesc;
					preconditionBoundVariables = ((SatisfiedBehavior) mostSpecificBeh
							.get(0)).preconditionBoundVariables;
				} else if (mostSpecificBeh.size() > 1) {
					int randomIndex = randomGen.nextInt(mostSpecificBeh.size());
					winningBeh = ((SatisfiedBehavior) mostSpecificBeh
							.get(randomIndex)).behDesc;
					preconditionBoundVariables = ((SatisfiedBehavior) mostSpecificBeh
							.get(randomIndex)).preconditionBoundVariables;
				} else
					throw new RuntimeError(
							"Expected mostSpecificBeh >= 1, instead < 1");

				// Construct the winnging behavior
				Object[] behFactoryArgs;
				if (debugLevel == GUI_DEBUGGER)
					behFactoryArgs = new Object[] {
							new Integer(winningBeh.behaviorID), args,
							preconditionBoundVariables, g,
							winningBeh.signature, winningBeh };
				else
					behFactoryArgs = new Object[] {
							new Integer(winningBeh.behaviorID), args,
							preconditionBoundVariables, g, winningBeh.signature };

				return (Behavior) winningBeh.factory.invoke(null,
						behFactoryArgs);
			} else
				return null;
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeError("Reflection error in chooseBehavior()");
		}
	}

	/*
	 * Given the current ABT, chooses a leaf step (from leafSteps, the set of
	 * leaf steps), and executes it. Implements the step arbitration algorithm
	 * described in Bryan Loyall's thesis: 1. Never choose wait steps, executing
	 * actions, or steps marked suspended. 2. Prefer steps with higher
	 * priorities. 3. Prefer to continue the same line of expansion. 4. If
	 * multiple steps still remain, randomly choose among them.
	 * 
	 * In this first implementation, conflicts are not handled; thus there are
	 * no suspended steps.
	 */
	private Step chooseStep() {

		if (!leafSteps.isEmpty() || !atomicSteps.isEmpty()) {
			// There are leaf steps to choose from.
			ArrayList conflictSet;
			if (!atomicSteps.isEmpty())
				conflictSet = new ArrayList(atomicSteps);
			else
				conflictSet = new ArrayList(leafSteps);

			// Remove suspended steps.
			if (bStepConflicts) {
				ListIterator iter = conflictSet.listIterator();
				while (iter.hasNext())
					if (((Step) iter.next()).isSuspended())
						iter.remove();
			}

			// Remove steps in the middle of negotiating
			ListIterator iter = conflictSet.listIterator();
			while (iter.hasNext()) {
				Step s = (Step) iter.next();
				if (s.getStepType() == Step.GOAL
						&& ((GoalStep) s).isJointGoal()
						&& ((JointGoalStep) s).getIsNegotiating()) {
					iter.remove();
				}
			}

			if (conflictSet.size() > 0) {
				// if, after removing suspended steps there are still some steps
				// left, choose one

				// Grab an array of all the leaf steps.
				Step[] conflictArray = (Step[]) conflictSet
						.toArray(new Step[conflictSet.size()]);

				// Sort the array by priority - highest priority first.
				Arrays.sort(conflictArray, stepPriorityComparator);

				conflictSet.clear(); // Clear the conflict set to prepare for
				// constructing the new set.
				conflictSet.add(conflictArray[0]); // The first step is always
				// highest priority.
				for (int i = 1; i < conflictArray.length; i++) {
					if (stepPriorityComparator.compare(conflictArray[0],
							conflictArray[i]) == 0)
						conflictSet.add(conflictArray[i]);
					else
						break;
				}

				if (bCurrentLineOfExpansion && (conflictSet.size() > 1)) {
					/*
					 * If the current line of expansion heuristic is turned on,
					 * and there is more than one highest priority step to
					 * choose from, sort the conflict set by the current line of
					 * expansion.
					 */

					conflictArray = (Step[]) conflictSet
							.toArray(new Step[conflictSet.size()]);
					Arrays.sort(conflictArray, stepExpansionComparator);

					// Remove all non-current line of expansion steps from the
					// conflict set.
					conflictSet.clear();

					/*
					 * The first step is always on the current line of expansion
					 * or no step is on the current line of expansion.
					 */
					conflictSet.add(conflictArray[0]);
					for (int i = 1; i < conflictArray.length; i++) {
						if (stepExpansionComparator.compare(conflictArray[0],
								conflictArray[i]) == 0)
							conflictSet.add(conflictArray[i]);
						else
							break;
					}
				}

				// Now randomly choose among the steps in the conflict set.
				assert (conflictSet.size() >= 1);
				if (conflictSet.size() > 1)
					return (Step) conflictSet.get(randomGen.nextInt(conflictSet
							.size()));
				else
					return (Step) conflictSet.get(0);
			} else
				return null; // no unsuspended or non-negotiating steps to
			// choose from
		} else
			return null; // No leaf steps.
	}

	// Given an ExecutableStep, determines if any currently executing
	// steps conflict with this step. Calls suspend on the argument
	// step for each conflicting executing step.
	void findConflictsWithCurrentlyExecutingSteps(ExecutableStep step,
			int comparisonIndex) {
		assert (comparisonIndex == BehavingEntity.suspendOnInstantiate || comparisonIndex == BehavingEntity.suspendOnExecute);
		String[] conflictArray = step.getConflicts();
		HashSet stepSet;
		for (int i = 0; i < conflictArray.length; i++) {
			stepSet = (HashSet) executingSteps.get(conflictArray[i]); // Get
			// the
			// conflict
			// set.
			if (stepSet != null) {
				Iterator iter = stepSet.iterator();
				while (iter.hasNext()) {
					// Suspend the step with each of the conflicting steps.
					ExecutableStep executingStep = (ExecutableStep) iter.next();
					switch (comparisonIndex) {
					case BehavingEntity.suspendOnInstantiate:
						// fixme: this case appears to be unused
						if (executingStep.getPriority() >= step.getPriority())
							step.suspend(executingStep);
						break;
					case BehavingEntity.suspendOnExecute:

						// When a step executes, it suspends any conflicting
						// executing step
						// of lower priority than it and in turn is
						// suspended by any conflicting executing step of
						// greater
						// or equal priority to it.

						if (executingStep.getPriority() < step.getPriority())
							executingStep.suspend(step);
						else
							step.suspend(executingStep);
						break;
					}
				}
			}
		}
	}

	void runSensors(List sensors, boolean isContinuous) {
		Iterator sensorIterator = sensors.iterator();
		SensorActivation activationRecord;
		Sensor s;
		Object[] args;
		LinkedList sensorThreadList = new LinkedList();

		// Thread class for running sensors.
		class SensorThread extends Thread {
			Sensor s;

			Object[] args;

			boolean isContinuous;

			SensorThread(Sensor sToSet, Object[] argsToSet,
					boolean isContinuousToSet) {
				s = sToSet;
				args = argsToSet;
				isContinuous = isContinuousToSet;
			}

			public void run() {
				if (isContinuous)
					s.senseContinuous(args);
				else
					s.senseOneShot(args);
			}
		}

		while (sensorIterator.hasNext()) {
			activationRecord = (SensorActivation) sensorIterator.next();
			s = activationRecord.activeSensor;
			args = activationRecord.arguments;
			if (s.canBeParallel()) {
				// Create a parallel thread to run the sensor
				SensorThread st = new SensorThread(s, args, isContinuous);
				st.setPriority(Thread.NORM_PRIORITY); // set priority in case
				// priority of parent
				// has been changed
				sensorThreadList.add(st);
				st.start();
			} else {
				
				/**
				 * Note: Ben Weber, 10-20-2009
				 * 
				 * A race condition with asynchronous serial sensors was discovered in Spring 2007.
				 * There did not appear to be any easy work-around, so Michael advised throwing 
				 * an exception if this situation arises.
				 */
				if (this.asynchronousSenseCycle) {
					throw new AblRuntimeError("Asynchronous serial sensors not currently supported");
				}
				
				// Run the sensor in series
				if (isContinuous) {
					s.senseContinuous(args);
				} else {
					s.senseOneShot(args);
				}
			}
		}

		// If some sensors have been run in parallel, wait for all the parallel
		// threads to have finished.
		if (sensorThreadList.size() != 0) {
			ListIterator threadIter = sensorThreadList.listIterator();
			while (threadIter.hasNext())
				try {
					((SensorThread) threadIter.next()).join();
				} catch (InterruptedException e) {
					throw new AblRuntimeError("Sensor thread interrupted");
				}
		}
	}

	// The default decision cycle SM call does nothing. Can be overridden on
	// concrete behaving entity classes.
	protected void decisionCycleSMCall() {
	}

	// If the behaving entity is running with the debugger, breaks.
	public void breakNextDecisionCycle() {
		if (debugLevel == GUI_DEBUGGER) {
			debuggerGUI.breakNextDecisionCycle();
		}
	}

	// A debugging utility for use in mental acts. Breaks immediately (during
	// the current mental act) and displays the tree.
	public void breakNow() {
		if (debugLevel == GUI_DEBUGGER) {
			debuggerGUI.breakNextDecisionCycle();
			debuggerGUI.debug(0); // Pass 0 for the continuousConditionTime.
		}
	}

	// If the behaving entity is running with the debugger, set the
	// trace-to-screen status
	public void setTraceToScreen(boolean b) {
		if (debugLevel == GUI_DEBUGGER)
			debuggerGUI.setTraceToScreen(b);
	}

	// If the behaving entity is running with the debugger, set the
	// trace-to-buffer status
	public void setTraceToBuffer(boolean b) {
		if (debugLevel == GUI_DEBUGGER)
			debuggerGUI.setTraceToBuffer(b);
	}

	// Starts WME reflection of the ABT. Should be called with the
	// RootCollectionBehavior in the constructor
	// of the concrete behaving entity class.
	protected void startWMEReflection(CollectionBehavior b) {
		final CollectionBehaviorWME root = new CollectionBehaviorWME(b, null);
		addWME(root);
		rootCollectionBehaviorWME = root;
	}

	// Returns true if ABT reflection has been enabled for this BehavingEntity,
	// false otherwise.
	boolean reflectionEnabled() {
		if (rootCollectionBehaviorWME != null)
			return true;
		else
			return false;
	}

	/*
	 * constantTrue() returns true regardless of the value of the argument. Used
	 * to force assignment in test expressions.
	 */
	public static boolean constantTrue(int i) {
		return true;
	}

	public static boolean constantTrue(float f) {
		return true;
	}

	public static boolean constantTrue(double d) {
		return true;
	}

	public static boolean constantTrue(short s) {
		return true;
	}

	public static boolean constantTrue(long l) {
		return true;
	}

	public static boolean constantTrue(char c) {
		return true;
	}

	public static boolean constantTrue(byte b) {
		return true;
	}

	public static boolean constantTrue(boolean b) {
		return true;
	}

	public static boolean constantTrue(Object o) {
		return true;
	}

	// A println which returns true; useful for debugging preconditions
	public static boolean truePrintln(String s) {
		System.out.println(s);
		return true;
	}

	// Returns a Set of registered individual behavior signatures
	Set getRegisteredIndividualBehaviors() {
		return individualBehaviorLibrary.getRegisteredBehaviors();
	}

	// Returns a Set of registered joint behavior signatures
	Set getRegisteredJointBehaviors() {
		return jointBehaviorLibrary.getRegisteredBehaviors();
	}

	Set getRegisteredBehaviors() {
		HashSet fullBehaviorSet = new HashSet(individualBehaviorLibrary
				.getRegisteredBehaviors());
		fullBehaviorSet.addAll(jointBehaviorLibrary.getRegisteredBehaviors());
		return fullBehaviorSet;
	}

	// Given a behavior sig, adds the ids of all behaviors with this sig to the
	// set of traced behaviors.
	void traceBehaviorSignature(String behaviorSig) {
		// grab a list containing the concatentation of all individual and joint
		// behaviors satisfying the signature
		List behaviorList = individualBehaviorLibrary
				.lookupBehavior(behaviorSig);
		behaviorList.addAll(jointBehaviorLibrary.lookupBehavior(behaviorSig));

		Iterator iter = behaviorList.iterator();
		while (iter.hasNext()) {
			__BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
			debuggerGUI.traceBehavior(behDesc.behaviorID);
		}
	}

	// Given a behavior sig, removes the ids of all behaviors with this sig from
	// the set of traced behaviors.
	void untraceBehaviorSignature(String behaviorSig) {
		// grab a list containing the concatentation of all individual and joint
		// behaviors satisfying the signature
		List behaviorList = individualBehaviorLibrary
				.lookupBehavior(behaviorSig);
		behaviorList.addAll(jointBehaviorLibrary.lookupBehavior(behaviorSig));

		Iterator iter = behaviorList.iterator();
		while (iter.hasNext()) {
			__BehaviorDesc behDesc = (__BehaviorDesc) iter.next();
			debuggerGUI.untraceBehavior(behDesc.behaviorID);
		}
	}

	// register a negotiation thread - package access
	void registerNegotiationThread(JointGoalNegotiationThread t) {
		assert (t != null);
		activeNegotiationThreads.add(t);
	}

	// unregister a negotiation thread = package access
	void unregisterNegotiationThread(JointGoalNegotiationThread t) {
		assert (t != null);
		activeNegotiationThreads.remove(t);
	}

	// Initiates negotiation for synchronizing on a joint goal.
	// g - the joint goal choosen by the initiating entity
	// args - the arguments to use for selecting joint behaviors
	// if negotiation is successful, returns the commit set (as a hashtable),
	// otherwise returns null
	Hashtable negotiateEntry(JointGoalStep g, Object[] args) {
		// Should only be called by a JointGoalNegotiationThread - this typecast
		// will throw an error if this method
		// has been called by some other thread type.
		JointGoalNegotiationThread test = (JointGoalNegotiationThread) Thread
				.currentThread();

		Long uniqueEntryNegotiationID;
		synchronized (currentEntryNegotiators) {
			uniqueEntryNegotiationID = g.negotiator
					.getUniqueEntryNegotiationID();
			currentEntryNegotiators.put(uniqueEntryNegotiationID, g);
		}
		// blocks on negotiation
		Hashtable commitSet = g.negotiator.negotiateEntry(args, g);
		synchronized (currentEntryNegotiators) {
			currentEntryNegotiators.remove(uniqueEntryNegotiationID);
		}
		return commitSet;
	}

	// Queues an intention to enter a joint goal
	// uniqueEntryNegotiationID - a wrapped long which uniquely identifies the
	// negotiation
	// sender - the entity which sent this intention-to-enter
	// args - the arguments to use for selecting a joint behavior
	// teamMembers - the set of team members participating in the joint behavior
	// g - the JointGoalStep chosen by the sender
	public void queueIntentionToEnter(final Long uniqueEntryNegotiationID,
			final BehavingEntity sender, final Object[] args,
			final boolean teamNeededForSuccess, final Set teamMembers,
			final JointGoalStep g) {
		JointGoalStep tempGoalStep;
		synchronized (currentEntryNegotiators) {
			tempGoalStep = (JointGoalStep) currentEntryNegotiators
					.get(uniqueEntryNegotiationID);
			if (tempGoalStep == null) {
				// first receipt of a message associated with this negotiation

				// Construct an initiated joint goal
				if (debugLevel == NO_DEBUG)
					tempGoalStep = getInitiatedJointGoalStep(this
							.getRootCollectionBehavior(), g.getSignature(),
							teamMembers, args, teamNeededForSuccess);
				else
					tempGoalStep = getInitiatedJointGoalStepDebug(this
							.getRootCollectionBehavior(), g.getSignature(),
							teamMembers, args, teamNeededForSuccess);
				currentEntryNegotiators.put(uniqueEntryNegotiationID,
						tempGoalStep);
			}
		}
		final JointGoalStep escroedGoalStep = tempGoalStep; // Lock it down for
		// easy reference
		// within inner
		// classes

		// Register the intention to enter.
		// Explicitly set entity so that it doesn't eroneously inherit from
		// sender
		// fixme: come up with a more general solution, probably employing RMI
		// for communication between entities

		final JointGoalNegotiationThread intentionToEnterThread = new JointGoalNegotiationThread(
				escroedGoalStep.negotiator, escroedGoalStep
						+ " queueIntentionToEnter") {
			public void run() {
				// explicitly set the entity in this thread as inheritance would
				// have it equal the sender's thread
				entity.set(BehavingEntity.this);
				escroedGoalStep.negotiator.processIntentionToEnter(
						uniqueEntryNegotiationID, sender, args,
						teamNeededForSuccess, teamMembers, g, escroedGoalStep);
			}
		};
		registerNegotiationThread(intentionToEnterThread);
	}

	// Queues an intention to refuse pursuit of a joint goal
	// uniqueEntryNegotiationID - a wrapped long which uniquely identifies the
	// negotiation
	// sender - the entity which sent this intention-to-refuse-entry
	// teamMembers - the set of team members which must coordinate on refusing
	// entry
	public void queueIntentionToRefuseEntry(
			final Long uniqueEntryNegotiationID, final BehavingEntity sender,
			final Set teamMembers) {
		JointGoalStep tempGoalStep;
		synchronized (currentEntryNegotiators) {
			tempGoalStep = (JointGoalStep) currentEntryNegotiators
					.get(uniqueEntryNegotiationID);
			if (tempGoalStep == null) {
				// First receipt of a message associated with this negotiation.
				// This strange case could occur if an entity is slow
				// to receive the original entry intention before some
				// other team member refuses entry.
				if (debugLevel == NO_DEBUG)
					tempGoalStep = getInitiatedJointGoalStep(this
							.getRootCollectionBehavior(),
							"_RefuseEntryDummy()", teamMembers, null, false);
				else
					tempGoalStep = getInitiatedJointGoalStepDebug(this
							.getRootCollectionBehavior(),
							"_RefuseEntryDummy()", teamMembers, null, false);
				currentEntryNegotiators.put(uniqueEntryNegotiationID,
						tempGoalStep);
			}
		}
		final JointGoalStep escroedGoalStep = tempGoalStep; // Lock it down for
		// easy reference
		// within inner
		// classes

		// Register the intention to refuse entry.
		// Explicitly set entity so that it doesn't eroneously inherit from
		// sender
		// fixme: come up with a more general solution, probably employing RMI
		// for communication between entities
		final JointGoalNegotiationThread intentionToRefuseEntryThread = new JointGoalNegotiationThread(
				escroedGoalStep.negotiator, escroedGoalStep
						+ " queueIntentionToRefuseEntry") {
			public void run() {
				// explicitly set the entity in this thread as inheritance would
				// have it equal the sender's thread
				entity.set(BehavingEntity.this);
				escroedGoalStep.negotiator.processIntentionToRefuseEntry(
						uniqueEntryNegotiationID, sender);
			}
		};
		registerNegotiationThread(intentionToRefuseEntryThread);
	}

	// Called when entry negotiation is finished. Clears the JointGoalNegotiator
	// from currentEntryNegotiators.
	public void terminateEntryNegotiation(Long uniqueEntryNegotiationID) {
		synchronized (currentEntryNegotiators) {
			assert (currentEntryNegotiators.get(uniqueEntryNegotiationID) != null);
			currentEntryNegotiators.remove(uniqueEntryNegotiationID);
		}
	}

	// Queues an intention to exit a goal.
	// Also used for unsuspend, which isn't really an exit.
	// sender - the entity which sent this intention-to-succeed
	// g - the JointGoalStep within this entity to succeed
	public void queueIntentionToExit(final BehavingEntity sender,
			final JointGoalStep g, final int intention) {
		// Explicitly set entity so that it doesn't eroneously inherit from
		// sender
		// fixme: come up with a more general solution, probably employing RMI
		// for communication between entities
		registerNegotiationThread(new JointGoalNegotiationThread(g.negotiator,
				g + " queueIntentionToExit("
						+ JointGoalNegotiator.formatIntention(intention) + ")") {
			public void run() {
				if (g.negotiator != null) {
					entity.set(BehavingEntity.this);
					switch (intention) {
					case JointGoalNegotiator.SUCCEED:
						g.negotiator.processIntentionToSucceed(sender);
						break;
					case JointGoalNegotiator.FAIL:
						g.negotiator.processIntentionToFail(sender);
						break;
					case JointGoalNegotiator.REMOVE:
						g.negotiator.processIntentionToRemove(sender);
						break;
					case JointGoalNegotiator.SUSPEND:
						g.negotiator.processIntentionToSuspend(sender);
						break;
					case JointGoalNegotiator.UNSUSPEND:
						g.negotiator.processIntentionToUnsuspend(sender);
						break;
					default:
						throw new AblRuntimeError("Unexpected intention");
					}
				}
			}
		});
	}

	private InitiatedJointGoalStep getInitiatedJointGoalStep(
			CollectionBehavior abtRoot, String signature, Set teamMembers,
			Object[] args, boolean teamNeededForSuccess) {
		return new InitiatedJointGoalStep(abtRoot, signature, teamMembers,
				args, teamNeededForSuccess);
	}

	// Return an InitiatedJointGoalStepDebug
	private InitiatedJointGoalStep getInitiatedJointGoalStepDebug(
			CollectionBehavior abtRoot, String signature, Set teamMembers,
			Object[] args, boolean teamNeededForSuccess) {
		return new InitiatedJointGoalStepDebug(abtRoot, signature, teamMembers,
				args, teamNeededForSuccess, debugLevel);
	}

	void traceAblExecutionEvent(int type, Step source, Object obj,
			int nestLevel, int behaviorID) {
		debuggerGUI.traceAblExecutionEvent(type, source, obj, nestLevel,
				behaviorID);
	}

	void traceAblExecutionEvent(int type, __BehaviorDesc source, Object obj,
			int nestLevel, int behaviorID) {
		debuggerGUI.traceAblExecutionEvent(type, source, obj, nestLevel,
				behaviorID);
	}

	void printLeafSteps() {
		Iterator iter = leafSteps.iterator();
		System.err.println("Leaf steps for " + getBehavingEntityShortName());
		while (iter.hasNext())
			System.err.println("    " + iter.next());
	}

	void printExecutingSteps() {
		Iterator iter = executingSteps.keySet().iterator();
		System.err.println("Executing steps for "
				+ getBehavingEntityShortName());
		while (iter.hasNext())
			System.err.println("    " + iter.next());
	}

	void printAtomicSteps() {
		Iterator iter = atomicSteps.iterator();
		System.err.println("Atomic steps for " + getBehavingEntityShortName());
		while (iter.hasNext())
			System.err.println("    " + iter.next());
	}

	// Debug support - given a step, walks upwards to the root collection
	// behavior, printing the nodes
	final static void printABTBranchUpwards(Step s) {
		while (s != null) {
			System.err.println(s);
			final Behavior b = s.getParent();
			System.err.println(b);
			s = b.getParent();
		}
	}

	// Debug support - given a behavior, walks upwards to the root collection
	// behavior, printing the nodes
	final static void printABTBranchUpwards(Behavior b) {
		if (b != null) {
			System.err.println(b);
			Step s = b.getParent();
			while (s != null) {
				System.err.println(s);
				b = s.getParent();
				System.err.println(b);
				s = b.getParent();
			}
		}
	}

    //Debug support - gets a reference to the debugger
    //Needed for ensuring that another element of the GUI doesn't steal focus
    //from the debugger
    public Debugger getDebuggerGUI()
    {
        return debuggerGUI;
    }
}
