// Sequential behaviors
package abl.runtime;

import java.lang.reflect.Method;
import java.util.*;

// AdaptiveBehaviors can be both sequential and parallel. But the sequential case is just a special case of the
// multi-step case, so inherit methods from the multi-step case.
public class AdaptiveBehavior extends MultiStepBehavior {
	// Initialize behaviorType as a static variable shared by all
	// AdaptiveBehaviors.
	protected final static short behaviorType = Behavior.ADAPTIVE;

	// Unlike non-adaptive behavior classes, adaptive behaviors can be
	// sequential or parallel
	private final short adaptiveBehaviorType;

	final private Method reinforcementSignal;

	final private Method reinforcementSignalSensorFactory;

	final private Method stateWME;

	final private Method stateWMESensorFactory;

	final private Method rlPolicy;

	// These refer to the previous state and action
	// This info is needed during the update of the learner
	static Object oldStateWMEList;

	static int prevAction;

	public AdaptiveBehavior(GoalStep arg_parent, Method arg_contextCondition,
			Method arg_contextConditionSensorFactory,
			Method arg_successCondition,
			Method arg_successConditionSensorFactory,
			Method arg_reinforcementSignal,
			Method arg_reinforcementSignalSensorFactory, Method arg_stateWME,
			Method arg_stateWMESensorFactory, Method arg_rlPolicy,
			boolean arg_isAtomic, String arg_signature, short arg_specificity,
			int arg_behaviorID, Object[] arg_behaviorVariableFrame,
			__StepDesc[] arg_stepDescs, short arg_adaptiveBehaviorType) {
		super(arg_parent, arg_contextCondition,
				arg_contextConditionSensorFactory, arg_successCondition,
				arg_successConditionSensorFactory, arg_isAtomic, arg_signature,
				arg_specificity, arg_behaviorID, arg_behaviorVariableFrame,
				arg_stepDescs, -1);
		reinforcementSignal = arg_reinforcementSignal;
		reinforcementSignalSensorFactory = arg_reinforcementSignalSensorFactory;
		stateWME = arg_stateWME;
		stateWMESensorFactory = arg_stateWMESensorFactory;
		rlPolicy = arg_rlPolicy;
		adaptiveBehaviorType = arg_adaptiveBehaviorType;
	}

	final int getNumberNeededForSuccess() {
		throw new AblRuntimeError(
				"numberNeededForSuccess is meaningless for AdaptiveBehaviors");
	}

	// Returns true if the AdaptiveBehavior is sequential and has 0 or 1
	// children, or is parallel (with no constraints on the number of children).
	private boolean childrenConsistencyTest() {
		return (behaviorType == Behavior.SEQUENTIAL && (children.size() == 0 || children
				.size() == 1))
				|| behaviorType != Behavior.PARALLEL;

	}

	// Recursively removes all children of this behavior.
	final protected void removeChildren() {
		assert childrenConsistencyTest();
		super.removeChildren();
	}

	// Adds a child step to the AdaptiveBehavior. Overrides addChild on
	// MultiStepBehavior as there is no need to do bookeeping for spawning a
	// child to
	// an adaptive behavior (not allowed) or for updating
	// numberNeededForSuccess.
	final void addChild(Step s) {
		// A sequential adaptive behavior should have no children when addChild
		// is called.
		assert behaviorType == Behavior.SEQUENTIAL && children.size() == 0
				|| behaviorType != Behavior.SEQUENTIAL;
		children.add(s);
		BehavingEntity.getBehavingEntity().addStep(s);
		changes.firePropertyChange("child", null, s); // Fire property change
														// for any listeners
	}

	// Adds behavior steps to children when a behavior is first added to the
	// ABT. Override MultiStepBehavior.addChildren(), instead of adding all the
	// steps
	// that could be sequenced by the policy, instead we add an initial
	// ExecutePolicy step.
	final void addChildren() {
		assert children.size() == 0;
		addChild(new ExecutePolicyStep(this));
	}

	// Overrides MultiStepBehavior.removeChild(Step, boolean) because there is
	// an additional assert test and because there is no need to maintain
	// numberOfCompletedSteps.
	final protected void removeChild(Step child, boolean isCallerChild) {
		// The step argument is not a child of this behavior; throw an error.
		assert children.contains(child) : "child == " + child
				+ " current children == " + children;
		assert childrenConsistencyTest();

		super.removeChild(child, isCallerChild);
		children.remove(child);

		/*
		 * If the child being removed is a GoalStep, setting lastPursuedGoal to
		 * null indicates that the behavior is no longer pursuing this goal. If
		 * the child being removed is not a GoalStep, lastPursuedGoal would
		 * already be null; setting it to null again does nothing.
		 */
		clearLastPursuedGoal();
	}

	// No need to ever call getAllSteps() on adaptive behaviors as steps are
	// only added by the policy.
	final protected Step[] getAllSteps() {
		throw new AblRuntimeError(
				"Don't call getAllSteps() on AdaptiveBehaviors");
	}

	// isSuccessful() always returns false since the condition for success of an
	// AdaptiveBehavior
	// is determined by the continuously monitored success_condition.
	final boolean isSuccessful() {
		return false;
	}

	// Overrides toString() on behavior to include "adaptive" in the behavior
	// description.
	public String toString() {
		try {
			String behaviorTypeString;
			switch (getAdaptiveBehaviorType()) {
			case SEQUENTIAL:
				behaviorTypeString = "adaptive sequential";
				break;
			case PARALLEL:
				behaviorTypeString = "adaptive parallel";
				break;
			case COLLECTION:
				// fixme: decide whether an adaptive collection behavior (vs.
				// adaptive parallel behavior) makes any sense.
				throw new AblRuntimeError(
						"Adaptive collection behaviors currently not supported");
			default:
				behaviorTypeString = "UNEXPECTED BEHAVIOR TYPE";
			}

			return behaviorTypeString + " " + getSignature() + " priority: "
					+ getPriority();
		} catch (Exception e) {
			throw new AblRuntimeError(
					"Reflection error in SequentialBehavior.toString()");
		}
	}

	/*
	 * Returns true if the current goal is in the current line of expansion,
	 * false otherwise.
	 */
	final boolean currentLineOfExpansion(GoalStep currentGoal) {
		assert parent != null : "AdaptiveBehavior found at the root of the ABT "
				+ this; // AdaptiveBehaviors can't be at the root.
		return super.currentLineOfExpansion(currentGoal);
	}

	void executePolicy() {
		try {
			// If the stateWMEs depend on sensed state, do appropriate sensing.
			final ArrayList sensors = new ArrayList();
			final SensorActivation[] stateWMEActivations = getStateWMESensorActivations();
			if (stateWMEActivations != null) {
				for (int activationCounter = 0; activationCounter < stateWMEActivations.length; activationCounter++)
					sensors.add(stateWMEActivations[activationCounter]);

				BehavingEntity.getBehavingEntity().runSensors(
						new ArrayList(sensors), false); // invoke the sensors -
														// senseOneShot
			}

			final Object[] args = { new Integer(behaviorID),
					behaviorVariableFrame, BehavingEntity.getBehavingEntity() };
			oldStateWMEList = (Object) stateWME.invoke(null, args);

			final Object[] args2 = { new Integer(behaviorID), oldStateWMEList,
					new Integer(0), null, new Double(0.0), new Boolean(false) };
			// stepIndex is an index into the stepDesc array - ranges over the
			// number of steps in the behavior.
			final int stepIndex = ((Integer) rlPolicy.invoke(null, args2))
					.intValue();
			prevAction = stepIndex;

			final Object[] factoryArgs = {
					new Integer(stepDescs[stepIndex].stepID), this,
					getBehaviorVariableFrame() };
			final Step selectedStep = (Step) stepDescs[stepIndex].factory
					.invoke(null, factoryArgs);
			addChild(selectedStep);
			selectedStep.execute(); // Execute the step.
		} catch (Exception e) {
			throw new AblRuntimeError("Error invoking executePolicy", e);
		}
	}

	// updatePolicy is called whenever a step of an adaptive behavior succeeds
	// or fails (call appears in Step).
	void updatePolicy() {
		try {
			// If the reinforcement signal or stateWMEs depend on sensed values,
			// collect the sensor activations and
			// run the sensors.
			final HashSet sensors = new HashSet();

			final SensorActivation[] reinforcementActivations = getReinforcementSignalSensorActivations();
			if (reinforcementActivations != null)
				for (int activationCounter = 0; activationCounter < reinforcementActivations.length; activationCounter++)
					sensors.add(reinforcementActivations[activationCounter]);

			final SensorActivation[] stateWMEActivations = getStateWMESensorActivations();
			if (stateWMEActivations != null)
				for (int activationCounter = 0; activationCounter < stateWMEActivations.length; activationCounter++)
					sensors.add(stateWMEActivations[activationCounter]);

			BehavingEntity.getBehavingEntity().runSensors(
					new ArrayList(sensors), false); // invoke the sensors -
													// senseOneShot

			final Object[] args = { new Integer(behaviorID),
					behaviorVariableFrame, BehavingEntity.getBehavingEntity() };
			final Object newStateWMEList = (Object) stateWME.invoke(null, args);

			final Object[] args2 = { new Integer(behaviorID),
					behaviorVariableFrame, BehavingEntity.getBehavingEntity() };
			final double reward = ((Double) reinforcementSignal.invoke(null,
					args2)).doubleValue();

			final Object[] args3 = { new Integer(behaviorID), oldStateWMEList,
					new Integer(prevAction), newStateWMEList,
					new Double(reward), new Boolean(true) };
			final int ingored = ((Integer) rlPolicy.invoke(null, args3))
					.intValue();
		} catch (Exception e) {
			throw new AblRuntimeError("Error invoking updatePolicy", e);
		}
	}

	// invokes the reinforcement signal
	final float reinforcementSignal() {
		assert reinforcementSignal != null;
		final Object[] args = { new Integer(behaviorID), behaviorVariableFrame,
				BehavingEntity.getBehavingEntity() };
		try {
			final Float f = (Float) reinforcementSignal.invoke(null, args);
			return f.floatValue();
		} catch (Exception e) {
			throw new AblRuntimeError(
					"Error invoking reinforcement signal function", e);
		}
	}

	// For those Behaviors with reinforcement signals referencing sensed wmes,
	// returns a list of
	// the SensorActivations needed by the reinforcement signal.
	final SensorActivation[] getReinforcementSignalSensorActivations() {
		if (reinforcementSignalSensorFactory == null)
			return null; // no sensor factory
		else {
			final Object[] args = { new Integer(behaviorID) };
			try {
				return (SensorActivation[]) reinforcementSignalSensorFactory
						.invoke(null, args);
			} catch (Exception e) {
				throw new AblRuntimeError(
						"Error invoking reinforcement signal sensor factory", e);
			}
		}
	}

	// For those behaviors with stateWMEs referencing sensed wmes, returns a
	// list of the SensorActivations needed
	// by the stateWMEs
	final SensorActivation[] getStateWMESensorActivations() {
		if (stateWMESensorFactory == null)
			return null; // no sensor factory
		else {
			final Object[] args = { new Integer(behaviorID) };
			try {
				return (SensorActivation[]) stateWMESensorFactory.invoke(null,
						args);
			} catch (Exception e) {
				throw new AblRuntimeError(
						"Error invoking stateWME sensor factory", e);
			}
		}
	}

	public short getBehaviorType() {
		return behaviorType;
	}

	public short getAdaptiveBehaviorType() {
		return adaptiveBehaviorType;
	}
}
