package abl.compiler.astchecks;

import abl.compiler.ASTBehaviorDefinition;
import abl.compiler.ASTBehaviorUnit;
import abl.compiler.ASTJavaStatement;
import abl.compiler.ASTMentalStep;
import abl.compiler.AblParseNode;
import abl.compiler.AblParserConstants;
import abl.compiler.AblParserTreeConstants;
import abl.compiler.CompileException;
import abl.compiler.Node;

/**
 * Perform sanity checks over the ABL AST.
 * <p>
 * We guarantee that if sanity checks pass, type-checking can proceed. That is
 * we check for things like: behavior definitions have greater than 0 steps,
 * etc.
 */
public class SanityCheckVisitor extends IdentityASTVisitor implements AblParserTreeConstants,
    AblParserConstants {
    
    // This may be used to implement certain checks.
    private final ASTBehaviorUnit unit;
    
    /**
     * Error messages included in errors thrown by this visitor.
     */
    public enum Messages {
        EMPTY_STEPS("No steps defined within behavior");
        
        
        private final String msg;
        private Messages(String msg) {
            this.msg = msg;
        }

        @Override
        public String toString() {
            return msg;
        }
        
    }

    public SanityCheckVisitor(final ASTBehaviorUnit unit) {
        this.unit = unit;
    }

    @Override
    public Node visit(final ASTBehaviorDefinition a) throws CompileException {
        behaviorDefinitionNonEmpty(a);
        return super.visit(a);
    }
    
    @Override
    public Node visit(ASTMentalStep a) throws CompileException {
        mentalActHasNoEscape(a);
        return super.visit(a);
    }

    /**
     * Check that a behavior definition is non-empty.
     * @param node the behavior definition
     */
    void behaviorDefinitionNonEmpty(final ASTBehaviorDefinition node) throws CompileException {
        final String behaviorName = node.isInitialTree() ? "initial_tree" : node.getBehaviorName();
        if (node.jjtGetNumChildren() == 0) {
            throw new CompileException(node.getFirstLineNumber(),
                                       Messages.EMPTY_STEPS + " " + behaviorName + ".");
                                       
        }
        
    }
    
    /**
     * Check that a mental act has no 'throws', 'return', or 'break' at the
     * toplevel.
     * <p>
     * TODO: This check isn't as strong as it needs to be. That is, if you open
     * a naked brace in a mental act (not as part of a try) and put a break in
     * there, ABL will compile the file, but the resultant java will not
     * compile. The solution is to walk the entire mental act & check that there
     * are:
     * <ul>
     * <li> no 'break's not inside declared loops/switch </li>
     * <li> no 'throw's not inside a try which catches the exception thrown </li>
     * <li> no 'return's not inside functions (?) </li>
     * @param node the mental act
     */
    void mentalActHasNoEscape(final ASTMentalStep node) throws CompileException {
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            final AblParseNode child = (AblParseNode) node.jjtGetChild(i);

            switch (child.getId()) {
            case JJTJAVASTATEMENT:
                final ASTJavaStatement stmt = (ASTJavaStatement) child;
                if (stmt.firstToken.kind == RETURN
                    || stmt.firstToken.kind == THROW
                    || stmt.firstToken.kind == BREAK) {
                    throw new CompileException(stmt.getFirstLineNumber(),
                                               tokenImage[stmt.firstToken.kind]
                                               + " not allowed "
                                               + " at toplevel in mental acts.");
                }
                break;
            }
        }
    }

}
