/* Parse node for behavior steps (mental, physical, subgoal, wait). */

package abl.compiler;

import java.util.*;
import jd.*;
import abl.runtime.Step;

abstract class GenericStep extends AblParseNode {

	/* True if the step has a success test; defaults to false. */
	boolean hasSuccessTest = false;

	/*
	 * Absolute priority of this step. The default of MIN_VALUE means "same as
	 * parent."
	 */
	int priority = Short.MIN_VALUE;

	/*
	 * The amount by which to change this step's priority relative to priority
	 * of the parent. The defaut of 0 means "same as parent."
	 */
	int priorityModifier = 0;

	/* Persistence modifiers. */
	boolean persistent = false;

	boolean persistentWhenSucceeds = false;

	boolean persistentWhenFails = false;

	/* Effect only modifier. */
	boolean effectOnly = false;

	// Team effect only modifier.
	boolean teamEffectOnly = false;

	/* Ignore failure modifier. */
	boolean ignoreFailure = false;

	// Post annotations
	private boolean post = false;

	private String postMemory = null;

	/* Name of the concrete class implementing this step. */
	protected String stepClassName;

	// Unique String name for this step - used in making comments.
	private String stepName;

	// compile time support for named properties
	private final Hashtable<String, ASTAblExpression> propertyTable;

	private String stepFactoryMethod_rField;

	private String executeMethod_rField;

	private String successTestMethod_rField;

	private String successTestSensorFactoryMethod_rField;

	// unique step id for this step
	protected int stepID = Integer.MIN_VALUE;

	GenericStep(int id) {
		super(id);
		propertyTable = new Hashtable<String, ASTAblExpression>();
	}

	GenericStep(AblParser p, int id) {
		super(p, id);
		propertyTable = new Hashtable<String, ASTAblExpression>();
	}

	/** Given a generic step, constructs a copy with the same step modifiers. */
	GenericStep(int id, GenericStep g) {
		super(id);
		hasSuccessTest = g.hasSuccessTest;
		priority = g.priority;
		priorityModifier = g.priorityModifier;
		persistent = g.persistent;
		persistentWhenSucceeds = g.persistentWhenSucceeds;
		persistentWhenFails = g.persistentWhenFails;
		effectOnly = g.effectOnly;
		teamEffectOnly = g.teamEffectOnly;
		ignoreFailure = g.ignoreFailure;
		post = g.post;
		postMemory = g.postMemory;
		propertyTable = new Hashtable<String, ASTAblExpression>(g.propertyTable);
	}
	
	int getStepID() {
		return stepID;
	}

	AblScopeMaintainer getEnclosingBehaviorScope() {
		return ((ASTBehaviorDefinition) jjtGetParent())
				.getEnclosingBehaviorScope();
	}

	String getUniqueName() {
		return stepName;
	}

	// Post accessors
	boolean getPost() {
		return post;
	}

	void setPost(boolean post) {
		this.post = post;
	}

	String getPostMemory() {
		return new String(postMemory);
	}

	void setPostMemory(String postMemory) {
		postMemory = new String(postMemory);
	}

	void setStepFactoryMethod_rField(String arg_stepFactoryMethod_rField) {
		if (stepFactoryMethod_rField != null)
			throw new CompileError(
					"Attempt to set stepFactoryMethod_rField multiple times: "
							+ stepFactoryMethod_rField);

		stepFactoryMethod_rField = arg_stepFactoryMethod_rField;
	}

	void setExecuteMethod_rField(String arg_executeMethod_rField) {
		if (executeMethod_rField != null)
			throw new CompileError(
					"Attempt to set executeMethod_rField multiple times: "
							+ executeMethod_rField);

		executeMethod_rField = arg_executeMethod_rField;
	}

	void setSuccessTestMethod_rField(String arg_successTestMethod_rField) {
		if (successTestMethod_rField != null)
			throw new CompileError(
					"Attempt to set successTestMethod_rField multiple times: "
							+ successTestMethod_rField);

		successTestMethod_rField = arg_successTestMethod_rField;
	}

	void setSuccessTestSensorFactoryMethod_rField(
			String arg_successTestSensorFactoryMethod_rField) {
		if (successTestSensorFactoryMethod_rField != null)
			throw new CompileError(
					"Attempt to set successTestSensorFactoryMethod_rField multiple times: "
							+ successTestSensorFactoryMethod_rField);

		successTestSensorFactoryMethod_rField = arg_successTestSensorFactoryMethod_rField;
	}

	String getStepFactoryMethod_rField() {
		return stepFactoryMethod_rField;
	}

	String getExecuteMethod_rField() {
		return executeMethod_rField;
	}

	String getSuccessTestMethod_rField() {
		return successTestMethod_rField;
	}

	String getSuccessTestSensorFactoryMethod_rField() {
		return successTestSensorFactoryMethod_rField;
	}

	// Returns a string containing the standard constructor arguments shared by
	// all step types. The specific step
	// nodes append their specific constructor arguments to the standard
	// constructor arguments to create the
	// complete constructor arguments.
	protected String standardConstructorArguments() throws CompileException {
		assert (stepID != Integer.MIN_VALUE);

		if (!((ASTBehaviorDefinition) jjtGetParent()).isJoint()
				&& teamEffectOnly == true)
			throw new CompileException(getFirstLineNumber(),
					"a step of a non-joint behavior was annotated with team_effect_only");

		if (effectOnly && teamEffectOnly)
			throw new CompileException(getFirstLineNumber(),
					"a step can't be annotated as both effect_only and team_effect_only");

		String execute = null;
		if (executeMethod_rField != null)
			execute = getBehavingEntityField(executeMethod_rField);

		String successTest = null;
		if (successTestMethod_rField != null)
			successTest = getBehavingEntityField(successTestMethod_rField);

		String successTestSensorFactory = null;
		if (successTestSensorFactoryMethod_rField != null)
			successTestSensorFactory = getBehavingEntityField(successTestSensorFactoryMethod_rField);

		StringBuffer argString = new StringBuffer(500);
		argString.append(stepID + ", __$behaviorParent, " + persistent + ", "
				+ persistentWhenSucceeds + ", " + persistentWhenFails + ", "
				+ ignoreFailure + ", " + effectOnly + ", " + teamEffectOnly
				+ ", " + "(short)" + priority + ", " + "(short)"
				+ priorityModifier + ", " + post + ", ");

		if (postMemory != null)
			argString.append("\"" + postMemory + "\"" + ", ");
		else
			argString.append("null" + ", ");

		argString.append(execute + ", " + successTest + ", "
				+ successTestSensorFactory + ", ");

		if (hasProperties())
			argString.append("__$propertyTable");
		else
			argString.append("null");

		return argString.toString();
	}

	// Returns a code block descriptor setting the property table.
	protected CodeSequenceDescriptor compilePropertyTable()
			throws CompileException {
		Enumeration propertyNames = propertyTable.keys();
		ASTBehaviorDefinition behaviorScope = (ASTBehaviorDefinition) getEnclosingBehaviorScope();
		CodeSequenceDescriptor createPropTableBlock = new CodeSequenceDescriptor();

		if (!propertyNames.hasMoreElements())
			return null; // If there are no properties defined on this step,
							// return null.

		createPropTableBlock
				.addToSequence(new CodeStringDescriptor(
						"final AblNamedPropertySupport __$propertyTable = new AblNamedPropertySupport();"));
		while (propertyNames.hasMoreElements()) {
			String propertyName = (String) propertyNames.nextElement();
			String declaredTypeName = ASTBehaviorUnit.getBehaviorUnit().lookupDeclaredProperty(propertyName);

			// check that property has been declared
			if (declaredTypeName == null)
				throw new CompileException(getFirstLineNumber(), "Property "
						+ propertyName + " has not been declared");

			ASTAblExpression property = propertyTable.get(propertyName);

			// check that the type of the property value matches the type of the
			// declared property
			if (!declaredTypeName.equals(property.getType(behaviorScope)))
				throw new CompileException(getFirstLineNumber(), "The type of "
						+ property.dumpTokens()
						+ " does not match the declared type of property "
						+ propertyName);

			String propertyValueAsObject = property.getVariableReferenceConstantOrLiteralAsObject(behaviorScope);
			// fixme: create a macro
			createPropTableBlock.addToSequence(new CodeStringDescriptor(
					"__$propertyTable.setProperty(\"" + propertyName + "\", "
							+ propertyValueAsObject + ");"));
		}
		return createPropTableBlock;
	}

	protected void compileSuccessTest() throws CompileException {
		AblParseNode n;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through children nodes looking for a success test.
			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTTESTEXPRESSION) {
				// The child is a test expression; it must be a success test.
				((ASTTestExpression) n).compileToJava();
				CodeBlockDescriptor successTestSensorFactory = ((ASTTestExpression) n)
						.compileSensorActivationFactory();

				if (successTestSensorFactory != null) {
					CodeBlockDescriptor factorySwitchCase = new CodeBlockDescriptor(
							"case " + stepID + ": {", "}");
					factorySwitchCase.addToBlockBody(successTestSensorFactory);
					ASTBehaviorUnit.getBehaviorUnit().writeSuccessTestSensorActivation(factorySwitchCase, this);
				}
				// fixme: This break means that only one success test is
				// compiled for a step.
				break;
			}
		}
	}

	void setProperty(String name, ASTAblExpression value) throws CompileException {
		ASTAblExpression oldProp = propertyTable.get(name);
		if (oldProp != null)
			throw new CompileException("Duplicate named step property: " + name);
		propertyTable.put(name, value);
	}

	protected boolean hasProperties() {
		return !propertyTable.isEmpty();
	}

	protected void initStep(int stepType) {
		final ASTBehaviorDefinition behaviorParent = (ASTBehaviorDefinition) jjtGetParent();
		final ASTBehaviorUnit behaviorUnitNode = ASTBehaviorUnit.getBehaviorUnit();

		// Create the stepName
		stepName = behaviorParent.getUniqueName() + "Step" + behaviorParent.incrementStepCount();

		// fixme: to support step sharing, modify getUniqueStepID to take the
		// step as an argument and set stepID on the
		// argument, returning true if a new step ID was created, false if an
		// old step ID was referenced.
		switch (stepType) {
		case Step.GOAL:
		case Step.PRIMITIVE:
		case Step.MENTAL:
		case Step.WAIT:
		case Step.FAIL:
		case Step.SUCCEED:
		case Step.MODIFY:
			stepID = behaviorUnitNode.getUniqueStepID();
			break;
		default:
			stepID = behaviorUnitNode.getSpecialStepID(stepType);
		}

		if (!effectOnly && !teamEffectOnly)
			behaviorParent.incrementNonEffectOnlySteps();

	}

	abstract void compileToJava() throws CompileException;

	// Returns true if the step has any modifiers
	protected boolean hasModifiers() {
		if (hasSuccessTest || priority != Short.MIN_VALUE
				|| priorityModifier != 0 || persistent
				|| persistentWhenSucceeds || persistentWhenFails || effectOnly
				|| teamEffectOnly || ignoreFailure || post || hasProperties())
			return true;
		else
			return false;
	}
}
