package abl.compiler;

import java.util.*;

/**
 * Abstract parse node for ABL steps which take arguments (namely, primitive
 * steps and subgoal steps.)
 */
abstract class ArgumentStep extends GenericStep {

	/** List of step arguments. Only used by primitive and subgoal steps. */
	protected List<ASTAblExpression> argList = new ArrayList<ASTAblExpression>(50);

	ArgumentStep(int id) {
		super(id);
	}

	ArgumentStep(AblParser p, int id) {
		super(p, id);
	}
	
	/** Constructs an ArgumentStep whose step modifiers are a copy of the step modifiers of GenericStep g. */
	ArgumentStep(int id, GenericStep g) {
		super(id, g);
	}

	/* accessor for adding arguments to primitive and subgoal
	   steps. Tokens are stored so as to keep token type
	   information. */
	void addArgument(ASTAblExpression arg) {
		argList.add(arg);
	}

	// Returns the name of the static field containing the conflict array for the
	// ExecutableStep. Returns null if the ExecutableStep has no
	// conflicts.
	protected String compileConflictArray(String executableStepName) {
		String[] conflictSet =
			ASTBehaviorUnit.getBehaviorUnit().lookupConflict(
				executableStepName);
		if (conflictSet != null)
			return ASTBehaviorUnit.getBehaviorUnit().addConflictSet(
				conflictSet);
		else
			return null;
	}

	/** Modify the variable arguments to refer to the appropriate
	   scope. Convert constant args into their typed
	   representations.*/
	protected List<AblArgument> processArgs() throws CompileException
    {
		final List<AblArgument> processedArgs = new ArrayList<AblArgument>(argList.size());
		final Iterator<ASTAblExpression> iter = argList.iterator();
		ASTAblExpression arg;
		String argType;
		while (iter.hasNext()) {
			arg = iter.next();
			if (arg.isIdentifier()) {
				// Store the dereferenced name in the ASTJavaName.

				ASTJavaName javaName = arg.getRef();
				ASTBehaviorDefinition behaviorParent =
					((ASTBehaviorDefinition) jjtGetParent());
				try {
					String scopeReference =
						javaName.getVariableReference(behaviorParent);
					argType = javaName.lookupFieldType(behaviorParent);
					if (argType.equals("java.lang.String"))
						argType = "String";
					// Treat String like a primitive type
					processedArgs.add(new AblArgument(scopeReference, argType, true));
				} catch (ScopeException e) {
					/* The argument is in neither global nor local
					   scope. It is either a constant declared in an
					   interface or an error. */
					Object value = javaName.getConstantValue();
					String constantType = value.getClass().getName();
					if (constantType.equals("java.lang.Integer"))
						processedArgs.add(
							new AblArgument((Integer) value, "int"));
					else if (constantType.equals("java.lang.Float"))
						processedArgs.add(
							new AblArgument((Float) value, "float"));
					else if (constantType.equals("java.lang.Double"))
						processedArgs.add(new AblArgument((Double) value, "double")); 
					else if (constantType.equals("java.lang.Character"))
						processedArgs.add(
							new AblArgument((Character) value, "char"));
					else if (constantType.equals("java.lang.String"))
						processedArgs.add(
							new AblArgument((String) value, "String"));
					else if (constantType.equals("java.lang.Boolean"))
						processedArgs.add(
							new AblArgument((Boolean) value, "boolean"));
					// fixme: make non-primitive arguments work
					/* else 
					   processedArgs.add(new AblArgument(value, constantType)); */
					else
						throw new CompileException(
							"Constant "
								+ javaName.dumpTokens()
								+ " of type "
								+ constantType
								+ " referenced in primitive step "
								+ dumpTokens(1, numberOfTokens())
								+ " has an unrecognized type.");
				}
			} else {
				// The AblExpression is a literal

				Object value = arg.getLiteralValue();
				String literalType = null;
				if (value == null)
					processedArgs.add(new AblArgument(null, "null"));
				else if (value != null) {
					literalType = value.getClass().getName();

					if (literalType.equals("java.lang.Integer"))
						processedArgs.add(
							new AblArgument((Integer) value, "int"));
					else if (literalType.equals("java.lang.Float"))
						processedArgs.add(
							new AblArgument((Float) value, "float"));
					else if (literalType.equals("java.lang.Double"))
						processedArgs.add(
							new AblArgument((Double) value, "double"));
					else if (literalType.equals("java.lang.Character"))
						processedArgs.add(
							new AblArgument((Character) value, "char"));
					else if (literalType.equals("java.lang.String"))
						processedArgs.add(
							new AblArgument((String) value, "String"));
					else if (literalType.equals("java.lang.Boolean"))
						processedArgs.add(
							new AblArgument((Boolean) value, "boolean"));
					else
						throw new CompileException(
							"Literal "
								+ arg.dumpTokens()
								+ " of type "
								+ literalType
								+ " referenced in primitive step "
								+ dumpTokens(1, numberOfTokens())
								+ " has an unrecognized type.");
				}
			}
		}
		return processedArgs;
	}

    public List<ASTAblExpression> getArgList() {
        return argList;
    }

}
