
package abl.compiler;

import java.util.*;
import jd.*;
// import debug.Assert;

class AblScopeMaintainer extends AblParseNode {

    protected Hashtable<String, Object[]> variableDecl = new Hashtable<String, Object[]>();

    protected List<FieldDescriptor> fieldDescriptors = new ArrayList<FieldDescriptor>();

    protected AblScopeMaintainer scopeParent = null;

    protected String scopeName;

    private int scopeType = NO_SCOPE;

    // Set to true if the behavior is an anonymous block and should inherit variables from the parent
    boolean isNestedScope = false;
    
    // Scope types. 
    final static int NO_SCOPE = -1;
    final static int GLOBAL_SCOPE = 0;
    final static int BEHAVIOR_SCOPE = 1;
    final static int METHOD_SCOPE = 2;

    AblScopeMaintainer(int id) {
        super(id);
    }

    AblScopeMaintainer(AblParser p, int id) {
        super(p, id);
    }

    // set accessor for scope name. 
    void setScopeName(String scopeNameToSet) {
        scopeName = scopeNameToSet;
    }

    // set accessor for the scope parent. 
    void setScopeParent(AblScopeMaintainer scopeParentToSet) {
        scopeParent = scopeParentToSet;
    }

    void setScopeType(int scopeTypeToSet) {
        if ((scopeTypeToSet == GLOBAL_SCOPE) || (scopeTypeToSet == BEHAVIOR_SCOPE) || (scopeTypeToSet == METHOD_SCOPE))
            scopeType = scopeTypeToSet;
        else
            throw new CompileError("Attempt to set illegal scope type " + scopeTypeToSet);

        if (scopeTypeToSet == BEHAVIOR_SCOPE)
            // If declaring a behavior scope, set the scope parent to the global scope. 
            scopeParent = (AblScopeMaintainer) jjtGetParent();
    }

    int getScopeType() {
        return scopeType;
    }

    // The enclosing scope of a AblScopeMaintainer is itself. 
    AblScopeMaintainer getEnclosingScope() {
        return this;
    }

    void addVariableDeclaration(FieldDescriptor field) {
        String[] fieldNames;

        int fieldIndex = fieldDescriptors.size(); // 0-based field index
        fieldDescriptors.add(field);
        fieldNames = field.getFieldNames();
        for (int j = 0; j < fieldNames.length; j++) {
            if (variableDecl.get(fieldNames[j]) != null)
                // fixme: should this be CompileException or CompileError?
                throw new CompileException("Variable " + fieldNames[j] + " multiply defined in scope " + scopeName);

            Object[] variableEntry = { field, new Integer(fieldIndex)};
            variableDecl.put(fieldNames[j], variableEntry);
        }
    }

    /* If the variable is declared within this scope or a parent
       scope, returns the type (as a string), otherwise returns
       null. */
    String lookupVariableType(String variableName) {
        if (variableDecl.containsKey(variableName)) {
            /* The variable was declared in this scope; return the
               type. */
            FieldDescriptor field = (FieldDescriptor) ((Object[]) variableDecl.get(variableName))[0];
            return field.fieldType;
        } else if (scopeParent != null)
            /* The variable was not declared in this scope and there
               is a parent scope; return the result of looking up the
               variable in the parent scope. */
            return scopeParent.lookupVariableType(variableName);
        else
            /* The variable was not declared in this scope and their
               is no scope parent; return null. */
            return null;
    }

    /* If the variable is declared within this scope or a parent
       scope, returns the scope type (AblScopeMaintainer.global or
       AblScopeMaintainer.behavior) of the scope within which the
       variable is declared. If the variable is not declared, returns
       AblScopeMaintainer.noScope). */
    int lookupVariableScope(String variableName) {
        if (variableDecl.containsKey(variableName))
            /* The variable was declared in this scope, return the
               scope type. */
            return scopeType;
        else if (scopeParent != null)
            /* The variable was not declared in this scope and there
               is a parent scope; return the result of looking up the
               variable in the parent scope. */
            return scopeParent.lookupVariableScope(variableName);
        else
            /* The variable was not declared in this scope and their
               is no scope parent; return AblScopeMaintainer.noScope. */
            return AblScopeMaintainer.NO_SCOPE;
    }

    // Given a variable name, returns the field descriptor describing the variable.
    // Returns null if variableName has not been defined in scope (or recursively in parent scopes).
    FieldDescriptor lookupVariable(String variableName) {
        if (variableDecl.containsKey(variableName)) {
            // The variable was declared in this scope, return the field descriptor.
            return (FieldDescriptor) ((Object[]) variableDecl.get(variableName))[0];
        } else if (scopeParent != null)
            /* The variable was not declared in this scope and there
               is a parent scope; return the result of looking up the
               variable in the parent scope. */
            return scopeParent.lookupVariable(variableName);
        else
            // The variable was not declared in this scope and there is no scope parent; return null.
            return null;
    }

    /* Returns a list of FieldDescriptors for the variables declared in this scope */
    List<FieldDescriptor> getDeclaredVariables() {
        return fieldDescriptors;
    }

    // Performs no rewriting of variable references appearing in the initializers
    protected void addChildVariableDeclarations() throws CompileException {
        FieldDescriptor[] fields;

        AblParseNode n;
        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through all the child nodes, looking for variable declarations.

            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTABLVARIABLEDECL) {
                // If the child node is a variable declaration, process it
            	
            	// fixme: remove when variable declarations handled within anonymous behaviors. 
            	if (isNestedScope) {
                	throw new CompileException(n.getFirstLineNumber(), "Attempt to declare a variable within an anonymous behavior. Currently this is not allowed. Future versions of ABL will lift this restriction");
                }
                fields = ((ASTAblVariableDecl) n).getFieldDescriptors();
				for (int x = 0; x < fields.length; x++)
					addVariableDeclaration(fields[x]);
            }
        }
    }

    /** Rewrites variables references appearing in the initializers as appropriate scope references */
    protected void addChildVariableDeclarationsWithReferenceRewrite() throws ScopeException, CompileException {
        FieldDescriptor[] fields;

        AblParseNode n;
        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through all the child nodes, looking for variable declarations.

            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTABLVARIABLEDECL) {
                // If the child node is a variable declaration, process it

            	// fixme: remove when variable declarations handled within anonymous behaviors. 
                if (isNestedScope) {
                	throw new CompileException(n.getFirstLineNumber(), "Attempt to declare a variable within an anonymous behavior. Currently this is not allowed. Future versions of ABL will lift this restriction");
                }

                fields = ((ASTAblVariableDecl) n).getFieldDescriptors(this);
				for (int x = 0; x < fields.length; x++)
					addVariableDeclaration(fields[x]);
            }
        }
    }

	/** Returns a list of field descriptors for the explicitly declared variables declared in this scope. If any of the variable declarations include an 
	 * initializer, variable references in the initializer are appropriately rewritten. */ 
	List getExplicitlyDeclaredVariableFieldDescriptors() throws ScopeException {
		final List<FieldDescriptor> fieldDescriptors = new ArrayList<FieldDescriptor>();
		
		AblParseNode n;
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through all the child nodes, looking for variable declarations.

			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTABLVARIABLEDECL) {
				// If the child node is a variable declaration, process it

				final FieldDescriptor[] fields = ((ASTAblVariableDecl) n).getFieldDescriptors(this);
				for (int x = 0; x < fields.length; x++)
					fieldDescriptors.add(fields[x]);
			}
		}
		return fieldDescriptors;
	}
	
    void initializeScope() {
        fieldDescriptors.clear();
        variableDecl.clear();
    }

    int getBehaviorFrameIndex(String varName) {
        assert scopeType != GLOBAL_SCOPE;

        if (scopeType != BEHAVIOR_SCOPE)
            return scopeParent.getBehaviorFrameIndex(varName);
        else if (!variableDecl.containsKey(varName))
            return scopeParent.getBehaviorFrameIndex(varName);
        else
            return ((Integer) ((Object[]) variableDecl.get(varName))[1]).intValue();
    }

    int getBehaviorFrameDepth(String varName, int depthSoFar) {
        assert scopeType != GLOBAL_SCOPE;
        
        if (scopeType != BEHAVIOR_SCOPE)
            return scopeParent.getBehaviorFrameDepth(varName, depthSoFar);
        else if (!variableDecl.containsKey(varName))
        {
            assert(isNestedScope);
            return scopeParent.getBehaviorFrameDepth(varName, depthSoFar + 1);
        }
        else
            return depthSoFar;
    }
}
