/* ABL parse nodes. */

package abl.compiler;

import java.util.*;
import wm.WorkingMemory;

public abstract class AblParseNode extends SimpleNode implements AblParserConstants, AblParserTreeConstants {
    public Token firstToken, lastToken;
    // Stores first and last tokens within the scope of this node.

    public AblParseNode (int id) {
	super(id);
    }
    
    public AblParseNode (AblParser p, int id) {
	super(p, id);
    }

    /* Public get accessors for getting the first and last source line
       numbers and columns subsumed by this node. */
    public int getFirstLineNumber() {
	return firstToken.beginLine;
    }

    public int getLastLineNumber() {
	return lastToken.endLine;
    }

    public int getFirstColumn() {
	return firstToken.beginColumn;
    }

    public int getLastColumn() {
	return lastToken.endColumn;
    }

    // Dumps the tokens, inserting a space between every token. 
    public String prettyPrintTokens() {
	StringBuffer image = new StringBuffer(); 
	Token currentToken;
	for(currentToken = firstToken; currentToken != lastToken; currentToken = currentToken.next) {
	    if ((!currentToken.next.image.equals(";")) &&
		(!currentToken.next.image.equals(".")) &&
		(!currentToken.next.image.equals("(")) &&
		(!currentToken.next.image.equals(")")) &&
		(!currentToken.image.equals(".")) && 
		(!currentToken.image.equals("(")))
		image.append(currentToken.image + " ");
	    else
		image.append(currentToken.image);
	}
	image.append(currentToken.image); // Add the last token to the image to return. 
	return image.toString();
    }

    // Dumps the tokens matched while this node was open. 
    public String dumpTokens() {
	StringBuffer image = new StringBuffer(); 
	Token currentToken;
	for(currentToken = firstToken; currentToken != lastToken; currentToken = currentToken.next) {
	    image.append(currentToken.image);
	}
	image.append(currentToken.image); // Add the last token to the image to return. 
	return image.toString();
    }

    // Returns the number of tokens matched by this parse node. 
    public int numberOfTokens() {
	Token currentToken;
	int tokenCount = 0;

	for(currentToken = firstToken; currentToken != lastToken; currentToken = currentToken.next) {
	    tokenCount++;
	}
	return ++tokenCount;
    }

    // Dumps the matched tokens in the range first to last. The
    // returned string contains the token images starting with token
    // first and extending to token last - 1. The token index starts at 0.
    public String dumpTokens(int first, int last) {
	StringBuffer image = new StringBuffer();
	Token currentToken;
	int tokenCounter = 0;

	assert last <= numberOfTokens();
	assert last >= 1;
	assert first >= 0;
	assert first < last;

	// Skip over tokens until you get to first.
	for(currentToken = firstToken; tokenCounter != first; currentToken = currentToken.next) {
	    tokenCounter++;
	}

	/* Store images of tokens between index first and last. Leave
           loop if we hit lastToken before hitting index last. */
	while(tokenCounter < last) {
	    image.append(currentToken.image);
	    currentToken = currentToken.next;
	    tokenCounter++;
	}

	return image.toString();
    }

    /* Nodes which compile to java classes need to uppercase the first
       character of the name associated with the node. This utility
       procedure provides this functionality. */
    public static String uppercaseFirstCharacter(String s) {
	StringBuffer tempStringBuffer = new StringBuffer(s);
	tempStringBuffer.setCharAt(0, Character.toUpperCase(tempStringBuffer.charAt(0)));
	return tempStringBuffer.toString();
    }

    /* Strips the leading and tailing quotes from strings of the form
       "\"<string>\"". The token images in
       AblParserConstants.tokenImage have this form and thus need to
       be stripped. */
    protected static String stripQuotes(String s) {
	return s.substring(1, s.length() - 1);
    }

    /* Given the string representation of a type, returns true if the
       type is primitive, false otherwise. */
    public static boolean primitiveType(String type) {
	if ((type.equals("int")) ||
	    (type.equals("float")) ||
	    (type.equals("char")) ||
	    (type.equals("boolean")) ||
	    (type.equals("long")) ||
	    (type.equals("short")) ||
	    (type.equals("byte")) ||
	    (type.equals("null")) ||
	    (type.equals("double")))
	    return true;
	else 
	    return false;
    }

    // Utility function available to all AblParseNodes. Given a class
    // name, returns the class object representing the type of that
    // name.
    protected Class getClass(String className) throws ClassNotFoundException
    {
	try {
	    if (className.indexOf(".") != -1) 
		// Already a full class name - try to look it up directly
		return Class.forName(className);
	    else 
		// A short class name - look in the behaving entity package.
		return Class.forName(ASTBehaviorUnit.getBehaviorUnit().behavingEntityPackage + "." + className);
	} catch (ClassNotFoundException e1) {
	    // Class wasn't found; look for the class in the user import packages. 
	    Iterator packageIter = ASTBehaviorUnit.getBehaviorUnit().getUserImportPackages();
	    while(packageIter.hasNext()) {
		try {
		    String packageName = (String)packageIter.next();
		    return Class.forName(packageName + "." + className);
		} catch (ClassNotFoundException e2) {
		    // Class wasn't found in this import package. Ignore exception and keep searching. 
		}
	    }
	    // Class wasn't found; look for class in the user import classes.
	    Iterator classIter = ASTBehaviorUnit.getBehaviorUnit().getUserImportClasses();
	    while(classIter.hasNext()) {
		String importedClassName = (String)classIter.next();
		if (className.equals(importedClassName.substring(importedClassName.lastIndexOf('.') + 1, importedClassName.length())))
		    return Class.forName(importedClassName);
	    }

	    // Searched through all the user import packages. Throw a ClassNotFoundException. 
	    throw new ClassNotFoundException(className);
	}
    }

    /* Utility function available to all AblParseNodes. Given a class
       name and field name, returns the Class object representing the
       type of the field name. */
    protected Class getFieldType(String className, String fieldName) 
	throws ClassNotFoundException, NoSuchFieldException 
    {
	return getClass(className).getField(fieldName).getType();
    }

    // Utility function available to all AblParseNodes. Given a wme
    // class name and a field name, returns the type of the field (the
    // type of the get accessor).
    protected Class getWMEFieldType(String wmeClassName, String fieldName)
	throws ClassNotFoundException, NoSuchMethodException
    {
	Class wmeClass;
	// fixme: isReflectionWME should move out of WorkingMemory since WorkingMemory is more general than abl.
	if (WorkingMemory.isReflectionWME(wmeClassName)) { 
	    // fixme: change to abl.runtime
	    wmeClass = Class.forName("abl.runtime." + wmeClassName);
	}
	else {
	    wmeClass = getClass(wmeClassName);
	} 
	return wmeClass.getMethod("get" + uppercaseFirstCharacter(fieldName), (Class[])null).getReturnType();
    }

    // getEnclosingBehaviorScope() is only defined on parse nodes
    // which can appear in a subtree rooted at an
    // ASTBehaviorDefinition. This definition of
    // getEnclosingBehaviorScope() catches any such calls and throws
    // an error.
    AblScopeMaintainer getEnclosingBehaviorScope() {
	throw new CompileError("getEnclosingBehaviorScope() called on a parse node which never appears in a subtree rooted at an ASTBehaviorDefinition");
    }

    // Returns a string reference to a field defined on the concrete behaving entity class. 
    protected String getBehavingEntityField(String fieldName)
    {
	return ASTBehaviorUnit.getBehaviorUnit().getBehavingEntityClass() + "." + fieldName;
    }
}



