/* The class ABL is the entry point for interpreting ABL programs. */

package abl.compiler;

import abl.compiler.astchecks.SanityCheckVisitor;
import abl.compiler.astchecks.TypeCheckVisitor;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.*;

public class Abl implements AblDebuggerConstants {

    private static List<String> sourceFiles = new ArrayList<String>(); // List of ABL source files to compile.
    /** The directory to which to write compiled Java classes.  */
    private static String objectDirectoryName = null;
    public static File objectDirectory;
    static byte debugLevel = NO_DEBUG; // 0 = no debug, 1 = debug level 1, 2 = debug level 2
    private static boolean dumpTree = false;
    static boolean currentLineOfExpansion = true;
    static boolean bMetaEnabled = true; // True if metaEnabled switch is used.
    static boolean noCodeGeneration = false; // True if no code is to be generated
    static boolean reflectionOptimization = true; // True if reflection optimization is turned off

    // fixme: added to experiment with compiling to one file
    // static PrintStream stepStream;
    // static PrintStream behaviorStream;

    private static void processCommandlineArgs(String args[]) {
        for(int i = 0; i < args.length; i++) {
            if (args[i].startsWith("-d")) {
                if (++i >= args.length) {
                    usage();
                    System.exit(1);
                }
                else
                    objectDirectoryName = args[i];
            }
            else if (args[i].startsWith("-X")) {
                if (++i >= args.length) {
                    usage();
                    System.exit(1);
                }
                else {
                    if (args[i].equals("dumpTree"))
                        dumpTree = true;
                    else if (args[i].equals("noCurrentLineOfExpansion"))
                        currentLineOfExpansion = false;
                    else if (args[i].equals("noReflectOpt"))
                        reflectionOptimization = false;
                    else {
                        usage();
                        System.exit(1);
                    }
                }
            }
            else if (args[i].startsWith("-g")) {
                try {
                    // See if there is a number following the -g
                    final byte level = Byte.parseByte(args[i+1]);
                    if (level != 1 && level != 2)
                    System.out.println("The debug level following -g should be either 1 or 2");
                    else {
                    debugLevel = level;
                    }
                }
                catch (NumberFormatException e) {
                    // The argument following g is not a number. Assume a max debug level (currently 2)
                    debugLevel = GUI_DEBUGGER;
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    // The -g is the last thing on the command line. Assume max debug level (currently 2)
                    debugLevel = GUI_DEBUGGER;
                }
            }
            else if (args[i].equals("-metaEnabled")) {
                bMetaEnabled = true;
            }
            else if (args[i].equals("-metaDisabled")) {
                bMetaEnabled = false;
            }
            else if (args[i].equals("-nc")) {
                // fixme: this should be moved into a -X command
                noCodeGeneration = true;
            }
            else {
                sourceFiles.add(args[i]);
            }
        }
        if (objectDirectoryName != null)
            objectDirectory = new File(objectDirectoryName);
        else
            objectDirectory = new File (".");
        }

        private static void usage() {
        System.out.println("abl [-d <object directory>] [-X dumpTree|noCurrentLineOfExpansion|noReflectOpt] [-g] [1|2] [-metaEnabled|metaDisabled] [-nc] <ABL source file>+");
    }

    public static void main(String args[]) throws CompileException {

        int errorcount = 0;
        long startTime = System.currentTimeMillis();

        processCommandlineArgs(args);

        // If called with no args, print help message.
        if (args.length == 0)
            usage();
        else {
            ListIterator sourceFileIter = sourceFiles.listIterator();
            while(sourceFileIter.hasNext()) {
                String sourceFile = (String)sourceFileIter.next();
                System.out.println("Reading from file " + sourceFile + " . . .");

                try {
                    compileFile(sourceFile);
                } catch (java.io.FileNotFoundException e) {
                    System.err.println("ABL Parser: File " + sourceFile + " not found.");
                    errorcount += 1;
                } catch (ParseException e) {
                    System.err.println(e.getMessage());
                    System.err.println("ABL Parser: Encountered errors during parse.");
                    errorcount += 1;
                } catch (CompileException e) {
                    System.err.println(e.getMessage());
                    System.err.println("ABL: Encountered errors during compilation, exiting.");
                    errorcount += 1;
                    System.exit(errorcount);
                }
            }
        }

        System.out.println("ABL compiler took: " + (System.currentTimeMillis() - startTime)/1000.0 + " seconds");
        if (errorcount != 0)
        {
            System.err.println("Encountered " + errorcount + " errors total");
            System.exit(errorcount);
        }
    }

    /**
     * Compile one source file, throwing the expected exceptions as they are encountered.
     * @param sourceFile the source file
     */
    public static void compileFile(final String sourceFile) throws FileNotFoundException,
                                                          ParseException, CompileException {
        AblParser parser = new AblParser(new java.io.FileInputStream(sourceFile)); // Create a new parser
        ASTBehaviorUnit n = parser.BehaviorUnit(); // Parse and return the root of the parse tree
        System.out.println(sourceFile + " parsed successfully.");
        if (dumpTree)
            n.dump("");

        n = ASTWalker.walk(new SanityCheckVisitor(n), n);
        n = ASTWalker.walk(new TypeCheckVisitor(n, sourceFile), n);

        System.out.println("Generating code . . .");
        n.compileToJava();
    }
}
