package abl.compiler;

import fun.FoldL;
import fun.Fun2;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

/**
 * Implements two generic AST walking methods parameterised by
 * {@link abl.compiler.ASTVisitor}s: breadth-first and depth-first walking. One
 * typically wants to do a breadth-first walk, processing the tree from the top
 * down (see {@link ASTWalker#walk(ASTVisitor, ASTBehaviorUnit)}). But
 * bottom-up processing (DFS) is also available (see
 * {@link ASTWalker#dwalk(ASTVisitor, ASTBehaviorUnit)}).
 * <p>
 * The most interesting method in this class is
 * {@link ASTWalker#walk(ASTVisitor, Node)}.
 */
public class ASTWalker {

    private ASTWalker() {
    }

    /**
     * Same as {@link ASTWalker#walk(ASTVisitor, ASTBehaviorUnit)}, but uses a
     * depth-first walk.
     * @param visitor the visitor
     * @param unit the behavior unit
     * @return the new, possibly-transformed behavior unit
     * @throws CompileError if the compiler has an error
     * @throws CompileException if the visiter throws the exception
     * @see ASTWalker#dwalk(ASTVisitor, Node)
     */
    public static ASTBehaviorUnit dwalk(final ASTVisitor visitor, final ASTBehaviorUnit unit)
        throws CompileError, CompileException {
        return (ASTBehaviorUnit) dwalk(visitor, (Node) unit);
    }

    /**
     * Same as {@link ASTWalker#walk(ASTVisitor, Node)}, but uses a
     * depth-first walk.
     * @param visitor the visitor to produce new nodes from the nodes in the
     *        AST.
     * @param ast the abstract syntax tree
     * @return whatever the visitor to the node returns
     * @throws CompileError if an error occurs in the machinery of the walking
     *         mechanism
     */
    public static Node dwalk(final ASTVisitor visitor, final Node ast) throws CompileError,
        CompileException {
        // For each of the children, walk it, and set its result as the
        // corresponding child of ast.
        for (int i = ast.jjtGetNumChildren() - 1; i >= 0; i--) {
            ast.jjtAddChild(dwalk(visitor, ast.jjtGetChild(i)), i);
        }

        return visitNode(visitor, ast);
    }

    /**
     * Walk a behavior unit. This is the common case. Including this method
     * avoids casting the return of {@link ASTWalker#dwalk(ASTVisitor, Node)}.
     * @param visitor the visitor
     * @param ast the tree
     * @return the possibly-transformed tree
     * @throws CompileError if there is an error during walking
     * @throws CompileException if the visitor throws this exception
     */
    public static ASTBehaviorUnit walk(final ASTVisitor visitor, final ASTBehaviorUnit ast)
        throws CompileError, CompileException {
        return (ASTBehaviorUnit) walk(visitor, (Node) ast);
    }

    /**
     * Walk the AST rooted at <code>ast</code>. This method does implicitly
     * side-effect the AST by setting the child nodes of the AST to the values
     * that the visitor returns. This does a breadth-first traversal of the AST.
     * <p>
     * If the visitor throws a {@link CompileException} on any node, traversal
     * is immediately halted & the same exception is re-thrown.
     * @param visitor the visitor
     * @param ast the ast
     * @return a possibly-transformed tree
     * @throws CompileError if there is an error during walking
     * @throws CompileException if the visitor throws this exception
     */
    public static Node walk(final ASTVisitor visitor, final Node ast) throws CompileError,
        CompileException {
        final Node newNode = visitNode(visitor, ast);

        // For each of the children, walk it, and set its result as the
        // corresponding child of ast.
        for (int i = newNode.jjtGetNumChildren() - 1; i >= 0; i--) {
            newNode.jjtAddChild(dwalk(visitor, newNode.jjtGetChild(i)), i);
        }

        return newNode;
    }

    /**
     * With a collection of visitors, walk the tree once.
     * @param visitors the visitors
     * @param ast the root node to walk
     * @return a possibly-transformed tree
     */
    public static Node walk(final List<ASTVisitor> visitors, final Node ast)
        throws CompileError, CompileException {

        final Node astPrime;
        try {
            // Call each visitor in succession on the current node, `ast', threading the results by
            // folding.
            astPrime = FoldL.foldl(new VisitNodeL(), ast, visitors.iterator());
        } catch (RuntimeException e) {
            if (e.getCause() instanceof CompileException) {
                throw (CompileException) e.getCause();
            } else {
                throw e;
            }
        }

        for (int i = astPrime.jjtGetNumChildren() - 1; i >= 0; i--) {
            astPrime.jjtAddChild(walk(visitors, astPrime.jjtGetChild(i)), i);
        }

        return astPrime;
    }

    /**
     * Left-associated visitNode wrapper. For use with {@link FoldL}.
     */
    static class VisitNodeL extends Fun2<Node, ASTVisitor, Node> {
        @Override
        public Node apply(Node curNode, ASTVisitor visitor) {
            try {
                return visitNode(visitor, curNode);
            } catch (CompileException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Right-associated visitNode wrapper. For use with {@link fun.FoldR}.
     *
     */
    static class VisitNodeR extends Fun2<ASTVisitor, Node, Node> {
        @Override
        public Node apply(ASTVisitor visitor, Node curNode) {
            try {
                return visitNode(visitor, curNode);
            } catch (CompileException e) {
                throw new RuntimeException(e);
            }
        }
    }



    private static Node visitNode(final ASTVisitor visitor, final Node n)
        throws CompileError, CompileException {

        final Method visit = getVisitMethod(visitor, n.getClass());
        // Process current node & get back a new node.
        try {
            Node ret = (Node) visit.invoke(visitor, n);
            return ret;
        } catch (final InvocationTargetException e) {
            // If the visitor throws a CompileException or an Error, throw the
            // original exception.
            if (e.getCause() instanceof CompileException) {
                throw (CompileException) e.getCause();
            } else if (e.getCause() instanceof Error) {
                throw (Error) e.getCause();
            } else {
                throw new RuntimeException(e.getCause());
            }
        } catch (final Exception e) {
            throw new CompileError("Compiler error", e);
        }
    }

    private static Method getVisitMethod(final ASTVisitor visitor, final Class astClass) {
        final Method visit; // the method we invoke on visitor
        try {
            visit = visitor.getClass().getMethod("visit", new Class[]{astClass});
        } catch (final Exception e) {
            throw new CompileError("Compiler error", e);
        }

        return visit;
    }
}
