/* Parse node for a WME test in a preconditions, context conditions or
   success tests. The child ASTWMEFieldTests represent the individual
   field tests. */

package abl.compiler;

import jd.*;
import java.util.*;
import macro.*;
import wm.WorkingMemory; // fixme: move isReflectionWME() somewhere else

public class ASTWMETest extends TestNode implements AblParserTreeConstants {

    // Currently ignoring negated. Add this after getting basic tests working. 
    boolean negated; // True if the WME test is negated, false otherwise. 

    String wmeClassName; // Name of the WME class to be tested. 

    /* If the whole WME is being grabbed in a test, the name of the
       variable this WME will be assigned to. */
    ASTJavaName wmeAssignmentVariable = null;

    /* Get the list of WMEs with a given WME class name that currently
       active in working memory. */
    private final static SimpleMacro wmeTestGetWMEListNoMemory = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListNoMemoryMacroString);

    private final static SimpleMacro wmeTestGetWMEListMemory = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListMemoryMacroString);

    private final static SimpleMacro wmeTestGetWMEListNoMemorySignatureOpt = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListNoMemorySignatureOptMacroString);

    private final static SimpleMacro wmeTestGetWMEListMemorySignatureOpt = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListMemorySignatureOptMacroString);

    private final static SimpleMacro wmeTestGetWMEListNoMemoryPropertyOpt = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListNoMemoryPropertyOptMacroString);

    private final static SimpleMacro wmeTestGetWMEListMemoryPropertyOpt = new SimpleMacro(MacroDefinitions.wmeTestGetWMEListMemoryPropertyOptMacroString);

    /* Iterate over the final WMEs in the list. */
    private final static SimpleMacro wmeTestWhile = new SimpleMacro(MacroDefinitions.wmeTestWhileMacroString);

    /* Get the next WME in the list with WME variable assignment. */
    private final static SimpleMacro wmeTestWhileNextAssign = new SimpleMacro(MacroDefinitions.wmeTestWhileNextAssignMacroString);

    /* Get the next WME in the list with no WME variable assignment. */
    private final static SimpleMacro wmeTestWhileNextNoAssign = new SimpleMacro(MacroDefinitions.wmeTestWhileNextNoAssignMacroString);

    // memory in which to satisfy this wme test
    String memoryName = null;

    ASTWMETest(int id) {
        super(id);
    }

    ASTWMETest(AblParser p, int id) {
        super(p, id);
    }

    // Returns the AblScopeMaintainer for the enclosing behavior. 
    AblScopeMaintainer getEnclosingBehaviorScope() {
        return ((ASTTestExpression) jjtGetParent()).getEnclosingBehaviorScope();
    }

    // Returns the tightest enclosing scope, either the method scope
    // (for success tests) or behavior scope.
    AblScopeMaintainer getEnclosingScope() {
        return ((ASTTestExpression) jjtGetParent()).getEnclosingScope();
    }

    // Set the memory in which to satisfy this wme test
    void setMemoryName(String memoryName) {
        this.memoryName = memoryName;
    }

	final Set getExplicitlyDeclaredVariableReferences() {
		final Set declaredVariableReferences = new HashSet();
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through all the child nodes looking for WMEFieldTests 

			final AblParseNode node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMEFIELDTEST) {
				// A WMEFieldTest was found 
				final ASTWMEFieldTest fieldTest = (ASTWMEFieldTest) node;
				final FieldDescriptor variable = fieldTest.getExplicitlyDeclaredVariableReference();
				if (variable != null) {
					declaredVariableReferences.add(variable);
				}
					
			}
		}
		
		return declaredVariableReferences;		
	}

    /* Iterates through the WME field tests looking for variable
    references. Returns a hash set of all variable references which
    were not defined in the enclosing behavior or global
    scope. Presumably the caller will use this information to define
    the variable in the appropriate scope (behavior scope for context
    conditions and preconditions, local scope for success tests). This
    should be called before compiling the WME test. */
    Set<FieldDescriptor> preprocessVariableRefs() throws CompileException {
        final Set<FieldDescriptor> newFieldDefs = new HashSet<FieldDescriptor>(); // Use HashSet for efficient no-duplicates collection. 
        AblParseNode node;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through all the child nodes looking for WME field tests. 

            node = (AblParseNode) jjtGetChild(i);
            if (node.id == JJTWMEFIELDTEST) {
                // A WME field test was found. 

                FieldDescriptor f = ((ASTWMEFieldTest) node).preprocessVariableRef();
                if (f != null)
                    newFieldDefs.add(f);
            }
        }
        if (wmeAssignmentVariable != null) {
            if (!wmeAssignmentVariable.isVariableReference(getEnclosingBehaviorScope())) {
                if (wmeAssignmentVariable.isSimpleRef()) {
                    // Variable has not been declared in behavior or
                    // global scope. Add it to the newFieldDefs. 
                    FieldDescriptor newVar = new FieldDescriptor();
                    newVar.addFieldName(wmeAssignmentVariable.getName());
                    newVar.fieldType = wmeClassName; // WME var has the type of the WME class being tested. 
                    newFieldDefs.add(newVar);
                } else
                    // not a simple reference - insufficient information to implicitly declare variable
                    throw new CompileException("Line " + getFirstLineNumber() + "Can not implicitly declare WME assignment variable (non-simple reference): " + prettyPrintTokens());
            } else if (!wmeAssignmentVariable.lookupVariableType(getEnclosingBehaviorScope()).equals(wmeClassName))
                throw new CompileException("Line " + getFirstLineNumber() + "Type error in WME assignment: " + prettyPrintTokens());
        }

        return newFieldDefs;
    }

    Set<String> getBoundVariables() {
        final Set<String> boundVariables = new HashSet<String>(); // Use HashSet for efficient no-duplicates collection. 
        AblParseNode node;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through all the child nodes looking for WME field tests. 

            node = (AblParseNode) jjtGetChild(i);
            if (node.id == JJTWMEFIELDTEST) {
                /* A WME field test was found. */

                final String v = ((ASTWMEFieldTest) node).getBoundVariable();
                if (v != null)
                    boundVariables.add(v);
            }
        }
        if (wmeAssignmentVariable != null)
            boundVariables.add(wmeAssignmentVariable.getName());

        return boundVariables;
    }

    int getWMECounter() {
        return ((ASTTestExpression) jjtGetParent()).getTestCounter();
    }

    protected List<JavaCodeDescriptor> compileFieldTests() throws CompileException {
        AblParseNode node;
        List<JavaCodeDescriptor> compiledFieldTests = new ArrayList<JavaCodeDescriptor>();

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through all the child nodes, looking for WME field tests. 

            node = (AblParseNode) jjtGetChild(i);
            if (node.id == JJTWMEFIELDTEST) {
                // A WME field test was found. 
                compiledFieldTests.add(((ASTWMEFieldTest) node).compileToJava());
            }
        }
        if (compiledFieldTests.size() == 0) { // No field tests - return the null test "true
            compiledFieldTests.add(new CodeStringDescriptor("true"));
        }
        return compiledFieldTests;
    }

    int getTestType() {
        return ((ASTTestExpression) jjtGetParent()).getTestType();
    }
 
    /* Compiles a WME test into the block of code implementing that
       test. */
    JavaCodeDescriptor compileToJava() throws CompileException {

        // fixme: create a new macro that looks up reflection WMEs by signature or by user property if appropriate. 
        // Look up by user property if any field test tests a property != null, == <value != null), or any other test 
        // operator (e.g. >, <, etc.). This means that there needs to be an easy way to tell whether something is a
        // user property or not. If there are no property tests, see if there is a signature test. 

        int wmeCount = ((ASTTestExpression) jjtGetParent()).getTestCounter();
        // macroArgs1 used a couple of places below
        String[] macroArgs1 = { Integer.toString(wmeCount), wmeClassName };
        CodeBlockDescriptor wmeTest = new CodeBlockDescriptor();

        String signatureTest = null; // set to the signature if a signature optimization is applied
        ASTWMEFieldTest propertyTest = null; // set to the field test parse node if a property optimization is applied

        try {
        	
        	if (WorkingMemory.isReflectionWME(wmeClassName) && Abl.reflectionOptimization) {
        		// A reflection WME - may apply an optimization
        		
        		// Look through the field tests to see if there are any references to signature or user properties 
        		for (int i = 0; i < jjtGetNumChildren(); i++) {
        			// Loop through all the child nodes, looking for WME field tests. 
        			
        			AblParseNode node = (AblParseNode) jjtGetChild(i);
        			if (node.id == JJTWMEFIELDTEST) {
        				/* A WME field test was found. */
        				
        				ASTWMEFieldTest fieldTest = (ASTWMEFieldTest) node;
        				if (fieldTest.isProperty()) {
        					// The field test tests a property - a property optimization may be appropriate
        					String operand = resolveOperandInFieldTest(fieldTest);
        					if ((operand.equals("null") && fieldTest.testOp == NE) || !operand.equals("null")) {
        						// Conditions have been met for optimizing on a user property
        						propertyTest = fieldTest;
        						break; // break the loop
        					}
        				} else if (fieldTest.wmeFieldName.equals("signature") && fieldTest.testOp == EQ) {
        					// conditions have been met for optimizing on signature
        					signatureTest = resolveOperandInFieldTest(fieldTest);
        					break; // break the loop
        				}
        			}
        		}
        		if (signatureTest != null) {
        			if (memoryName == null) {
        				// default to BehavingEntity's memory
        				String[] sigMacroArgs = { Integer.toString(wmeCount), wmeClassName, signatureTest };
        				wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListNoMemorySignatureOpt.expand(sigMacroArgs)));
        			} else {
        				// a specific memory name was specified
        				String[] sigMacroArgs = { Integer.toString(wmeCount), wmeClassName, memoryName, signatureTest };
        				wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListMemorySignatureOpt.expand(sigMacroArgs)));
        			}
        		} else if (propertyTest != null) {
        			if (memoryName == null) {
        				// default to BehavingEntity's memory
        				String[] propertyMacroArgs = { Integer.toString(wmeCount), wmeClassName, propertyTest.wmeFieldName };
        				wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListNoMemoryPropertyOpt.expand(propertyMacroArgs)));
        			} else {
        				// a specific memory name was specified
        				String[] propertyMacroArgs = { Integer.toString(wmeCount), wmeClassName, memoryName, propertyTest.wmeFieldName };
        				wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListMemoryPropertyOpt.expand(propertyMacroArgs)));
        				
        			}
        		}
        	} // end if Abl.reflectionOptimization
        	
        	if (signatureTest == null && propertyTest == null) {
        		// If no reflection WME optimization applied, do normal WME lookup
        		
        		if (memoryName == null) {
        			// default to BehavingEntity's memory
        			wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListNoMemory.expand(macroArgs1)));
        		} else {
        			// a specific memory name was specified
        			String[] specfiedMemoryMacroArgs = { Integer.toString(wmeCount), wmeClassName, memoryName };
        			wmeTest.addToBlockBody(new CodeStringDescriptor(wmeTestGetWMEListMemory.expand(specfiedMemoryMacroArgs)));
        		}
        		
        	}
        	
        	// Compile the rest of the WME test
        	CodeBlockDescriptor testLoop = new CodeBlockDescriptor();
        	String[] macroArgs2 = { Integer.toString(wmeCount)};
        	testLoop.setBlockHeader(wmeTestWhile.expand(macroArgs2));
        	testLoop.setBlockFooter("}");
        	if (wmeAssignmentVariable != null) {
        		String[] macroArgs3 = { Integer.toString(wmeCount), wmeClassName, wmeAssignmentVariable.getVariableReference(getEnclosingScope())};
        		testLoop.addToBlockBody(new CodeStringDescriptor(wmeTestWhileNextAssign.expand(macroArgs3)));
        	} else {
        		testLoop.addToBlockBody(new CodeStringDescriptor(wmeTestWhileNextNoAssign.expand(macroArgs1)));
        	}
        	
        	CodeBlockDescriptor ifTest = new CodeBlockDescriptor("if (", ")");
        	
        	List wmeFieldTests = compileFieldTests();
        	Iterator wmeFieldTestIter = wmeFieldTests.iterator();
        	if (wmeFieldTestIter.hasNext())
        		// Add the first WME field test
        		ifTest.addToBlockBody((CodeStringDescriptor) wmeFieldTestIter.next());
        	
        	// Now add the rest of the tests, prepending &&
        	while (wmeFieldTestIter.hasNext()) {
        		ifTest.addToBlockBody(new CodeStringDescriptor("&&"));
        		ifTest.addToBlockBody((CodeStringDescriptor) wmeFieldTestIter.next());
        	}
        	
        	CodeBlockDescriptor ifBody = new CodeBlockDescriptor("{", "}");
        	
        	if (!negated) {
        		// Test is not negated. 
        		if (((ASTTestExpression) jjtGetParent()).hasNextTest())
        			ifBody.addToBlockBody(((ASTTestExpression) jjtGetParent()).compileNextTest());
        		else {
        			// No more WME tests 
        			
        			// If the test is a precondition, add code to copy variables in _variableTable. 
        			if (((ASTTestExpression) jjtGetParent()).getTestType() == ASTTestExpression.PRECONDITION)
        				((ASTTestExpression) jjtGetParent()).compileOutgoingVariables(ifBody);
        			
        			// Return true 
        			ifBody.addToBlockBody(new CodeStringDescriptor("return true;"));
        		}
        	} else
        		// Test is negated - return false;
        		ifBody.addToBlockBody(new CodeStringDescriptor("return false;"));
        	
        	testLoop.addToBlockBody(ifTest);
        	testLoop.addToBlockBody(ifBody);
        	wmeTest.addToBlockBody(testLoop);
        	if (negated) {
        		// If the test is negated, falling through the loop is success
        		if (((ASTTestExpression) jjtGetParent()).hasNextTest())
        			wmeTest.addToBlockBody(((ASTTestExpression) jjtGetParent()).compileNextTest());
        		else {
        			// No more WME tests 
        			
        			// If the test is a precondition, add code to copy variables in _variableTable. 
        			if (((ASTTestExpression) jjtGetParent()).getTestType() == ASTTestExpression.PRECONDITION)
        				((ASTTestExpression) jjtGetParent()).compileOutgoingVariables(wmeTest);
        			
        			// Return true 
        			wmeTest.addToBlockBody(new CodeStringDescriptor("return true;"));
        		}
        	}
        	return wmeTest;
        } catch (SimpleMacroException e) {
        	throw new CompileError("Error in ASTWMETest.compileToJava()", e);
        } catch (ClassNotFoundException e) {
        	throw new CompileError("Error in ASTWMETest.compileToJava()", e);
        } catch (NoSuchFieldException e) {
        	throw new CompileError("Error in ASTWMETest.compileToJava()", e);
        }
    }

    // Returns a field representation of references appearing in the condition 
    String resolveOperandInFieldTest(ASTWMEFieldTest fieldTest) throws CompileException {
        AblScopeMaintainer behaviorScope = getEnclosingBehaviorScope();

        if ((getTestType() == ASTTestExpression.PRECONDITION)) {
            // Behavior scope variables are reference as locals 

            if (fieldTest.wmeTestOperand.isIdentifier()) {
                // could be a global reference or a constant
                if (fieldTest.wmeTestOperand.getRef().lookupVariableScope(behaviorScope) == AblScopeMaintainer.GLOBAL_SCOPE)
                    return fieldTest.wmeTestOperand.getRef().getVariableReference(behaviorScope);
                else if (fieldTest.wmeTestOperand.getRef().isConstant(behaviorScope))
                    // The operand is a declared constant.
                    return fieldTest.wmeTestOperand.getRef().getConstantString();
                else
                    // The identifier is neither a global reference nor a constant
                    return fieldTest.wmeTestOperand.getFullNameOrLiteral();
            } else
                // The operand is not an identifier
                return fieldTest.wmeTestOperand.getFullNameOrLiteral();
        } else {
            // All scope references are resolved
            AblScopeMaintainer localScope = getEnclosingScope();
            return fieldTest.wmeTestOperand.getVariableReferenceConstantOrLiteral(localScope);
        }

    }

}
