/* Parse tree node for WME declarations */
package abl.compiler;

import jd.*;
import macro.*;
import java.util.Hashtable;

public class ASTWMEDecl extends AblParseNode {
    public String wmeClassName; // Name of the WME class

    public String wmeExtendsClass = null;

    // Field assignment used in WME constructor. 
    private final static SimpleMacro  wmeConstructorAssignment = 
	new SimpleMacro(MacroDefinitions.wmeConstructorAssignmentMacroString);
    
    // get accessor body
    private final static SimpleMacro wmeGetAccessorBody =
	new SimpleMacro(MacroDefinitions.wmeGetAccessorBodyMacroString);

    // set accessor body
    private final static SimpleMacro wmeSetAccessorBody =
	new SimpleMacro(MacroDefinitions.wmeSetAccessorBodyMacroString);

    public ASTWMEDecl(int id) {
	super(id);
    }
    
    public ASTWMEDecl(AblParser p, int id) {
	super(p, id);
    }

    // The default constructor for WMEs has an empty body.
    // Currently not used, but leaving it in in case I need it later.
    private MethodDescriptor defineDefaultConstructor()
    {
	MethodDescriptor constructor = new MethodDescriptor();
	
	constructor.methodName = uppercaseFirstCharacter(wmeClassName);
	constructor.addModifier("public");
	
	return constructor;
    } 

    private MethodDescriptor defineInitConstructor(FieldDescriptor[] fields) {
	MethodDescriptor constructor = new MethodDescriptor();
	
	try {
	    constructor.methodName = uppercaseFirstCharacter(wmeClassName);
	    constructor.addModifier("public");
	
	    for(int i = 0; i < fields.length; i++) {
		constructor.addArgument(fields[i].fieldType, fields[i].getFieldNames()[0]);
	    }
	    StringBuffer methodBodyBuf = new StringBuffer();
	    for(int i = 0; i < fields.length; i++) {
		String fieldName = fields[i].getFieldNames()[0];
		String[] macroArgs = {fieldName};
		
		methodBodyBuf.append(wmeConstructorAssignment.expand(macroArgs));
	    }
	    constructor.addToBlockBody(new CodeStringDescriptor(methodBodyBuf.toString()));
	    
	    return constructor;
	}
	catch (SimpleMacroException e) {
	    throw new CompileError("Error expanding macro in ASTWMEDecl.defineInitConstructor(): " + e.getMessage());
	}
    } 

    // Returns a Hashtable with field names as keys and field types as values.
    public Hashtable<String, String> getFieldTypeTable()
    {
		Hashtable<String, String> typeTable = new Hashtable<String, String>();
		int numberOfChildren = jjtGetNumChildren();
		for(int i = 0; i < numberOfChildren; i++) {
			FieldDescriptor[] tempFields = ((ASTAblVariableDecl)jjtGetChild(i)).getFieldDescriptors();
			for(int j = 0; j < tempFields.length; j++) {
				String[] fieldNames = tempFields[j].getFieldNames();
				typeTable.put(fieldNames[0], tempFields[j].fieldType);
			}
		}
		return typeTable;
    }

    // Given a FieldDescriptor, returns a MethodDescriptor for the get accessor
    private MethodDescriptor defineGetAccessor(FieldDescriptor field) {
	MethodDescriptor getAccessor = new MethodDescriptor();
	getAccessor.methodName = "get" + uppercaseFirstCharacter(field.getFieldNames()[0]);
	getAccessor.addModifier("public");
	getAccessor.addModifier("synchronized");
	getAccessor.addModifier(field.fieldType);
	String[] macroArgs = {field.getFieldNames()[0]};
	try {
	    getAccessor.addToBlockBody(new CodeStringDescriptor(wmeGetAccessorBody.expand(macroArgs)));
	} catch (SimpleMacroException e) {
	    throw new CompileError("Error expanding macro: " + e.getMessage());
	}
	return getAccessor;
    }
	
    // Given a FieldDescriptor, returns a MethodDescriptor for the set accessor
    private MethodDescriptor defineSetAccessor(FieldDescriptor field) {
	MethodDescriptor setAccessor = new MethodDescriptor();
	setAccessor.methodName = "set" + uppercaseFirstCharacter(field.getFieldNames()[0]);
	setAccessor.addModifier("public");
	setAccessor.addModifier("synchronized");
	setAccessor.addModifier("void");
	setAccessor.addArgument(field.fieldType, field.getFieldNames()[0]);
	String[] macroArgs = {field.getFieldNames()[0]};
	try {
	    setAccessor.addToBlockBody(new CodeStringDescriptor(wmeSetAccessorBody.expand(macroArgs)));
	} catch (SimpleMacroException e) {
	    throw new CompileError("Error expanding macro: " + e.getMessage());
	}
	return setAccessor;
    } 
    
    public JavaCodeDescriptor compileToJava() throws CompileException {

	ClassDescriptor newWME = new ClassDescriptor();

	// WME declared in behaving entity package. 
	newWME.packageName = ASTBehaviorUnit.getBehaviorUnit().behavingEntityPackage;

	// Imports wm.WME
	newWME.addPackageImport("wm.WME");
	
	// Declare the wme as a class extending WME
	newWME.className = uppercaseFirstCharacter(wmeClassName);
	if (wmeExtendsClass == null)
	    newWME.extendsClass = "WME";
	else
	    newWME.extendsClass = newWME.packageName + "." + wmeExtendsClass;
	
	// The wme is accessible outside of its package. Necessary for the debugger. 
	newWME.addClassModifier("public");

	// Declare the wme fields.
	int numberOfChildren = jjtGetNumChildren();
	for(int i = 0; i < numberOfChildren; i++) {
	    FieldDescriptor[] tempFields; //  = new FieldDescriptor();
	    tempFields = ((ASTAblVariableDecl)jjtGetChild(i)).getFieldDescriptors();
		for (int x = 0; x < tempFields.length; x++) {
			tempFields[x].addFieldModifier("private");
			newWME.addField(tempFields[x]);
		}
	}

	// Declare the constructors
	FieldDescriptor[] fieldNames = newWME.getFields();
	newWME.addMethod(defineInitConstructor(fieldNames));
	newWME.addMethod(defineDefaultConstructor());

	// Define the get accessors
	for(int i = 0; i < fieldNames.length; i++)
	    newWME.addMethod(defineGetAccessor(fieldNames[i]));

	// Define the set accessors
	for(int i = 0; i < fieldNames.length; i++)
	    newWME.addMethod(defineSetAccessor(fieldNames[i]));

	return newWME;
    }
    
}



