package abl.compiler;

import java.util.*;
import jd.*;
import macro.*;

public class ASTTestExpression extends AblScopeMaintainer implements AblParserTreeConstants {

	/* Constants defining the types of conditions. */
    private static final int MIN_CONDITION_VAL = 0;
	static final int PRECONDITION = 0;
	static final int CONTEXT_CONDITION = 1;
	static final int SUCCESS_TEST = 2;
    static final int ENTRY_CONDITION = 3;
    static final int SUCCESS_CONDITION = 4;
    static final int STATE_CONDITION = 5;
    static final int REINFORCEMENT_TEST = 6;
    private static final int MAX_CONDITION_VAL = 6;

	/* The type of test expression. */
	private int testType = -1;

	/* Counter and array used during compilation of wme tests. */
	private int testCounter;
	private LinkedList testList = new LinkedList();
	private ListIterator testIter;

	private ASTAblExpression signalValue = null;

	private final static SimpleMacro sensorActivation = new SimpleMacro(MacroDefinitions.sensorActivationMacroString);

	ASTTestExpression(int id) {
		super(id);
	}

	ASTTestExpression(AblParser p, int id) {
		super(p, id);
	}

	/* Package set accessor for testType. */
	void setTestType(int testTypeToSet) {
		if ((testTypeToSet < MIN_CONDITION_VAL) || (testTypeToSet > MAX_CONDITION_VAL))
			throw new CompileError("Illegal test type in call to ASTTestExpression.setTestType(int)");
		testType = testTypeToSet;
	}

	/* Package get accessor for test type. */
	int getTestType() {
		if (testType == -1)
			throw new CompileError("Attempt to get the test type of an ASTTestExpression node before the test type was set.");
		return testType;
	}

	void setSignalValue(ASTAblExpression value)
	{
		if (testType != REINFORCEMENT_TEST)
			throw new CompileError("Attempt to set a signal value of an ASTTestExpression node without test type of REINFORCEMENT_TEST.");
		signalValue = value;
	}

	ASTAblExpression getSignalValue()
	{
		return signalValue;
	}
	
	/* Returns the parent (ASTBehaviorDefinition) as a AblScopeMaintainer. */
	AblScopeMaintainer getEnclosingBehaviorScope() {
		return ((AblParseNode) jjtGetParent()).getEnclosingBehaviorScope();
	}

	/* Iterates through the WME tests calling preprocessVariableRefs()
	   on each test. Returns a hash set of all the variables
	   referenced in all the tests which are not currently declared in
	   any variable scope. */
	private Set<FieldDescriptor> preprocessVariableRefs() throws CompileException {
		final Set<FieldDescriptor> variables = new HashSet<FieldDescriptor>();
		// Use a HashSet for efficient implementation of a no-duplicates collection.
		AblParseNode node;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through the children looking for WME tests. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMETEST) {
				// A WME test was found.
				variables.addAll(((ASTWMETest) node).preprocessVariableRefs());
			}
		}
		return variables;
	}

	private void compilePreconditionArguments() {
		final Object[] behaviorArgs = ((ASTBehaviorDefinition) jjtGetParent()).getFormalArguments();

		for (int i = 0; i < behaviorArgs.length; i++) {
			/* Iterate through the behavior args, initializing each one. */
			final FieldDescriptor variableDecl = new FieldDescriptor();
			final MethodArgDescriptor arg = (MethodArgDescriptor) behaviorArgs[i];
			variableDecl.fieldType = arg.argType;
			variableDecl.addFieldName(arg.argName);
			if (primitiveType(arg.argType)) {
				if (arg.argType.equals("int"))
					variableDecl.initializer = "((Integer)__$args[" + i + "]).intValue()";
				else if (arg.argType.equals("float"))
					variableDecl.initializer = "((Float)__$args[" + i + "]).floatValue()";
				else if (arg.argType.equals("char"))
					variableDecl.initializer = "((Char)__$args[" + i + "]).charValue()";
				else if (arg.argType.equals("boolean"))
					variableDecl.initializer = "((Boolean)__$args[" + i + "]).booleanValue()";
				else if (arg.argType.equals("long"))
					variableDecl.initializer = "((Long)__$args[" + i + "]).longValue()";
				else if (arg.argType.equals("short"))
					variableDecl.initializer = "((Short)__$args[" + i + "]).shortValue()";
				else if (arg.argType.equals("byte"))
					variableDecl.initializer = "((Byte)__$args[" + i + "]).byteValue()";
				else if (arg.argType.equals("double"))
					variableDecl.initializer = "((Double)__$args[" + i + "]).doubleValue()";
			} else
				variableDecl.initializer = "(" + arg.argType + ")__$args[" + i + "]";
			addVariableDeclaration(variableDecl);
		}
	}

	private void compileExplicitBehaviorScopeVariables() { 
		final Set variables = getBehaviorScopeVariableReferences(); // Set of field descriptors for explicitly declared variables. 
		final Iterator iter = variables.iterator();
		while(iter.hasNext()) {
			// System.out.println((FieldDescriptor)iter.next());
			addVariableDeclaration((FieldDescriptor)iter.next());
		}
	}
	
	private void compileExplicitStateScopeVariables() { 
		final Set variables = getStateScopeVariableReferences(); // Set of field descriptors for explicitly declared variables. 
		final Iterator iter = variables.iterator();
		while(iter.hasNext()) {
			// System.out.println((FieldDescriptor)iter.next());
			addVariableDeclaration((FieldDescriptor)iter.next());
		}
	}
	
	private NestableCodeDescriptor compileTests() throws CompileException {
		testCounter = 0;
		AblParseNode n;
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through child nodes, accumulating all WME test. */

			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTWMETEST || n.id == JJTCONDITIONALEXPRESSION)
				testList.add((TestNode) n);
		}

		testIter = testList.listIterator();
		if (testIter.hasNext()) {
			return (NestableCodeDescriptor) ((TestNode) testIter.next()).compileToJava();
		} else
			// If the test expression is empty, return null
			return null;
	}

	/* Returns true if there are any more tests to compile, false otherwise. */
	boolean hasNextTest() {
		return testIter.hasNext();
	}

	/* Co-routine with ASTWMETest.compileToJava() and
	   ConditionalExpresion.compileToJava(). Calls compileToJava() on
	   the next test. */
	NestableCodeDescriptor compileNextTest() throws CompileException {
		++testCounter;
		if (!testIter.hasNext())
			// If there are no more tests; throw an error
			throw new CompileError("Call to ASTTestExpression.compileNextTest() with no more tests.");

		return (NestableCodeDescriptor) ((TestNode) testIter.next()).compileToJava();
	}

	/* package get accessor for testCounter. Called by
	   ASTWMETest.compileToJava() to grab the counter as a macro
	   argument. */
	int getTestCounter() {
		return testCounter;
	}

	// Returns the set of WME classes referenced in this test. 
	private HashSet getReferencedWMEs() {
		HashSet referencedWMEs = new HashSet();
		AblParseNode node;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through the children looking for WME tests. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMETEST) {
				// A WME test was found.
				referencedWMEs.add(((ASTWMETest) node).wmeClassName);
			}
		}
		return referencedWMEs;
	}

	Set<String> getBoundVariables() {
		Set<String> boundVariables = new HashSet<String>();
		AblParseNode node;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through the children looking for WME tests. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMETEST || node.id == JJTCONDITIONALEXPRESSION) {
				// A WME test was found.
                Set<String> varSet = ((TestNode) node).getBoundVariables();
				boundVariables.addAll(varSet);
			}
		}
		return boundVariables;
	}
	
	// Returns a Set of FieldDescriptors for the explicitly declared behavior scope variables referenced in this test expression. Only used for preconditions. 
	// fixme: should this be behavior and global scope?
	private Set getBehaviorScopeVariableReferences() {
		final Set explicitlyDeclaredVariables = new HashSet();
		AblParseNode node;
		for(int i = 0; i < jjtGetNumChildren(); i++) {
			node = (AblParseNode) jjtGetChild(i);
			assert (node.id == JJTWMETEST || node.id == JJTCONDITIONALEXPRESSION); 
			explicitlyDeclaredVariables.addAll(((TestNode)node).getExplicitlyDeclaredVariableReferences());
		}
		
		// Remove formal arguments from the explicitly declared variables
		final Object[] behaviorArgs = ((ASTBehaviorDefinition) jjtGetParent()).getFormalArguments();
		for(int i = 0; i < behaviorArgs.length; i++) {
			final MethodArgDescriptor arg = (MethodArgDescriptor)behaviorArgs[i];
			final FieldDescriptor field = new FieldDescriptor();
			field.fieldType = arg.argType;
			field.addFieldName(arg.argName);
			explicitlyDeclaredVariables.remove(field);
		}
		return explicitlyDeclaredVariables;
	}

	// Returns a Set of FieldDescriptors for the explicitly declared behavior scope variables referenced in this test expression. Only used for state conditions. 
	// fixme: should this be behavior and global scope?
	private Set getStateScopeVariableReferences() {
		final Set explicitlyDeclaredVariables = new HashSet();
		AblParseNode node;
		for(int i = 0; i < jjtGetNumChildren(); i++) {
			node = (AblParseNode) jjtGetChild(i);
			assert (node.id == JJTWMETEST || node.id == JJTCONDITIONALEXPRESSION); 
			explicitlyDeclaredVariables.addAll(((TestNode)node).getExplicitlyDeclaredVariableReferences());
		}
		
		// Remove formal arguments from the explicitly declared variables
		final Object[] behaviorArgs = ((ASTBehaviorDefinition) jjtGetParent().jjtGetParent()).getFormalArguments();
		for(int i = 0; i < behaviorArgs.length; i++) {
			final MethodArgDescriptor arg = (MethodArgDescriptor)behaviorArgs[i];
			final FieldDescriptor field = new FieldDescriptor();
			field.fieldType = arg.argType;
			field.addFieldName(arg.argName);
			explicitlyDeclaredVariables.remove(field);
		}
		return explicitlyDeclaredVariables;
	}
	
	/* Adds CodeStringDescriptors to the passed CodeBlockDescriptor
	   making appropriate outgoing variable assignments to
	   __$variableTable. */
	void compileOutgoingVariables(CodeBlockDescriptor desc) {
		final Set boundVariables = getBoundVariables();
		final Iterator iter = boundVariables.iterator();
		while (iter.hasNext()) {
			String variable = (String) iter.next();
			String variableType = lookupVariableType(variable);
			if (primitiveType(variableType)) {
				if (variableType.equals("int")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Integer(" + variable + "));"));
				}
				else if (variableType.equals("float")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Float(" + variable + "));"));
				}
				else if (variableType.equals("char")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Char(" + variable + "));"));
				}
				else if (variableType.equals("boolean")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Boolean(" + variable + "));"));
				}
				else if (variableType.equals("long")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Long(" + variable + "));"));
				}
				else if (variableType.equals("short")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Short(" + variable + "));"));
				}
				else if (variableType.equals("byte")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Byte(" + variable + "));"));
				}
				else if (variableType.equals("double")) {
					desc.addToBlockBody(
						new CodeStringDescriptor(
							"__$variableTable.put(" + "\"" + variable + "\"" + ", new Double(" + variable + "));"));
				}
			} else
				desc.addToBlockBody(
					new CodeStringDescriptor(
						"__$variableTable.put(" + "\"" + variable + "\"" + ", new ObjectWrapper(" + variable + "));"));

		}
	}

	// Returns a code block descriptor containing the sensor activations for this 
	CodeBlockDescriptor compileSensorActivationFactory() {
		HashSet referencedWMEs = getReferencedWMEs();
		if (referencedWMEs == null) {
			return null;
		} else {
			Iterator wmeIter = referencedWMEs.iterator();
			List activations = new Vector(referencedWMEs.size());
			try {
				while (wmeIter.hasNext()) {
					String sensorClassName =
						ASTBehaviorUnit.getBehaviorUnit().lookupRegisteredWME((String) wmeIter.next());
					if (sensorClassName != null) {
						String[] macroArgs = { sensorClassName };

						/* Currently the sensorActivation macro adds null
						   args to the SensorActivation instance. Need to
						   do something here to handle variable
						   arguments. Sensor registration needs to keep
						   track of arguments. */
						activations.add(sensorActivation.expand(macroArgs));
					}
				}
			} catch (SimpleMacroException e) {
				throw new CompileError("Error compiling getSensorActivations", e);
			}
			if (activations.size() != 0) {
				Iterator activationIter = activations.iterator();
				CodeBlockDescriptor arrayInit =
					new CodeBlockDescriptor("SensorActivation[] __$activationArray = {", "};");
				while (activationIter.hasNext()) {
					String init = (String) activationIter.next();
					if (activationIter.hasNext())
						// There is another initialization after this one; add a comma.
						arrayInit.addToBlockBody(new CodeStringDescriptor(init + ","));
					else
						// There isn't anther initialization after this one; don't add a comma.
						arrayInit.addToBlockBody(new CodeStringDescriptor(init));
				}
				CodeBlockDescriptor sensorActivationBlock = new CodeBlockDescriptor();
				sensorActivationBlock.addToBlockBody(arrayInit);
				sensorActivationBlock.addToBlockBody(new CodeStringDescriptor("return __$activationArray;"));
				return sensorActivationBlock;
			} else
				// No registered WMEs referenced in test
				return null;
		}
	}

	// Returns true if all the child WME tests are negated, otherwise returns false. 
	protected boolean allNegatedTests() {
		AblParseNode node;
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through the children looking for WME tests. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMETEST) {
				// A WME test was found.
				if (!((ASTWMETest) node).negated)
					return false;
			} else if (node.id == JJTCONDITIONALEXPRESSION)
				return false;
		}
		return true;
	}

	// Compiles the test into a switch case implementing the test. Writes the switch case to the appropriate class. 
	void compileToJava() throws CompileException {
		if (testType == -1)
			// Test type hasn't been set yet; throw an error.
			throw new CompileError("ASTTestExpression.compileToJava() called before setting the test type.");

		setScopeType(AblScopeMaintainer.METHOD_SCOPE);
		setScopeParent(getEnclosingBehaviorScope());

		CodeBlockDescriptor test;
		if (   testType == PRECONDITION
			|| testType == CONTEXT_CONDITION
			|| testType == ENTRY_CONDITION
			|| testType == SUCCESS_CONDITION)
		{
			final ASTBehaviorDefinition beh = (ASTBehaviorDefinition) jjtGetParent();
			final int behaviorID = beh.getID();
			test = new CodeBlockDescriptor("case " + behaviorID + ": {", "}");
			test.addToBlockBody(new CodeStringDescriptor("// " + beh.getUniqueName()));
		}
		else if (testType == REINFORCEMENT_TEST)
		{
			test = new CodeBlockDescriptor("{//begin reinforcement signal test: " + signalValue.dumpTokens(),
			                               "}//end reinforcement signal test: " + signalValue.dumpTokens());
		} 
		else if (testType == STATE_CONDITION)
		{
			final ASTBehaviorDefinition beh = (ASTBehaviorDefinition) jjtGetParent().jjtGetParent();
			final int behaviorID = beh.getID();
		//	test = new CodeBlockDescriptor("case " + behaviorID + ": {", "}");
		//	test.addToBlockBody(new CodeStringDescriptor("// " + beh.getUniqueName()));
			test = new CodeBlockDescriptor("class stateTest"+behaviorID+"{//begin state condition", "}}//end state condition");
		} 
		else {
			// A success test 
			final GenericStep step = (GenericStep) jjtGetParent();
			final int stepID = step.getStepID();
			test = new CodeBlockDescriptor("case " + stepID + ": {", "}");
			test.addToBlockBody(new CodeStringDescriptor("// " + step.getUniqueName()));
		}

		if (   testType == PRECONDITION
			|| testType == ENTRY_CONDITION)
		{
			/* Include variable declarations in the precondition()
			       method for every formal argument in the enclosing
			       behavior. Include code to assign each of the elements
			       of the argument to precondition (Object[]) to each of
			       these declared variables. */
			compilePreconditionArguments();
			// Include variabe declarations in the preconditions() method for every explicitly declared behavior scope 
			// variable referenced in the precondition.
			compileExplicitBehaviorScopeVariables();
		}
		else if (testType == STATE_CONDITION)
		{
			compileExplicitStateScopeVariables();
		}

		final Set variableRefs = preprocessVariableRefs();
		final Iterator variableRefsIter = variableRefs.iterator();
		while (variableRefsIter.hasNext()) {
			final FieldDescriptor newVarDecl = (FieldDescriptor) variableRefsIter.next();
			if (   (testType == PRECONDITION)
				|| (testType == SUCCESS_TEST)
				|| (testType == STATE_CONDITION))
			{
				// Add the variable declaration to the local scope for preconditions and success tests. 
				addVariableDeclaration(newVarDecl);
			}
			if (   (testType == PRECONDITION)
				|| (testType == CONTEXT_CONDITION)
				|| (testType == ENTRY_CONDITION)
				|| (testType == SUCCESS_CONDITION)
				|| (testType == STATE_CONDITION))
			{
				// For preconditions and context conditions, add the variable decls to the enclosing behavior. 
				// For state conditions, the parent is an ASTReinforcementState
				 ((AblScopeMaintainer) jjtGetParent()).addVariableDeclaration(newVarDecl);
			}
		}

		// Add the variables declared in the local scope to the method.
		final Iterator varIter = fieldDescriptors.listIterator();
		while (varIter.hasNext()) {
			test.addToBlockBody((FieldDescriptor) varIter.next());
		}

		if (testType == STATE_CONDITION) {
			test.addToBlockBody(new CodeStringDescriptor("boolean doTest(){"));
		}

		NestableCodeDescriptor testBody = compileTests();
		test.addToBlockBody(testBody);
		if (!allNegatedTests())
			// If the test expression doesn't consist of all negated tests, add a "return false" at the end. 
			// Need to make this test because the "return false" becomes an unreachable statement in the case
			// of all negated tests. The java compiler chokes on the unreachable statement. 
			test.addToBlockBody(new CodeStringDescriptor("return false;"));

		final ASTBehaviorUnit behUnit = ASTBehaviorUnit.getBehaviorUnit();
		switch (testType) {
			case PRECONDITION :
				behUnit.writePrecondition(test, (ASTBehaviorDefinition) jjtGetParent());
				break;
			case CONTEXT_CONDITION :
				behUnit.writeContextCondition(test, (ASTBehaviorDefinition) jjtGetParent());
				break;
			case SUCCESS_CONDITION :
				behUnit.writeSuccessCondition(test, (ASTBehaviorDefinition) jjtGetParent());
				break;
			case SUCCESS_TEST :
				behUnit.writeSuccessTest(test, (GenericStep) jjtGetParent());
				break;
            case ENTRY_CONDITION :
                // no special output, just side-effects
                break;
			case REINFORCEMENT_TEST :
				behUnit.writeReinforcementTest(test, (ASTBehaviorDefinition) jjtGetParent().jjtGetParent());
                break;
			case STATE_CONDITION :
				behUnit.writeStateWME(test, (ASTBehaviorDefinition) jjtGetParent().jjtGetParent());
                break;
			default:
				throw new CompileError("Invalid testType " + testType);
		}
	}
}
